#include "tags_manager_impl.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/service_session.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/evlog.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/intersect.h>

#include <util/string/cast.h>
#include <util/system/env.h>

namespace {
    bool CheckPerformGroups() {
        if (!NDrive::HasServer()) {
            return false;
        }
        const auto& settings = NDrive::GetServer().GetSettings();
        return settings.GetValue<bool>("tags_manager.check_perform_groups").GetOrElse(true);
    }
}

using TDurationGuard = TTimeGuardImpl<false, TLOG_INFO, true>;

TDriveTagsManager::TDriveTagsManager(NStorage::IDatabase::TPtr database, const TTagsManagerConfig& daConfig)
    : TDatabaseSessionConstructor(database)
    , DeviceTagPropositionsConfig(daConfig.GetDeviceTagPropositionsConfig())
{
    HistoryContext.Reset(new THistoryContext(database));
    TagsMeta.Reset(new TTagsMeta(*HistoryContext, daConfig.GetTagDescriptionsHistoryConfig()));
    TagsMeta->InitPropositions(daConfig.GetTagDescriptionsPropositionsConfig());
    UserTags.Reset(new TUserTagsManager(*TagsMeta));
    DeviceTags.Reset(new TDeviceTagsManager(*TagsMeta, daConfig.GetDeviceTagsHistoryConfig()));
    DeviceTags->GetHistoryManager().RegisterSessionBuilder(new TBillingSessionSelector(*this));
    DeviceTags->GetHistoryManager().RegisterSessionBuilder(new TServiceSessionSelector(*TagsMeta, "service"));
    DeviceTags->InitPropositions(DeviceTagPropositionsConfig);
    TraceTags.Reset(new TTraceTagsManager(*TagsMeta));
    AccountTags.Reset(new TAccountTagsManager(*TagsMeta));
}

bool TDriveTagsManager::DoStop() {
    return
        DeviceTags->Stop() &&
        UserTags->Stop() &&
        TagsMeta->Stop();
}

bool TDriveTagsManager::DoStart() {
    TDurationGuard dgg("StartComponent DriveTagsManager");
    auto threadCount = FromStringWithDefault<size_t>(GetEnv("TAGS_MANAGER_START_THREADS"), 2);
    auto threadPool = CreateThreadPool(threadCount);
    auto userTagsStarted = NThreading::Async([this] {
        TDurationGuard dg("StartComponent UserTags");
        return UserTags->Start();
    }, *threadPool);
    auto deviceTagsStarted = NThreading::Async([this] {
        TDurationGuard dg("StartComponent DeviceTags");
        return DeviceTags->Start();
    }, *threadPool);
    return
        userTagsStarted.GetValueSync() &&
        deviceTagsStarted.GetValueSync();
}

const ITagsHistoryContext& TDriveTagsManager::GetContext() const {
    return *TagsMeta;
}

const IEntityTagsManager& TDriveTagsManager::GetEntityTagManager(NEntityTagsManager::EEntityType entityType) const {
    switch (entityType) {
    case NEntityTagsManager::EEntityType::Car:
        return GetDeviceTags();
    case NEntityTagsManager::EEntityType::Area:
        throw yexception() << "unsupported";
    case NEntityTagsManager::EEntityType::Trace:
        return GetTraceTags();
    case NEntityTagsManager::EEntityType::User:
        return GetUserTags();
    case NEntityTagsManager::EEntityType::Account:
        return GetAccountTags();
    case NEntityTagsManager::EEntityType::Undefined:
        return GetDeviceTags();
    }
}

bool TDeviceTagsManager::CheckInvariants(const TString& objectId, NDrive::TEntitySession& session) const {
    if (!CheckPerformGroups()) {
        return true;
    }
    auto optionalObject = RestoreObject(objectId, session);
    if (!optionalObject) {
        session.AddErrorMessage("DeviceTagsManager::CheckInvariants", "cannot RestoreObject");
        return false;
    }
    auto performGroups = optionalObject->GetPerformGroups();
    for (auto&& [group, performers] : performGroups) {
        if (performers.size() > 1) {
            for (auto&& performer : performers) {
                session.AddErrorMessage("performers", performer);
            }
            session.SetErrorInfo("DeviceTagsManager::CheckInvariants", "multiple performers in group " + group);
            return false;
        }
    }
    return true;
}

NDrive::IObjectSnapshot::TPtr TDeviceTagsManager::BuildSnapshot(const TString& objectId, const NDrive::IServer* server) const {
    if (server) {
        return server->GetSnapshotsManager().GetSnapshotPtr(objectId);
    } else {
        return nullptr;
    }
}

TString TDeviceTagsManager::MakeTagPerformerCondition(const TDBTag& tag, const TString& userId, bool force, ui32 lockedLimit, NDrive::TEntitySession& session, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform) const {
    if (!CheckPerformGroups()) {
        return TBase::MakeTagPerformerCondition(tag, userId, force, lockedLimit, session, overrideMultiPerform);
    }
    auto condition = TStringBuilder() << "tag_id = " << session->Quote(tag.GetTagId());
    if (overrideMultiPerform.GetOrElse(GetMultiplePerformersPolicy()) == NEntityTagsManager::EMultiplePerformersPolicy::Deny) {
        if (!force) {
            condition << " AND NOT EXISTS (SELECT 1 FROM " << GetTableName()
                << " WHERE performer != " << session->Quote(userId)
                << " AND tag_id = " << session->Quote(tag.GetTagId())
                << " AND performer != '') ";
        }
    }
    if (lockedLimit) {
        condition << " AND ((SELECT COUNT(DISTINCT object_id) FROM " << GetTableName()
            << " WHERE performer = " << session->Quote(userId)
            << " AND object_id != " << session->Quote(tag.GetObjectId())
            << ") <= " << lockedLimit - 1 << ")";
    }
    return condition;
}

TMaybe<TTaggedObject> TDeviceTagsManager::GetCachedOrRestoreObject(const TString& objectId, NDrive::TEntitySession& session) const {
    auto cached = GetObject(objectId);
    if (cached) {
        return cached;
    }
    return RestoreObject(objectId, session);
}

TMaybe<TSet<TString>> TDeviceTagsManager::PrefilterObjects(const TTagsFilter& tFilter, const TSet<TString>* objectIds, TInstant reqActuality) const {
    if (!RefreshCache(reqActuality)) {
        return {};
    }
    if (!TagNameToObjects.RefreshCache(reqActuality)) {
        return {};
    }
    auto rg = MakeObjectReadGuard();
    TTagsInvIndex::TIndexReader::TPtr invIndex = TagNameToObjects.GetInvIndex();
    TMergeIterator<TIntersectIterator<TSetIterator<TString>>> itResult;
    for (auto&& i : tFilter.GetMatchConditions()) {
        TIntersectIterator<TSetIterator<TString>> it;
        for (auto&& tag : i.GetMatchTokens()) {
            if (tag.IsInObjectTags) {
                if (!objectIds && tFilter.GetMatchConditions().size() == 1 && i.GetMatchTokens().size() == 1) {
                    return invIndex->Get(TString(tag.TagName));
                }
                it.AddInclude(&invIndex->Get(TString(tag.TagName)));
            } else {
                it.AddExclude(&invIndex->Get(TString(tag.TagName)));
            }
        }
        if (objectIds) {
            it.AddInclude(objectIds);
        }
        itResult.AddIterator(std::move(it));
    }
    TSet<TString> result;
    if (!itResult.Start()) {
        return result;
    }
    do {
        result.emplace(*itResult.GetValue());
    } while (itResult.Next(nullptr));
    return result;
}

TTraceTagsManager::TOptionalTags TTraceTagsManager::RestoreTags(NDrive::TEntitySession& session, TQueryOptions&& queryOptions) const {
    if (queryOptions.GetObjectIds()) {
        queryOptions.SetSecondaryIndex("trace_tags_object_id_index");
    }
    return TBase::RestoreTags(session, std::move(queryOptions));
}

TUserTagsManager::TUserTagsManager(const ITagsHistoryContext& context)
    : TBase(context)
    , TTagCache(*this, "user_tags", TDuration::Seconds(100))
    , TableName("user_tags")
    , HistoryManager(context)
{
}

bool TUserTagsManager::RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& objects, NDrive::TEntitySession& session) const {
    bool result = TBase::RestoreObjects(objectId, objects, session);
    if (result) {
        for (auto&& [id, object] : objects) {
            Update(object);
        }
    }
    return result;
}

TAccountTagsManager::TAccountTagsManager(const ITagsHistoryContext& context)
    : TBase(context)
    , TTagCache(*this, "account_tags", TDuration::Seconds(100))
    , TableName("account_tags")
    , HistoryManager(context)
{
}

bool TAccountTagsManager::RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& objects, NDrive::TEntitySession& session) const {
    bool result = TBase::RestoreObjects(objectId, objects, session);
    if (result) {
        for (auto&& [id, object] : objects) {
            Update(object);
        }
    }
    return result;
}


IEntityTagsManager::IEntityTagsManager(const ITagsHistoryContext& context)
    : TDatabaseSessionConstructor(context.GetDatabase())
    , TagsHistoryContext(context)
{
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::AddTag(ITag::TPtr tag, const TString& userId, const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session, EUniquePolicy uniquePolicy, const TString& filter) const {
    auto optionalTags = AddTags(NContainer::Scalar(tag), userId, objectId, server, session, uniquePolicy, filter);
    if (!optionalTags) {
        return {};
    }
    if (optionalTags->size() > 1) {
        session.SetErrorInfo("EntityTagsManager::AddTag", TStringBuilder() << "incorrect tags count: " << optionalTags->size());
        return {};
    }
    for (auto&& tag : *optionalTags) {
        Y_ASSERT(tag);
    }
    return optionalTags;
}

bool IEntityTagsManager::CheckInvariants(const TString& objectId, NDrive::TEntitySession& session) const {
    Y_UNUSED(objectId);
    Y_UNUSED(session);
    return true;
}

bool IEntityTagsManager::RejectPropositions(const TSet<TString>& propositionIds, const TString& userId, NDrive::TEntitySession& session) const {
    if (!GetPropositionsPtr()) {
        session.SetErrorInfo("ConfirmPropositions", "No propositions manager", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    for (auto&& i : propositionIds) {
        if (GetPropositionsPtr()->Reject(i, userId, session) != EPropositionAcceptance::Rejected) {
            return false;
        }
    }
    return true;
}

bool IEntityTagsManager::ConfirmPropositions(const TSet<TString>& propositionIds, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!GetPropositionsPtr()) {
        session.SetErrorInfo("ConfirmPropositions", "No propositions manager", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto optionalSessions = GetPropositionsPtr()->Get(propositionIds, session);
    if (!optionalSessions) {
        return false;
    }
    auto sessions = *optionalSessions;
    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        if (it == sessions.end()) {
            WARNING_LOG << "cannot find proposition " << i << Endl;
            continue;
        }
        const EPropositionAcceptance confirmResult = GetPropositionsPtr()->Confirm(i, userId, session);
        if (confirmResult == EPropositionAcceptance::ConfirmWaiting) {
        } else if (confirmResult == EPropositionAcceptance::ReadyForCommit) {
            if (!AddTag(it->second.GetData(), userId, it->second.GetObjectId(), server, session, EUniquePolicy::Rewrite)) {
                return false;
            }
        } else {
            return false;
        }
    }
    return true;
}

TPropositionId IEntityTagsManager::ProposeTag(const TDBTag& dbTag, const TString& userId, NDrive::TEntitySession& session) const {
    if (!GetPropositionsPtr()) {
        session.SetErrorInfo("ConfirmPropositions", "No propositions manager", EDriveSessionResult::InconsistencySystem);
        return {};
    }
    auto td = TagsHistoryContext.GetTagsManager().GetDescriptionByName(dbTag->GetName());
    if (!td) {
        session.SetErrorInfo("ProposeTag", "Incorrect tag description", EDriveSessionResult::IncorrectRequest);
        return {};
    }
    const TConfirmableTag::TDescription* ctd = dynamic_cast<const TConfirmableTag::TDescription*>(td.Get());
    TObjectProposition<TDBTag> proposition(dbTag, ctd ? ctd->GetConfirmationsCount() : GetPropositionsPtr()->GetConfig().GetDefaultConfirmationsNeed());
    if (GetPropositionsPtr()->Propose(proposition, userId, session) == EPropositionAcceptance::Problems) {
        return {};
    }
    return proposition.GetPropositionId();
}

bool IEntityTagsManager::InitPerformer(const TSet<TString>& tagIds, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TVector<TDBTag> initialTags;
    if (!RestoreTags(tagIds, initialTags, session)) {
        return false;
    }
    return InitPerformer(initialTags, permissions, server, session);
}

bool IEntityTagsManager::InitPerformer(const TVector<TDBTag>& initialTagsExt, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool replaceSnapshots) const {
    TVector<TDBTag> initialTags;

    TSet<TString> tagIds;
    const TString& userId = permissions.GetUserId();
    for (auto&& i : initialTagsExt) {
        if (i->GetPerformer() == userId) {
            continue;
        }
        if (tagIds.emplace(i.GetTagId()).second) {
            initialTags.emplace_back(i);
        }
    }

    TVector<TDBTag> tags;
    {
        TMap<TString, TVector<TDBTag>> performObjects;
        if (!GetPerformObjects({permissions.GetUserId()}, performObjects, session)) {
            return false;
        }

        TSet<TString> lockedResources;
        for (auto&& i : initialTags) {
            performObjects[i.GetObjectId()].push_back(i);
        }

        if (performObjects.size() > permissions.LockedResourcesLimit()) {
            session.SetErrorInfo("init_perform", "locked resources limit reached", EDriveSessionResult::LockedResourcesLimitEnriched);
            return false;
        }

        TString lockingGroup;
        if (permissions.LockedTagsLimitExceeded(performObjects, lockingGroup)) {
            session.SetErrorInfo("init_perform", "tags limit exceeded by " + lockingGroup, EDriveSessionResult::LockedResourcesLimitEnriched);
            return false;
        }
    }

    TInstant timestamp;
    {
        TSet<TString> objects;
        for (auto&& i : initialTags) {
            objects.emplace(i.GetObjectId());
        }
        timestamp = Now();
        if (!objects.empty() && !RestoreTags(objects, {}, tags, session)) {
            return false;
        }
    }

    TMap<TString, TTaggedObject> devices;
    for (auto&& i : tags) {
        auto it = devices.find(i.GetObjectId());
        if (it == devices.end()) {
            TTaggedDevice td(i.GetObjectId(), TDBTags{}, timestamp);
            it = devices.emplace(i.GetObjectId(), std::move(td)).first;
        }
        it->second.MutableTags().push_back(i);
    }

    for (auto&& i : devices) {
        const auto performChecker = permissions.GetPerformable(i.second, tagIds, GetMultiplePerformersPolicy());
        if (performChecker == TUserPermissions::EPermormability::NoPerformableByPriorities) {
            session.SetError(NDrive::MakeError("resource_has_major_problems"));
            session.SetErrorInfo("no_permissions_perform_by_major_tag", i.first, EDriveSessionResult::ResourceHasMajorProblems);
            return false;
        }
        if (performChecker == TUserPermissions::EPermormability::NoPerformAbilities) {
            session.SetErrorInfo("no_permissions", i.first, EDriveSessionResult::NoPermissionsForPerform);
            return false;
        }
    }

    TMap<NEntityTagsManager::EMultiplePerformersPolicy, TVector<TDBTag>> forceTags;
    TMap<NEntityTagsManager::EMultiplePerformersPolicy, TVector<TDBTag>> gracefulTags;
    {
        TMap<TString, NDrive::IObjectSnapshot::TPtr> snapshots;
        for (auto&& i : initialTags) {
            const NEntityTagsManager::EMultiplePerformersPolicy mrp = i->GetMultiPerformingAbility().GetOrElse(GetMultiplePerformersPolicy());
            if (!i->OnBeforePerform(i, permissions, server, session)) {
                return false;
            }
            auto it = snapshots.find(i.GetObjectId());
            if (it == snapshots.end()) {
                it = snapshots.emplace(i.GetObjectId(), BuildSnapshot(i.GetObjectId(), server)).first;
            }
            if (i->GetObjectSnapshot() == nullptr || replaceSnapshots) {
                i->SetObjectSnapshot(it->second);
            }
            if (i->GetPerformer() == "") {
                gracefulTags[mrp].emplace_back(i);
            } else {
                forceTags[mrp].emplace_back(i);
            }
        }
    }

    for (auto&& [policy, tag] : forceTags) {
        if (!SetTagsPerformer(tag, userId, true, session, server, permissions.LockedResourcesLimit(), policy)) {
            return false;
        }
    }
    for (auto&& [policy, tag] : gracefulTags) {
        if (!SetTagsPerformer(tag, userId, false, session, server, permissions.LockedResourcesLimit(), policy)) {
            return false;
        }
    }

    for (auto&& i : initialTags) {
        if (!i->OnAfterPerform(i, permissions, server, session)) {
            return false;
        }
    }

    return true;
}

bool IEntityTagsManager::DropPerformer(const TSet<TString>& removeTagIds, const TSet<TString>& cancelTagIds, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool force) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "DropPerformer")
            ("cancel_tag_ids", NJson::ToJson(cancelTagIds))
            ("remove_tag_ids", NJson::ToJson(removeTagIds))
            ("force", force)
        );
    }
    TTimeGuard tg("DropPerformer");
    TVector<TDBTag> carTags;
    TSet<TString> tagIds(removeTagIds.begin(), removeTagIds.end());
    tagIds.insert(cancelTagIds.begin(), cancelTagIds.end());

    const TString& userId = permissions.GetUserId();
    if (!RestoreTags(tagIds, carTags, session)) {
        session.AddErrorMessage("DropPerformer", "cannot RestoreTags");
        return false;
    }
    const auto pred = [](const TDBTag& item) ->bool {
        return item->GetPerformer() == "";
    };
    carTags.erase(std::remove_if(carTags.begin(), carTags.end(), pred), carTags.end());

    TVector<TDBTag> notFinishedTags;
    TVector<TDBTag> tagsToRemove;

    TMap<TString, NDrive::IObjectSnapshot::TPtr> snapshots;

    auto history = GetEvents({}, {}, session, TQueryOptions().SetTagIds(tagIds));
    if (!history) {
        session.AddErrorMessage("DropPerformer", "cannot GetEvents");
        return false;
    }
    TMap<TString, TObjectEvents<TConstDBTag>> historyByTag;
    for (auto&& id : tagIds) {
        historyByTag.emplace(id, TObjectEvents<TConstDBTag>());
    }
    for (auto&& ev : *history) {
        historyByTag[ev.GetTagId()].emplace_back(std::move(ev));
    }
    TSet<TString> objectIds;
    for (auto&& tag : carTags) {
        objectIds.insert(tag.GetObjectId());
    }
    TTaggedObjectsSnapshot taggedObjects;
    if (!RestoreObjects(objectIds, taggedObjects, session)) {
        session.AddErrorMessage("DropPerformer", "cannot RestoreObjects");
        return false;
    }
    for (auto&& tag : carTags) {
        if (!tag) {
            continue;
        }
        auto objectTags = taggedObjects.Get(tag.GetObjectId());
        if (!objectTags) {
            session.SetErrorInfo("DropPerformer", "no object tags for " + tag.GetObjectId(), EDriveSessionResult::InternalError);
            return false;
        }
        auto historyIt = historyByTag.find(tag.GetTagId());
        if (historyIt == historyByTag.end()) {
            session.SetErrorInfo("DropPerformer", "no tag history", EDriveSessionResult::InternalError);
            return false;
        }
        if (removeTagIds.contains(tag.GetTagId())) {
            if (!permissions.CheckObjectTagAction(TTagAction::ETagAction::RemovePerform, tag, *objectTags, historyIt->second)) {
                session.SetErrorInfo("DropPerformer", "no permissions to RemovePerform tag " + tag.GetTagId());
                return false;
            }
            tagsToRemove.push_back(tag);
        } else if (cancelTagIds.contains(tag.GetTagId())) {
            if (!permissions.CheckObjectTagAction(TTagAction::ETagAction::DropPerform, tag, *objectTags, historyIt->second)) {
                session.SetErrorInfo("DropPerformer", "no permissions to DropPerform tag " + tag.GetTagId());
                return false;
            }
            notFinishedTags.push_back(tag);
        } else {
            continue;
        }
        if ((tag->GetPerformer() != userId) && !permissions.CheckObjectTagAction(TTagAction::ETagAction::ForcePerform, tag, *objectTags, historyIt->second)) {
            session.SetErrorInfo("DropPerformer", "no permissions to ForcePerform tag " + tag.GetTagId());
            return false;
        }
        if (tag->GetPerformer() && tag->GetPerformer() != userId) {
            force = true;
        }
        auto it = snapshots.find(tag.GetObjectId());
        if (it == snapshots.end()) {
            it = snapshots.emplace(tag.GetObjectId(), BuildSnapshot(tag.GetObjectId(), server)).first;
        }
        tag->SetObjectSnapshot(it->second);
    }

    for (auto&& tag : notFinishedTags) {
        if (!tag->OnBeforeDropPerform(tag, server, session)) {
            session.AddErrorMessage("DropPerformer", "cannot OnBeforeDropPerform for " + tag->GetName());
            return false;
        }
    }

    if (!DropTagsPerformer(notFinishedTags, userId, session, force)) {
        session.AddErrorMessage("DropPerformer", "cannot DropTagsPerformer");
        return false;
    }

    for (auto&& tag : notFinishedTags) {
        if (!tag->OnAfterDropPerform(tag, permissions, server, session)) {
            session.AddErrorMessage("DropPerformer", "cannot OnAfterDropPerform for " + tag->GetName());
            return false;
        }
    }

    if (!RemoveTags(tagsToRemove, permissions.GetUserId(), server, session)) {
        session.AddErrorMessage("DropPerformer", "cannot RemoveTags");
        return false;
    }

    return true;
}

bool IEntityTagsManager::GetPerformObjects(const TString& userId, TMap<TString, TVector<TDBTag>>& tagsByObject, NDrive::TEntitySession& session) const {
    auto tx = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> records(&TagsHistoryContext);

    NStorage::TTableRecord row;
    row.Set("performer", userId);
    TQueryResultPtr queryResult = tagsTable->GetRows(row.BuildCondition(*tx), records, tx);
    if (!queryResult->IsSucceed()) {
        return false;
    }

    for (auto tag : records) {
        tagsByObject[tag.GetObjectId()].push_back(tag);
    }
    return true;
}

bool IEntityTagsManager::GetObjectsByTagIds(const TSet<TString>& tagIds, TVector<TTaggedObject>& result, NDrive::TEntitySession& session) const {
    result.clear();
    TTransactionPtr transaction = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());

    TQueryResultPtr queryResult;
    TStringStream condition = "tag_id IN (" + session->Quote(tagIds) + ")";

    NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> tags(&TagsHistoryContext);

    TStringStream conditionFull;
    conditionFull << "SELECT a.* FROM " << GetTableName() << " AS a INNER JOIN (SELECT object_id FROM " << GetTableName()
        << " WHERE " << condition.Str() << " GROUP BY object_id) as b USING(object_id) ";
    auto timestamp = Now();
    queryResult = transaction->Exec(conditionFull.Str(), &tags);

    if (!queryResult->IsSucceed()) {
        return false;
    }

    {
        const auto pred = [](const TDBTag& l, const TDBTag& r) ->bool {
            return l.GetObjectId() < r.GetObjectId();
        };
        std::sort(tags.begin(), tags.end(), pred);

        for (auto&& tag : tags) {
            if (result.empty() || result.back().GetId() != tag.GetObjectId()) {
                result.emplace_back(tag.GetObjectId(), TDBTags{}, timestamp);
            }
            result.back().MutableTags().emplace_back(std::move(tag));
        }
    }
    return true;
}

bool IEntityTagsManager::RemoveTagsSimple(const TSet<TString>& tagIds, const TString& userId, NDrive::TEntitySession& session, const bool force) const {
    if (tagIds.empty()) {
        return true;
    }
    auto optionalTags = RestoreTags(tagIds, session);
    if (!optionalTags) {
        return false;
    }
    return RemoveTagsSimple(*optionalTags, userId, session, force);
}

bool IEntityTagsManager::RemoveTagsSimple(const TString& objectId, const TString& userId, const TVector<TString>& tagNames, NDrive::TEntitySession& session, const bool force) const {
    TVector<TDBTag> tags;
    if (!RestoreEntityTags(objectId, tagNames, tags, session)) {
        return false;
    }
    return RemoveTagsSimple(tags, userId, session, force);
}

bool IEntityTagsManager::RemoveTagSimple(const TDBTag& tag, const TString& userId, NDrive::TEntitySession& session, bool force) const {
    return RemoveTagsSimple({ tag }, userId, session, force);
}

bool IEntityTagsManager::RemoveTags(TConstArrayRef<TDBTag> tags, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session, const bool force, const bool tryRemove) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "RemoveTags")
            ("tags", NJson::ToJson(tags))
        );
    }
    for (auto&& tag : tags) {
        auto copied = tag;
        if (!copied->GetObjectSnapshot()) {
            copied->SetObjectSnapshot(BuildSnapshot(tag.GetObjectId(), server));
        }
        if (!copied->OnBeforeRemove(tag, userId, server, session)) {
            return false;
        }
    }
    if (!RemoveTagsSimple(tags, userId, session, force, tryRemove)) {
        return false;
    }
    for (auto&& tag : tags) {
        if (!tag->OnAfterRemove(tag, userId, server, session)) {
            return false;
        }
    }
    return true;
}

bool IEntityTagsManager::RestoreEntityTags(const TString& objectId, TConstArrayRef<TString> tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    return RestoreTags({objectId}, tagNames, result, session);
}

bool IEntityTagsManager::RestoreTags(const TSet<TString>& objectIds, TConstArrayRef<TString> tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    auto optionalTags = RestoreTags(objectIds, tagNames, session);
    if (optionalTags) {
        optionalTags->swap(result);
        return true;
    } else {
        return false;
    }
}

bool IEntityTagsManager::RestoreTagsRobust(NSQL::TStringContainer objectIds, NSQL::TStringContainer tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    auto optionalTags = RestoreTagsRobust(std::move(objectIds), std::move(tagNames), session);
    if (optionalTags) {
        optionalTags->swap(result);
        return true;
    } else {
        return false;
    }
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreEntityTags(const TString& objectId, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session) const {
    return RestoreTags(TVector<TString>(1, objectId), tagNames, session);
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreTags(const TVector<TString>& objectIds, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session) const {
    TSet<TString> vectorObjects(objectIds.begin(), objectIds.end());

    return RestoreTags(vectorObjects, tagNames, session);
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreTags(const TSet<TString>& tagIds, NDrive::TEntitySession& session) const {
    if (tagIds.empty()) {
        return TDBTags();
    }

    TQueryOptions queryOptions;
    queryOptions.SetTagIds(tagIds);
    return RestoreTags(session, std::move(queryOptions));
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreTags(const TString& tagId, NDrive::TEntitySession& session) const {
    return RestoreTags(TSet<TString>({tagId}), session);
}

TMaybe<TTaggedObject> IEntityTagsManager::GetCachedOrRestoreObject(const TString& objectId, NDrive::TEntitySession& session) const {
    return RestoreObject(objectId, session);
}

TMaybe<TTaggedObject> IEntityTagsManager::RestoreObject(const TString& objectId, NDrive::TEntitySession& session) const {
    TTaggedObject result;
    if (!RestoreObject(objectId, result, session)) {
        return {};
    }
    Y_ASSERT(result.GetId() == objectId);
    return result;
}

bool IEntityTagsManager::RestoreObject(const TString& objectId, TTaggedObject& result, NDrive::TEntitySession& session) const {
    TMap<TString, TTaggedObject> objects;
    TInstant timestamp = Now();
    if (!RestoreObjects({objectId}, objects, session)) {
        return false;
    }
    Y_ASSERT(objects.size() == 1);
    if (!objects.empty()) {
        result = std::move(objects.begin()->second);
    } else {
        result = TTaggedObject(objectId, TDBTags{}, timestamp);
    }
    return true;
}

bool IEntityTagsManager::RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& result, NDrive::TEntitySession& session) const {
    if (objectId.empty()) {
        result.clear();
        return true;
    }
    TVector<TDBTag> tags;
    TInstant timestamp = Now();
    if (!RestoreTags(objectId, {}, tags, session)) {
        return false;
    }
    for (auto&& i : objectId) {
        result.emplace(i, TTaggedObject(i, TDBTags{}, timestamp));
    }
    for (auto&& i : tags) {
        auto it = result.find(i.GetObjectId());
        Y_ASSERT(it != result.end());
        if (it != result.end()) {
            it->second.MutableTags().push_back(std::move(i));
        }
    }
    return true;
}

bool IEntityTagsManager::RestoreObjects(const TSet<TString>& ids, TTaggedObjectsSnapshot& result, NDrive::TEntitySession& session) const {
    TMap<TString, TTaggedObject> objects;
    if (!RestoreObjects(ids, objects, session)) {
        return false;
    }
    result.Reset(std::move(objects));
    return true;
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreTags(const TSet<TString>& objectIds, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session, const ui32 limit) const {
    if (tagNames.empty() && objectIds.empty()) {
        session.SetErrorInfo("EntityTagsManager::RestoreTags", "both empty tag names and object_ids are not permitted");
        return {};
    }

    TQueryOptions queryOptions(limit);
    if (tagNames.size()) {
        queryOptions.SetTags(MakeSet<TString>(tagNames));
    }
    if (objectIds.size()) {
        queryOptions.SetObjectIds(objectIds);
    }
    return RestoreTags(session, std::move(queryOptions));
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestoreTagsRobust(NSQL::TStringContainer objectIds, NSQL::TStringContainer tagNames, NDrive::TEntitySession& session, const ui32 limit) const {
    if (!objectIds && !tagNames) {
        session.SetErrorInfo("EntityTagsManager::RestoreTagsRobust", "both undefined objectIds and tagNames are not permitted");
        return {};
    }

    if (objectIds.Empty() || tagNames.Empty()) {
        return TVector<TDBTag>{};
    }

    TQueryOptions queryOptions(limit);
    queryOptions.SetObjectIds(std::move(objectIds));
    queryOptions.SetTags(std::move(tagNames));
    return RestoreTags(session, std::move(queryOptions));
}

IEntityTagsManager::TOptionalTag IEntityTagsManager::RestoreTag(const TString& tagId, NDrive::TEntitySession& session) const {
    auto tags = RestoreTags(tagId, session);
    if (!tags) {
        return {};
    }
    if (tags->empty()) {
        return TDBTag();
    }
    if (tags->size() > 1) {
        session.SetErrorInfo("EntityTagsManager::RestoreTag", TStringBuilder() << "multiple instances: " << tags->size());
        return {};
    }
    return std::move(tags->front());
}

IEntityTagsManager::TOptionalTags IEntityTagsManager::RestorePerformedTags(TOptionalStrings tagNames, TOptionalStrings performerIds, NDrive::TEntitySession& session) const {
    TVector<TDBTag> result;
    if (performerIds && performerIds->empty()) {
        return result;
    }
    if (tagNames && tagNames->empty()) {
        return result;
    }
    auto effectivePerformerIds = performerIds.GetOrElse({});
    bool success = false;
    if (tagNames) {
        success = RestorePerformerTags(*tagNames, effectivePerformerIds, result, session);
    } else {
        success = RestorePerformerTags(effectivePerformerIds, result, session);
    }
    if (!success) {
        return {};
    }
    return result;
}

bool IEntityTagsManager::RestorePerformerTags(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> performerIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    if (tagNames.empty()) {
        return true;
    }

    TQueryOptions queryOptions;
    queryOptions.SetTags(MakeSet<TString>(tagNames));
    if (!performerIds.empty()) {
        queryOptions.SetPerformers(MakeSet<TString>(performerIds));
    } else {
        queryOptions.AddCustomCondition("performer != ''");
    }

    auto optionalTags = RestoreTags(session, std::move(queryOptions));
    if (!optionalTags) {
        return false;
    }

    optionalTags->swap(result);
    return true;
}

bool IEntityTagsManager::RestorePerformerTags(TConstArrayRef<TString> performerIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    TQueryOptions queryOptions;
    if (!performerIds.empty()) {
        queryOptions.SetPerformers(MakeSet<TString>(performerIds));
    } else {
        queryOptions.AddCustomCondition("performer != ''");
    }

    auto optionalTags = RestoreTags(session, std::move(queryOptions));
    if (!optionalTags) {
        return false;
    }

    optionalTags->swap(result);
    return true;
}

bool IEntityTagsManager::RestorePerformerTags(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> performerIds, TConstArrayRef<TString> objectIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    TQueryOptions queryOptions;
    queryOptions.SetTags(MakeSet<TString>(tagNames));
    queryOptions.SetObjectIds(MakeSet<TString>(objectIds));
    if (!performerIds.empty()) {
        queryOptions.SetPerformers(MakeSet<TString>(performerIds));
    } else {
        queryOptions.AddCustomCondition("performer != ''");
    }

    auto optionalTags = RestoreTags(session, std::move(queryOptions));
    if (!optionalTags) {
        return false;
    }

    optionalTags->swap(result);
    return true;
}

bool IEntityTagsManager::RestoreTagsWithoutPerformer(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> objectIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const {
    TQueryOptions queryOptions;
    queryOptions.AddCustomCondition("performer = ''");
    queryOptions.SetTags(MakeSet<TString>(tagNames));
    if (!objectIds.empty()) {
        queryOptions.SetObjectIds(MakeSet<TString>(objectIds));
    }

    auto optionalTags = RestoreTags(session, std::move(queryOptions));
    if (!optionalTags) {
        return false;
    }

    optionalTags->swap(result);
    return true;
}

bool IEntityTagsManager::RestoreEvolutionTagsByUser(const TUserPermissions& permissions, const TString& targetTagName, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const {
    TVector<TDBTag> userTags;
    if (!RestorePerformerTags({ permissions.GetUserId() }, userTags, session)) {
        return false;
    }

    tags.clear();
    for (auto&& i : userTags) {
        if (i->GetPerformer() == permissions.GetUserId() && permissions.GetEvolutionPtr(i->GetName(), targetTagName)) {
            tags.push_back(i);
        }
    }
    return true;
}

IEntityTagsManager::TOptionalTag IEntityTagsManager::EvolveTag(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const NDrive::ITag::TEvolutionContext* eContext) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "EvolveTag")
            ("from", NJson::ToJson(fromTag))
            ("to", NJson::ToJson(toTag))
        );
    }
    if (!toTag) {
        session.SetErrorInfo("IEntityTagsManager::EvolveTag", "incorrect dest tag for evolution", EDriveSessionResult::InconsistencySystem);
        return {};
    }
    if (!fromTag->OnBeforeEvolve(fromTag, toTag, permissions, server, session, eContext)) {
        return {};
    }
    if (!toTag->ProvideDataOnEvolve(fromTag, permissions, server, session)) {
        return {};
    }

    if (!toTag->GetObjectSnapshot()) {
        auto snapshot = BuildSnapshot(fromTag.GetObjectId(), server);
        toTag->SetObjectSnapshot(std::move(snapshot));
    }
    auto evolved = DirectEvolveTag(permissions.GetUserId(), fromTag, toTag, session);
    if (!evolved) {
        return {};
    }
    if (!toTag->OnAfterEvolve(fromTag, toTag, permissions, server, session, eContext)) {
        return {};
    }
    if (!CheckInvariants(fromTag.GetObjectId(), session)) {
        return {};
    }
    if (eContext) {
        session.Committed().Subscribe([
            objectId = fromTag.GetObjectId(),
            toTag,
            permissions = permissions.Self(),
            server,
            policies = eContext->GetPolicies()
        ](const NThreading::TFuture<void>& w) {
            if (w.HasValue()) {
                for (auto&& policy : policies) try {
                    if (!policy) {
                        continue;
                    }
                    if (!policy->ExecuteAfterEvolveCommit(objectId, toTag.Get(), *Yensured(permissions), server)) {
                        NDrive::TEventLog::Log("PolicyAfterEvolveCommitError", NJson::TMapBuilder
                            ("object_id", objectId)
                            ("to", toTag->GetName())
                            ("user_id", permissions->GetUserId())
                            ("type", TypeName(*policy))
                        );
                    }
                } catch (...) {
                    NDrive::TEventLog::Log("PolicyAfterEvolveCommitException", NJson::TMapBuilder
                        ("object_id", objectId)
                        ("to", toTag->GetName())
                        ("user_id", permissions->GetUserId())
                        ("info", CurrentExceptionInfo())
                        ("type", TypeName(*policy))
                    );
                }
            }
        });
    }
    return evolved;
}

bool IEntityTagsManager::UpdateTagsData(TArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& tx) const {
    for (auto&& tag : tags) {
        if (!UpdateTagData(tag, userId, tx)) {
            return false;
        }
    }
    return true;
}

bool IEntityTagsManager::UpdateTagData(const TDBTag& tag, ITag::TPtr data, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto conditionBuilder = [](const TDBTag& tag) -> NStorage::TTableRecord {
        return NStorage::TRecordBuilder
            ("tag_id", tag.GetTagId())
        ;
    };
    auto updateBuilder = [data](const TDBTag& /*tag*/) -> NStorage::TTableRecord {
        NStorage::TTableRecord result;
        result.Set("data", Yensured(data)->GetStringData());
        if (data->HasTagPriority()) {
            result.Set("priority", data->GetTagPriorityRef());
        }
        return result;
    };
    if (!tag->OnBeforeUpdate(tag, data, userId, server, session)) {
        return false;
    }
    if (!UpdateTagsExtCondition({ tag }, userId, conditionBuilder, updateBuilder, session)) {
        return false;
    }
    if (!data->OnAfterUpdate(tag, data, userId, server, session)) {
        return false;
    }
    return true;
}

bool IEntityTagsManager::UpdateTagData(TDBTag& tag, const TString& userId, NDrive::TEntitySession& tx) const {
    auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
    return UpdateTagData(tag, tag.GetData(), userId, server, tx);
}

bool IEntityTagsManager::UpdateTagsExtCondition(TConstArrayRef<TDBTag> tags, const TString& userId, const TConditionBuilder& conditionBuilder, const TUpdateBuilder& updateBuilder, NDrive::TEntitySession& session) const {
    for (auto&& i : tags) {
        if (!UpdateTagExtCondition(i, userId, conditionBuilder, updateBuilder, session)) {
            return false;
        }
    }
    return true;
}

TMaybe<TMap<TString, TInstant>> IEntityTagsManager::RestoreTagsCreationTimes(const TSet<TString>& tagIds, NDrive::TEntitySession& session) const {
    auto maybeEvents = GetEvents(TRange<TInstant>{}, session, TQueryOptions()
        .SetTagIds(tagIds).SetActions({ EObjectHistoryAction::Add })
    );
    if (!maybeEvents) {
        return Nothing();
    }
    TMap<TString, TInstant> result;

    for (const auto& ev : *maybeEvents) {
        if (ev) {
            result[ev.GetTagId()] = ev.GetHistoryTimestamp();
        }
    }
    return result;
}

TMaybe<TMap<TString, TInstant>> IEntityTagsManager::RestoreTagsCreationTimes(const TVector<TDBTag>& tags, NDrive::TEntitySession& session) const {
    TSet<TString> tagIds;
    for (const auto& tag : tags) {
        tagIds.insert(tag.GetTagId());
    }
    return RestoreTagsCreationTimes(tagIds, session);
}

TMaybe<TMultiSet<TString>> IEntityTagsManager::SimpleRollbackTagsState(const TString& objectId, TInstant dateTo, NSQL::TStringContainer tagsFilter, NDrive::TEntitySession& tx) const {
    TMultiSet<TString> state;
    if (auto dbTags = RestoreTagsRobust({ objectId }, tagsFilter, tx)) {
        Transform(dbTags->begin(), dbTags->end(), std::inserter(state, state.begin()), [](auto&& item) { return std::move(item->GetName()); });
    } else {
        return {};
    }
    IEntityTagsManager::TQueryOptions options;
    options.SetTags(tagsFilter);
    options.SetObjectIds({ objectId });
    options.SetActions({ EObjectHistoryAction::Add, EObjectHistoryAction::Remove });
    auto events = GetEvents({ dateTo }, tx, options);
    if (!events) {
        return {};
    }
    for (auto&& event : *events) {
        if (!event) {
            continue;
        }
        switch (event.GetHistoryAction()) {
            case EObjectHistoryAction::Add:
                state.erase(event->GetName());
                continue;
            case EObjectHistoryAction::Remove:
                state.insert(event->GetName());
                continue;
            default:
                continue;
        }
    }
    return state;
}

TString TDBTagMetaConditionConstructor::BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
    return "name IN (" + session->Quote(ids) + ")";
}

NStorage::TTableRecord TDBTagMetaConditionConstructor::BuildCondition(const TString& id) {
    NStorage::TTableRecord trCondition;
    trCondition.Set("name", id);
    return trCondition;
}
NStorage::TTableRecord TDBTagMetaConditionConstructor::BuildCondition(const TDBTagMeta& object) {
    return BuildCondition(object.GetDescription().GetName());
}

TTagsMeta::TTagsMeta(const IHistoryContext& historyContext, const THistoryConfig& config)
    : TBase(historyContext, config)
{
    Y_ENSURE_BT(Start());

    TMap<TString, TDBTagMeta> tagsData;
    {
        auto rg = MakeObjectReadGuard();
        tagsData = Objects;
    }

    TSet<TString> registeredTypes;
    TSet<TString> registeredKeys;
    ITag::TFactory::GetRegisteredKeys(registeredKeys);
    {
        auto session = GetHistoryManager().BuildSession(false);
        for (auto&& type : registeredKeys) {
            if (!tagsData.contains(type)) {
                THolder<ITag> instance(ITag::TFactory::Construct(type));
                CHECK_WITH_LOG(!!instance);
                INFO_LOG << "Register service type: " << type << Endl;
                auto description = instance->GetMetaDescription(type);
                if (description) {
                    CHECK_WITH_LOG(RegisterTagImpl(description, "tags_meta", session, /*preserveType=*/true)) << session.GetReport();
                }
            }
        }
        CHECK_WITH_LOG(session.Commit()) << session.GetReport() << Endl;
    }
}

TTagsMeta::~TTagsMeta() {
    if (IsActive() && !Stop()) {
        ERROR_LOG << "cannot stop TagsMeta" << Endl;
    }
}

EUniquePolicy TTagsMeta::GetTagUniquePolicy(ITag::TConstPtr tag, TTagDescription::TConstPtr description, const EUniquePolicy overridePolicy) const {
    if (overridePolicy != EUniquePolicy::Undefined) {
        return overridePolicy;
    }
    if (!tag) {
        return EUniquePolicy::Undefined;
    }
    if (!description) {
        description = GetDescriptionByName(tag->GetName());
    }
    if (!!description) {
        return (description->GetUniquePolicy() == EUniquePolicy::Undefined) ? tag->GetUniquePolicy() : description->GetUniquePolicy();
    }
    return tag->GetUniquePolicy();
}

ITag::TPtr TTagsMeta::CreateTag(const TString& name, const TString& comment, TInstant reqActuality) const {
    auto description = GetDescriptionByName(name, reqActuality);
    if (!description) {
        ERROR_LOG << "unknown tag: " << name << Endl;
        return nullptr;
    }
    ITag::TPtr tag = ITag::TFactory::Construct(description->GetType());
    if (!tag) {
        ERROR_LOG << "cannot construct tag type: " << description->GetType() << Endl;
        return nullptr;
    }
    tag->SetName(name);
    tag->SetComment(comment);
    tag->SetTagPriority(description->GetDefaultPriority());
    return tag;
}

ITagsMeta::TOptionalTagDescription TTagsMeta::GetDescription(const TString& name, NDrive::TEntitySession& tx) const {
    auto database = GetDatabase();
    auto table = Yensured(database)->GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TDBTagMeta> descriptions;
    NStorage::TTableRecord condition = NStorage::TRecordBuilder
        ("name", name)
    ;
    auto queryResult = table->GetRows(condition, descriptions, tx.GetTransaction());
    if (!ParseQueryResult(queryResult, tx)) {
        return {};
    }
    for (auto&& description : descriptions) {
        return description.GetDescriptionPtr();
    }
    return nullptr;
}

bool TTagsMeta::RegisterTag(TTagDescription::TConstPtr description, const TString& userId, NDrive::TEntitySession& session, bool force) const {
    if (!ITag::TFactory::Has(description->GetType())) {
        session.SetErrorInfo("input", "unknown tag type " + description->GetType(), EDriveSessionResult::InternalError);
        return false;
    }
    return RegisterTagImpl(description, userId, session, /*preserveType=*/!force);
}

ITagsMeta::TTagDescriptions TTagsMeta::GetTagsByType(const TString& tagType, const TInstant reqActuality) const {
    TTagDescriptions result;
    TVector<TDBTagMeta> objects;
    if (!GetAllObjectsFromCache(objects, reqActuality)) {
        return result;
    }
    for (auto&& it : objects) {
        if (it->GetType() == tagType) {
            result.push_back(it.GetDescriptionPtr());
        }
    }
    return result;
}

bool TTagsMeta::UnregisterTag(const TString& tagName, const TString& userId, NDrive::TEntitySession& session) const {
    TTagDescription::TConstPtr td;
    {
        auto tagDescriptions = GetRegisteredTags(Now());
        auto it = tagDescriptions.find(tagName);
        if (it == tagDescriptions.end()) {
            return true;
        }
        td = it->second;
    }

    auto database = GetDatabase();
    auto tagsTable = Yensured(database)->GetTable(GetTableName());
    auto transaction = session.GetTransaction();
    TString condition = "name='" + tagName + "'";
    TQueryResultPtr queryResult = tagsTable->RemoveRow(condition, transaction);
    if (!queryResult || !queryResult->IsSucceed()) {
        session.SetErrorInfo("RemoveTag", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!queryResult->GetAffectedRows()) {
        return true;
    }
    return HistoryManager->AddHistory(TDBTagMeta(td), userId, EObjectHistoryAction::Remove, session);
}

TString TTagsMeta::GetTagTypeByName(const TString& name, const TInstant reqActuality) const {
    auto td = GetDescriptionByName(name, reqActuality);
    return !!td ? td->GetType() : "";
}

TTagDescription::TConstPtr TTagsMeta::GetDescriptionByIndex(ui32 index) const {
    auto rg = MakeObjectReadGuard();
    auto it = ObjectsByIndex.find(index);
    if (it == ObjectsByIndex.end()) {
        return nullptr;
    }
    return it->second.GetDescriptionPtr();
}

TTagDescription::TConstPtr TTagsMeta::GetDescriptionByName(const TString& name, const TInstant reqActuality) const {
    TVector<TDBTagMeta> tags;
    if (!GetCustomObjectsFromCache(tags, NContainer::Scalar(name), reqActuality)) {
        return nullptr;
    }
    if (tags.size() == 1) {
        return tags.front().GetDescriptionPtr();
    }
    return nullptr;
}

ITagsMeta::TTagDescriptionsByName TTagsMeta::GetRegisteredTags(const TInstant reqActuality) const {
    TTagDescriptionsByName result;
    TVector<TDBTagMeta> tags;
    if (GetAllObjectsFromCache(tags, reqActuality)) {
        for (auto&& tag : tags) {
            result.emplace(tag->GetName(), tag.GetDescriptionPtr());
        }
    }
    return result;
}

ITagsMeta::TTagDescriptionsByName TTagsMeta::GetRegisteredTags(NEntityTagsManager::EEntityType type, const TSet<TString>& tagTypes, const TInstant actuality) const {
    TTagDescriptionsByName result;
    TVector<TDBTagMeta> tags;
    if (GetAllObjectsFromCache(tags, actuality)) {
        for (auto&& tag : tags) {
            auto description = tag.GetDescriptionPtr();
            if (!description) {
                continue;
            }
            if (!tagTypes.empty() && !tagTypes.contains(description->GetType())) {
                continue;
            }
            ITag::TPtr instance = ITag::TFactory::Construct(description->GetType());
            if (!instance || !instance->GetObjectType().contains(type)) {
                continue;
            }
            result.emplace(tag->GetName(), description);
        }
    }
    return result;
}

TSet<TString> TTagsMeta::GetRegisteredTagNames(const TSet<TString>& tagTypes, const TInstant reqActuality) const {
    auto registeredTags = GetRegisteredTags(reqActuality);
    TSet<TString> result;
    for (auto&&[name, description] : registeredTags) {
        if (description && tagTypes.contains(description->GetType())) {
            result.insert(name);
        }
    }
    return result;
}

bool TTagsMeta::RegisterTagImpl(TTagDescription::TConstPtr description, const TString& userId, NDrive::TEntitySession& session, bool preserveType) const {
    auto database = GetDatabase();
    auto tagsTable = Yensured(database)->GetTable(GetTableName());
    NStorage::TTableRecord tRecord = description->SerializeToTableRecord();
    NStorage::TTableRecord tRecordUnique;
    tRecordUnique.Set("name", description->GetName());
    if (preserveType) {
        tRecordUnique.Set("type", description->GetType());
    }
    bool isUpdate = false;
    NStorage::TObjectRecordsSet<TDBTagMeta> descriptions;
    TQueryResultPtr queryResult = tagsTable->Upsert(tRecord, session.GetTransaction(), tRecordUnique, &isUpdate, &descriptions);
    if (!queryResult->IsSucceed()) {
        session.SetErrorInfo("RegisterTagImpl", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }
    if (descriptions.size() == 0) {
        session.SetErrorInfo("RegisterTagImpl", "no parsed objects", EDriveSessionResult::TransactionProblem);
        WARNING_LOG << "no updated objects" << Endl;
        return false;
    }
    return HistoryManager->AddHistory(descriptions.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData: EObjectHistoryAction::Add, session);
}

bool TTagsMeta::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TDBTagMeta> descriptions;
    {
        auto database = GetDatabase();
        auto tagsTable = Yensured(database)->GetTable(GetTableName());
        auto transaction = Yensured(database)->CreateTransaction(true);
        TQueryResultPtr queryResult = tagsTable->GetRows("", descriptions, transaction);
        if (!queryResult->IsSucceed()) {
            return false;
        }
    }

    for (auto&& tRecord : descriptions) {
        Objects[tRecord->GetName()] = tRecord;
        ObjectsByIndex[tRecord->GetIndex()] = tRecord;
    }
    return true;
}

void TTagsMeta::AcceptHistoryEventUnsafe(const TObjectEvent<TDBTagMeta>& ev) const {
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        ObjectsByIndex.erase(ev->GetIndex());
        Objects.erase(ev->GetName());
    } else if (ev.HasDescription()) {
        auto deepCopy = ev->Clone();
        if (!!deepCopy) {
            Objects[ev->GetName()] = TDBTagMeta(deepCopy);
            ObjectsByIndex[ev->GetIndex()] = TDBTagMeta(deepCopy);
        } else {
            ERROR_LOG << "Incorrect TAG - cannot copying: " << ev->GetType() << Endl;
        }
    } else {
        ERROR_LOG << "Incorrect TAG - empty event" << Endl;
    }
}

NStorage::TTableRecord TDBTagMeta::SerializeToTableRecord() const {
    return GetDescription().SerializeToTableRecord();
}

NJson::TJsonValue TDBTagMeta::BuildJsonReport() const{
    return GetDescription().SerializeToTableRecord().SerializeToJson();
}

bool TDBTagMeta::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    TString objectType;
    READ_DECODER_VALUE_TEMP(decoder, values, objectType, Type);

    TTagDescription::TPtr description = TTagDescription::TFactory::Construct(objectType, "default");
    if (!description) {
        ERROR_LOG << "Incorrect TagDescription type: " << objectType << Endl;
        return false;
    }
    if (!description->DeserializeWithDecoder(decoder, values, hContext)) {
        ERROR_LOG << "Incorrect TagDescription data for " << objectType << Endl;
        return false;
    }

    Description = description;

    return true;
}

TInstant TTagPropositions::GetDeadline(const TDBTag& obj, const TInstant start) const {
    if (!!obj) {
        auto td = Context.GetTagsManager().GetDescriptionByName(obj->GetName());
        const TConfirmableTag::TDescription* ctd = dynamic_cast<const TConfirmableTag::TDescription*>(td.Get());
        if (ctd && ctd->GetLivetime()) {
            return start + ctd->GetLivetime();
        }
    }
    return TBase::GetDeadline(obj, start);
}

ESelfConfirmationPolicy TTagPropositions::GetSelfConfirmationPolicy(const TDBTag& obj) const {
    if (!!obj) {
        auto td = Context.GetTagsManager().GetDescriptionByName(obj->GetName());
        const TConfirmableTag::TDescription* ctd = dynamic_cast<const TConfirmableTag::TDescription*>(td.Get());
        if (ctd) {
            return ctd->GetSelfConfirmationPolicy();
        }
    }
    return TBase::GetSelfConfirmationPolicy(obj);
}

EDoubleConfirmationPolicy TTagPropositions::GetDoubleConfirmationPolicy(const TDBTag& obj) const {
    if (!!obj) {
        auto td = Context.GetTagsManager().GetDescriptionByName(obj->GetName());
        const TConfirmableTag::TDescription* ctd = dynamic_cast<const TConfirmableTag::TDescription*>(td.Get());
        if (ctd) {
            return ctd->GetDoubleConfirmationPolicy();
        }
    }
    return TBase::GetDoubleConfirmationPolicy(obj);
}

template class TEntityTagsManager<TTaggedDevice, TCarTagsHistoryManager>;
template class TEntityTagsManager<TTaggedTrace, TTraceTagsHistoryManager>;
template class TEntityTagsManager<TTaggedUser, TUserTagsHistoryManager>;
