#pragma once

#include "config.h"
#include "history.h"
#include "tags.h"
#include "tags_cache.h"
#include "tags_filter.h"

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/db_entities.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/util/types/expected.h>

class TTagPropositions: public TPropositionsManager<TDBTag> {
private:
    using TBase = TPropositionsManager<TDBTag>;
    const ITagsHistoryContext& Context;

public:
    virtual TInstant GetDeadline(const TDBTag& obj, const TInstant start) const override;

    virtual ESelfConfirmationPolicy GetSelfConfirmationPolicy(const TDBTag& obj) const override;
    virtual EDoubleConfirmationPolicy GetDoubleConfirmationPolicy(const TDBTag& obj) const override;

    TTagPropositions(const ITagsHistoryContext& context, const TPropositionsManagerConfig& config)
        : TBase(context, "car_tag_propositions", config)
        , Context(context)
    {
    }

    TMaybe<TMap<TString, TReport>> GetByObjectId(const NSQL::TStringContainer& objectIds, NDrive::TEntitySession& session) const {
        if (objectIds.Empty()) {
            return MakeMaybe<TMap<TString, TReport>>();
        }
        NSQL::TQueryOptions options;
        if (objectIds) {
            options.SetGenericCondition("object_id", objectIds.BuildCondition());
        }
        return Get(options, session);
    }
};

class TTaggedObjectsSnapshot {
private:
    TMap<TString, TTaggedObject> Objects;

public:
    TTaggedObjectsSnapshot() = default;

    void Reset(TMap<TString, TTaggedObject>&& objects) {
        Objects = std::move(objects);
    }

    const TTaggedObject* Get(const TString& id) const {
        auto it = Objects.find(id);
        if (it == Objects.end()) {
            return nullptr;
        } else {
            return &it->second;
        }
    }

    auto begin() const {
        return Objects.begin();
    }

    auto end() const {
        return Objects.end();
    }
};

class IEntityTagsManager: public TDatabaseSessionConstructor {
public:
    enum ETagError {
        Absent,
        MultipleInstances,
        LogicError,
        ParsingError,
        QueryError,
    };
    using TExpectedConstTag = TExpected<TConstDBTag, ETagError>;
    using TExpectedConstTags = TExpected<TVector<TExpectedConstTag>, ETagError>;
    using TExpectedTag = TExpected<TDBTag, ETagError>;
    using TExpectedTags = TExpected<TVector<TExpectedTag>, ETagError>;

    using TOptionalTag = TMaybe<TDBTag>;
    using TOptionalTags = TMaybe<TVector<TDBTag>>;

    using TConditionBuilder = std::function<NStorage::TTableRecord(const TDBTag&)>;
    using TUpdateBuilder = TConditionBuilder;
    using TOptionalStrings = TMaybe<TConstArrayRef<TString>>;
    using TQueryOptions = TTagEventsManager::TQueryOptions;

protected:
    const ITagsHistoryContext& TagsHistoryContext;

protected:
    virtual bool CheckInvariants(const TString& objectId, NDrive::TEntitySession& session) const;
    virtual NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& objectId, const NDrive::IServer* server) const = 0;
    virtual NEntityTagsManager::EMultiplePerformersPolicy GetMultiplePerformersPolicy() const {
        return NEntityTagsManager::EMultiplePerformersPolicy::Deny;
    }

public:
    IEntityTagsManager(const ITagsHistoryContext& context);
    virtual ~IEntityTagsManager() = default;

    virtual NEntityTagsManager::EEntityType GetEntityType() const = 0;
    virtual TSessionsBuilder<TTagHistoryEvent>::TPtr GetSessionsBuilder(TStringBuf name, TInstant actuality = TInstant::Zero()) const = 0;
    virtual const TString& GetTableName() const = 0;

    virtual TMaybe<TTaggedObject> GetCachedOrRestoreObject(const TString& objectId, NDrive::TEntitySession& session) const;
    virtual TMaybe<TTaggedObject> RestoreObject(const TString& objectId, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestoreObject(const TString& objectId, TTaggedObject& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestoreObjects(const TSet<TString>& ids, TTaggedObjectsSnapshot& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] virtual bool RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& result, NDrive::TEntitySession& session) const;

    [[nodiscard]] bool RejectPropositions(const TSet<TString>& propositionIds, const TString& userId, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool ConfirmPropositions(const TSet<TString>& propositionIds, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    [[nodiscard]] TPropositionId ProposeTag(const TDBTag& dbTag, const TString& userId, NDrive::TEntitySession& session) const;

    [[nodiscard]] virtual TOptionalTagHistoryEvent GetEvent(TTagHistoryEvent::TEventId id, NDrive::TEntitySession& session) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEvents(TRange<TTagHistoryEvent::TEventId> idRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEvents(TRange<TTagHistoryEvent::TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEvents(TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEventsByObject(const TString& objectId, NDrive::TEntitySession& session, TTagHistoryEvent::TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEventsByTag(const TString& tagId, NDrive::TEntitySession& session, TTagHistoryEvent::TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEvents GetEventsSince(TTagHistoryEvent::TEventId id, NDrive::TEntitySession& session, ui64 limit) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEventId GetLockedMaxEventId(NDrive::TEntitySession& session) const = 0;
    [[nodiscard]] virtual TOptionalTagHistoryEventId GetMaxEventIdOrThrow(NDrive::TEntitySession& session) const = 0;

    [[nodiscard]] virtual bool SetTagsPerformer(TArrayRef<TDBTag> tags, const TString& userId, bool force, NDrive::TEntitySession& session, const NDrive::IServer* server, ui32 lockedLimit = 0, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform = {}) const = 0;
    [[nodiscard]] bool SetTagPerformer(TDBTag& tag, const TString& userId, const bool force, NDrive::TEntitySession& session, const NDrive::IServer* server, const ui32 lockedLimit = 0, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform = {}) const {
        return SetTagsPerformer(NContainer::Scalar(tag), userId, force, session, server, lockedLimit, overrideMultiPerform);
    }

    [[nodiscard]] virtual TOptionalTag AddSnapshot(const TDBTag& tag, NDrive::IObjectSnapshot::TPtr snapshot, const TString& userId, NDrive::TEntitySession& session) const = 0;
    [[nodiscard]] virtual TOptionalTags AddTags(TConstArrayRef<ITag::TPtr> tags, const TString& userId, const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session, EUniquePolicy uniquePolicy = EUniquePolicy::Undefined, const TString& filter = "") const = 0;
    [[nodiscard]] TOptionalTags AddTag(ITag::TPtr tag, const TString& userId, const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session, EUniquePolicy uniquePolicy = EUniquePolicy::Undefined, const TString& filter = "") const;

    [[nodiscard]] virtual bool RemoveTagsSimple(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, const bool force, const bool tryRemove = false) const = 0;
    [[nodiscard]] bool RemoveTagsSimple(const TSet<TString>& tagIds, const TString& userId, NDrive::TEntitySession& session, const bool force) const;
    [[nodiscard]] bool RemoveTagsSimple(const TString& objectId, const TString& userId, const TVector<TString>& tagNames, NDrive::TEntitySession& session, const bool force) const;
    [[nodiscard]] bool RemoveTagSimple(const TDBTag& tag, const TString& userId, NDrive::TEntitySession& session, bool force) const;

    [[nodiscard]] bool RestoreEntityTags(const TString& objectId, TConstArrayRef<TString> tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestoreTags(const TSet<TString>& objectIds, TConstArrayRef<TString> tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestoreTagsRobust(NSQL::TStringContainer objectIds, NSQL::TStringContainer tagNames, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;

    [[nodiscard]] bool RestoreTags(const TSet<TString>& tagIds, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const {
        auto optionalTags = RestoreTags(tagIds, session);
        if (!optionalTags) {
            return false;
        }
        optionalTags->swap(tags);
        return true;
    }
    [[nodiscard]] bool RestoreTags(const TString& tagId, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const {
        auto optionalTags = RestoreTags(tagId, session);
        if (!optionalTags) {
            return false;
        }
        optionalTags->swap(tags);
        return true;
    }

    [[nodiscard]] bool RemoveTag(const TDBTag& tag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session, const bool force = false, const bool tryRemove = false) const {
        return RemoveTags(NContainer::Scalar(tag), userId, server, session, force, tryRemove);
    };
    [[nodiscard]] bool RemoveTags(TConstArrayRef<TDBTag> tags, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session, const bool force = false, const bool tryRemove = false) const;

    virtual TOptionalTags RestoreTags(NDrive::TEntitySession& session, TQueryOptions&& queryOptions) const = 0;
    TOptionalTags RestoreEntityTags(const TString& objectId, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session) const;
    TOptionalTags RestorePerformedTags(TOptionalStrings tagNames, TOptionalStrings performerIds, NDrive::TEntitySession& session) const;
    TOptionalTags RestoreTags(const TVector<TString>& objectIds, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session) const;
    TOptionalTags RestoreTags(const TSet<TString>& objectIds, TConstArrayRef<TString> tagNames, NDrive::TEntitySession& session, const ui32 limit = 0) const;
    TOptionalTags RestoreTagsRobust(NSQL::TStringContainer objectIds, NSQL::TStringContainer tagNames, NDrive::TEntitySession& session, const ui32 limit = 0) const;

    TOptionalTags RestoreTags(const TSet<TString>& tagIds, NDrive::TEntitySession& session) const;
    TOptionalTags RestoreTags(const TString& tagId, NDrive::TEntitySession& session) const;
    TOptionalTag RestoreTag(const TString& tagId, NDrive::TEntitySession& session) const;

    [[nodiscard]] bool RestorePerformerTags(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> performerIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestorePerformerTags(TConstArrayRef<TString> performerIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestorePerformerTags(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> performerIds, TConstArrayRef<TString> objectIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool RestoreTagsWithoutPerformer(TConstArrayRef<TString> tagNames, TConstArrayRef<TString> objectIds, TVector<TDBTag>& result, NDrive::TEntitySession& session) const;

    [[nodiscard]] bool RestoreEvolutionTagsByUser(const TUserPermissions& permissions, const TString& targetTagName, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const;

    [[nodiscard]] virtual TOptionalTag DirectEvolveTag(const TString& userId, const TDBTag& fromTag, ITag::TPtr toTag, NDrive::TEntitySession& session) const = 0;
    [[nodiscard]] TOptionalTag EvolveTag(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const NDrive::ITag::TEvolutionContext* eContext = nullptr) const;

    [[nodiscard]] bool UpdateTagsData(TArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& tx) const;
    [[nodiscard]] bool UpdateTagData(TDBTag& tag, const TString& userId, NDrive::TEntitySession& tx) const;
    [[nodiscard]] bool UpdateTagData(const TDBTag& tag, ITag::TPtr data, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool UpdateTagsExtCondition(TConstArrayRef<TDBTag> tags, const TString& userId, const TConditionBuilder& conditionBuilder, const TUpdateBuilder& updateBuilder, NDrive::TEntitySession& session) const;
    [[nodiscard]] virtual bool UpdateTagExtCondition(const TDBTag& tag, const TString& userId, const TConditionBuilder& conditionBuilder, const TUpdateBuilder& updateBuilder, NDrive::TEntitySession& session) const = 0;

    virtual const TTagPropositions* GetPropositionsPtr() const = 0;

    [[nodiscard]] bool InitPerformer(const TSet<TString>& tagIds, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool InitPerformer(const TVector<TDBTag>& initialTagsExt, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool replaceSnapshots = true) const;

    [[nodiscard]] virtual bool DropTagsPerformer(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, bool force = false) const = 0;
    [[nodiscard]] bool DropPerformer(const TSet<TString>& removeTagIds, const TSet<TString>& cancelTagIds, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool force = false) const;
    [[nodiscard]] bool DropTagPerformer(const TDBTag& tag, const TString& userId, NDrive::TEntitySession& session, bool force = false) const {
        return DropTagsPerformer({tag}, userId, session, force);
    }

    [[nodiscard]] bool GetObjectsByTagIds(const TSet<TString>& tagIds, TVector<TTaggedObject>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool GetPerformObjects(const TString& userId, TMap<TString, TVector<TDBTag>>& tagsByObject, NDrive::TEntitySession& session) const;

    [[nodiscard]] TMaybe<TMap<TString, TInstant>> RestoreTagsCreationTimes(const TSet<TString>& tagIds, NDrive::TEntitySession& session) const;
    [[nodiscard]] TMaybe<TMap<TString, TInstant>> RestoreTagsCreationTimes(const TVector<TDBTag>& tags, NDrive::TEntitySession& session) const;
    TMaybe<TMultiSet<TString>> SimpleRollbackTagsState(const TString& objectId, TInstant dateTo, NSQL::TStringContainer tagsFilter, NDrive::TEntitySession& session) const;
};

template <class TEntity, class THistoryManagerExternal>
class TEntityTagsManager: public IEntityTagsManager {
private:
    using TBase = IEntityTagsManager;
    class TTagEntityManager;

public:
    using TOptionalTags = typename TBase::TOptionalTags;

protected:
    using THistoryManager = THistoryManagerExternal;

protected:
    using TBase::BuildSnapshot;
    using TBase::GetTableName;

    virtual const THistoryManager& GetHistoryManagerImpl() const = 0;
    virtual THistoryManager& GetHistoryManagerImpl() = 0;

    virtual TMaybe<TString> GetDefaultFilter(const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
        Y_UNUSED(objectId);
        Y_UNUSED(server);
        Y_UNUSED(tx);
        return MakeMaybe<TString>();
    }
    virtual TSessionsBuilder<TTagHistoryEvent>::TPtr GetSessionsBuilder(TStringBuf name, TInstant actuality = TInstant::Zero()) const override {
        Y_UNUSED(name);
        Y_UNUSED(actuality);
        return nullptr;
    }

public:
    virtual const TTagPropositions* GetPropositionsPtr() const override {
        return Propositions.Get();
    }

    virtual TOptionalTagHistoryEvent GetEvent(TTagHistoryEvent::TEventId id, NDrive::TEntitySession& session) const override {
        return GetHistoryManagerImpl().GetEvent(id, session);
    }

    virtual TOptionalTagHistoryEvents GetEvents(TRange<TTagHistoryEvent::TEventId> idRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const override {
        return GetHistoryManagerImpl().GetEventsByRanges(std::move(idRange), {}, session, queryOptions);
    }

    virtual TOptionalTagHistoryEvents GetEvents(TRange<TTagHistoryEvent::TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const override {
        return GetHistoryManagerImpl().GetEventsByRanges(std::move(idRange), std::move(timestampRange), session, queryOptions);
    }

    virtual TOptionalTagHistoryEvents GetEvents(TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const override {
        return GetHistoryManagerImpl().GetEventsByRanges({}, std::move(timestampRange), session, queryOptions);
    }

    virtual TOptionalTagHistoryEvents GetEventsByObject(const TString& objectId, NDrive::TEntitySession& session, TTagHistoryEvent::TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const override {
        return GetHistoryManagerImpl().GetEventsByObject(objectId, session, sinceId, sinceTimestamp, queryOptions);
    }

    virtual TOptionalTagHistoryEvents GetEventsByTag(const TString& tagId, NDrive::TEntitySession& session, TTagHistoryEvent::TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const override {
        return GetHistoryManagerImpl().GetEventsByTag(tagId, session, sinceId, sinceTimestamp, queryOptions);
    }

    virtual TOptionalTagHistoryEvents GetEventsSince(TTagHistoryEvent::TEventId id, NDrive::TEntitySession& session, ui64 limit) const override {
        return GetHistoryManagerImpl().GetEvents(id, session, limit);
    }

    virtual TOptionalTagHistoryEventId GetLockedMaxEventId(NDrive::TEntitySession& session) const override {
        Y_UNUSED(session);
        return GetHistoryManagerImpl().GetLockedMaxEventId();
    }

    virtual TOptionalTagHistoryEventId GetMaxEventIdOrThrow(NDrive::TEntitySession& session) const override {
        return GetHistoryManagerImpl().GetMaxEventIdOrThrow(session);
    }

    virtual TString MakeTagPerformerCondition(const TDBTag& tag, const TString& userId, bool force, ui32 lockedLimit, NDrive::TEntitySession& session, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform) const;

public:
    TEntityTagsManager(const ITagsHistoryContext& context);
    ~TEntityTagsManager();

    using TBase::RemoveTagsSimple;
    using TBase::RestoreTags;

    [[nodiscard]] TOptionalTag AddSnapshot(const TDBTag& tag, NDrive::IObjectSnapshot::TPtr snapshot, const TString& userId, NDrive::TEntitySession& session) const override;
    [[nodiscard]] TOptionalTags AddTags(TConstArrayRef<ITag::TPtr> tags, const TString& userId, const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session, EUniquePolicy uniquePolicy = EUniquePolicy::Undefined, const TString& filter = "") const override;
    [[nodiscard]] TOptionalTags RestoreTags(NDrive::TEntitySession& session, TQueryOptions&& queryOptions) const override;
    [[nodiscard]] TOptionalTag DirectEvolveTag(const TString& userId, const TDBTag& fromTag, ITag::TPtr toTag, NDrive::TEntitySession& session) const override;
    [[nodiscard]] bool DropTagsPerformer(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, bool force = false) const override;
    [[nodiscard]] bool SetTagsPerformer(TArrayRef<TDBTag> tags, const TString& userId, bool force, NDrive::TEntitySession& session, const NDrive::IServer* server, ui32 lockedLimit = 0, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform = {}) const override;
    [[nodiscard]] bool RemoveTagsSimple(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, bool force, bool tryRemove = false) const override;
    [[nodiscard]] bool UpdateTagExtCondition(const TDBTag& tag, const TString& userId, const TConditionBuilder& conditionBuilder, const TUpdateBuilder& updateBuilder, NDrive::TEntitySession& session) const override;

    void InitPropositions(const TPropositionsManagerConfig& propositionsConfig) {
        Y_ENSURE_BT(!Propositions);
        Propositions = MakeHolder<TTagPropositions>(TagsHistoryContext, propositionsConfig);
    }

    [[nodiscard]] bool GetObjects(const TSet<TString>& objectIds, const TSet<TString>& tagNames, bool getAllTags, TVector<TEntity>& result, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool GetObjects(const TSet<TString>& tagNames, bool getAllTags, TVector<TEntity>& result, NDrive::TEntitySession& session) const {
        return GetObjects({}, tagNames, getAllTags, result, session);
    }

private:
    THolder<TTagEntityManager> EntityManager;
    THolder<TTagPropositions> Propositions;
};

template <class TEntity, class THistoryManagerExternal>
class TCachedEntityTagsManager
    : public TEntityTagsManager<TEntity, THistoryManagerExternal>
    , public TDBCacheWithHistoryOwner<THistoryManagerExternal, TEntity>
{
private:
    using TBase = TEntityTagsManager<TEntity, THistoryManagerExternal>;
    using TCacheBase = TDBCacheWithHistoryOwner<THistoryManagerExternal, TEntity>;

protected:
    using TBase::Database;
    using TBase::GetTableName;
    using TBase::TagsHistoryContext;
    using TCacheBase::ForObjectsList;
    using TCacheBase::HistoryManager;
    using TCacheBase::Objects;

public:
    using ETagError = typename TBase::ETagError;
    using TExpectedConstTag = typename TBase::TExpectedConstTag;
    using TExpectedConstTags = typename TBase::TExpectedConstTags;

    class TCurrentSnapshot {
    private:
        using TData = TMap<TString, TAtomicSharedPtr<TEntity>>;

    private:
        TData Data;

    public:
        TCurrentSnapshot() = default;
        TCurrentSnapshot(const TMap<TString, TAtomicSharedPtr<TEntity>>& data)
            : Data(data)
        {
        }

        const TData& Get() const {
            return Data;
        }

        TData& Mutable() {
            return Data;
        }

        auto begin() const {
            return Data.begin();
        }

        auto end() const {
            return Data.end();
        }

        template <class TKey>
        auto find(const TKey& id) const {
            return Data.find(id);
        }

        auto size() const {
            return Data.size();
        }

        template <class TFilter>
        void ApplyFilter(const TFilter& filter, const TSet<TString>* filterIds = nullptr) {
            TSet<TString>::const_iterator it;
            if (filterIds) {
                it = filterIds->begin();
            }
            for (auto dIt = Data.begin(); dIt != Data.end();) {
                if ((filterIds && !AdvanceSimple(it, filterIds->end(), dIt->first)) || !filter(*dIt->second)) {
                    dIt = Data.erase(dIt);
                } else {
                    ++dIt;
                }
            }
        }
    };

public:
    TCachedEntityTagsManager(const TString& tableName, const ITagsHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context)
        , TCacheBase(tableName, context, hConfig)
        , TableName(tableName)
    {
    }

    using TBase::BuildTx;

    const TString& GetTableName() const override {
        return TableName;
    }

    bool GetCurrentSnapshot(TCurrentSnapshot& result, const TInstant reqActuality = TInstant::Zero()) const {
        return GetCurrentSnapshot(result, nullptr, reqActuality);
    }

    bool GetCurrentSnapshot(TCurrentSnapshot& result, const TSet<TString>* ids, const TInstant reqActuality = TInstant::Zero()) const {
        if (!TCacheBase::RefreshCache(reqActuality)) {
            return false;
        }
        TReadGuard rg(MutexSnapshots);
        if (ids && ids->size() < CurrentObjects.size() / 10) {
            for (auto&& i : *ids) {
                auto it = CurrentObjects.find(i);
                if (it != CurrentObjects.end()) {
                    result.Mutable().emplace(i, it->second);
                }
            }
        } else {
            result = CurrentObjects;
        }
        return true;
    }

    bool IsPerformer(const TString& userId) const {
        auto rg = TCacheBase::MakeObjectReadGuard();
        return PerformedTagsByUser.contains(userId);
    }

    bool GetPerformedTags(const TString& userId, TVector<TDBTag>& result, TInstant reqActuality = TInstant::Zero()) const {
        if (!TCacheBase::RefreshCache(reqActuality)) {
            return false;
        }
        auto rg = TCacheBase::MakeObjectReadGuard();
        auto it = PerformedTagsByUser.find(userId);
        if (it == PerformedTagsByUser.end()) {
            return true;
        }
        for (auto&& i : it->second) {
            auto itTag = TagsInfo.find(i);
            if (itTag != TagsInfo.end()) {
                result.emplace_back(itTag->second);
            }
        }
        return true;
    }

    bool GetObjectsFromCache(const TSet<TString>& objectIds, const TSet<TString>& tagNames, TVector<TEntity>& result, const TDuration maxAge) const {
        return GetObjectsFromCache(objectIds, tagNames, result, Now() - maxAge);
    }

    bool GetObjectsFromCache(const TSet<TString>& tagNames, TVector<TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        return GetObjectsFromCache({}, tagNames, result, reqActuality);
    }

    template <class TAction>
    bool GetObjectsFromCacheImpl(const TSet<TString>& objectIds, const TSet<TString>& tagNames, TAction& actionExt, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&tagNames, &actionExt](const TEntity& obj) {
            if (obj.FilterTags(tagNames, nullptr)) {
                actionExt(obj);
            }
        };
        return ForObjectsList(action, reqActuality, objectIds.empty() ? nullptr : &objectIds);
    }

    bool GetObjectsFromCache(const TSet<TString>& objectIds, const TSet<TString>& tagNames, TVector<TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TEntity& obj) {
            result.emplace_back(obj);
        };
        return GetObjectsFromCacheImpl(objectIds, tagNames, action, reqActuality);
    }

    bool GetObjectsFromCache(const TSet<TString>& objectIds, const TSet<TString>& tagNames, TVector<TEntity>& result, const TSet<TString>& tagsForFilter, TInstant reqActuality = TInstant::Zero()) const {
        TVector<TDBTag> tagsFiltered;
        const auto action = [&result, &tagsFiltered, &tagNames, &tagsForFilter](const TEntity& obj) {
            tagsFiltered.clear();
            if (obj.FilterTags(tagNames, nullptr)) {
                obj.FilterTags(tagsForFilter, &tagsFiltered);
                TEntity eFiltered(obj);
                eFiltered.MutableTags() = std::move(tagsFiltered);
                result.emplace_back(std::move(eFiltered));
            }
        };
        return ForObjectsList(action, reqActuality, objectIds.empty() ? nullptr : &objectIds);
    }

    bool GetObjectsFromCacheByIds(const TSet<TString>& objectIds, TVector<TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TEntity& obj) {
            result.emplace_back(obj);
        };
        return ForObjectsList(action, reqActuality, objectIds.empty() ? nullptr : &objectIds);
    }

    bool GetObjectsFromCacheByIds(const TSet<TString>& objectIds, TTaggedObjectsSnapshot& result, TInstant reqActuality = TInstant::Zero()) const {
        TMap<TString, TEntity> map;
        const auto action = [&map](const TEntity& obj) {
            map[obj.GetId()] = obj;
        };
        if (!ForObjectsList(action, reqActuality, objectIds.empty() ? nullptr : &objectIds)) {
            return false;
        }
        result.Reset(std::move(map));
        return true;
    }

    bool GetObjectsFromCache(const TSet<TString>& tagNames, TVector<TEntity>& result, const TSet<TString>& tagsForFilter, TInstant reqActuality = TInstant::Zero()) const {
        return GetObjectsFromCache({}, tagNames, result, tagsForFilter, reqActuality);
    }

    bool GetIdsFromCache(const TSet<TString>& objectIds, const TSet<TString>& tagNames, TSet<TString>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TEntity& obj) {
            result.emplace(obj.GetId());
        };
        return GetObjectsFromCacheImpl(objectIds, tagNames, action, reqActuality);
    }

    bool GetIdsFromCache(const TSet<TString>& objectIds, const TTagsFilter& filter, TSet<TString>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result, &filter](const TEntity& obj) {
            if (filter.IsMatching(obj)) {
                result.emplace(obj.GetId());
            }
        };
        return ForObjectsList(action, reqActuality, objectIds.empty() ? nullptr : &objectIds);
    }

    bool GetIdsFromCache(const TTagsFilter& filter, TSet<TString>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result, &filter](const TEntity& obj) {
            if (filter.IsMatching(obj)) {
                result.emplace(obj.GetId());
            }
        };
        return ForObjectsList(action, reqActuality);
    }

    bool GetIdsFromCache(const TSet<TString>& tagNames, TSet<TString>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TEntity& obj) {
            result.emplace(obj.GetId());
        };
        return GetObjectsFromCacheImpl({}, tagNames, action, reqActuality);
    }

    TExpectedConstTag GetTagFromCache(const TString& objectId, const TString& tagName, TInstant actuality = TInstant::Zero()) const {
        TVector<TEntity> entities;
        if (!GetObjectsFromCache({objectId}, {tagName}, entities, actuality)) {
            return MakeUnexpected(ETagError::QueryError);
        }
        if (entities.empty()) {
            return MakeUnexpected(ETagError::Absent);
        }
        if (entities.size() != 1) {
            return MakeUnexpected(ETagError::MultipleInstances);
        }
        TConstDBTag result;
        for (auto&& tag : entities[0].GetTags()) {
            if (!tag) {
                continue;
            }
            if (tag->GetName() == tagName) {
                if (!result) {
                    result = tag;
                } else {
                    return MakeUnexpected(ETagError::MultipleInstances);
                }
            }
        }
        if (result) {
            return result;
        } else {
            return MakeUnexpected(ETagError::Absent);
        }
    }
    TExpectedConstTag GetTagFromCache(const TString& tagId, const TInstant reqActuality = TInstant::Zero()) const {
        if (!TCacheBase::RefreshCache(reqActuality)) {
            return MakeUnexpected(ETagError::QueryError);
        }
        auto rg = TCacheBase::MakeObjectReadGuard();
        auto tagIt = TagsInfo.find(tagId);
        if (tagIt == TagsInfo.end()) {
            return MakeUnexpected(ETagError::Absent);
        }
        return tagIt->second;
    }

protected:
    virtual const THistoryManagerExternal& GetHistoryManagerImpl() const override {
        return TCacheBase::GetHistoryManager();
    }

    virtual THistoryManagerExternal& GetHistoryManagerImpl() override {
        return TCacheBase::GetHistoryManager();
    }

    virtual TTagsInvIndex::TIndexWriter MutableInvIndexTagName() const {
        return {};
    }

    virtual void OnObjectConstructUnsafe(const TObjectEvent<TConstDBTag>& /*ev*/, const TEntity& /*object*/) const {
    }

    virtual TString GetRebuildRequest() const {
        return "SELECT * FROM " + GetTableName();
    }

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TConstDBTag>& ev) const override {
        return ev.GetTagId();
    }

    void AcceptHistoryEventUnsafe(const TObjectEvent<TConstDBTag>& ev) const override;
    bool DoRebuildCacheUnsafe() const override;

private:
    const TString TableName;

    mutable TMap<TString, TAtomicSharedPtr<TEntity>> CurrentObjects;
    mutable TMap<TString, TSet<TString>> PerformedTagsByUser;
    mutable TMap<TString, TDBTag> TagsInfo;
    TRWMutex MutexSnapshots;
};

class TDeviceTagsManager: public TCachedEntityTagsManager<TTaggedDevice, TCarTagsHistoryManager> {
private:
    using TBase = TCachedEntityTagsManager<TTaggedDevice, TCarTagsHistoryManager>;

private:
    mutable TTagsInvIndex TagNameToObjects;

protected:
    virtual bool DoStart() override {
        return TBase::DoStart() && TagNameToObjects.Start();
    }

    virtual bool DoStop() override {
        return TBase::DoStop() && TagNameToObjects.Stop();
    }

    TTagsInvIndex::TIndexWriter MutableInvIndexTagName() const override {
        return TagNameToObjects.BuildModifier();
    }

    virtual bool CheckInvariants(const TString& objectId, NDrive::TEntitySession& session) const override;
    virtual NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& objectId, const NDrive::IServer* server) const override;
    virtual TString MakeTagPerformerCondition(const TDBTag& tag, const TString& userId, bool force, ui32 lockedLimit, NDrive::TEntitySession& session, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform) const override;

public:
    TDeviceTagsManager(const ITagsHistoryContext& context, const THistoryConfig& hConfig)
        : TBase("car_tags", context, hConfig)
    {
    }

    TMaybe<TTaggedObject> GetCachedOrRestoreObject(const TString& objectId, NDrive::TEntitySession& session) const override;
    TMaybe<TSet<TString>> PrefilterObjects(const TTagsFilter& tagsFilter, const TSet<TString>* objectIds = nullptr, TInstant actuality = TInstant::Zero()) const;

    virtual NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Car;
    }

    virtual TSessionsBuilder<TTagHistoryEvent>::TPtr GetSessionsBuilder(TStringBuf name, TInstant actuality = TInstant::Zero()) const override {
        return GetHistoryManagerImpl().GetSessionsBuilder(name, actuality);
    }
};

class TTraceTagsManager: public TEntityTagsManager<TTaggedTrace, TTraceTagsHistoryManager> {
private:
    using TBase = TEntityTagsManager<TTaggedTrace, TTraceTagsHistoryManager>;

private:
    const TString TableName = "trace_tags";

protected:
    NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& /*objectId*/, const NDrive::IServer* /*server*/) const override {
        return nullptr;
    }

    const THistoryManager& GetHistoryManagerImpl() const override {
        return HistoryManager;
    }

    THistoryManager& GetHistoryManagerImpl() override {
        return HistoryManager;
    }

    const TString& GetTableName() const override {
        return TableName;
    }

    NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Trace;
    }

public:
    TTraceTagsManager(const ITagsHistoryContext& context)
        : TBase(context)
        , HistoryManager(context)
    {
    }

    const THistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }

    using TBase::RestoreTags;

    [[nodiscard]] TOptionalTags RestoreTags(NDrive::TEntitySession& session, TQueryOptions&& queryOptions) const override;

private:
    THistoryManager HistoryManager;
};

class TUserTagsManager
    : public TEntityTagsManager<TTaggedUser, TUserTagsHistoryManager>
    , public TTagCache
{
private:
    using TBase = TEntityTagsManager<TTaggedUser, TUserTagsHistoryManager>;

protected:
    virtual NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& /*objectId*/, const NDrive::IServer* /*server*/) const override {
        return nullptr;
    }

    virtual const THistoryManager& GetHistoryManagerImpl() const override {
        return HistoryManager;
    }

    virtual THistoryManager& GetHistoryManagerImpl() override {
        return HistoryManager;
    }

    virtual const TString& GetTableName() const override {
        return TableName;
    }

    virtual NEntityTagsManager::EMultiplePerformersPolicy GetMultiplePerformersPolicy() const override {
        return NEntityTagsManager::EMultiplePerformersPolicy::Allow;
    }

public:
    TUserTagsManager(const ITagsHistoryContext& context);

    TMaybe<TTaggedObject> GetCachedOrRestoreObject(const TString& objectId, NDrive::TEntitySession& session) const override {
        return GetCachedObject(objectId, session);
    }

    virtual NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::User;
    }

    const THistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }

    [[nodiscard]] bool RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& result, NDrive::TEntitySession& session) const override;

private:
    const TString TableName;

    THistoryManager HistoryManager;
};

class TAccountTagsManager
    : public TEntityTagsManager<TTaggedObject, TAccountTagsHistoryManager>
    , public TTagCache
{
private:
    using TBase = TEntityTagsManager<TTaggedObject, TAccountTagsHistoryManager>;

private:
    const TString TableName;

protected:
    virtual NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& /*objectId*/, const NDrive::IServer* /*server*/) const override {
        return nullptr;
    }

    virtual const THistoryManager& GetHistoryManagerImpl() const override {
        return HistoryManager;
    }

    virtual THistoryManager& GetHistoryManagerImpl() override {
        return HistoryManager;
    }

    virtual const TString& GetTableName() const override {
        return TableName;
    }

    virtual NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Account;
    }

public:
    TAccountTagsManager(const ITagsHistoryContext& context);

    const THistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }

    [[nodiscard]] bool RestoreObjects(const TSet<TString>& objectId, TMap<TString, TTaggedObject>& result, NDrive::TEntitySession& session) const override;

private:
    THistoryManager HistoryManager;
};

class TDBTagMeta {
public:
    using TId = TString;

private:
    TTagDescription::TConstPtr Description;

public:
    TDBTagMeta() = default;
    TDBTagMeta(TTagDescription::TConstPtr description)
        : Description(description)
    {
        Y_ENSURE_BT(Description);
    }

    TTagDescription::TConstPtr GetDescriptionPtr() const {
        return Description;
    }

    const TTagDescription& GetDescription() const {
        Y_ENSURE_BT(Description);
        return *Description;
    }

    bool HasDescription() const {
        return !!Description;
    }

    const TTagDescription* operator->() const {
        return Description.Get();
    }

    explicit operator bool() const {
        return HasDescription();
    }

public:
    const TString& GetInternalId() const {
        return GetDescription().GetName();
    }

    static TString GetTableName() {
        return "tags_description";
    }

    static TString GetPropositionsTableName() {
        return GetTableName() + "_propositions";
    }

    static TString GetHistoryTableName() {
        return GetTableName() + "_standart_history";
    }

    class TDecoder: public TTagDescription::TDecoder {
    private:
        using TBase = TTagDescription::TDecoder;

    private:
        R_FIELD(i32, Type, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase)
            : TBase(decoderBase)
        {
            Type = GetFieldDecodeIndex("type", decoderBase);
        }
    };

    const TMaybe<ui32>& OptionalRevision() const {
        return GetDescription().OptionalRevision();
    }

    NStorage::TTableRecord SerializeToTableRecord() const;
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);

    NJson::TJsonValue BuildJsonReport() const;
};

class TDBTagMetaConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session);
    static NStorage::TTableRecord BuildCondition(const TString& id);
    static NStorage::TTableRecord BuildCondition(const TDBTagMeta& object);
};

class TDBTagMetaPropositions: public TPropositionsManager<TDBTagMeta> {
private:
    using TBase = TPropositionsManager<TDBTagMeta>;

public:
    TDBTagMetaPropositions(const IHistoryContext& context, const TPropositionsManagerConfig& config)
        : TBase(context, TDBTagMeta::GetPropositionsTableName(), config)
    {
    }
};

class TTagsMeta final
    : public TDBEntitiesManager<TDBTagMeta, TDBTagMetaConditionConstructor>
    , public IDBEntitiesWithPropositionsManager<TDBTagMeta>
    , public ITagsMeta
    , public ITagsHistoryContext
{
public:
    using TBase = TDBEntitiesManager<TDBTagMeta, TDBTagMetaConditionConstructor>;

private:
    THolder<TDBTagMetaPropositions> Propositions;
    mutable TMap<ui32, TDBTagMeta> ObjectsByIndex;

private:
    bool RegisterTagImpl(TTagDescription::TConstPtr description, const TString& userId, NDrive::TEntitySession& session, bool preserveType) const;

protected:
    bool DoRebuildCacheUnsafe() const override;

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TDBTagMeta>& ev) const override {
        return ev->GetName();
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TDBTagMeta>& ev) const override;

public:
    virtual TSet<TString> GetDependencesAfter() const override {
        return {"drive_actions_standart_history"};
    }

    TTagsMeta(const IHistoryContext& historyContext, const THistoryConfig& config);
    ~TTagsMeta();

    virtual EUniquePolicy GetTagUniquePolicy(ITag::TConstPtr tag, TTagDescription::TConstPtr description, const EUniquePolicy overridePolicy) const override;

    virtual const ITagsMeta& GetTagsManager() const override {
        return *this;
    }

    virtual NStorage::IDatabase::TPtr GetDatabase() const override {
        return HistoryCacheDatabase;
    }

    const IDBEntitiesWithPropositionsManager<TDBTagMeta>& GetEntityManager() const override {
        return *this;
    }

    const TPropositionsManager<TDBTagMeta>* GetPropositions() const override {
        Y_ENSURE_BT(Propositions);
        return Propositions.Get();
    }

    const TPropositionsManager<TDBTagMeta>& GetPropositionManager() const override {
        Y_ENSURE_BT(Propositions);
        return *Propositions;
    }

    void InitPropositions(const TPropositionsManagerConfig& propositionsConfig) {
        Y_ENSURE_BT(!Propositions);
        Propositions = MakeHolder<TDBTagMetaPropositions>(GetHistoryManager().GetContext(), propositionsConfig);
    }

    virtual TTagDescriptions GetTagsByType(const TString& tagType, const TInstant reqActuality = TInstant::Zero()) const override;
    virtual TTagDescription::TConstPtr GetDescriptionByName(const TString& name, const TInstant reqActuality = TInstant::Zero()) const override;
    virtual TTagDescription::TConstPtr GetDescriptionByIndex(ui32 index) const override;
    virtual TString GetTagTypeByName(const TString& name, const TInstant reqActuality = TInstant::Zero()) const override;

    virtual TOptionalTagDescription GetDescription(const TString& name, NDrive::TEntitySession& tx) const override;
    virtual TTagDescriptionsByName GetRegisteredTags(const TInstant reqActuality = TInstant::Zero()) const override;
    virtual TTagDescriptionsByName GetRegisteredTags(NEntityTagsManager::EEntityType type, const TSet<TString>& tagType = {}, const TInstant actuality = TInstant::Zero()) const override;
    virtual TSet<TString> GetRegisteredTagNames(const TSet<TString>& tagTypes, const TInstant reqActuality = TInstant::Zero()) const override;

    virtual ITag::TPtr CreateTag(const TString& name, const TString& comment = {}, TInstant reqActuality = TInstant::Zero()) const override;
    virtual bool RegisterTag(TTagDescription::TConstPtr description, const TString& userId, NDrive::TEntitySession& session, bool force = false) const override;
    virtual bool UnregisterTag(const TString& tagName, const TString& userId, NDrive::TEntitySession& session) const override;
};

class TDriveTagsManager
   : public ITagsHistoryContext
   , public IDriveTagsManager
   , public TDatabaseSessionConstructor
{
private:
    THolder<TTagsMeta> TagsMeta;
    THolder<TDeviceTagsManager> DeviceTags;
    THolder<TTraceTagsManager> TraceTags;
    THolder<TUserTagsManager> UserTags;
    THolder<TAccountTagsManager> AccountTags;
    THolder<THistoryContext> HistoryContext;
    const TPropositionsManagerConfig DeviceTagPropositionsConfig;

public:
    TDriveTagsManager(NStorage::IDatabase::TPtr database, const TTagsManagerConfig& daConfig);
    ~TDriveTagsManager() = default;

    virtual const ITagsHistoryContext& GetContext() const override;
    virtual const IEntityTagsManager& GetEntityTagManager(NEntityTagsManager::EEntityType entityType) const override;

    virtual const ITagsMeta& GetTagsManager() const override {
        return *TagsMeta;
    }
    virtual const ITagsMeta& GetTagsMeta() const override {
        return *TagsMeta;
    }

    virtual bool DoStop() override;
    virtual bool DoStart() override;

    virtual NStorage::IDatabase::TPtr GetDatabase() const override {
        return Database;
    }

    virtual const TDeviceTagsManager& GetDeviceTags() const override {
        return *DeviceTags;
    }

    virtual const TTraceTagsManager& GetTraceTags() const override {
        return *TraceTags;
    }

    virtual const TUserTagsManager& GetUserTags() const override {
        return *UserTags;
    }

    virtual const TAccountTagsManager& GetAccountTags() const override {
        return *AccountTags;
    }
};
