#pragma once

#include "tags_manager.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/logging/evlog.h>

class TTxTagHistoryContext: public ITagsHistoryContext {
public:
    TTxTagHistoryContext(const ITagsHistoryContext& ctx, NDrive::TEntitySession& tx)
        : Ctx(ctx)
        , Tx(tx)
    {
    }

    NDrive::TEntitySession* GetCurrentTx() const override {
        return &Tx;
    }
    TAtomicSharedPtr<NStorage::IDatabase> GetDatabase() const override {
        return Ctx.GetDatabase();
    }
    const ITagsMeta& GetTagsManager() const override {
        return Ctx.GetTagsManager();
    }

private:
    const ITagsHistoryContext& Ctx;
    NDrive::TEntitySession& Tx;
};

template <class TEntity, class THistoryManagerExternal>
class TEntityTagsManager<TEntity, THistoryManagerExternal>::TTagEntityManager: public TBaseEntityManager<TDBTag> {
private:
    using TBase = TBaseEntityManager<TDBTag>;
    using TParent = TEntityTagsManager<TEntity, THistoryManagerExternal>;

public:
    TTagEntityManager(const TParent& parent)
        : TBase(parent.GetDatabasePtr())
        , Parent(parent)
    {
    }

    TString GetTableName() const override {
        return Parent.GetTableName();
    }

private:
    const TParent& Parent;
};

template <class TEntity, class THistoryManagerExternal>
TEntityTagsManager<TEntity, THistoryManagerExternal>::TEntityTagsManager(const ITagsHistoryContext& context)
    : TBase(context)
    , EntityManager(MakeHolder<TTagEntityManager>(*this))
{
}

template <class TEntity, class THistoryManagerExternal>
TEntityTagsManager<TEntity, THistoryManagerExternal>::~TEntityTagsManager() {
}

template <class TEntity, class THistoryManagerExternal>
IEntityTagsManager::TOptionalTag TEntityTagsManager<TEntity, THistoryManagerExternal>::AddSnapshot(const TDBTag& tag, NDrive::IObjectSnapshot::TPtr snapshot, const TString& userId, NDrive::TEntitySession& session) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "AddSnapshot")
            ("tag", NJson::ToJson(tag))
            ("snapshot", snapshot ? snapshot->SerializeToJson() : NJson::JSON_NULL)
            ("user_id", userId)
        );
    }

    auto result = tag;
    result->SetObjectSnapshot(snapshot);
    if (!GetHistoryManagerImpl().AddHistory(NContainer::Scalar(tag), userId, EObjectHistoryAction::AddSnapshot, session)) {
        return {};
    }
    return result;
}

template <class TEntity, class THistoryManagerExternal>
IEntityTagsManager::TOptionalTags TEntityTagsManager<TEntity, THistoryManagerExternal>::AddTags(TConstArrayRef<ITag::TPtr> tags, const TString& userId, const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session, EUniquePolicy uniquePolicy, const TString& filter) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "AddTags")
            ("tags", NJson::ToJson(tags))
            ("object_id", objectId)
            ("user_id", userId)
            ("filter", filter)
            ("unique_policy", ToString(uniquePolicy))
        );
    }

    if (tags.empty()) {
        return TVector<TDBTag>();
    }

    auto effectiveFilter = filter;
    if (!effectiveFilter) {
        auto optionalFilter = GetDefaultFilter(objectId, server, session);
        if (!optionalFilter) {
            return {};
        }
        effectiveFilter = std::move(*optionalFilter);
        if (evlog && effectiveFilter) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "GetDefaultFilter")
                ("filter", effectiveFilter)
            );
        }
    }

    auto tagsTable = Database->GetTable(GetTableName());
    auto transaction = session.GetTransaction();

    TQueryResultPtr queryResult;

    for (auto&& tag : tags) {
        Y_ENSURE(tag);
        if (!tag->GetObjectType().contains(GetEntityType())) {
            session.SetErrorInfo("add_tag", tag->GetName() + "(" + JoinSeq(", ", tag->GetObjectType()) + " / " + ToString(GetEntityType()) + ")", EDriveSessionResult::IncorrectTagObjectType);
            return {};
        }
        if (!tag->OnBeforeAdd(objectId, userId, server, session)) {
            return {};
        }
    }

    TVector<TDBTag> result;
    TVector<TDBTag> tagsAdded;
    TVector<TDBTag> tagsRewritten;
    auto snapshot = BuildSnapshot(objectId, server);
    for (auto&& tag : tags) {
        auto description = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
        if (description) {
            if (!tag->HasTagPriority()) {
                tag->SetTagPriority(description->GetDefaultPriority());
            }
            if (tag && !tag->HasSLAInstant() && description->HasSLADuration()) {
                tag->SetSLAInstant(ModelingNow() + description->GetSLADurationUnsafe());
            }
        }
        if (!tag->GetObjectSnapshot()) {
            tag->SetObjectSnapshot(snapshot);
        }
        NStorage::TTableRecord record = tag->SerializeToTableRecord();
        record.Set("object_id", objectId);
        if (effectiveFilter) {
            record.Set("filter", effectiveFilter);
        }
        bool isUpdate = false;
        TTxTagHistoryContext ctx(TagsHistoryContext, session);
        NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> records(&ctx);
        const EUniquePolicy policy = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetTagUniquePolicy(tag, description, uniquePolicy);
        switch (policy) {
            case EUniquePolicy::NoUnique: {
                queryResult = tagsTable->AddRow(record, transaction, "", &records);
                break;
            }
            case EUniquePolicy::SkipIfExists: {
                NStorage::TTableRecord uniqueConstraint;
                uniqueConstraint.Set("tag", tag->GetName());
                uniqueConstraint.Set("object_id", objectId);
                queryResult = tagsTable->AddIfNotExists(record, transaction, uniqueConstraint, &records);
                break;
            }
            case EUniquePolicy::Rewrite: {
                NStorage::TTableRecord uniqueConstraint;
                uniqueConstraint.Set("tag", tag->GetName());
                uniqueConstraint.Set("object_id", objectId);
                uniqueConstraint.Set("performer", "");
                queryResult = tagsTable->UpdateRow(uniqueConstraint, record, transaction, &records);
                if (records.size() == 0) {
                    uniqueConstraint.Remove("performer");
                    queryResult = tagsTable->AddIfNotExists(record, transaction, uniqueConstraint, &records);
                } else {
                    isUpdate = true;
                }
                break;
            }
            default:
                session.SetErrorInfo("AddTags", "incorrect unique policy: " + ::ToString(policy), EDriveSessionResult::InternalError);
                return {};
        }

        if (!queryResult->IsSucceed()) {
            session.SetErrorInfo("AddTags", "query failed", EDriveSessionResult::TransactionProblem);
            return {};
        }

        if (records.size() != queryResult->GetAffectedRows()) {
            session.SetErrorInfo("AddTags", "Parsing problems", EDriveSessionResult::InternalError);
            return {};
        }

        for (auto&& dbTag : records) {
            result.push_back(dbTag);
            if (!isUpdate) {
                tagsAdded.emplace_back(std::move(dbTag));
            } else {
                tagsRewritten.push_back(std::move(dbTag));
            }
        }
    }
    if (!CheckInvariants(objectId, session)) {
        return {};
    }
    if (!GetHistoryManagerImpl().AddHistory(tagsAdded, userId, EObjectHistoryAction::Add, session)) {
        return {};
    }
    if (!GetHistoryManagerImpl().AddHistory(tagsRewritten, userId, EObjectHistoryAction::UpdateData, session)) {
        return {};
    }

    for (auto&& tag : tagsAdded) {
        if (!tag->OnAfterAdd(tag, userId, server, session)) {
            return {};
        }
    }

    return result;
}

template <class TEntity, class THistoryManagerExternal>
IEntityTagsManager::TOptionalTags TEntityTagsManager<TEntity, THistoryManagerExternal>::RestoreTags(NDrive::TEntitySession& session, TQueryOptions&& queryOptions) const {
    if (queryOptions.GetPerformers()) {
        queryOptions.SetGenericCondition("performer", queryOptions.GetPerformers().BuildCondition());
    }
    if (queryOptions.GetTagIds()) {
        queryOptions.SetGenericCondition("tag_id", queryOptions.GetTagIds().BuildCondition());
    }
    if (queryOptions.GetTags()) {
        queryOptions.SetGenericCondition("tag", queryOptions.GetTags().BuildCondition());
    }
    if (queryOptions.GetObjectIds()) {
        queryOptions.SetGenericCondition("object_id", queryOptions.GetObjectIds().BuildCondition());
    }
    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    return EntityManager->Fetch(session, queryOptions, &ctx);
}

template <class TEntity, class THistoryManagerExternal>
IEntityTagsManager::TOptionalTag TEntityTagsManager<TEntity, THistoryManagerExternal>::DirectEvolveTag(const TString& userId, const TDBTag& fromTag, ITag::TPtr toTag, NDrive::TEntitySession& session) const {
    if (!toTag) {
        session.SetErrorInfo("DirectEvolveTag", "incorrect dest tag for evolution", EDriveSessionResult::InconsistencySystem);
        return {};
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "DirectEvolveTag")
            ("user_id", userId)
        );
    }

    NStorage::TTableRecord condition;
    condition.Set("tag_id", fromTag.GetTagId());
    condition.Set("tag", fromTag->GetName());
    condition.Set("performer", fromTag->GetPerformer());

    NStorage::TTableRecord update(toTag->SerializeToTableRecord());

    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    NStorage::TObjectRecordsSet<TDBTag, ITagsHistoryContext> records(&ctx);
    auto tagsTable = Database->GetTable(GetTableName());
    auto transaction = session.GetTransaction();
    auto queryResult = tagsTable->UpdateRow(condition, update, transaction, &records);
    if (queryResult->IsSucceed()) {
        if (records.size() != 1) {
            session.SetErrorInfo("EvolveTag", "can't evolve tag " + fromTag.GetTagId(), EDriveSessionResult::InconsistencyUser);
            return {};
        }
    } else {
        session.SetErrorInfo("EvolveTag", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
        return {};
    }
    auto tags = records.DetachObjects();
    if (!GetHistoryManagerImpl().AddHistory(tags, userId, EObjectHistoryAction::TagEvolve, session)) {
        return {};
    }

    Y_ASSERT(tags.size() == 1);
    Y_ASSERT(tags[0].GetObjectId() == fromTag.GetObjectId());
    Y_ASSERT(tags[0].GetTagId() == fromTag.GetTagId());
    Y_ASSERT(tags[0]->GetName() == toTag->GetName());

    auto result = fromTag;
    result.SetData(toTag);
    return result;
}

template <class TEntity, class THistoryManagerExternal>
bool TEntityTagsManager<TEntity, THistoryManagerExternal>::DropTagsPerformer(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, bool force) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "DropTagsPerformer")
            ("tags", NJson::ToJson(tags))
            ("user_id", userId)
            ("force", force)
        );
    }

    if (tags.empty()) {
        return true;
    }
    TSet<TString> tagsSet;
    for (auto&& i : tags) {
        tagsSet.emplace(i.GetTagId());
    }

    auto transaction = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());
    const TString update = "performer=''";

    const TString tagsSetStr = session->Quote(tagsSet);
    TString condition;
    if (force) {
        condition = "tag_id IN (" + tagsSetStr + ")";
    } else {
        condition = "tag_id IN (" + tagsSetStr + ") AND NOT EXISTS (SELECT 1 FROM " + GetTableName() + " WHERE tag_id IN (" + tagsSetStr + ") AND performer!='" + userId + "' AND performer!='')";
    }

    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    NStorage::TObjectRecordsSet<TDBTag, ITagsHistoryContext> tagsAffected(&ctx);
    TQueryResultPtr queryResult = tagsTable->UpdateRow(condition, update, transaction, &tagsAffected);
    if (queryResult->IsSucceed()) {
        if (queryResult->GetAffectedRows() != tagsSet.size()) {
            session.SetErrorInfo("DropTagsPerformer", "can't set tags {" + tagsSetStr + "} performer", EDriveSessionResult::TransactionProblem);
            return false;
        }
    } else {
        session.SetErrorInfo("DropTagsPerformer", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    for (auto&& tag : tags) {
        if (!CheckInvariants(tag.GetObjectId(), session)) {
            return false;
        }
    }
    if (!GetHistoryManagerImpl().AddHistory(tagsAffected.GetObjects(), userId, EObjectHistoryAction::DropTagPerformer, session)) {
        return false;
    }

    return true;
}

template <class TEntity, class THistoryManagerExternal>
bool TEntityTagsManager<TEntity, THistoryManagerExternal>::RemoveTagsSimple(TConstArrayRef<TDBTag> tags, const TString& userId, NDrive::TEntitySession& session, bool force, bool tryRemove) const {
    if (tags.empty()) {
        return true;
    }

    auto tagsTable = Database->GetTable(GetTableName());
    auto transaction = session.GetTransaction();

    TStringStream ss;
    ui32 idx = 0;
    for (auto&& tag : tags) {
        if (idx++) {
            ss << ",";
        }
        ss << "'" << tag.GetTagId() << "'";
    }

    const TString condition = "tag_id IN ( " + ss.Str() + ") " + (force ? "" : (" AND (performer = '' OR performer = '" + userId + "')"));

    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> tagsReallyRemoved(&ctx);
    TQueryResultPtr queryResult = tagsTable->RemoveRow(condition, transaction, &tagsReallyRemoved);
    if (!queryResult->IsSucceed()) {
        session.SetErrorInfo("RemoveTags", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (queryResult->GetAffectedRows() != tags.size() && !tryRemove) {
        session.SetErrorInfo("RemoveTags", Sprintf("locked tags/%u/%lu", queryResult->GetAffectedRows(), tags.size()), EDriveSessionResult::InconsistencyUser);
        return false;
    }
    TMap<TString, NDrive::IObjectSnapshot::TPtr> snapshots;
    for (auto&& i : tags) {
        snapshots.emplace(i.GetTagId(), i->GetObjectSnapshot());
    }
    for (auto&& i : tagsReallyRemoved) {
        auto it = snapshots.find(i.GetTagId());
        if (it != snapshots.end()) {
            i->SetObjectSnapshot(it->second);
        }
    }
    if (!GetHistoryManagerImpl().AddHistory(tagsReallyRemoved.GetObjects(), userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }

    return true;
}

template <class TEntity, class THistoryManagerExternal>
bool TEntityTagsManager<TEntity, THistoryManagerExternal>::SetTagsPerformer(TArrayRef<TDBTag> tags, const TString& userId, bool force, NDrive::TEntitySession& session, const NDrive::IServer* server, ui32 lockedLimit, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "SetTagsPerformer")
            ("force", force)
            ("locked_limit", lockedLimit)
            ("override_multiperform", overrideMultiPerform ? NJson::TJsonValue{ToString(overrideMultiPerform)} : NJson::JSON_NULL)
            ("tag", NJson::ToJson(tags))
            ("user_id", NJson::ToJson(userId))
        );
    }

    if (tags.empty()) {
        return true;
    }
    auto transaction = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());

    for (auto&& i : tags) {
        if (i->GetPerformer() == userId) {
            continue;
        }

        const TString update = "performer='" + userId + "', data = '" + i->GetStringData() + "', snapshot='" + NDrive::ITag::GetStringSnapshot(BuildSnapshot(i.GetObjectId(), server)) + "'";
        const TString condition = MakeTagPerformerCondition(i, userId, force, lockedLimit, session, overrideMultiPerform);

        TTxTagHistoryContext ctx(TagsHistoryContext, session);
        NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> updatedRecords(&ctx);
        TQueryResultPtr queryResult = tagsTable->UpdateRow(condition, update, transaction, &updatedRecords);
        if (queryResult->IsSucceed()) {
            if (queryResult->GetAffectedRows() != 1 || updatedRecords.GetObjects().size() != 1) {
                session.SetErrorInfo("SetTagsPerformer", "can't set tags {" + i.GetTagId() + "} performer (check limits and abilities)", EDriveSessionResult::ResourceLocked);
                return false;
            }
        } else {
            session.SetErrorInfo("SetTagsPerformer", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!GetHistoryManagerImpl().AddHistory(updatedRecords.GetObjects(), userId, i->IsBusyFor(userId) ? EObjectHistoryAction::ForceTagPerformer : EObjectHistoryAction::SetTagPerformer, session)) {
            return false;
        }
    }
    for (auto&& i : tags) {
        i->SetPerformer(userId);
    }
    for (auto&& tag : tags) {
        if (!CheckInvariants(tag.GetObjectId(), session)) {
            return false;
        }
    }
    return true;
}

template <class TEntity, class THistoryManagerExternal>
bool TEntityTagsManager<TEntity, THistoryManagerExternal>::UpdateTagExtCondition(const TDBTag& tag, const TString& userId, const TConditionBuilder& conditionBuilder, const TUpdateBuilder& updateBuilder, NDrive::TEntitySession& session) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "UpdateTagExtCondidition")
            ("tag", NJson::ToJson(tag))
            ("user_id", NJson::ToJson(userId))
        );
    }
    auto transaction = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());

    TRecordsSet updatedTags;
    TQueryResultPtr queryResult = tagsTable->UpdateRow(conditionBuilder(tag), updateBuilder(tag), transaction, &updatedTags);
    if (queryResult->IsSucceed()) {
        if (queryResult->GetAffectedRows() != 1 || updatedTags.size() != 1) {
            session.SetErrorInfo("UpdateTagData", "cannot update tag data {" + tag.GetTagId() + "}", EDriveSessionResult::InternalError);
            return true;
        }
    } else {
        session.SetErrorInfo("UpdateTagData", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    TConstDBTag updatedTag;
    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    if (!TConstDBTag::TDecoder::DeserializeFromTableRecordCommon(updatedTag, *updatedTags.begin(), &ctx, true)) {
        session.SetErrorInfo("UpdateTagData", "could not deserialize updated tag", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    if (!GetHistoryManagerImpl().AddHistory(updatedTag, userId, EObjectHistoryAction::UpdateData, session)) {
        return false;
    }

    return true;
}

template <class TEntity, class THistoryManagerExternal>
bool TEntityTagsManager<TEntity, THistoryManagerExternal>::GetObjects(const TSet<TString>& objectIds, const TSet<TString>& tagNames, bool getAllTags, TVector<TEntity>& result, NDrive::TEntitySession& session) const {
    result.clear();
    if (tagNames.empty()) {
        return true;
    }
    TTransactionPtr transaction = session.GetTransaction();
    auto tagsTable = Database->GetTable(GetTableName());

    TQueryResultPtr queryResult;
    TStringStream condition = "tag IN (" + session->Quote(tagNames) + ")";

    if (!objectIds.empty()) {
        condition << " AND object_id IN (" << session->Quote(objectIds) << ")";
    }

    TTxTagHistoryContext ctx(TagsHistoryContext, session);
    NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> tags(&ctx);

    auto timestamp = Now();
    if (getAllTags) {
        TStringStream conditionFull;
        conditionFull << "SELECT a.* FROM " << GetTableName() << " AS a INNER JOIN (SELECT object_id FROM " << GetTableName()
                        << " WHERE " << condition.Str() << " GROUP BY object_id) as b USING(object_id) ";
        queryResult = transaction->Exec(conditionFull.Str(), &tags);
    } else {
        queryResult = tagsTable->GetRows(condition.Str(), tags, transaction);
    }

    if (!queryResult->IsSucceed()) {
        return false;
    }

    {
        const auto pred = [](const TDBTag& l, const TDBTag& r) -> bool {
            return l.GetObjectId() < r.GetObjectId();
        };
        std::sort(tags.begin(), tags.end(), pred);

        for (auto&& tag : tags) {
            if (result.empty() || result.back().GetId() != tag.GetObjectId()) {
                result.emplace_back(tag.GetObjectId(), TDBTags{}, timestamp);
            }
            result.back().MutableTags().emplace_back(std::move(tag));
        }
    }
    return true;
}

template <class TEntity, class THistoryManagerExternal>
TString TEntityTagsManager<TEntity, THistoryManagerExternal>::MakeTagPerformerCondition(const TDBTag& tag, const TString& userId, bool force, ui32 lockedLimit, NDrive::TEntitySession& session, TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> overrideMultiPerform) const {
    auto condition = TStringBuilder() << "tag_id = " << session->Quote(tag.GetTagId());
    if (overrideMultiPerform.GetOrElse(GetMultiplePerformersPolicy()) == NEntityTagsManager::EMultiplePerformersPolicy::Deny) {
        if (force) {
            condition << " AND NOT EXISTS (SELECT 1 FROM " << GetTableName()
                << " WHERE tag_id != " << session->Quote(tag.GetTagId())
                << " AND object_id = " << session->Quote(tag.GetObjectId())
                << " AND performer != " << session->Quote(userId)
                << " AND performer != '') ";
        } else {
            condition << " AND NOT EXISTS (SELECT 1 FROM " << GetTableName()
                << " WHERE performer != " << session->Quote(userId)
                << " AND object_id = " << session->Quote(tag.GetObjectId())
                << " AND performer != '') ";
        }
    }
    if (lockedLimit) {
        condition << " AND ((SELECT COUNT(DISTINCT object_id) FROM " << GetTableName()
            << " WHERE performer = " << session->Quote(userId)
            << " AND object_id != " << session->Quote(tag.GetObjectId())
            << ") <= " << lockedLimit - 1 << ")";
    }
    return condition;
}

template <class TEntity, class THistoryManagerExternal>
void TCachedEntityTagsManager<TEntity, THistoryManagerExternal>::AcceptHistoryEventUnsafe(const TObjectEvent<TConstDBTag>& ev) const {
    auto it = Objects.find(ev.TConstDBTag::GetObjectId());
    if (it == Objects.end()) {
        if (ev.GetHistoryAction() != EObjectHistoryAction::Remove) {
            auto entity = MakeAtomicShared<TEntity>(ev.TConstDBTag::GetObjectId(), TDBTags{}, TInstant::Zero());
            CHECK_WITH_LOG(entity);
            auto modifier = MutableInvIndexTagName();
            entity->RefreshTag(ev, TagsHistoryContext, &PerformedTagsByUser, &modifier, &TagsInfo);
            Objects.emplace(ev.TConstDBTag::GetObjectId(), *entity);
            OnObjectConstructUnsafe(ev, *entity);

            TWriteGuard rg(MutexSnapshots);
            CurrentObjects.emplace(ev.TConstDBTag::GetObjectId(), std::move(entity));
        }
    } else {
        auto modifier = MutableInvIndexTagName();
        auto& entity = it->second;
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            bool removed = entity.RemoveTag(ev, &PerformedTagsByUser, &modifier, &TagsInfo);
            Y_UNUSED(removed);
        } else {
            bool refreshed = entity.RefreshTag(ev, TagsHistoryContext, &PerformedTagsByUser, &modifier, &TagsInfo);
            Y_UNUSED(refreshed);
        }
        OnObjectConstructUnsafe(ev, entity);
        auto current = MakeAtomicShared<TEntity>(entity);
        TWriteGuard rg(MutexSnapshots);
        CurrentObjects[it->first] = std::move(current);
    }
}

template <class TEntity, class THistoryManagerExternal>
bool TCachedEntityTagsManager<TEntity, THistoryManagerExternal>::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TDBTag, const ITagsHistoryContext> tags(&TagsHistoryContext);
    {
        auto session = TBase::template BuildTx<NSQL::ReadOnly>();
        auto tagsTable = Database->GetTable(GetTableName());
        auto request = GetRebuildRequest();
        if (!request) {
            return true;
        }

        auto queryResult = session->Exec(GetRebuildRequest(), &tags);
        if (!queryResult || !queryResult->IsSucceed()) {
            return false;
        }
    }

    {
        const auto pred = [](const TDBTag& l, const TDBTag& r) -> bool {
            return l.GetObjectId() < r.GetObjectId();
        };
        std::sort(tags.begin(), tags.end(), pred);

        TEntity* entity = nullptr;

        Objects.clear();
        PerformedTagsByUser.clear();
        auto modifier = MutableInvIndexTagName();
        for (auto&& tag : tags) {
            TagsInfo.emplace(tag.GetTagId(), tag);
            if (!entity || entity->GetId() != tag.GetObjectId()) {
                TEntity entityLocal(tag.GetObjectId(), {}, TInstant::Zero());
                entity = &Objects.emplace(tag.GetObjectId(), std::move(entityLocal)).first->second;
            }
            CHECK_WITH_LOG(entity);
            modifier.AddPosition(tag->GetName(), tag.GetObjectId());
            if (!!tag->GetPerformer()) {
                PerformedTagsByUser[tag->GetPerformer()].emplace(tag.GetTagId());
            }
            entity->MutableTags().emplace_back(std::move(tag));
        }
    }
    CurrentObjects.clear();
    for (auto&& [id, entity] : Objects) {
        CurrentObjects.emplace(id, MakeAtomicShared<TEntity>(entity));
    }
    return true;
}
