#pragma once

#include "tags.h"
#include "tags_manager.h"

#include <rtline/library/unistat/signals.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>
#include <util/string/vector.h>

namespace NMatchCheck {
    using TCheckPolicy = ui32;
    enum ECheckPolicy : TCheckPolicy {
        CheckRequired = 1 << 0,
        CheckForbidden = 1 << 1,
        CheckPerformer = 1 << 2,
        CheckOneOfTags = 1 << 3
    };
    static constexpr TCheckPolicy CheckAll = ECheckPolicy::CheckRequired | ECheckPolicy::CheckForbidden | ECheckPolicy::CheckPerformer | ECheckPolicy::CheckOneOfTags;
};

class TTagsSearchResult {
private:
    using TTags = TMap<TString, TVector<TConstDBTag>>;

private:
    R_FIELD(TVector<TString>, MatchedIds);
    R_OPTIONAL(size_t, TotalMatched);
    R_READONLY(TTags, AssociatedTags);

public:
    TTagsSearchResult() = default;
    TTagsSearchResult(const ui64 resultsLimit)
        : ResultsLimit(resultsLimit)
    {
    }

    template<class T>
    void AddMatchedId(const TString& id, T&& associatedTags) {
        if (!AssociatedTags.contains(id) && MatchedIds.size() < ResultsLimit) {
            MatchedIds.emplace_back(id);
            AssociatedTags.emplace(id, associatedTags);
        }
    }

    bool IsLimitReached() const;

    template<class T>
    NJson::TJsonValue BuildJsonReport(const T* dbEntities, bool withTags = true) const {
        Y_ENSURE(dbEntities);
        auto fetchResult = dbEntities->FetchInfo(MatchedIds);
        NJson::TJsonValue resultEntities = NJson::JSON_ARRAY;
        for (auto&& id : MatchedIds) {
            NJson::TJsonValue singleReport;
            auto ptr = fetchResult.GetResultPtr(id);
            if (ptr) {
                singleReport = ptr->GetSearchReport();
            } else {
                singleReport["id"] = id;
            }
            if (withTags) {
                NJson::TJsonValue tagsList = NJson::JSON_ARRAY;
                auto associatedTagsIt = AssociatedTags.find(id);
                if (associatedTagsIt != AssociatedTags.end()) {
                    for (auto&& tag : associatedTagsIt->second) {
                        auto tagReport = tag.BuildJsonReport();
                        if (tag->GetPerformer()) {
                            tagReport["performer"] = tag->GetPerformer();
                        } else {
                            tagReport["performer"] = NJson::JSON_NULL;
                        }
                        tagsList.AppendValue(std::move(tagReport));
                    }
                }
                singleReport["tags"] = std::move(tagsList);
            }
            resultEntities.AppendValue(std::move(singleReport));
        }
        return resultEntities;
    }

private:
    ui64 ResultsLimit = 100;
};

class TTagsSearchRequest {
    R_READONLY(TVector<TString>, HasAllOf);
    R_READONLY(TVector<TString>, HasNoneOf);
    R_READONLY(TVector<TString>, HasOneOf);
    R_READONLY(ui64, Limit, 100);
    R_OPTIONAL(TString, Performer);

public:
    TTagsSearchRequest() = default;
    TTagsSearchRequest(const TVector<TString>& hasAllOf, const TVector<TString>& hasNoneOf, const TVector<TString>& hasOneOf, const ui64 limit = 100)
        : HasAllOf(hasAllOf)
        , HasNoneOf(hasNoneOf)
        , HasOneOf(hasOneOf)
        , Limit(limit)
    {
    }

    bool IsSimpleQuery() const;

    TVector<TString> GetObservedTagNames() const;

    void Prepare(const ITagsMeta& tagsMeta);

    bool DeserializeFromJson(const NJson::TJsonValue& json);
    NJson::TJsonValue SerializeToJson() const;

    static NDrive::TScheme GetScheme(const IServerBase& /*server*/);

private:
    TVector<TString> ResolveSpecials(const TVector<TString>& rawTagsList, const ITagsMeta& tagsMeta) const;
};

class TTagsSearch {
private:
    using TBase = IAutoActualization;

public:
    struct TTagStat {
        size_t Count = 0;
        TInstant Timestamp;
    };

private:
    mutable NUtil::TConcurrentCache<TString, TTagStat> TagStatCache;

    const IEntityTagsManager& ObservedObject;
    const ITagsMeta& TagsMeta;

    TUnistatSignal<> SearchCount;
    TUnistatSignal<> SearchErrors;
    TUnistatSignal<> SearchTimes;

public:
    TTagsSearch(const IEntityTagsManager& observedObject, const IDriveTagsManager& tagsManager);

    ui64 Count(NDrive::TEntitySession& tx, const NSQL::TQueryOptions& queryOptions) const;
    ui64 GetCachedCount(const TString& tag, NDrive::TEntitySession& tx) const;

    TTagsSearchResult Search(TTagsSearchRequest& request, TDuration timeout = TDuration::Zero()) const;
    TTagsSearchResult Search(TTagsSearchRequest& request, NDrive::TEntitySession& tx) const;

private:
    TTagsSearchResult Search2(TTagsSearchRequest& request, NDrive::TEntitySession& tx) const;
};
