#include "matcher.h"

#include <library/cpp/json/writer/json_value.h>

#include <rtline/library/json/parse.h>

NDrive::TMatcherEvent::TMatcherEvent(const TObjectEvent<TDriveCarInfo>& event)
    : HistoryEventId(event.GetHistoryEventId())
    , HistoryAction(event.GetHistoryAction())
    , ObjectId(event.GetId())
    , Imei(event.GetIMEI())
    , Timestamp(event.GetHistoryTimestamp())
    {}

NDrive::TMatcherEvent::TMatcherEvent(i64 historyEventId, EObjectHistoryAction action, const TString& objectId, const TString& imei, const TInstant timestamp) 
    : HistoryEventId(historyEventId)
    , HistoryAction(action)
    , ObjectId(objectId)
    , Imei(imei)
    , Timestamp(timestamp)
    {}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::TMatcherEvent>(const NDrive::TMatcherEvent& event) {
    NJson::TJsonValue json;
    json.InsertValue("history_event_id", event.GetHistoryEventId());
    json.InsertValue("history_action", ::ToString(event.GetHistoryAction()));
    json.InsertValue("object_id", event.GetObjectId());
    json.InsertValue("imei", event.GetImei());
    json.InsertValue("timestamp", event.GetTimestamp().Seconds());
    return json;
}

template <>
bool NJson::TryFromJson<NDrive::TMatcherEvent>(const NJson::TJsonValue& json, NDrive::TMatcherEvent& event) {
    return NJson::TryFromJson(json["history_event_id"], event.MutableHistoryEventId()) &&
        NJson::TryFromJson(json["history_action"], event.MutableHistoryAction()) &&
        NJson::TryFromJson(json["object_id"], event.MutableObjectId()) &&
        NJson::TryFromJson(json["imei"], event.MutableImei()) &&
        NJson::TryFromJson(json["timestamp"], event.MutableTimestamp());
}

TString NDrive::TYTObjectIdMatcherHistory::GetObjectId(const TString& imei, TMaybe<TInstant> timestamp) {
    if (Imeis[imei].size() == 0) {
        return "";
    }
    if (!timestamp) {
        return ImeisHistory[Imeis[imei].back()].ObjectId;
    }
    for (const auto& index : Imeis[imei]) {
        if (ImeisHistory[index].BeginTime >= timestamp && timestamp <= ImeisHistory[index].EndTime) {
            return ImeisHistory[index].ObjectId;
        }
    }
    return "";
}

TVector<NDrive::TMatcherEvent> NDrive::TYTObjectIdMatcherHistory::GetDroppedImeiHistory() {
    return {
        NDrive::TMatcherEvent(0, EObjectHistoryAction::UpdateData, "6737719f-538b-48e1-9144-a34184f4ed9b", "867962041541114", TInstant::Seconds(1550747215)),
        NDrive::TMatcherEvent(0, EObjectHistoryAction::UpdateData, "996d6a1d-6025-4a1a-fad1-4cb32a8fcbd2", "867962041506943", TInstant::Seconds(1551799141)),
        NDrive::TMatcherEvent(0, EObjectHistoryAction::UpdateData, "996d6a1d-6025-4a1a-fad1-4cb32a8fcbd2", "867962041480149", TInstant::Seconds(1551800611)),
        NDrive::TMatcherEvent(0, EObjectHistoryAction::UpdateData, "c6faae18-fa60-e047-deb5-694480a4db9d", "867962041480289", TInstant::Seconds(1551796147)),
        NDrive::TMatcherEvent(0, EObjectHistoryAction::UpdateData, "fda3de64-6b5d-717f-3ef1-71e799e64265", "867962041487821", TInstant::Seconds(1554570291))
    };
}

bool NDrive::TYTObjectIdMatcherHistory::UpdateHistory(const NJson::TJsonValue& json, bool needDroppedHistory) {
    TVector<TMatcherEvent> events;
    if (!NJson::TryFromJson(json["last_event_id"], LastEventId) ||
        !NJson::TryFromJson(json["actuality"], Actuality) ||
        !NJson::TryFromJson(json["events"], events)) {
        return false;
    }

    if (needDroppedHistory) {
        for (const auto& history : GetDroppedImeiHistory()) {
            events.push_back(history);
        }
    }

    for (const auto& event : events) {
        AddEvent(event);
    }
    return true;
}

void NDrive::TYTObjectIdMatcherHistory::AddEvent(const TMatcherEvent& event) {
    if (!ObjectIds[event.GetObjectId()].empty()) {
        auto index = ObjectIds[event.GetObjectId()].back();
        ImeisHistory[index].EndTime = event.GetTimestamp();
    }
    if (event.GetHistoryAction() == EObjectHistoryAction::Remove) {
        return;
    }
    auto index = ImeisHistory.size();
    ObjectIds[event.GetObjectId()].push_back(index);
    Imeis[event.GetImei()].push_back(index);
    ImeisHistory.push_back(ImeiInfo(event.GetImei(), event.GetObjectId(), event.GetTimestamp()));
    return;
}

void NDrive::TYTObjectIdMatcherHistory::Load(IInputStream* in) {
    NJson::TJsonValue json;
    NJson::ReadJsonTree(in, &json, true);

    NJson::TryFromJson(json["last_event_id"], LastEventId);
    NJson::TryFromJson(json["actuality"], Actuality);
    NJson::TryFromJson(json["imeis_history"], ImeisHistory);

    for (size_t i = 0; i < ImeisHistory.size(); ++i) {
        ObjectIds[ImeisHistory[i].ObjectId].push_back(i);
        Imeis[ImeisHistory[i].Imei].push_back(i);
    }
}

void NDrive::TYTObjectIdMatcherHistory::Save(IOutputStream* out) const {
    NJson::TJsonValue json;
    json.InsertValue("last_event_id", LastEventId);
    json.InsertValue("actuality", Actuality.Seconds());
    json.InsertValue("imeis_history", NJson::ToJson(ImeisHistory));

    NJson::WriteJson(out, &json);
}

NDrive::ImeiInfo::ImeiInfo() : BeginTime(TInstant::Zero()), EndTime(TInstant::Max()) {}

NDrive::ImeiInfo::ImeiInfo(const TString& imei, const TString& objectId, TInstant begin) : Imei(imei), ObjectId(objectId), BeginTime(begin), EndTime(TInstant::Max()) {}

template <>
NJson::TJsonValue NJson::ToJson<NDrive::ImeiInfo>(const NDrive::ImeiInfo& info) {
    NJson::TJsonValue json;
    json.InsertValue("begin_time", info.BeginTime.Seconds());
    json.InsertValue("end_time", info.EndTime.Seconds());
    json.InsertValue("imei", info.Imei);
    json.InsertValue("object_id", info.ObjectId);
    return json;
}

template <>
bool NJson::TryFromJson<NDrive::ImeiInfo>(const NJson::TJsonValue& json, NDrive::ImeiInfo& info) {
    return NJson::TryFromJson(json["object_id"], info.ObjectId) &&
        NJson::TryFromJson(json["imei"], info.Imei) &&
        NJson::TryFromJson(json["begin_time"], info.BeginTime) &&
        NJson::TryFromJson(json["end_time"], info.EndTime);
}
