#include "report.h"

#include <drive/backend/report/json.h>

#include <drive/library/cpp/geojson/value.h>
#include <drive/library/cpp/tracks/tio.h>

#include <util/generic/adaptor.h>

namespace {
    class TGeoJsonAnalyzerReport : public NDrive::IAnalyzerReport {
    public:
        void Fill(TJsonReport& report) override {
            auto extReport = Collection.ToJson();
            report.SetExternalReport(std::move(extReport));
        }
        void Fill(NJson::TJsonValue& value) override {
            value = Collection.ToJson();
        }

        void OnStartTrack(const NDrive::TTrack& track) override {
            Name = track.Name;
            Status = track.Status;
            CreateFeature();
        }
        void OnStartSegment(const NGraph::TRouter::TMatch& match, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            Y_UNUSED(match);
            Y_UNUSED(coordinates);
        }
        void OnSegmentRange(const TSpeedLimitRange& range, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            if (range.IsSpeedLimitExceeded()) {
                Violating = true;
                CreateFeature();
            } else if (Violating) {
                Violating = false;
                CreateFeature();
            }
            NGeoJson::TFeature& feature = GetFeature();
            if (!feature.GetGeometry()) {
                feature.SetGeometry(NGeoJson::TGeometry::LineString);
            }
            const NGeoJson::TGeometry* geometry = feature.GetGeometry();
            TGeoCoord last = geometry->GetCoordinates().size() > 0 ? geometry->GetCoordinates().back() : TGeoCoord();
            for (auto&& point : range.Points) {
                if (std::abs(last.X - point.X) > 0.00001 || std::abs(last.Y - point.Y) > 0.00001) {
                    feature.MutableGeometry().GetCoordinates().push_back(point);
                    last = point;
                }
            }
            if (range.IsSpeedLimitExceeded()) {
                NJson::TJsonValue violation;
                SerializeViolationMeta(violation, range, coordinates);
                TGeoPolyLine polyline(range.Points);
                violation["center"] = NDrive::SerializeCoordinate(polyline.GetCoordByLength(polyline.GetLength() / 2));
                feature.GetProperties().emplace("violation", std::move(violation));
                if (range.GeocodedCenter) {
                    violation["geocoded_center"] = range.GeocodedCenter;
                }
            }
            Ranges += 1;
        }
        void OnFinishSegment(const NGraph::TRouter::TMatch& match, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            Y_UNUSED(match);
            if (Ranges) {
                Ranges = 0;
            } else {
                NGeoJson::TFeature& feature = GetFeature();
                if (!feature.GetGeometry()) {
                    feature.SetGeometry(NGeoJson::TGeometry::LineString);
                }
                NGeoJson::TGeometry::TCoordinates& featureCoordinates = feature.MutableGeometry().GetCoordinates();
                featureCoordinates.insert(featureCoordinates.end(), coordinates.begin(), coordinates.end());
            }
        }
        void OnFinishTrack(const NDrive::TTrack& track) override {
            Y_UNUSED(track);
        }
        bool UseDirectOrder() const override {
            return true;
        }

    private:
        NGeoJson::TFeature& CreateFeature() {
            auto& features = Collection.GetFeatures();
            if (!features.empty() && !features.back().GetGeometry()) {
                return features.back();
            }

            size_t id = Collection.GetFeatures().size();
            Collection.GetFeatures().emplace_back();

            auto& feature = Collection.GetFeatures().back();
            feature.SetId(id);

            auto& properties = feature.GetProperties();
            properties.emplace("description", Name);
            properties.emplace("status", ToString(Status));
            return feature;
        }
        NGeoJson::TFeature& GetFeature() {
            if (Collection.GetFeatures().empty()) {
                CreateFeature();
            }
            return Collection.GetFeatures().back();
        }

    private:
        NGeoJson::TFeatureCollection Collection;
        TString Name;
        size_t Ranges = 0;
        NDrive::ECarStatus Status = NDrive::ECarStatus::csUnknown;
        bool Violating = false;
    };

    class TLegacyAnalyzerReport : public NDrive::IAnalyzerReport {
    public:
        TLegacyAnalyzerReport(bool debug = false, bool reportRaw = false)
            : Debug(debug)
            , ReportRaw(reportRaw)
        {
        }

        void Fill(TJsonReport& report) override {
            report.AddReportElement("tracks", std::move(Tracks));
        }

        void Fill(NJson::TJsonValue& value) override {
            value = std::move(Tracks);
        }

        void OnStartTrack(const NDrive::TTrack& track) override {
            NJson::TJsonValue& t = Tracks.AppendValue(NJson::JSON_MAP);
            NDrive::SerializeMetaInfo(t, track);
            if (ReportRaw) {
                t["raw"] = NJson::ToJson(track);
            }
        }

        void OnStartSegment(const NGraph::TRouter::TMatch& match, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            TotalLength += CalcLength(match);
            if (Debug) {
                NJson::TJsonValue& t = Tracks.GetArraySafe().back();
                t["segment"].AppendValue(TGeoCoord::SerializeVector(coordinates));
            }
            if (ReportRaw) {
                NJson::TJsonValue& t = Tracks.GetArraySafe().back();
                NJson::TJsonValue linked;
                linked["coordinates"] = NJson::ToJson(coordinates);
                linked["match"] = NJson::ToJson(match);
                t["linked"].AppendValue(std::move(linked));
            }
        }
        void OnSegmentRange(const TSpeedLimitRange& range, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            NJson::TJsonValue& t = Tracks.GetArraySafe().back();
            NJson::TJsonValue& trace = t["track"].SetType(NJson::JSON_ARRAY);
            NJson::TJsonValue& violations = t["violations"].SetType(NJson::JSON_ARRAY);

            TGeoCoord last;
            if (range.Points.size() > 0) {
                NDrive::SerializeCoordinate(trace, range.Points.front());
                last = range.Points.front();
            }
            for (size_t i = 1; i < range.Points.size() - 1; ++i) {
                auto index = GetPointIndex(range.PointIndices[i]);
                if (index) {
                    continue;
                }
                const auto& point = range.Points[i];
                if (std::abs(last.X - point.X) > 0.00001 || std::abs(last.Y - point.Y) > 0.00001) {
                    NDrive::SerializeCoordinate(trace, point);
                    last = point;
                }
            }
            if (range.Points.size() > 1) {
                NDrive::SerializeCoordinate(trace, range.Points.back());
                last = range.Points.back();
            }

            if (range.IsSpeedLimitExceeded()) {
                TotalViolation += range;
                NDrive::SerializeViolationIntervals(violations, coordinates, range);
            }
            TotalRanges++;
        }
        void OnFinishSegment(const NGraph::TRouter::TMatch& match, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) override {
            Y_UNUSED(match);
            if (TotalRanges == 0) {
                NJson::TJsonValue& t = Tracks.GetArraySafe().back();
                NJson::TJsonValue& trace = t["track"].SetType(NJson::JSON_ARRAY);
                NDrive::SerializeTrace(trace, coordinates);
            }
            TotalRanges = 0;
        }
        void OnFinishTrack(const NDrive::TTrack& track) override {
            Y_UNUSED(track);
            NJson::TJsonValue& t = Tracks.GetArraySafe().back();
            NJson::TJsonValue& summary = t["summary"].SetType(NJson::JSON_MAP);
            {
                summary["length"] = TotalLength;
                summary["violation_parts"] = TotalViolation.Parts;
                summary["violation_length"] = TotalViolation.Length;
            }

            TotalViolation = {};
            TotalLength = 0;
        }

        bool UseDirectOrder() const override {
            return false;
        }

    private:
        NJson::TJsonValue Tracks = NJson::JSON_ARRAY;

        TSpeedLimitRange TotalViolation;
        float TotalLength = 0;
        ui32 TotalRanges = 0;
        bool Debug = false;
        bool ReportRaw = false;
    };
}

THolder<NDrive::IAnalyzerReport> NDrive::CreateAnalyzerReport(TStringBuf format) {
    if (format == "legacy" || format.empty()) {
        return MakeHolder<TLegacyAnalyzerReport>();
    }
    if (format == "geojson") {
        return MakeHolder<TGeoJsonAnalyzerReport>();
    }
    if (format == "legacy_debug") {
        return MakeHolder<TLegacyAnalyzerReport>(/*debug=*/true);
    }
    if (format == "legacy_with_raw") {
        return MakeHolder<TLegacyAnalyzerReport>(/*debug=*/false, /*reportRaw=*/true);
    }
    return nullptr;
}


NJson::TJsonValue NDrive::ToGeoJsonFormat(const TTracks &tracks) {
    NGeoJson::TFeatureCollection Collection;

    auto& features = Collection.GetFeatures();
    auto& feature = features.emplace_back();

    feature.SetId(0);
    feature.SetGeometry(NGeoJson::TGeometry::LineString);

    auto& properties = feature.GetProperties();
    properties.emplace("description", tracks.front().Name);
    properties.emplace("status", ToString(tracks.front().Status));


    for (const auto& track : tracks) {
        for (const auto& coordinate : track.Coordinates) {
            feature.MutableGeometry().GetCoordinates().push_back(coordinate);
        }
    }

    return Collection.ToJson();
}


NJson::TJsonValue NDrive::ToLegacyFormat(const TTracks& tracks) {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (const auto& track : tracks) {
        NJson::TJsonValue& t = result.AppendValue(NJson::JSON_MAP);
        NDrive::SerializeMetaInfo(t, track);

        NJson::TJsonValue& trace = t["track"].SetType(NJson::JSON_ARRAY);
        for (const auto& coordinate : track.Coordinates) {
            NDrive::SerializeCoordinate(trace, coordinate);
        }
    }

    return result;
}


template <>
NJson::TJsonValue NDrive::GetAnalyzerReport<NJson::TJsonValue>(TConstArrayRef<TTracksLinker::TResult> results, TStringBuf format) {
    auto report = NDrive::CreateAnalyzerReport(format);
    if (!report) {
        return NJson::JSON_NULL;
    }
    auto process = [&](const TTracksLinker::TResult& i) {
        report->OnStartTrack(i.Track);
        for (auto&& segment : i.Segments) {
            report->OnStartSegment(segment.Linked, segment.Coordinates);
            for (auto&& range : segment.Processed) {
                report->OnSegmentRange(range, segment.Coordinates);
            }
            report->OnFinishSegment(segment.Linked, segment.Coordinates);
        }
        report->OnFinishTrack(i.Track);
    };
    if (report->UseDirectOrder()) {
        for (auto&& i : Reversed(results)) {
            process(i);
        }
    } else {
        for (auto&& i : results) {
            process(i);
        }
    }
    NJson::TJsonValue result;
    report->Fill(result);
    return result;
}
