#include "yagr.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/history_iterator/history_iterator.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>



TYAGRTrackClient::TYAGRTrackClient(const NDrive::IServer& server)
    : Server(server)
    , Client(server.GetTaxiTrackStoryClient())
{
}

NThreading::TFuture<NDrive::TTracks> TYAGRTrackClient::GetTracks(const NDrive::TTrackQuery& query, TDuration timeout) const {
    Y_ENSURE(Client);

    const auto isQueryOk = query.DeviceId || query.SessionId || query.UserId;
    if (!isQueryOk) {
        DEBUG_LOG << "Cannot get tracks from yagr: cannot get sessions for query";
        return NThreading::MakeFuture<NDrive::TTracks>();
    }

    THistoryRidesContext context(Server, query.Since);
    auto tx = Server.GetDriveDatabase().GetCompiledSessionManager().BuildTx<NSQL::ReadOnly | NSQL::Deferred>(timeout, timeout);
    auto& carManager = Server.GetDriveDatabase().GetCarManager();
    auto until = query.Until != TInstant::Max() ? MakeMaybe(query.Until) : Nothing();
    auto ydbTx = Server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("yagr_track_client", &Server);
    R_ENSURE(
        context.Initialize(query.SessionId, query.UserId, query.DeviceId, tx, ydbTx, until, query.NumDoc),
        {},
        "cannot Initialize HistoryRidesContext",
        tx
    );

    auto allTracks = NThreading::TFutures<NDrive::TTrack>();
    auto sessions = context.GetSessions(query.Until, query.NumDoc);
    const auto maxInterval = Client->GetMaxInterval();
    for (auto&& session : sessions) {
        const auto deviceId = session.GetObjectId();
        const auto car = carManager.FetchInfo(deviceId, TInstant{});

        const auto imei = car.GetResult().at(deviceId).GetIMEI();
        const auto sessionSince = session.GetStartTS();
        const auto sessionUntil = session.GetLastTS();

        for (auto since = sessionSince; since < sessionUntil; since += maxInterval) {
            const auto until = std::min(since + maxInterval, sessionUntil);
            auto rawTrack = Client->GetTrack(imei, since, until);
            auto track = rawTrack.Apply([deviceId, sessionId = session.GetSessionId(), userId = session.GetUserId()](const NThreading::TFuture<NDrive::TTrack>& t) {
                auto track = t.GetValue();
                track.DeviceId = deviceId;
                track.SessionId = sessionId;
                track.UserId = userId;
                return track;
            });

            allTracks.push_back(std::move(track));
        }
    }
    if (allTracks.empty()) {
        return NThreading::MakeFuture<NDrive::TTracks>();
    }
    auto waiter = NThreading::WaitAll(allTracks);
    auto merged = waiter.Apply([allTracks = std::move(allTracks)](const NThreading::TFuture<void>& /*w*/) mutable {
        NDrive::TTracks result;
        for (auto&& track : allTracks) {
            result.push_back(track.ExtractValue());
        }
        return result;
    });
    return merged;
}

TYAGRTrackClientV2::TYAGRTrackClientV2(const NDrive::IServer& server)
    : Server(server)
    , Client(server.GetTaxiTrackStoryClient())
{
}

NThreading::TFuture<NDrive::TTracks> TYAGRTrackClientV2::GetTracks(const NDrive::TTrackQuery& query, TDuration timeout) const {
    Y_ENSURE(Client);

    const auto isQueryOk = query.DeviceId || query.SessionId || query.UserId;
    if (!isQueryOk) {
        return NThreading::TExceptionFuture() << "Cannot get tracks from yagr for signalq: cannot get sessions for query";
    }

    THistoryRidesContext context(Server, query.Since);
    auto tx = Server.GetDriveDatabase().GetCompiledSessionManager().BuildTx<NSQL::ReadOnly | NSQL::Deferred>(timeout, timeout);
    auto until = query.Until != TInstant::Max() ? MakeMaybe(query.Until) : Nothing();
    auto ydbTx = Server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("yagr_track_client", &Server);
    R_ENSURE(
        context.Initialize(query.SessionId, query.UserId, query.DeviceId, tx, ydbTx, until, query.NumDoc),
        {},
        "cannot Initialize HistoryRidesContext",
        tx
    );

    auto allTracks = NThreading::TFutures<NDrive::TTrack>();
    auto sessions = context.GetSessions(query.Until, query.NumDoc);
    const auto signalqDrivematicsApiClient = Server.GetTaxiSignalqDrivematicsApiClient();
    const auto maxInterval = Client->GetMaxInterval();
    for (auto&& session : sessions) {
        auto deviceId = session.GetObjectId();
        auto sessionId = session.GetSessionId();
        auto userId = session.GetUserId();
        const auto sessionSince = session.GetStartTS();
        const auto sessionUntil = session.GetLastTS();
        TString parkId;
        TString serialNumber;
        auto attachments = Server.GetDriveAPI()->GetCarAttachmentAssignments().GetAssignmentsOfType(TVector{deviceId}, EDocumentAttachmentType::CarSignalDevice, TInstant::Zero());
        if (!attachments.empty()) {
            auto carSignalDeviceImpl = dynamic_cast<const TCarSignalDevice*>(attachments.begin()->second.Get());
            if (carSignalDeviceImpl) {
                NSignalq::TV1DeviceParkGetRequest getParkRequest;
                serialNumber = carSignalDeviceImpl->GetSerialNumber();
                getParkRequest.SetSerialNumber(serialNumber);
                auto responseFuture = signalqDrivematicsApiClient->GetPark(getParkRequest);
                responseFuture.Wait();
                if (!responseFuture.HasValue() || responseFuture.HasException()) {
                    continue;
                } else {
                    parkId = responseFuture.ExtractValue().GetParkId();
                }
            } else {
                continue;
            }
        } else {
            continue;
        }

        for (auto since = sessionSince; since < sessionUntil; since += maxInterval) {
            const auto until = std::min(since + maxInterval, sessionUntil);
            auto rawTrack = Client->GetCameraTrack(serialNumber, parkId, since, until);
            auto track = rawTrack.Apply([deviceId, sessionId = session.GetSessionId(), userId = session.GetUserId()](const NThreading::TFuture<NDrive::TTrack>& t) {
                auto track = t.GetValue();
                track.DeviceId = deviceId;
                track.SessionId = sessionId;
                track.UserId = userId;
                return track;
            });

            allTracks.push_back(std::move(track));
        }
    }
    if (allTracks.empty()) {
        return NThreading::MakeFuture<NDrive::TTracks>();
    }
    auto waiter = NThreading::WaitAll(allTracks);
    auto merged = waiter.Apply([allTracks = std::move(allTracks)](const NThreading::TFuture<void>& /*w*/) mutable {
        NDrive::TTracks result;
        for (auto&& track : allTracks) {
            result.push_back(track.ExtractValue());
        }
        return result;
    });
    return merged;
}
