#include "client.h"
#include "matched_signal.h"

#include <drive/backend/tracks/object_id_matcher/matcher.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/proto/matcher.pb.h>
#include <drive/backend/tags/tag.h>

#include <contrib/restricted/boost/boost/date_time/posix_time/conversion.hpp>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/json/json_reader.h>

#include <mapreduce/yt/interface/client.h>

#include <maps/analyzer/libs/data/include/gpssignal.h>
#include <maps/analyzer/libs/graphmatching/include/reasons.h>
#include <maps/analyzer/libs/graphmatching/include/match_signals.h>
#include <maps/analyzer/libs/graphmatching/include/stream_matcher.h>
#include <maps/libs/road_graph/include/graph.h>
#include <maps/libs/succinct_rtree/include/rtree.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/library/storage/ydb/structured.h>

#include <yweb/blender/lib/yql/yql.h>

#include <util/datetime/base.h>
#include <util/stream/output.h>
#include <util/string/subst.h>
#include <util/system/env.h>

using namespace NYT;

class TTelematicsLogMapper
    : public IMapper<TTableReader<TNode>, TTableWriter<TNode>>
    , public ISerializableForJob
{
public:
    TTelematicsLogMapper(const NDrive::TYTObjectIdMatcherHistory& matcher, ui64 begin, ui64 end)
        : Matcher(std::move(matcher))
        , BeginTime(begin)
        , EndTime(end)
    {
    }
    TTelematicsLogMapper() = default;

    Y_SAVELOAD_JOB(BeginTime, EndTime, Matcher);

    void Do(TReader* reader, TWriter* writer) override {
        for (auto&& cursor : *reader) {
            auto row = cursor.GetRow();
            if (!row["type"].IsString() || !row["event"].IsString() || row["type"].AsString() != "BLACKBOX_RECORDS" || row["event"].AsString() != "incoming") {
                continue;
            }

            auto timestamp = row["timestamp"].AsUint64();
            if (timestamp > Matcher.GetActuality().Seconds()) {
                Cerr << "Future record " << timestamp << " > " << Matcher.GetActuality().Seconds() << Endl;
                continue;
            }
            if (timestamp < BeginTime || timestamp >= EndTime) {
                Cerr << "Record with ts " << timestamp << " doesn't belong to a period [" << BeginTime << "; " << EndTime << ")." << '\n';
                continue;
            }
            // check for missing ids. If there is no car then match by imei
            auto objectId = Matcher.GetObjectId(ToString(row["imei"].AsInt64()));
            TNode result;
            result["object_id"] = objectId ? objectId : ToString(row["imei"].AsInt64());
            result["imei"] = row["imei"].AsInt64();
            result["timestamp"] = timestamp;
            result["data"] = row["data"];
            // this field is need for future sorting two tables in reducer
            result["history_timestamp"] = static_cast<i64>(timestamp);
            writer->AddRow(result);
        }
    }

private:
    NDrive::TYTObjectIdMatcherHistory Matcher;
    ui64 BeginTime;
    ui64 EndTime;
};
REGISTER_MAPPER(TTelematicsLogMapper);

struct TSessionState {
    TString SessionId;
    TString State;
    TString UserId;
};

class TTelematicsLogReducer
    : public IReducer<TTableReader<TNode>, TTableWriter<TNode>>
{
public:
    void Do(TReader* reader, TWriter* writer) override {
        for (auto& cursor : *reader) {
            const auto& curRow = cursor.GetRow();
            auto tableIndex = cursor.GetTableIndex();

            auto objectId = curRow["object_id"].AsString();
            if (!objectId) {
                Cerr << "TTelematicsLogReducer. No object id. Table index: " << tableIndex << " .Imei: " << ToString(curRow["imei"].AsInt64()) << '\n';
            }
            if (tableIndex == 0) { /*car_tags_history*/
                TString tag = curRow["tag"].AsString();
                auto historyAction = curRow["history_action"].AsString();

                auto tagType = tag.substr(0, 9);
                if (tagType == "old_state" || tagType == "servicing") {
                    TString tagId = curRow["tag_id"].AsString();

                    if (tag == "old_state_reservation") {
                        TChargableTag chargableTag;
                        chargableTag.DeserializeSpecialData(curRow["data"].AsString(), "");

                        bool isForceTagPerformer = false;
                        if (historyAction ==  ::ToString(EObjectHistoryAction::ForceTagPerformer)) {
                            isForceTagPerformer = true;
                        }
                        if (historyAction == ::ToString(EObjectHistoryAction::DropTagPerformer) ||
                            historyAction == ::ToString(EObjectHistoryAction::Remove) ||
                            isForceTagPerformer) {
                            auto iter = SessionStates.find(tagId);
                            if (iter != SessionStates.end()) {
                                auto currentState = iter->second;
                                currentState.State = "post";
                                SessionStates.erase(iter);
                                CarSessions[objectId].erase(tagId);
                                LastFinishedCarSession[objectId] = currentState;
                            }
                        }
                        if (historyAction == ::ToString(EObjectHistoryAction::SetTagPerformer) ||
                            isForceTagPerformer) {
                            ICommonOffer::TPtr offer = chargableTag.GetOffer();
                            if (!offer) {
                                Cerr << "Coudn't get offer_id! Tag id:" << curRow["tag_id"].AsString() << '\n';
                            } else {
                                TSessionState state;
                                state.SessionId = offer->GetOfferId();
                                state.State = tag;
                                state.UserId = offer->GetUserId();
                                SessionStates[tagId] = state;
                                CarSessions[objectId].insert(tagId);
                            }
                        }
                    } else if (tag != SessionStates[tagId].State) {
                        SessionStates[tagId].State = tag;
                    }
                }
            } else if (tableIndex == 1) { /*telematics events*/
                auto data = curRow["data"].AsMap();
                if (!data["records"].IsList()) {
                    continue;
                }
                auto records = data["records"].AsList();
                for (auto record : records) {
                    if (record["type"].AsString() != "record") {
                        continue;
                    }
                    TNode result;
                    result["timestamp"] = record["timestamp"];
                    result["groupstamp"] = (record["timestamp"].AsInt64() / 3600) * 3600;
                    result["object_id"] = objectId;
                    // result["data"] = curRow["data"];
                    auto subrecords = record["subrecords"].AsList();
                    for (auto subrecord : subrecords) {
                        auto subrecordType = subrecord["type"].AsString();
                        if (subrecordType == "position_data") {
                            result["lat"] = subrecord["lat"];
                            result["lon"] = subrecord["lon"];
                            result["speed"] = subrecord["speed"];
                            result["course"] = subrecord["course"];
                        } else if (subrecordType == "custom_parameters") {
                            auto params = subrecord["params"].AsMap();
                            if (params["2111"].HasValue()) {
                                result["can_speed"] = params["2111"].ConvertTo<i64>();
                            }
                            if (params["2209"].HasValue()) {
                                result["can_hand_break"] = params["2209"].ConvertTo<i64>();
                            }
                            if (params["2125"].HasValue()) {
                                result["can_brake_accelerator"] = params["2125"].ConvertTo<i64>();
                            }
                            if (params["2135"].HasValue()) {
                                result["can_steering_wheel"] = params["2135"].ConvertTo<i64>();
                            }
                        }
                    }
                    if (!CarSessions[objectId].empty()) {
                        for (const auto& tagId : CarSessions[objectId]) {
                            result["session_id"] = SessionStates[tagId].SessionId;
                            result["tag"] = SessionStates[tagId].State;
                            result["user_id"] = SessionStates[tagId].UserId;
                            writer->AddRow(result);
                        }
                    } else {
                        auto it = LastFinishedCarSession.find(objectId);
                        if (it != LastFinishedCarSession.end()) {
                            auto&& lastSession = it->second;
                            result["session_id"] = lastSession.SessionId;
                            result["tag"] = lastSession.State;
                            result["user_id"] = lastSession.UserId;
                        } else {
                            result["session_id"] = "";
                            result["tag"] = "";
                            result["user_id"] = "";
                        }
                        writer->AddRow(result);
                    }
                }
            }
        }
        SessionStates.clear();
        CarSessions.clear();
        LastFinishedCarSession.clear();
    }

private:
    TMap<TString /*tag_id*/, TSessionState /*session_state*/> SessionStates;
    TMap<TString /*objectId*/, TSet<TString> /*tag_id*/> CarSessions;
    TMap<TString /*objectId*/, TSessionState /*session_state*/> LastFinishedCarSession;
};
REGISTER_REDUCER(TTelematicsLogReducer);

namespace ma = maps::analyzer;
namespace mag = maps::analyzer::graphmatching;

class TMatcherReducer
    : public IReducer<TTableReader<TNode>, TTableWriter<TNode>>
{
public:
    void Do(TReader* reader, TWriter* writer) override {
        auto filterConfig = ma::graphmatching::FilterConfig::loadConfig("filter_config");
        auto matcherConfig = mag::MatcherConfig::MatcherConfig::loadConfig("matcher_config");

        auto roadGraph = maps::road_graph::Graph("road_graph", EMappingMode::Locked);
        auto rtree = maps::succinct_rtree::Rtree("rtree", roadGraph);
        mag::StreamSignalsMatcher m{
            [&](const mag::Signal& signal, const mag::Candidate& candidate) {
                AddMatchedPoints(Data, signal.gpsSignal, candidate, roadGraph);
            },
            mag::OnSegmentMatchedFn{},
            mag::OnSignalProcessedFn{},
            mag::OnPathMatchedFn{},
            roadGraph, rtree, matcherConfig, filterConfig,
        };
        for (auto&& cursor: *reader) {
            try {
                auto curRow = cursor.GetRow();
                maps::analyzer::data::GpsSignal signal;
                signal.setTime(boost::posix_time::from_time_t(curRow["timestamp"].AsInt64()));
                if (!curRow["lon"].IsDouble()) {
                    Cerr << "Invalid signal with lon " << Endl;
                    continue;
                }
                if (!curRow["lat"].IsDouble()) {
                    Cerr << "Invalid signal with lat " << Endl;
                    continue;
                }
                signal.setLon(curRow["lon"].AsDouble());
                signal.setLat(curRow["lat"].AsDouble());
                signal.setDirection(curRow["course"].AsInt64());
                signal.setAverageSpeed(curRow["speed"].AsInt64());

                auto uuid = TStringBuilder() << curRow["session_id"].AsString() << "@" << curRow["object_id"].AsString();
                auto clid = TStringBuilder() << curRow["user_id"].AsString() << "@" << curRow["tag"].AsString();
                signal.setUuid(uuid);
                signal.setClid(clid);

                m.push(signal);
            }  catch (const yexception& e) {
                Cerr << "Invalid signal: " << e.what() << '\n';
            }
        }
        m.done();

        if (!Data.empty()) {
            std::stable_sort(Data.begin(), Data.end());
            auto record = SerializeToYtNode(Data);
            writer->AddRow(record);
            Data.clear();
        }
    }
private:
    TVector<TMatchedSignal> Data;
};
REGISTER_REDUCER(TMatcherReducer);

TString GetYqlQuery();

NYT::TTableSchema GetObjectIdTableSchema() {
    return NYT::TTableSchema()
        .AddColumn(NYT::TColumnSchema().Name("object_id").Type(NYT::VT_STRING, /*required*/ false))
        .AddColumn(NYT::TColumnSchema().Name("history_timestamp").Type(NYT::VT_INT64, /*required*/ true))
        .AddColumn(NYT::TColumnSchema().Name("data").Type(NYT::VT_ANY, /*required*/ false))
        .AddColumn(NYT::TColumnSchema().Name("imei").Type(NYT::VT_INT64, /*required*/ false))
        .AddColumn(NYT::TColumnSchema().Name("timestamp").Type(NYT::VT_UINT64, /*required*/ true));
}

int main(int argc, const char* argv[])
{
    Initialize(argc, argv);
    NLastGetopt::TOpts opts;
    opts.SetFreeArgsNum(0);
    opts.AddHelpOption('h');

    TString cluster = "hahn";

    TVector<NYT::TYPath> srcTables;
    opts.AddLongOption("input", "input tables")
        .Required()
        .RequiredArgument("YT_PATH")
        .AppendTo(&srcTables);

    ui64 beginTs;
    opts.AddLongOption("begin_ts", "begin timestamp of the period")
        .Required()
        .StoreResult(&beginTs);

    ui64 endTs;
    opts.AddLongOption("end_ts", "end timestamp of the period")
        .Required()
        .StoreResult(&endTs);

    NYT::TYPath objectIdsTable;
    opts.AddLongOption("objectIds", "telematics logs with object id")
        .Required()
        .RequiredArgument("YT_PATH")
        .StoreResult(&objectIdsTable);

    NYT::TYPath carsTagsHistoryTable;
    opts.AddLongOption("history", "cars tags history table")
        .Required()
        .RequiredArgument("YT_PATH")
        .StoreResult(&carsTagsHistoryTable);

    NYT::TYPath carsTagsHistorySorted;
    opts.AddLongOption("sorted", "cars tags history sorted table")
        .Required()
        .RequiredArgument("YT_PATH")
        .StoreResult(&carsTagsHistorySorted);

    NYT::TYPath sessionIdsTable;
    opts.AddLongOption("sessionIds", "table with tags and session ids")
        .Required()
        .RequiredArgument("YT_PATH")
        .StoreResult(&sessionIdsTable);

    NYT::TYPath matchedTable;
    opts.AddLongOption("matched", "table with road graph matched data")
        .Required()
        .RequiredArgument("YT_PATH")
        .StoreResult(&matchedTable);

    TString endpoint = "https://prestable.carsharing.yandex.net";
    opts.AddLongOption("endpoint", "backend endpoint")
        .Optional()
        .StoreResult(&endpoint);

    TString matcherConfigPath = "/maps/analyzer/libs/graphmatching/conf/offline.json";
    opts.AddLongOption("matcher", "path to matcher config")
        .Optional()
        .StoreResult(&matcherConfigPath);

    TString filterConfigPath = "/maps/analyzer/libs/graphmatching/conf/signal_filter.json";
    opts.AddLongOption("filter", "path to filter config")
        .Optional()
        .StoreResult(&filterConfigPath);

    bool needMatching = false;
    opts.AddLongOption("needMatching", "need to match data")
        .Optional()
        .NoArgument()
        .SetFlag(&needMatching);

    bool useYDB = false;
    opts.AddLongOption("useYDB", "use ydb")
        .Optional()
        .NoArgument()
        .SetFlag(&useYDB);

    TString ydbEndpoint = "ydb-ru-prestable.yandex.net:2135";
    opts.AddLongOption("ydbEndpoint", "ydb endpoint")
        .Optional()
        .StoreResult(&ydbEndpoint);

    TString ydbDatabase = "/ru-prestable/yandexcarsharing/development/drive_backend";
    opts.AddLongOption("ydbDatabase", "path to ydb database")
        .Optional()
        .StoreResult(&ydbDatabase);

    TString ydbTableName = "/tracks";
    opts.AddLongOption("ydbTable", "ydb table for final dumping")
        .Optional()
        .StoreResult(&ydbTableName);

    NLastGetopt::TOptsParseResult parseOpts(&opts, argc, argv);

    if (needMatching) {
        NDrive::TYTObjectIdMatcherHistory matcher;
        NDrive::TObjectIdMatcherClient::TOptions options;
        options.Endpoint = endpoint;
        options.Timeout = TDuration::MilliSeconds(10000);
        NDrive::TObjectIdMatcherClient historyClient(options);

        bool needDroppedHistory = true;
        if (endpoint.find("testing") != std::string::npos) {
            needDroppedHistory = false;
        }
        matcher.UpdateHistory(historyClient.GetHistory().GetValueSync(), needDroppedHistory);

        auto ytClient = CreateClient(cluster, TCreateClientOptions().Token(GetEnv("YT_TOKEN")));

        if (ytClient->Exists(objectIdsTable)) {
            ytClient->Remove(objectIdsTable);
        }
        ytClient->Create(objectIdsTable, NYT::NT_TABLE);

        NYT::TMapOperationSpec spec;
        for (const auto &table: srcTables) {
            spec.AddInput<NYT::TNode>(table);
        }
        auto schema = GetObjectIdTableSchema();
        spec.AddOutput<NYT::TNode>(NYT::TRichYPath(objectIdsTable).CompressionCodec("lz4").Schema(schema));
        auto operation = ytClient->Map(
            spec,
            new TTelematicsLogMapper(matcher, beginTs, endTs),
            TOperationOptions()
                .InferOutputSchema(true));

        if (ytClient->Exists(carsTagsHistorySorted)) {
            ytClient->Remove(carsTagsHistorySorted);
        }

        ytClient->Sort(
            TSortOperationSpec()
                .AddInput(carsTagsHistoryTable)
                .Output(carsTagsHistorySorted)
                .SortBy({"object_id", "history_timestamp", "history_event_id"}));

        ytClient->Sort(
            TSortOperationSpec()
                .AddInput(objectIdsTable)
                .Output(objectIdsTable)
                .SortBy({"object_id", "history_timestamp"}));

        if (ytClient->Exists(sessionIdsTable)) {
            ytClient->Remove(sessionIdsTable);
        }
        ytClient->Create(sessionIdsTable, NYT::NT_TABLE);

        ytClient->Reduce(
        TReduceOperationSpec()
            .SortBy({"object_id", "history_timestamp"})
            .ReduceBy({"object_id"})
            .AddInput<TNode>(carsTagsHistorySorted)
            .AddInput<TNode>(objectIdsTable)
            .AddOutput<TNode>(sessionIdsTable),
        new TTelematicsLogReducer);

        ytClient->Sort(
        TSortOperationSpec()
            .AddInput(sessionIdsTable)
            .Output(sessionIdsTable)
            .SortBy({"session_id", "object_id", "groupstamp", "timestamp"}));

        NYT::TYPath graphPath = "//home/maps/graph/latest/road_graph.fb";
        NYT::TYPath rtreePath = "//home/maps/graph/latest/rtree.fb";

        if (ytClient->Exists(matchedTable)) {
            ytClient->Remove(matchedTable);
        }

        ui64 memoryLimit = 10737418240;
        ytClient->Reduce(
        TReduceOperationSpec()
            .ReduceBy({"session_id", "object_id", "groupstamp"})
            .AddInput<TNode>(sessionIdsTable)
            .AddOutput<TNode>(matchedTable)
            .ReducerSpec(TUserJobSpec()
                .AddFile(TRichYPath(graphPath).FileName("road_graph"))
                .AddFile(TRichYPath(rtreePath).FileName("rtree"))
                .AddLocalFile(matcherConfigPath, TAddLocalFileOptions().PathInJob("matcher_config"))
                .AddLocalFile(filterConfigPath, TAddLocalFileOptions().PathInJob("filter_config"))
                .MemoryLimit(memoryLimit)),
        new TMatcherReducer()
        );
    }

    if (!useYDB) {
        return 0;
    }

    TString script = GetYqlQuery();
    SubstGlobal(script, "#YDB_ENDPOINT#", ydbEndpoint);
    SubstGlobal(script, "#YDB_DATABASE#", ydbDatabase);
    SubstGlobal(script, "#YDB_TABLENAME#", ydbTableName);
    SubstGlobal(script, "##INPUT_TABLEPATH##", matchedTable);

    auto op = RunYQLScript(script, GetEnv("YQL_TOKEN"), DEFAULT_YQL_API_URL, "SQLv1");
    try {
        WaitForOp(op);
    } catch (const yexception& e) {
        Cerr << "Yql failed. Error: " << e.what() << Endl;
        Cerr << op.GetFailureReason() << Endl;
        Cerr << script << Endl;
    }
    return 0;
}


TString GetYqlQuery() {
    TString script = R"(
        $ydb_endpoint = "#YDB_ENDPOINT#";
        $ydb_database = "#YDB_DATABASE#";
        $ydb_table = "#YDB_DATABASE##YDB_TABLENAME#";

        $ydb_batch_size_bytes = 104857;
        $ydb_oauth_token = AsTuple("token", SecureParam("token:default_ydb"));
        $ydb_batch_size_rows = 104857;
        $ydb_max_retries = 10;
        PRAGMA yt.InferSchema = '1';
        PRAGMA yt.QueryCacheMode = "disable";
        PRAGMA yt.DataSizePerJob = "500485760";
        PRAGMA yt.MaxJobCount = "100000";
        PRAGMA yt.UserSlots = "1000";
        PRAGMA yt.DefaultMaxJobFails = "5";

        SELECT
            SUM(Bytes) AS TotalBytes,
            SUM(Rows) AS TotalRows,
            SUM(Batches) AS TotalBatches,
            SUM(Retries) AS TotalRetries
        FROM (
            PROCESS (
                SELECT
                    CAST(object_id AS String) AS object_id,
                    CAST(user_id AS String) AS user_id,
                    CAST(session_id AS String) AS session_id,
                    CAST(groupstamp AS UInt64) AS groupstamp,
                    CAST(data AS String) AS data
                FROM hahn.`##INPUT_TABLEPATH##`
            )
            USING YDB::PushData(
                TableRows(),
                $ydb_endpoint,
                $ydb_database,
                $ydb_table,
                $ydb_oauth_token,
                $ydb_batch_size_bytes,
                $ydb_batch_size_rows,
                $ydb_max_retries
            )
        );
    )";
    return script;
}
