#include "matched_signal.h"

#include <cmath>

#include <drive/library/cpp/common/status.h>

#include <google/protobuf/util/json_util.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/yson/node/node_io.h>

#include <rtline/util/algorithm/container.h>

TMatchedSignal::TMatchedSignal(const maps::analyzer::data::GpsSignal& signal, const maps::geolib3::Point2& point, const maps::road_graph::EdgeData& data, const double& length)
    : Timestamp(TInstant::Seconds(maps::toTimestamp(signal.time())))
    , Coordinate(TGeoCoord(point.x(), point.y()))
    , LenghtToPreviuosPoint(length)
{
    auto clid = MakeVector(StringSplitter(signal.clid()).SplitBySet("@").ToList<TString>());
    UserId = clid[0];
    Status = ::ToString(NDrive::GetStatus(clid[1]));

    auto uuid = MakeVector(StringSplitter(signal.uuid()).SplitBySet("@").ToList<TString>());
    SessionId = uuid[0];
    ObjectId = uuid[1];

    double speedLimit = 0;
    EdgeId = data.edgeId();
    if (auto limit = data.speedLimit(maps::road_graph::AccessId::Automobile); limit) {
        speedLimit = *limit;
    }
    FC = data.category();
    Speed = data.speed();
};

bool TMatchedSignal::operator < (const TMatchedSignal& signal) const {
    if (signal.Status == Status) {
        return Timestamp < signal.Timestamp;
    }
    return Status < signal.Status;
}

NDrive::NProto::TMatchedCoordinate TMatchedSignal::SerializeToProto() const {
    NDrive::NProto::TMatchedCoordinate coord;
    coord.SetTimestamp(Timestamp.Seconds());
    coord.SetSpeed(std::round(Speed * 3.6)); // km/h
    coord.SetSpeedLimit(std::round(SpeedLimit * 3.6)); // km/h
    *coord.MutableCoord() = Coordinate.Serialize();
    coord.SetFC(FC);
    coord.SetLength(LenghtToPreviuosPoint);
    return coord;
}

NYT::TNode SerializeToYtNode(const TVector<TMatchedSignal>& signals) {
    NYT::TNode node;
    if (signals.empty()) {
        return node;
    }
    node["object_id"] = signals.front().ObjectId;
    node["session_id"] = signals.front().SessionId;
    node["user_id"] = signals.front().UserId;
    node["groupstamp"] = signals.front().Timestamp.Hours() * 3600;

    NDrive::NProto::TTrackPartData record;
    TVector<NDrive::NProto::TMatchedCoordinate> coords;
    size_t beginIdx = 0;
    coords.push_back(signals[beginIdx].SerializeToProto());

    for (auto curIdx = beginIdx; curIdx < signals.size(); ++curIdx) {
        if (signals[curIdx].Status == signals[beginIdx].Status) {
            coords.push_back(signals[curIdx].SerializeToProto());
        } else {
            NDrive::NProto::TTrackSegment segment;
            segment.SetStatus(signals[beginIdx].Status);
            segment.SetBeginTimestamp(signals[beginIdx].Timestamp.Seconds());
            segment.SetEndTimestamp(signals[curIdx - 1].Timestamp.Seconds());
            for (const auto& coord : coords) {
                *segment.AddMatchedCoords() = coord;
            }
            coords.clear();
            *record.AddSegments() = segment;

            beginIdx = curIdx;
            coords.push_back(signals[beginIdx].SerializeToProto());
        }
    }
    if (!coords.empty()) {
        NDrive::NProto::TTrackSegment segment;
        segment.SetStatus(signals[beginIdx].Status);
        segment.SetBeginTimestamp(signals[beginIdx].Timestamp.Seconds());
        segment.SetEndTimestamp(signals[signals.size() - 1].Timestamp.Seconds());
        for (const auto& coord : coords) {
            *segment.AddMatchedCoords() = coord;
        }
        *record.AddSegments() = segment;
    }

    TString proto;
    auto res = record.SerializeToString(&proto);
    Y_UNUSED(res);
    node["data"] = proto;

    // fields for debug
    TString asJson;
    ::google::protobuf::util::MessageToJsonString(record, &asJson);
    NJson::TJsonValue value;
    NJson::ReadJsonTree(asJson, &value);
    node["json_data"] = NYT::NodeFromYsonString(NJson2Yson::SerializeJsonValueAsYson(value));
    return node;
}

void AddMatchedPoints(TVector<TMatchedSignal>& allPoints, const maps::analyzer::data::GpsSignal& signal, const maps::analyzer::graphmatching::Candidate& candidate, const maps::road_graph::Graph& roadGraph) {
    if (!candidate.pathFromPrevious || !candidate.pathFromPrevious->trace || candidate.pathFromPrevious->trace->edges.empty()) {
        if (allPoints.empty() || allPoints.back().EdgeId != candidate.graphPoint.segmentId.edgeId) {
            auto data = roadGraph.edgeData(candidate.graphPoint.segmentId.edgeId);
            allPoints.push_back(TMatchedSignal(signal, candidate.point, data, data.length()));
        }
        return;
    }

    auto routeEdges = candidate.pathFromPrevious->trace->edges;
    i32 beginIdx = 0;
    for (i32 i = routeEdges.size() - 1; i >= 0; --i) {
        if (routeEdges[i] == allPoints.back().EdgeId) {
            beginIdx = i + 1;
            break;
        }
    }

    for (ui32 i = beginIdx; i < routeEdges.size(); ++i) {
        const auto& edgeData = roadGraph.edgeData(routeEdges[i]);
        auto points = maps::geolib3::Polyline2(edgeData.geometry()).points();
        for (const auto& point : points) {
            auto length = maps::geolib3::length(maps::geolib3::Segment2(maps::geolib3::Point2(allPoints.back().Coordinate.X, allPoints.back().Coordinate.Y), point)) ;
            allPoints.push_back(TMatchedSignal(signal, point, edgeData, length));
        }
    }
}
