#include "manager.h"

#include <drive/backend/logging/evlog.h>

#include <drive/library/cpp/openssl/abstract.h>

#include <library/cpp/case_insensitive_string/case_insensitive_string.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/sql/query.h>
#include <rtline/library/unistat/cache.h>

#include <util/random/random.h>

constexpr TCaseInsensitiveStringBuf AuthorizationHeader = "Authorization";

TString TFakeEventsRegistratorConfig::Type = "fake";
IEventsRegistratorConfig::TFactory::TRegistrator<TFakeEventsRegistratorConfig> TFakeEventsRegistratorConfig::Registrator(TFakeEventsRegistratorConfig::Type);

TString TAdjustEventRegistratorConfig::Type = "adjust";
IEventsRegistratorConfig::TFactory::TRegistrator<TAdjustEventRegistratorConfig> TAdjustEventRegistratorConfig::Registrator(TAdjustEventRegistratorConfig::Type);

TShortUserDevice::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    DeviceId = GetFieldDecodeIndex("device_id", decoderBase);
    UserId = GetFieldDecodeIndex("user_id", decoderBase);
    Enabled = GetFieldDecodeIndex("enabled", decoderBase);
    Verified = GetFieldDecodeIndex("verified", decoderBase);

    AdvertisingToken = GetFieldDecodeIndex("advertising_token", decoderBase);
    AdvertisingTokenType = GetFieldDecodeIndex("advertising_token_type", decoderBase);
    AdvertisingTokenTimestamp = GetFieldDecodeIndex("advertising_token_timestamp", decoderBase);
}

bool TShortUserDevice::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, DeviceId);
    READ_DECODER_VALUE(decoder, values, Enabled);
    READ_DECODER_VALUE(decoder, values, Verified);

    READ_DECODER_VALUE(decoder, values, AdvertisingToken);
    READ_DECODER_VALUE_DEF(decoder, values, AdvertisingTokenType, EAdvertisingTokenType::UNDEFINED);
    if (decoder.GetAdvertisingTokenTimestamp() >= 0) {
        TStringBuf activateDateStr = values[decoder.GetAdvertisingTokenTimestamp()];
        if (!!activateDateStr) {
            ui32 activateDate = 0;
            if (TryFromString(activateDateStr, activateDate)) {
                AdvertisingTokenTimestamp = TInstant::Seconds(activateDate);
            }
        }
    }
    return true;
}

void TShortUserDevice::DoBuildReportItem(NJson::TJsonValue& report) const {
    report.InsertValue("user_id", UserId);
    report.InsertValue("device_id", DeviceId);
    report.InsertValue("id", DeviceId);
    report.InsertValue("enabled", Enabled);
    report.InsertValue("verified", Verified);

    if (AdvertisingToken) {
        report.InsertValue("advertising_token", AdvertisingToken);
        report.InsertValue("advertising_token_type", ::ToString(AdvertisingTokenType));
        report.InsertValue("advertising_token_timestamp", AdvertisingTokenTimestamp.Seconds());
    }
}

NStorage::TTableRecord TShortUserDevice::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("user_id", UserId).Set("device_id", DeviceId).Set("verified", Verified).Set("enabled", Enabled);
    if (AdvertisingToken) {
        record.Set("advertising_token", AdvertisingToken).Set("advertising_token_type", ::ToString(AdvertisingTokenType)).Set("advertising_token_timestamp", AdvertisingTokenTimestamp.Seconds());
    }
    return record;
}

TUserDevice::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase)
    : TBase(decoderBase)
{
    Token = GetFieldDecodeIndex("token", decoderBase);
    Description = GetFieldDecodeIndex("description", decoderBase);
    Phone = GetFieldDecodeIndex("phone", decoderBase);
}

void TUserDevice::DoBuildReportItem(NJson::TJsonValue& report) const {
    TShortUserDevice::DoBuildReportItem(report);
    report.InsertValue("phone", Phone);
    NJson::TJsonValue jsonDescription;
    if (NJson::ReadJsonFastTree(Description, &jsonDescription)) {
        report.InsertValue("description", jsonDescription);
    } else {
        report.InsertValue("description", Description);
    }
}

NJson::TJsonValue TUserDevice::GetJsonReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    DoBuildReportItem(result);
    return result;
}

NJson::TJsonValue TUserDevice::GenerateDescriptionJson(IReplyContext::TPtr context) {
    NJson::TJsonValue jsonDescription = NJson::JSON_MAP;
    NJson::TJsonValue& headers = jsonDescription.InsertValue("headers", NJson::JSON_MAP);
    if (context) {
        for (auto&& i : context->GetRequestData().HeadersIn()) {
            TCaseInsensitiveStringBuf headerName(i.first);
            if (headerName != AuthorizationHeader) {
                headers.InsertValue(i.first, i.second);
            }
        }
    }
    return jsonDescription;
}

void TUserDevice::InitDescription(IReplyContext::TPtr context) {
    Description = GenerateDescriptionJson(context).GetStringRobust();
}

NStorage::TTableRecord TUserDevice::SerializeToTableRecord() const {
    NStorage::TTableRecord record = TShortUserDevice::SerializeToTableRecord();
    if (Description) {
        record.Set("description", Description);
    } else {
        record.Set("description", "get_null()");
    }
    if (Phone) {
        record.Set("phone", Phone);
    } else {
        record.Set("phone", "get_null()");
    }
    return record;
}

bool TUserDevice::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    if (!TBase::DeserializeWithDecoder(decoder, values, hContext)) {
        return false;
    }
    READ_DECODER_VALUE(decoder, values, Token);
    READ_DECODER_VALUE(decoder, values, Description);
    READ_DECODER_VALUE(decoder, values, Phone);
    return true;
}

namespace {
    TUnistatSignal<double> CacheHit { { "user_devices-cache-hit" }, false };
    TUnistatSignal<double> CacheMiss { { "user_devices-cache-miss" }, false };
    TUnistatSignal<double> CacheExpired { { "user_devices-cache-expired" }, false };
    TUnistatSignal<double> CacheInvalidated { { "user_devices-cache-invalidated" }, false };
}

TUserDevicesDB::TUserDevicesDB(const IHistoryContext& context)
    : IAutoActualization("user_devices-cache", TDuration::Seconds(1))
    , TBaseEntityManager(context.GetDatabase())
    , HistoryManager(context)
    , DefaultLifetime(TDuration::Seconds(1000))
    , TableName("user_devices")
    , ObjectCache(128 * 1024)
{
}

TUserDevicesDB::TExpectedUserDevices TUserDevicesDB::GetCachedUserDevices(const TString& userId, TInstant statementDeadline) const {
    auto impl = [&] {
        auto eventLogger = NDrive::GetThreadEventLogger();
        auto lifetimeKey = "user_devices.cache_lifetime";
        auto lifetime = NDrive::HasServer()
            ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
            : Nothing();
        auto now = Now();
        auto threshold = now - lifetime.GetOrElse(DefaultLifetime);
        auto optionalObject = ObjectCache.find(userId);
        if (optionalObject && optionalObject->Timestamp > threshold) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "UserDevicesCacheHit")
                    ("user_id", userId)
                    ("timestamp", NJson::ToJson(optionalObject->Timestamp))
                );
            }
            CacheHit.Signal(1);
            return std::move(*optionalObject);
        }
        if (optionalObject) {
            CacheExpired.Signal(1);
        }

        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "UserDevicesCacheMiss")
                ("user_id", userId)
            );
        }

        auto lockTimeout = TDuration::Zero();
        auto statementTimeout = statementDeadline - now;
        auto session = BuildSession(true, false, lockTimeout, statementTimeout);
        auto restoredObject = RestoreUserDevices(userId, session);
        if (!restoredObject) {
            session.Check();
        }

        ObjectCache.update(userId, *restoredObject);
        CacheMiss.Signal(1);
        return std::move(*restoredObject);
    };
    return WrapUnexpected<TCodedException>(impl);
}

TUserDevicesDB::TOptionalUserDevices TUserDevicesDB::RestoreUserDevices(const TString& userId, NDrive::TEntitySession& session) const {
    auto queryOptions = TQueryOptions()
        .AddGenericCondition("user_id", userId);
    auto timestamp = Now();
    auto optionalDevices = Fetch(session, queryOptions);
    if (!optionalDevices) {
        return {};
    }
    TUserDevices result;
    result.Values = std::move(*optionalDevices);
    result.Timestamp = timestamp;
    return result;
}

bool TUserDevicesDB::Refresh() {
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!LastEventId) {
        LastEventId = HistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = HistoryManager.GetEvents<TShortUserDevice>(since, session, 1000);
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEventsSince " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& ev : *optionalEvents) {
        LastEventId = std::max(LastEventId.GetOrElse(0), ev.GetHistoryEventId());
        bool erased = ObjectCache.erase(ev.GetUserId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << ev.GetUserId() << Endl;
        }
    }
    return true;
}

bool TUserDevicesDB::GetUserDevices(const TSet<TString>& userIds, TSet<TShortUserDevice>& result, NDrive::TEntitySession& session) const {
    auto queryOptions = TQueryOptions()
        .SetGenericCondition("user_id", userIds)
    ;
    auto optionalDevices = Fetch(session, queryOptions);
    if (!optionalDevices) {
        return false;
    }
    result.insert(optionalDevices->begin(), optionalDevices->end());
    return true;
}

bool TUserDevicesDB::GetUsersByDeviceIds(const TSet<TString>& deviceIds, TSet<TString>& userIds, NDrive::TEntitySession& session, bool verifiedOnly) const {
    {
        auto queryOptions = TQueryOptions()
            .SetGenericCondition("device_id", deviceIds)
        ;
        if (verifiedOnly) {
            queryOptions.SetGenericCondition("verified", true);
        }
        auto optionalDevices = Fetch(session, queryOptions);
        if (!optionalDevices) {
            NDrive::TEventLog::Log("GetUsersByDeviceIdsError", NJson::TMapBuilder
                ("device_ids", NJson::ToJson(deviceIds))
                ("verified_only", verifiedOnly)
                ("errors", session.GetReport())
            );
            return false;
        }
        for (auto&& device : *optionalDevices) {
            userIds.insert(device.GetUserId());
        }
        return true;
    }
}

TUserDevicesManager::TUserDevicesManager(const IServerBase& server, const TUserDevicesManagerConfig& config)
    : TDatabaseSessionConstructor(server.GetDatabase(config.GetDBName()))
    , Config(config)
{
    CHECK_WITH_LOG(!!Database);
    Table = Database->GetTable("user_devices");
    CHECK_WITH_LOG(!!Table);
    HistoryContext = MakeHolder<THistoryContext>(Database);
    UserDevicesDB = MakeHolder<TUserDevicesDB>(*HistoryContext);
    Y_ENSURE_BT(UserDevicesDB->Start());
    HistoryManager = &UserDevicesDB->GetHistoryManager();
    if (Config.GetEventsRegistratorConfig()) {
        EventRegistrator.Reset(Config.GetEventsRegistratorConfig()->BuildRegistrator());
    }
}

TUserDevicesManager::~TUserDevicesManager() {
    if (!UserDevicesDB->Stop()) {
        ERROR_LOG << "cannot stop UserDevicesDB" << Endl;
    }
}

void TPassportUserDevicesManagerConfig::Init(const TYandexConfig::Section* section) {
    TSimpleAsyncRequestSender::TConfig::Init(section);
    TUserDevicesManagerConfig::Init(section);
    Consumer = section->GetDirectives().Value("Consumer", Consumer);
    SubmitUri = section->GetDirectives().Value("SubmitUri", SubmitUri);
    CommitUri = section->GetDirectives().Value("CommitUri", CommitUri);
    ValidateUri = section->GetDirectives().Value("ValidateUri", ValidateUri);
    CreateTrackUri = section->GetDirectives().Value("CreateTrackUri", CreateTrackUri);
}

void TPassportUserDevicesManagerConfig::ToString(IOutputStream& os) const {
    TSimpleAsyncRequestSender::TConfig::ToString(os);
    TUserDevicesManagerConfig::ToString(os);
    os << "Consumer: " << Consumer << Endl;
    os << "SubmitUri: " << SubmitUri << Endl;
    os << "CommitUri: " << CommitUri << Endl;
    os << "ValidateUri: " << ValidateUri << Endl;
    os << "CreateTrackUri: " << CreateTrackUri << Endl;
}

THolder<IEventsRegistrator> TFakeEventsRegistratorConfig::BuildRegistrator() const {
    return THolder(new TFakeEventsRegistrator(*this));
}

void TFakeEventsRegistratorConfig::Init(const TYandexConfig::Section* section) {
    const TYandexConfig::TSectionsMap sections = section->GetAllChildren();
    auto itEvents = sections.find("Events");
    if (itEvents == sections.end()) {
        return;
    }

    for (const auto& globalEvent : itEvents->second->GetDirectives()) {
        AvailableEvents[IUserDevicesManager::EEventGlobalTypes::Global].insert(globalEvent.first);
    }
    for (const auto& eventType : itEvents->second->GetAllChildren()) {
        IUserDevicesManager::EEventGlobalTypes enumEventType = IUserDevicesManager::EEventGlobalTypes::Global;
        if (!TryFromString(eventType.first, enumEventType)) {
            continue;
        }
        for (const auto& event : eventType.second->GetDirectives()) {
            AvailableEvents[enumEventType].insert(event.first);
        }
    }
}

void TFakeEventsRegistratorConfig::ToString(IOutputStream& os) const {
    os << "<Events>" << Endl;
    for (auto&&[key, name] : GetEnumNames<IUserDevicesManager::EEventGlobalTypes>()) {
        auto it = AvailableEvents.find(key);
        if (it != AvailableEvents.end()) {
            if (key != IUserDevicesManager::EEventGlobalTypes::Global) {
                os << "<" << name << ">" << Endl;
            }
            for (const auto& event : it->second) {
                os << event << " : 1" << Endl;
            }
            if (key != IUserDevicesManager::EEventGlobalTypes::Global) {
                os << "</" << name << ">" << Endl;
            }
        }
    }
    os << "</Events>" << Endl;
}

TExpected<bool, IUserDevicesManager::EEventResult>  TFakeEventsRegistrator::RegisterEvent(const IUserDevicesManager::EEventGlobalTypes type, const TString& key, const TShortUserDevice& /*userDevice*/, const TInstant /*timestamp*/) const {
    auto itType = Config.GetAvailableEvents().find(type);
    if (itType != Config.GetAvailableEvents().end()) {
        auto itEvent = itType->second.find(key);
        if (itEvent != itType->second.end()) {
            ++EventsCounter[type][key];
            return true;
        }
    }
    return MakeUnexpected(IUserDevicesManager::EEventResult::UnknownSignal);
}

TAdjustEventRegistratorConfig::TTokenTypeNames TAdjustEventRegistratorConfig::DefaultTokenNames = { { EAdvertisingTokenType::ADID, "gps_adid" },{ EAdvertisingTokenType::IDFA, "idfa" },{ EAdvertisingTokenType::OAID, "oaid" } };

void TAdjustEventRegistratorConfig::Init(const TYandexConfig::Section* section) {
    ClientConfig.Init(section, nullptr);
    const TYandexConfig::TSectionsMap sections = section->GetAllChildren();
    TokenTypeNames = DefaultTokenNames;
    auto itNames = sections.find("TokenTypes");
    for (auto&&[key, name] : GetEnumNames<EAdvertisingTokenType>()) {
        if (key == EAdvertisingTokenType::UNDEFINED) {
            continue;
        }

        TString typeName;
        if (itNames != sections.end() && itNames->second->GetDirectives().GetValue(name, typeName)) {
            TokenTypeNames[key] = typeName;
        }
        AssertCorrectConfig(!!TokenTypeNames[key], "incorrect token type for " + name);
    }

    auto itEvents = sections.find("Events");
    AssertCorrectConfig(itEvents != sections.end(), "absent events");

    for (const auto& globalEvent : itEvents->second->GetDirectives()) {
        AssertCorrectConfig(EventTokens[IUserDevicesManager::EEventGlobalTypes::Global].emplace(globalEvent.first, globalEvent.second).second, "double event");
    }
    for (const auto& eventType : itEvents->second->GetAllChildren()) {
        IUserDevicesManager::EEventGlobalTypes enumEventType = IUserDevicesManager::EEventGlobalTypes::Global;
        AssertCorrectConfig(TryFromString(eventType.first, enumEventType), "undefined event type " + eventType.first);
        for (const auto& event : eventType.second->GetDirectives()) {
            AssertCorrectConfig(EventTokens[enumEventType].emplace(event.first, event.second).second, "double event");
        }
    }
};

void TAdjustEventRegistratorConfig::ToString(IOutputStream& os) const {
    ClientConfig.ToString(os);
    os << "<TokenTypes>" << Endl;
    for (const auto&[type, name] : TokenTypeNames) {
        os << ::ToString(type) << " : " << name << Endl;
    }
    os << "</TokenTypes>" << Endl;

    os << "<Events>" << Endl;
    for (auto&&[key, name] : GetEnumNames<IUserDevicesManager::EEventGlobalTypes>()) {
        if (key == IUserDevicesManager::EEventGlobalTypes::Undefined) {
            continue;
        }
        auto it = EventTokens.find(key);
        if (it != EventTokens.end()) {
            if (key != IUserDevicesManager::EEventGlobalTypes::Global) {
                os << "<" << name << ">" << Endl;
            }
            for (const auto& event : it->second) {
                os << event.first << " : " << event.second << Endl;
            }
            if (key != IUserDevicesManager::EEventGlobalTypes::Global) {
                os << "</" << name << ">" << Endl;
            }
        }
    }
    os << "</Events>" << Endl;
};

THolder<IEventsRegistrator> TAdjustEventRegistratorConfig::BuildRegistrator() const {
    return THolder(new TAdjustEventRegistrator(*this));
};

TExpected<bool, IUserDevicesManager::EEventResult>  TAdjustEventRegistrator::RegisterEvent(const IUserDevicesManager::EEventGlobalTypes type, const TString& key, const TShortUserDevice& userDevice, const TInstant timestamp) const {
    if (!userDevice.GetAdvertisingToken()) {
        TUnistatSignalsCache::SignalAdd("check_adv", "empty", 1);
        return MakeUnexpected(IUserDevicesManager::EEventResult::NoToken);
    }
    auto it = Config.GetTokenTypeNames().find(userDevice.GetAdvertisingTokenType());
    if (it == Config.GetTokenTypeNames().end()) {
        TUnistatSignalsCache::SignalAdd("check_adv", "undefined_" + ::ToString(userDevice.GetAdvertisingTokenType()), 1);
        return MakeUnexpected(IUserDevicesManager::EEventResult::NoToken);
    }

    TUnistatSignalsCache::SignalAdd("register_adv", ::ToString(type) + "_" + key, 1);

    if (type == IUserDevicesManager::EEventGlobalTypes::Undefined) {
        if (!Client.RegisterEvent(key, userDevice.GetAdvertisingToken(), it->second, timestamp)) {
            return MakeUnexpected(IUserDevicesManager::EEventResult::ClientError);
        }
        return true;
    }

    auto itType = Config.GetEventTokens().find(type);
    if (itType != Config.GetEventTokens().end()) {
        auto itEvent = itType->second.find(key);
        if (itEvent != itType->second.end()) {
            if (!Client.RegisterEvent(itEvent->second, userDevice.GetAdvertisingToken(), it->second, timestamp)) {
                return MakeUnexpected(IUserDevicesManager::EEventResult::ClientError);
            }
            return true;
        }
    }

    TUnistatSignalsCache::SignalAdd("check_adv", "undefined_token_for_" + ::ToString(type) + "_" + key, 1);
    return MakeUnexpected(IUserDevicesManager::EEventResult::UnknownSignal);
}

void TUserDevicesManagerConfig::Init(const TYandexConfig::Section* section) {
    DBName = section->GetDirectives().Value("DBName", DBName);
    MinPropositionsInterval = section->GetDirectives().Value("MinPropositionsInterval", MinPropositionsInterval);
    AssertCorrectConfig(!!DBName, "Incorrect DBName for user devices manager");
    for (auto&& child : section->GetAllChildren()) {
        if (child.first == "Headers") {
            for (auto&& type : GetEnumNames<EAdvertisingTokenType>()) {
                auto it = child.second->GetDirectives().find(type.second);
                if (it != child.second->GetDirectives().end()) {
                    AssertCorrectConfig(TypeHeaders.emplace(type.first, it->second).second, "Double header" + type.second);
                }
            }
        }
    }
    TString ignoredTokens = section->GetDirectives().Value<TString>("IgnoredTokens");
    StringSplitter(ignoredTokens).SplitBySet(", ").SkipEmpty().Collect(&IgnoredTokens);
    auto subSections = section->GetAllChildren();
    auto it = subSections.find("EventsRegistrator");
    if (it != subSections.end()) {
        TString registratorType;
        AssertCorrectConfig(it->second->GetDirectives().GetValue("Type", registratorType), "registrator type absent");
        EventsRegistratorConfig.Reset(IEventsRegistratorConfig::TFactory::Construct(registratorType));
        AssertCorrectConfig(!!EventsRegistratorConfig, "incorrect registrator type " + registratorType);
        EventsRegistratorConfig->Init(it->second);
    }
}

void TUserDevicesManagerConfig::ToString(IOutputStream& os) const {
    os << "DBName: " << DBName << Endl;
    if (TypeHeaders) {
        os << "<Headers>" << Endl;
        for (auto&& type : TypeHeaders) {
            os << ::ToString(type.first) << " : " << type.second << Endl;
        }
        os << "</Headers>" << Endl;
    }
    os << "IgnoredTokens: " << JoinSeq(",", IgnoredTokens) << Endl;
    if (EventsRegistratorConfig) {
        os << "<EventsRegistrator>" << Endl;
        os << "Type : " << EventsRegistratorConfig->GetType() << Endl;
        EventsRegistratorConfig->ToString(os);
        os << "</EventsRegistrator>" << Endl;
    }
}

TString TUserDevicesManagerConfig::GetAdvertisingHeader(const EAdvertisingTokenType type) const {
    auto it = TypeHeaders.find(type);
    if (it != TypeHeaders.end()) {
        return it->second;
    }
    return ::ToString(type);
}

bool TUserDevicesManager::GetFullUserDevices(const TString& userId, TVector<TUserDevice>& devices, NDrive::TEntitySession& session) const {
    auto optionalDevices = UserDevicesDB->RestoreUserDevices(userId, session);
    if (!optionalDevices) {
        return false;
    }
    devices = std::move(optionalDevices->Values);
    return true;
}

ENewDeviceStatus TUserDevicesManager::CheckDeviceId(const TString& userId, const TString& deviceId) const {
    const auto expectedDevices = UserDevicesDB->GetCachedUserDevices(userId);
    const auto& devices = expectedDevices->Values;
    if (devices.empty()) {
        return ENewDeviceStatus::NoDevices;
    }
    for (auto&& i : devices) {
        if (i.GetDeviceId() == deviceId) {
            if (!i.GetEnabled()) {
                return ENewDeviceStatus::Disabled;
            }
            return i.GetVerified() ? ENewDeviceStatus::Verified : ENewDeviceStatus::Verification;
        }
    }
    return ENewDeviceStatus::New;
}

void TUserDevicesManager::AddAdvertisingInfo(TUserDevice& device, IReplyContext::TPtr context) const {
    if (!context) {
        return;
    }
    for (auto&& type : GetEnumAllValues<EAdvertisingTokenType>()) {
        if (type == EAdvertisingTokenType::UNDEFINED) {
            continue;
        }
        TStringBuf deviceTokenEncode = context->GetRequestData().HeaderInOrEmpty(Config.GetAdvertisingHeader(type));
        if (deviceTokenEncode) {
            TString deviceToken = Base64Decode(deviceTokenEncode);
            if (!Config.GetIgnoredTokens().contains(deviceToken)) {
                device.SetAdvertisingToken(deviceToken);
                device.SetAdvertisingTokenType(type);
                device.SetAdvertisingTokenTimestamp(context->GetRequestStartTime());
            }
            break;
        }
    }
}

TMaybe<TUserDevice> TUserDevicesManager::StartDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, NDrive::TEntitySession& session, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes, const bool dropVerifiedStatus) const {
    auto optionalEvents = HistoryManager->GetUser(userId, session, ModelingNow() - Config.GetMinPropositionsInterval());
    if (!optionalEvents) {
        return {};
    } else {
        for (auto&& ev : *optionalEvents) {
            if (ev.GetDeviceId() == deviceId && ev.GetHistoryAction() == EObjectHistoryAction::Proposition) {
                TUnistatSignalsCache::SignalAdd("frontend", "device-verification-start-fail", 1);
                errorCodes.emplace("too_often");
                return Nothing();
            }
        }
    }
    TString token;
    if (!StartPhoneVerification(token, phoneNumber, divc, context, verificationMethod, errorCodes)) {
        return Nothing();
    }
    TUserDevice uDevice(userId, deviceId, phoneNumber);
    uDevice.SetVerified(!dropVerifiedStatus);
    uDevice.InitDescription(context);
    AddAdvertisingInfo(uDevice, context);
    NStorage::TTableRecord record = uDevice.SerializeToTableRecord();
    record.Set("token", token);

    NStorage::TTableRecord condition;
    condition.Set("user_id", userId).Set("device_id", deviceId);
    auto result = Table->Upsert(record, session.GetTransaction(), condition);
    if (!HistoryManager->AddHistory(uDevice, userId, EObjectHistoryAction::Proposition, session)) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-start-fail", 1);
        ERROR_LOG << "Cannot start device verification: " << session.GetStringReport() << Endl;
        return Nothing();
    }
    if (!result || !result->IsSucceed()) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-start-fail", 1);
        ERROR_LOG << "Cannot start device verification: " << session.GetStringReport() << Endl;
        return Nothing();
    }
    TUnistatSignalsCache::SignalAdd("frontend", "device-verification-start-success", 1);
    return uDevice;
}

bool TUserDevicesManager::StartPhoneVerification(TString& token, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, const EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const {
    auto deadline = context ? context->GetRequestDeadline() : TInstant::Zero();
    auto optionalToken = CreateTrack(divc.ClientIp, deadline, errorCodes);
    if (!optionalToken) {
        TUnistatSignalsCache::SignalAdd("frontend", "phone-verification-start-fail", 1);
        ERROR_LOG << JoinSeq(", ", errorCodes) << Endl;
        return false;
    }
    if (verificationMethod == IUserDevicesManager::EVerificationMethod::PhoneCall || verificationMethod == IUserDevicesManager::EVerificationMethod::FlashCall) {
        if (!ValidatePhoneNumber(phoneNumber, *optionalToken, divc.ClientIp, deadline, verificationMethod, errorCodes)) {
            TUnistatSignalsCache::SignalAdd("frontend", "phone-verification-start-fail", 1);
            ERROR_LOG << JoinSeq(", ", errorCodes) << Endl;
            return false;
        }
    }
    if (!DoStartPhoneVerification(phoneNumber, *optionalToken, divc, deadline, verificationMethod, errorCodes)) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-start-fail", 1);
        ERROR_LOG << JoinSeq(", ", errorCodes) << Endl;
        return false;
    }
    token = *optionalToken;
    return true;
}

bool TUserDevicesManager::FinishDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes, NDrive::TEntitySession& session, TString* confirmedPhoneNumber, bool skipVerifiedStatus) const {
    auto transaction = session.GetTransaction();
    NStorage::TObjectRecordsSet<TUserDevice> records;
    {
        auto query = "user_id=" + transaction->Quote(userId) + " AND device_id=" + transaction->Quote(deviceId) + " AND token != ''";
        if (!skipVerifiedStatus) {
            query += " AND NOT verified";
        }
        auto result = Table->GetRows(query, records, transaction);
        if (!result || !result->IsSucceed()) {
            TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-fail", 1);
            ERROR_LOG << "Cannot check user_device: " << transaction->GetErrors().GetStringReport() << Endl;
            return false;
        }
        if (records.size() != 1) {
            TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-fail", 1);
            ERROR_LOG << "verification not started for user " << userId << ": " << transaction->GetErrors().GetStringReport() << Endl;
            return false;
        }
    }
    if (!FinishPhoneVerification(records.front().GetToken(), code, clientIp, deadline, errorCodes)) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-fail", 1);
        return false;
    }

    TUserDevice uDeviceNew = records.front();
    uDeviceNew.SetVerified(true);

    NStorage::TTableRecord condition;
    condition.Set("user_id", userId).Set("device_id", deviceId);
    auto result = Table->UpdateRow(condition, uDeviceNew.SerializeToTableRecord(), transaction);
    if (!HistoryManager->AddHistory(uDeviceNew, userId, EObjectHistoryAction::Confirmation, session)) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-fail", 1);
        ERROR_LOG << "Cannot start device verification: " << session.GetStringReport() << Endl;
        return false;
    }
    if (!!result && result->IsSucceed()) {
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-success", 1);
        if (confirmedPhoneNumber) {
            *confirmedPhoneNumber = uDeviceNew.GetPhone();
        }
        return true;
    } else {
        ERROR_LOG << "Cannot start device verification: " << session.GetStringReport() << Endl;
        TUnistatSignalsCache::SignalAdd("frontend", "device-verification-finish-fail", 1);
        return false;
    }
}

bool TUserDevicesManager::RemoveDevices(const TString& userId, const TSet<TString>& devices, const TString& historyUserId) const {
    if (devices.empty()) {
        return true;
    }
    auto session = BuildTx<NSQL::Writable>();

    for (auto&& i : devices) {
        NStorage::TTableRecord condition;
        condition.Set("user_id", userId).Set("device_id", i);
        NStorage::TObjectRecordsSet<TUserDevice> records;
        auto result = Table->RemoveRow(condition, session.GetTransaction(), &records);
        if (records.size() != 1) {
            ERROR_LOG << "Incorrect records.size: " << records.size() << Endl;
            return false;
        }
        if (!result || !result->IsSucceed()) {
            ERROR_LOG << "Cannot remove device: " << session.GetStringReport() << Endl;
            return false;
        }
        if (!HistoryManager->AddHistory(records.front(), historyUserId, EObjectHistoryAction::Remove, session)) {
            ERROR_LOG << "Cannot write history about device removing: " << session.GetStringReport() << Endl;
            return false;
        }
    }
    if (!session.Commit()) {
        ERROR_LOG << "Cannot commit devices removing: " << session.GetStringReport() << Endl;
        return false;
    }
    return true;
}

bool TUserDevicesManager::SetDevicesFeatures(const TString& userId, const TSet<TString>& devices, const TMaybe<bool> enabled, const TMaybe<bool> verified, const TString& historyUserId) const {
    auto session = BuildTx<NSQL::Writable>();

    NStorage::TTableRecord record;
    if (enabled) {
        record.Set("enabled", *enabled);
    }
    if (verified) {
        record.Set("verified", *verified);
    }
    if (record.Empty()) {
        return true;
    }

    for (auto&& i : devices) {
        NStorage::TTableRecord condition;
        condition.Set("user_id", userId).Set("device_id", i);
        NStorage::TObjectRecordsSet<TUserDevice> records;
        auto result = Table->UpdateRow(condition, record, session.GetTransaction(), &records);
        if (records.size() != 1) {
            ERROR_LOG << "Incorrect records.size: " << records.size() << Endl;
            return false;
        }
        if (!result || !result->IsSucceed()) {
            ERROR_LOG << "Cannot change device enabled/verified: " << session.GetStringReport() << Endl;
            return false;
        }
        if (!HistoryManager->AddHistory(records.front(), historyUserId, EObjectHistoryAction::UpdateData, session)) {
            ERROR_LOG << "Cannot write history about device features changing: " << session.GetStringReport() << Endl;
            return false;
        }
    }
    if (!session.Commit()) {
        ERROR_LOG << "Cannot commit devices features change: " << session.GetStringReport() << Endl;
        return false;
    }
    return true;
}

bool TUserDevicesManager::GetUserDevices(const TString& userId, TVector<TShortUserDevice>& devices) const {
    NStorage::TObjectRecordsSet<TShortUserDevice> records;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        NStorage::ITransaction::TPtr transaction = session.GetTransaction();
        auto result = Table->GetRows("user_id=" + transaction->Quote(userId), records, session.GetTransaction());
        if (!result || !result->IsSucceed()) {
            ERROR_LOG << "Cannot check user_device: " << session.GetStringReport() << Endl;
            return false;
        }
    }
    for (auto&& i : records) {
        devices.emplace_back(std::move(i));
    }
    return true;
}

bool TUserDevicesManager::AddFirstDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context) const {
    auto session = BuildTx<NSQL::Writable>();
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    TUserDevice uDevice(userId, deviceId);
    uDevice.InitDescription(context);
    AddAdvertisingInfo(uDevice, context);
    NStorage::TTableRecord record = uDevice.SerializeToTableRecord();
    NStorage::TObjectRecordsSet<TUserDevice> records;
    auto result = Table->AddRow(record, transaction, "NOT EXISTS (SELECT device_id FROM user_devices WHERE user_id=" + transaction->Quote(userId) + ")", &records);
    if (!result || !result->IsSucceed()) {
        ERROR_LOG << "Cannot insert first user_device: " << session.GetStringReport() << Endl;
        return false;
    }
    if (records.size() == 0) {
        return false;
    }
    for (auto&& i : records) {
        if (!HistoryManager->AddHistory(i, userId, EObjectHistoryAction::Add, session)) {
            ERROR_LOG << "Cannot insert first user_device_history: " << session.GetStringReport() << Endl;
            return false;
        }
    }
    if (!session.Commit()) {
        ERROR_LOG << "Cannot commit first user_device: " << session.GetStringReport() << Endl;
        return false;
    }
    TUnistatSignalsCache::SignalAdd("frontend", "device-verification-first-device", 1);
    return true;
}

bool TUserDevicesManager::GetFullHistoryByUser(const TString& userId, TVector<TObjectEvent<TUserDevice>>& events, NDrive::TEntitySession& session) const {
    return UserDevicesDB->GetFullHistoryByUser(userId, events, session);
}

bool TUserDevicesManager::GetUserDevices(const TSet<TString>& userIds, TSet<TShortUserDevice>& devices, NDrive::TEntitySession& session) const {
    return UserDevicesDB->GetUserDevices(userIds, devices, session);
}

bool TUserDevicesManager::GetUsersByDeviceIds(const TSet<TString>& deviceIds, TSet<TString>& userIds, NDrive::TEntitySession& session, bool verifiedOnly) const {
    return UserDevicesDB->GetUsersByDeviceIds(deviceIds, userIds, session, verifiedOnly);
}


TDuration TUserDevicesManager::GetMinPropositionsInterval() const {
    return Config.GetMinPropositionsInterval();
}

bool TUserDevicesManager::GetLastAdvertisingDevice(const TString& userId, TMaybe<TShortUserDevice>& result, const TInstant reqActuality, const TSet<EAdvertisingTokenType>& typeFilter) const {
    Y_UNUSED(reqActuality);
    const auto expectedDevices = UserDevicesDB->GetCachedUserDevices(userId);
    const auto& allDevices = expectedDevices->Values;
    for (auto&& device : allDevices) {
        if (!device.GetAdvertisingToken() || device.GetAdvertisingTokenType() == EAdvertisingTokenType::UNDEFINED || Config.GetIgnoredTokens().contains(device.GetAdvertisingToken()) || !device.GetEnabled() || !device.GetVerified()) {
            continue;
        }
        if (typeFilter && !typeFilter.contains(device.GetAdvertisingTokenType())) {
            continue;
        }
        if (!result.Defined() || result->GetAdvertisingTokenTimestamp() < device.GetAdvertisingTokenTimestamp()) {
            result = device;
        }
    }
    return true;
}

bool TUserDevicesManager::RegisterAdvertisingDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context, const TInstant activatingTime, const NOpenssl::IAbstractCipher* cipher) const {
    if (!context) {
        return true;
    }
    const auto expectedDevices = UserDevicesDB->GetCachedUserDevices(userId);
    const auto& allDevices = expectedDevices->Values;
    const TShortUserDevice* itDevice = nullptr;
    for (auto&& i : allDevices) {
        if (i.GetDeviceId() == deviceId) {
            itDevice = &i;
            break;
        }
    }

    if (!itDevice) {
        TUnistatSignalsCache::SignalAdd("user_device", "add_advertising_first", 1);
        return true;
    }

    for (auto&& type : GetEnumAllValues<EAdvertisingTokenType>()) {
        if (type == EAdvertisingTokenType::UNDEFINED) {
            continue;
        }
        TStringBuf deviceTokenEncode = context->GetRequestData().HeaderInOrEmpty(Config.GetAdvertisingHeader(type));
        if (deviceTokenEncode) {
            TString deviceToken = Base64Decode(deviceTokenEncode);
            if (cipher) {
                TString decryptToken;
                if (!cipher->Decrypt(deviceToken, decryptToken)) {
                    TUnistatSignalsCache::SignalAdd("user_device", "incorrect_ecryption", 1);
                    return false;
                }
                deviceToken = decryptToken;
            }
            if (!Config.GetIgnoredTokens().contains(deviceToken) && (itDevice->GetAdvertisingToken() != deviceToken || itDevice->GetAdvertisingTokenType() == EAdvertisingTokenType::UNDEFINED)) {
                NStorage::TTableRecord record;
                record.Set("advertising_token", deviceToken);
                record.Set("advertising_token_type", ::ToString(type));
                record.Set("advertising_token_timestamp", activatingTime);

                auto session = BuildTx<NSQL::Writable>();
                NStorage::TTableRecord condition;
                condition.Set("user_id", userId).Set("device_id", deviceId);

                NStorage::TObjectRecordsSet<TUserDevice> records;
                auto result = Table->UpdateRow(condition, record, session.GetTransaction(), &records);
                if (records.size() != 1) {
                    ERROR_LOG << "Incorrect records.size: " << records.size() << Endl;
                    return false;
                }
                if (!result || !result->IsSucceed()) {
                    ERROR_LOG << "Cannot change device advertising token: " << session.GetStringReport() << Endl;
                    return false;
                }
                if (!HistoryManager->AddHistory(records.front(), userId, EObjectHistoryAction::UpdateData, session)) {
                    ERROR_LOG << "Cannot write history about device features changing: " << session.GetStringReport() << Endl;
                    return false;
                }
                if (!session.Commit()) {
                    ERROR_LOG << "Cannot commit devices features change: " << session.GetStringReport() << Endl;
                    return false;
                }
                if (itDevice->GetAdvertisingToken()) {
                    TUnistatSignalsCache::SignalAdd("user_device", "change_advertising_token_" + ::ToString(type), 1);
                } else {
                    TUnistatSignalsCache::SignalAdd("user_device", "add_advertising_token_" + ::ToString(type), 1);
                }
            }
            break;
        }
    }
    return true;
}

TExpected<bool, IUserDevicesManager::EEventResult> TUserDevicesManager::RegisterEvent(const TUserDevicesManager::EEventGlobalTypes type, const TString& key, const TString& userId, const TInstant timestamp) const {
    if (!EventRegistrator) {
        return MakeUnexpected(EEventResult::NoEventRegistrator);
    }
    TMaybe<TShortUserDevice> userDevice;
    if (!GetLastAdvertisingDevice(userId, userDevice, timestamp) || !userDevice.Defined()) {
        TUnistatSignalsCache::SignalAdd("frontend", "no-adv-before-" + ::ToString(type) + "_" + key, 1);
        return MakeUnexpected(EEventResult::NoToken);
    }
    return EventRegistrator->RegisterEvent(type, key, *userDevice, timestamp);
}

bool TUserDevicesManager::GetLastDeviceEvent(const TSet<TString>& userIds, const TSet<EObjectHistoryAction>& includeActions, const TSet<EObjectHistoryAction>& excludeActions, NDrive::TEntitySession& session, TMaybe<TObjectEvent<TUserDevice>>& result) const {
    TMaybe<TObjectEvent<TUserDevice>> resultEvent;
    auto queryOptions = NSQL::TQueryOptions().SetGenericCondition("user_id", userIds);
    auto events = HistoryManager->GetEvents({}, {}, session, queryOptions);
    if (!events) {
        return false;
    }
    if (events->size() == 0) {
        result = Nothing();
        return true;
    }
    for (auto&& event : *events) {
        if ((!resultEvent || event.GetHistoryTimestamp() > resultEvent->GetHistoryTimestamp()) &&
            (includeActions.empty() || includeActions.contains(event.GetHistoryAction())) &&
            !excludeActions.contains(event.GetHistoryAction())) {
            resultEvent = std::move(event);
        }
    }
    result = resultEvent;
    return true;
}

namespace {
    TMutex Mutex;
    TMap<TString, TString> TokensInfo;
}

const TMap<TString, TString>& TFakeUserDevicesManager::GetCodes() {
    TGuard<TMutex> g(Mutex);
    return TokensInfo;
}

TString TFakeUserDevicesManager::GetCodeByToken(const TString& token) {
    TGuard<TMutex> g(Mutex);
    auto it = TokensInfo.find(token);
    if (it == TokensInfo.end()) {
        return "";
    }
    return it->second;
}

bool TFakeUserDevicesManager::FinishPhoneVerification(const TString& token, const TString& code, const TString& /*clientIp*/, TInstant /*deadline*/, TSet<TString>& /*errorCodes*/) const {
    TGuard<TMutex> g(Mutex);
    auto it = TokensInfo.find(token);
    if (it == TokensInfo.end()) {
        return false;
    }
    return it->second == code;
}

bool TFakeUserDevicesManager::DoStartPhoneVerification(const TString& /*number*/, const TString& /*token*/, const TDeviceIdVerificationContext& /*divc*/, TInstant /*deadline*/, const IUserDevicesManager::EVerificationMethod /*verificationMethod*/, TSet<TString>& /*errorCodes*/) const {
    return true;
}

TMaybe<TString> TFakeUserDevicesManager::CreateTrack(const TString& /*clientIp*/, TInstant /*deadline*/, TSet<TString>& /*errorCodes*/) const {
    TGuard<TMutex> g(Mutex);
    const ui64 tokenInt = RandomNumber<ui64>();
    const ui64 code = RandomNumber<ui16>();
    const auto token = ToString(tokenInt);
    TokensInfo.emplace(token, ToString(code));
    return token;
}

bool TFakeUserDevicesManager::ValidatePhoneNumber(const TString& /*number*/, const TString& /*token*/, const TString& /*clientIp*/, TInstant /*deadline*/, const TMaybe<IUserDevicesManager::EVerificationMethod> /*verificationMethod*/, TSet<TString>& /*errorCodes*/) const {
    return true;
}

THolder<IUserDevicesManager> TPassportUserDevicesManagerConfig::BuildManager(const IServerBase& server) const {
    return THolder(new TPassportUserDevicesManager(server, *this));
}

THolder<IUserDevicesManager> TFakeUserDevicesManagerConfig::BuildManager(const IServerBase& server) const {
    return THolder(new TFakeUserDevicesManager(server, *this));
}

TMaybe<NJson::TJsonValue> TPassportUserDevicesManager::SendRequest(const TString& uri, const TString& postData, const TString& clientIp, const TInstant deadline, const TPassportUserDevicesManager::LogData& logData, TSet<TString>& errorCodes) const {
    NNeh::THttpRequest req;
    req.SetUri(uri);
    req.SetCgiData("consumer=" + Config.GetConsumer());
    req.SetPostData(postData);
    req.AddHeader("Ya-Consumer-Client-Ip", clientIp);
    req.AddHeader("Content-Type", "application/x-www-form-urlencoded");

    TInstant requestDeadline = deadline ? deadline : Config.GetRequestConfig().CalcRequestDeadline();
    TInstant start = Now();
    NUtil::THttpReply result = Sender->SendMessageSync(req, requestDeadline);
    TInstant finish = Now();

    auto report = logData.LogReport;
    NJson::InsertField(report, "http_code", result.Code());
    NJson::InsertField(report, "response", result.Serialize());
    NJson::InsertField(report, "duration", (finish - start));
    NDrive::TEventLog::Log(logData.EventName, report);
    TUnistatSignalsCache::SignalAdd("user_device", logData.SignalPrefix + "-code-" + ::ToString(result.Code()), 1);

    NJson::TJsonValue jsonInfo;
    if (!NJson::ReadJsonFastTree(result.Content(), &jsonInfo)) {
        NDrive::TEventLog::Log(logData.EventName, report);
        return Nothing();
    }

    const auto& status = jsonInfo["status"];
    NJson::InsertField(report, "status", status);
    NDrive::TEventLog::Log(logData.EventName, report);

    if (status.GetString() == "ok") {
        TUnistatSignalsCache::SignalAdd("user_device", logData.SignalPrefix + "-success", 1);
        return jsonInfo;
    } else if (jsonInfo["errors"].IsArray()) {
        for (auto&& i : jsonInfo["errors"].GetArray()) {
            if (i.IsString()) {
                errorCodes.emplace(i.GetString());
            }
        }
    }
    TUnistatSignalsCache::SignalAdd("user_device", logData.SignalPrefix + "-fail", 1);
    ERROR_LOG << result.Content() << Endl;
    return Nothing();
}

bool TPassportUserDevicesManager::FinishPhoneVerification(const TString& token, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const {
    TPassportUserDevicesManager::LogData logData("finish-verification", "PassportFinishVerification", NJson::TMapBuilder
        ("client_ip", clientIp)
        ("code", code)
        ("track_id", token)
    );

    auto jsonResult = SendRequest(Config.GetCommitUri(), "track_id=" + token + "&code=" + code, clientIp, deadline, logData, errorCodes);
    return jsonResult.Defined();
}

bool TPassportUserDevicesManager::DoStartPhoneVerification(const TString& number, const TString& token, const TDeviceIdVerificationContext& divc, TInstant deadline, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const {
    const auto& applicationId = divc.ApplicationId;
    const auto& clientIp = divc.ClientIp;
    auto postData = TStringBuilder() << "display_language=ru"
                                     << "&number=" << UrlEscapeRet(number)
                                     << "&confirm_method=" << ToString(verificationMethod)
                                     << "&track_id=" << token;
    bool enableGpsPackageName = divc.EnableGpsPackageName;
    if (enableGpsPackageName) {
        postData << "&gps_package_name=" << applicationId;
    }
    TPassportUserDevicesManager::LogData logData("start-verification", "PassportStartVerification", NJson::TMapBuilder
        ("application_id", applicationId)
        ("client_ip", clientIp)
        ("enable_gps_package_name", enableGpsPackageName)
        ("number", number)
        ("track_id", token)
    );
    auto jsonResult = SendRequest(Config.GetSubmitUri(), postData, clientIp, deadline, logData, errorCodes);
    return jsonResult.Defined();
}

TMaybe<TString> TPassportUserDevicesManager::CreateTrack(const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const {
    TPassportUserDevicesManager::LogData logData("create-track", "PassportCreateTrack", NJson::TMapBuilder
        ("client_ip", clientIp)
    );
    auto jsonResult = SendRequest(Config.GetCreateTrackUri(), "track_type=register", clientIp, deadline, logData, errorCodes);
    if (!jsonResult) {
        return {};
    }
    TString trackId;
    if (!NJson::ParseField(*jsonResult, "id", trackId, true)) {
        errorCodes.emplace("no_track_data");
        return {};
    }
    return trackId;
}

bool TPassportUserDevicesManager::ValidatePhoneNumber(const TString& number, const TString& token, const TString& clientIp, TInstant deadline, const TMaybe<IUserDevicesManager::EVerificationMethod> verificationMethod, TSet<TString>& errorCodes) const {
    auto postData = TStringBuilder() << "phone_number=" << UrlEscapeRet(number)
                                     << "&track_id=" << token;
    if (verificationMethod) {
        postData << "&validate_for_call=true";
    }
    TPassportUserDevicesManager::LogData logData("start-verification-validate-number", "PassportValidatePhoneNumber", NJson::TMapBuilder
        ("number", number)
        ("track_id", token)
    );
    auto jsonResult = SendRequest(Config.GetValidateUri(), postData, clientIp, deadline, logData, errorCodes);
    if (!jsonResult) {
        return false;
    }
    if (!verificationMethod) {
        return true;
    }
    bool isPhoneValid;
    switch (*verificationMethod) {
    case IUserDevicesManager::EVerificationMethod::FlashCall:
        if (!NJson::ParseField(*jsonResult, "valid_for_flash_call", isPhoneValid, true)) {
            errorCodes.emplace("unavailable_for_call");
            return false;
        }
        break;
    case IUserDevicesManager::EVerificationMethod::PhoneCall:
        if (!NJson::ParseField(*jsonResult, "valid_for_call", isPhoneValid, true)) {
            errorCodes.emplace("unavailable_for_call");
            return false;
        }
        break;
    default:
        return true;
    }
    if (!isPhoneValid) {
        errorCodes.emplace("unavailable_for_call");
    }
    return isPhoneValid;
}

bool TUserDevicesDB::GetFullHistoryByUser(const TString& userId, TVector<TObjectEvent<TUserDevice>>& events, NDrive::TEntitySession& session) const {
    auto optionalEvents = GetHistoryManager().GetUser(userId, session);
    if (!optionalEvents) {
        return false;
    }
    events = std::move(*optionalEvents);
    return true;
}
