#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/manager.h>

#include <drive/library/cpp/adjust/client.h>
#include <drive/library/cpp/searchserver/context/replier.h>
#include <drive/library/cpp/threading/concurrent_cache.h>

#include <rtline/util/network/simple.h>

namespace NOpenssl {
    class IAbstractCipher;
}

enum class ENewDeviceStatus {
    Verified,
    Verification,
    New,
    CannotDeterm,
    Disabled,
    NoDevices
};

enum class EAdvertisingTokenType {
    UNDEFINED,
    IDFA,
    ADID,
    OAID,
};

struct TDeviceIdVerificationContext {
    TString ApplicationId;
    TString ClientIp;
    bool EnableGpsPackageName = false;
};

class TShortUserDevice {
    R_FIELD(TString, UserId);
    R_FIELD(TString, DeviceId);
    R_FIELD(bool, Enabled, true);
    R_FIELD(bool, Verified, true);

    R_FIELD(TString, AdvertisingToken);
    R_FIELD(EAdvertisingTokenType, AdvertisingTokenType, EAdvertisingTokenType::UNDEFINED);
    R_FIELD(TInstant, AdvertisingTokenTimestamp, TInstant::Zero());

public:
    TShortUserDevice() = default;
    TShortUserDevice(const TString& userId, const TString& deviceId)
        : UserId(userId)
        , DeviceId(deviceId)
    {
    }

    void DoBuildReportItem(NJson::TJsonValue& report) const;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, DeviceId, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Enabled, -1);
        R_FIELD(i32, Verified, -1);

        R_FIELD(i32, AdvertisingToken, -1);
        R_FIELD(i32, AdvertisingTokenType, -1);
        R_FIELD(i32, AdvertisingTokenTimestamp, -1);

    public:
        static TString GetFieldsForRequest() {
            return "device_id, user_id, enabled, verified, advertising_token, advertising_token_type, advertising_token_timestamp";
        }

        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;

    bool operator<(const TShortUserDevice& item) const {
        if (UserId == item.UserId) {
            return DeviceId < item.DeviceId;
        } else {
            return UserId < item.UserId;
        }
    }
};

class TUserDevice: public TShortUserDevice {
private:
    using TBase = TShortUserDevice;

private:
    R_FIELD(TString, Token);
    R_FIELD(TString, Description);
    R_FIELD(TString, Phone);

public:
    NJson::TJsonValue GetJsonReport() const;
    void DoBuildReportItem(NJson::TJsonValue& report) const;
    void InitDescription(IReplyContext::TPtr context);

    static NJson::TJsonValue GenerateDescriptionJson(IReplyContext::TPtr context);

    class TDecoder: public TShortUserDevice::TDecoder {
    private:
        using TBase = TShortUserDevice::TDecoder;

    private:
        R_FIELD(i32, Token, -1);
        R_FIELD(i32, Description, -1);
        R_FIELD(i32, Phone, -1);

    public:
        static TString GetFieldsForRequest() {
            return "device_id, user_id, enabled, verified, advertising_token, advertising_token_type, advertising_token_timestamp, token, description, phone";
        }

        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);
    NStorage::TTableRecord SerializeToTableRecord() const;

    TUserDevice() = default;
    TUserDevice(const TString& userId, const TString& deviceId, const TString& phoneNumber = "")
        : TBase(userId, deviceId)
        , Phone(phoneNumber)
    {
    }
};

struct TUserDevices {
    TVector<TUserDevice> Values;
    TInstant Timestamp;
};

class IUserDevicesManager {
public:
    enum EEventGlobalTypes {
        Undefined,
        Global,
        FirstTripOfferType,
        FirstTripTag,
        TripOfferType,
        TripTag,
        RegistrationPhotos,
    };

    enum EOfferTypeEvents {
        Success /* "success" */,
        SuccessStandart /* "success_standart" */,
        SuccessPack /* "success_pack" */,
        SuccessFix /* "success_fix" */,
    };

    enum EEventType {
        SuccessRegistration /* "success_registration" */,
        SuccessFirstTrip /* "success_first_trip" */,
        SuccessStandartFirstTrip /* "success_first_trip_standart" */,
        SuccessPackFirstTrip /* "success_first_trip_pack" */,
        SuccessFixFirstTrip /* "success_first_trip_fix" */,
        SuccessCargoTrip /* "success_cargo_trip" */,
        SuccessShuttleTrip /* "success_shuttle_trip" */,
        ReferralCodeGeneration /* "referral_code_generation" */,
    };

    enum EEventResult {
        UnknownSignal,
        NoToken,
        ClientError,
        NoEventRegistrator
    };

    enum class EDeviceVerificationCheckType {
        Verified /* "verified" */,
        NotVerified /* "not_verfied" */,
        NotCheck /* "not_check" */,
    };

    enum class EVerificationMethod {
        Sms /* "by_sms" */,
        PhoneCall /* "by_call" */,
        FlashCall /* "by_flash_call" */,
    };

public:
    virtual ~IUserDevicesManager() = default;
    virtual ENewDeviceStatus CheckDeviceId(const TString& userId, const TString& deviceId) const = 0;
    virtual bool AddFirstDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context) const = 0;
    virtual TMaybe<TUserDevice> StartDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, NDrive::TEntitySession& session, const EVerificationMethod verificationMethod, TSet<TString>& errorCodes, bool dropVerifiedStatus = true) const = 0;
    virtual bool StartPhoneVerification(TString& token, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, const EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const = 0;
    virtual bool FinishPhoneVerification(const TString& token, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const = 0;
    virtual bool FinishDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes, NDrive::TEntitySession& session, TString* confirmedPhoneNumber = nullptr, bool skipVerifiedStatus = false) const = 0;
    virtual bool SetDevicesFeatures(const TString& userId, const TSet<TString>& devices, const TMaybe<bool> enabled, const TMaybe<bool> verified, const TString& historyUserId) const = 0;
    virtual bool RemoveDevices(const TString& userId, const TSet<TString>& devices, const TString& historyUserId) const = 0;
    virtual bool GetFullUserDevices(const TString& userId, TVector<TUserDevice>& devices, NDrive::TEntitySession& session) const = 0;
    virtual bool GetUserDevices(const TString& userId, TVector<TShortUserDevice>& devices) const = 0;
    virtual bool GetUserDevices(const TSet<TString>& userIds, TSet<TShortUserDevice>& devices, NDrive::TEntitySession& session) const = 0;
    virtual bool GetUsersByDeviceIds(const TSet<TString>& deviceIds, TSet<TString>& userIds, NDrive::TEntitySession& session, bool verifiedOnly = false) const = 0;
    virtual TDuration GetMinPropositionsInterval() const = 0;
    virtual bool GetLastAdvertisingDevice(const TString& userId, TMaybe<TShortUserDevice>& result, const TInstant reqActuality, const TSet<EAdvertisingTokenType>& typeFilter = {}) const = 0;
    virtual bool RegisterAdvertisingDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context, const TInstant activatingTime, const NOpenssl::IAbstractCipher* cipher = nullptr) const = 0;
    virtual TExpected<bool, EEventResult> RegisterEvent(const EEventGlobalTypes type, const TString& key, const TString& userId, const TInstant timestamp) const = 0;
    virtual TExpected<bool, EEventResult> RegisterEvent(const EEventType type, const TString& userId, const TInstant timestamp) const {
        return RegisterEvent(EEventGlobalTypes::Global, ::ToString(type), userId, timestamp);
    }
    virtual bool GetFullHistoryByUser(const TString& userId, TVector<TObjectEvent<TUserDevice>>& events, NDrive::TEntitySession& session) const = 0;
    virtual bool GetLastDeviceEvent(const TSet<TString>& userIds, const TSet<EObjectHistoryAction>& includeActions, const TSet<EObjectHistoryAction>& excludeActions, NDrive::TEntitySession& session, TMaybe<TObjectEvent<TUserDevice>>& result) const = 0;
};

class IUserDevicesManagerConfig {
public:
    virtual ~IUserDevicesManagerConfig() = default;
    virtual void Init(const TYandexConfig::Section* section) = 0;
    virtual void ToString(IOutputStream& os) const = 0;
    virtual THolder<IUserDevicesManager> BuildManager(const IServerBase& server) const = 0;

    virtual TDuration GetMinPropositionsInterval() const {
        return MinPropositionsInterval;
    }

protected:
    TDuration MinPropositionsInterval = TDuration::Minutes(1);
};

class IEventsRegistrator;
class IEventsRegistratorConfig {
public:
    virtual ~IEventsRegistratorConfig() = default;
    virtual void Init(const TYandexConfig::Section* section) = 0;
    virtual void ToString(IOutputStream& os) const = 0;
    virtual THolder<IEventsRegistrator> BuildRegistrator() const = 0;
    virtual TString GetType() const = 0;
    typedef NObjectFactory::TObjectFactory<IEventsRegistratorConfig, TString> TFactory;
};

class IEventsRegistrator {
public:
    virtual ~IEventsRegistrator() = default;
    virtual TExpected<bool, IUserDevicesManager::EEventResult> RegisterEvent(const IUserDevicesManager::EEventGlobalTypes type, const TString& key, const TShortUserDevice& userDevice, const TInstant timestamp) const = 0;
};

class TFakeEventsRegistratorConfig : public IEventsRegistratorConfig {
private:
    using TEvents = TMap<IUserDevicesManager::EEventGlobalTypes, TSet<TString>>;

private:
    R_READONLY(TEvents, AvailableEvents);

public:
    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;
    virtual THolder<IEventsRegistrator> BuildRegistrator() const override;
    virtual TString GetType() const override {
        return Type;
    }

public:
    static TFactory::TRegistrator<TFakeEventsRegistratorConfig> Registrator;
    static TString Type;
};

class TFakeEventsRegistrator : public IEventsRegistrator {
private:
    using TEventsCounter = TMap<IUserDevicesManager::EEventGlobalTypes, TMap<TString, TAtomic>>;

private:
    R_READONLY(TEventsCounter, EventsCounter, {}, mutable);

public:
    TFakeEventsRegistrator(const TFakeEventsRegistratorConfig& config)
        : Config(config)
    {}

    virtual TExpected<bool, IUserDevicesManager::EEventResult> RegisterEvent(const IUserDevicesManager::EEventGlobalTypes type, const TString& key, const TShortUserDevice& /*userDevice*/, const TInstant /*timestamp*/) const override;

private:
    const TFakeEventsRegistratorConfig& Config;
};

class TAdjustEventRegistratorConfig : public IEventsRegistratorConfig {
private:
    using TTokenTypeNames = TMap<EAdvertisingTokenType, TString>;
    using TEventTokens = TMap<IUserDevicesManager::EEventGlobalTypes, TMap<TString, TString>>;

private:
    R_READONLY(TTokenTypeNames, TokenTypeNames);
    R_READONLY(TAdjustClientConfig, ClientConfig);
    R_READONLY(TEventTokens, EventTokens);

public:
    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;
    virtual THolder<IEventsRegistrator> BuildRegistrator() const override;

    virtual TString GetType() const override {
        return Type;
    }

public:
    static TFactory::TRegistrator<TAdjustEventRegistratorConfig> Registrator;
    static TString Type;
    static TTokenTypeNames DefaultTokenNames;
};

class TAdjustEventRegistrator : public IEventsRegistrator {
public:
    TAdjustEventRegistrator(const TAdjustEventRegistratorConfig& config)
        : Client(config.GetClientConfig())
        , Config(config)
    {}

    virtual TExpected<bool, IUserDevicesManager::EEventResult>  RegisterEvent(const IUserDevicesManager::EEventGlobalTypes type, const TString& key, const TShortUserDevice& userDevice, const TInstant timestamp) const override;

private:
    const TAdjustClient Client;
    const TAdjustEventRegistratorConfig& Config;
};

class TUserDevicesManagerConfig: public IUserDevicesManagerConfig {
private:
    using TTypeHeaders = TMap<EAdvertisingTokenType, TString>;

private:
    R_FIELD(TString, DBName);
    R_FIELD(TTypeHeaders, TypeHeaders);
    R_FIELD(TSet<TString>, IgnoredTokens);

public:
    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;
    TString GetAdvertisingHeader(const EAdvertisingTokenType type) const;

    const IEventsRegistratorConfig* GetEventsRegistratorConfig() const {
        return EventsRegistratorConfig.Get();
    }

private:
    THolder<IEventsRegistratorConfig> EventsRegistratorConfig;
};

class TUserDevicesHistoryManager: public TDatabaseHistoryManager<TUserDevice> {
private:
    using TBase = TDatabaseHistoryManager<TUserDevice>;

public:
    TUserDevicesHistoryManager(const IHistoryContext& context)
        : TBase(context, "user_devices_history")
    {
    }

    template <class T = TUserDevice>
    TOptionalObjectEvents<T> GetUser(const TString& userId, NDrive::TEntitySession& session, TInstant since = TInstant::Zero()) const {
        auto queryOptions = TQueryOptions().AddGenericCondition("user_id", userId);
        return GetEvents<T>({}, since, session, queryOptions);
    }
};

class TUserDevicesDB
    : public IAutoActualization
    , public TBaseEntityManager<TUserDevice>
{
public:
    using THistoryManager = TUserDevicesHistoryManager;
    using TExpectedUserDevices = TExpected<TUserDevices, TCodedException>;
    using TOptionalUserDevices = TMaybe<TUserDevices>;

public:
    TUserDevicesDB(const IHistoryContext& context);

    const THistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }
    THistoryManager& GetHistoryManager() {
        return HistoryManager;
    }

    virtual TString GetTableName() const override {
        return TableName;
    }

    TExpectedUserDevices GetCachedUserDevices(const TString& userId, TInstant statementDeadline = TInstant::Zero()) const;
    TOptionalUserDevices RestoreUserDevices(const TString& userId, NDrive::TEntitySession& session) const;

    bool GetFullHistoryByUser(const TString& userId, TVector<TObjectEvent<TUserDevice>>& events, NDrive::TEntitySession& session) const;
    bool GetUserDevices(const TSet<TString>& userIds, TSet<TShortUserDevice>& result, NDrive::TEntitySession& session) const;
    bool GetUsersByDeviceIds(const TSet<TString>& deviceIds, TSet<TString>& userIds, NDrive::TEntitySession& session, bool verifiedOnly = false) const;

protected:
    bool GetStartFailIsProblem() const override {
        return false;
    }
    bool Refresh() override;

private:
    THistoryManager HistoryManager;
    const TDuration DefaultLifetime;
    const TString TableName;

    TOptionalObjectEventId<TUserDevice> LastEventId;

    mutable NUtil::TConcurrentCache<TString, TUserDevices> ObjectCache;
};

class TUserDevicesManager
    : public IUserDevicesManager
    , public TDatabaseSessionConstructor
{
private:
    const TUserDevicesManagerConfig& Config;
    NStorage::ITableAccessor::TPtr Table;
    THolder<TUserDevicesDB> UserDevicesDB;
    THolder<THistoryContext> HistoryContext;
    THolder<IEventsRegistrator> EventRegistrator;
    const TUserDevicesHistoryManager* HistoryManager = nullptr;

protected:
    virtual bool DoStartPhoneVerification(const TString& phoneNumber, const TString& token, const TDeviceIdVerificationContext& divc, TInstant deadline, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const = 0;
    virtual bool ValidatePhoneNumber(const TString& number, const TString& token, const TString& clientIp, TInstant deadline, const TMaybe<IUserDevicesManager::EVerificationMethod> verificationMethod, TSet<TString>& errorCodes) const = 0;
    virtual TMaybe<TString> CreateTrack(const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const = 0;

public:
    TUserDevicesManager(const IServerBase& server, const TUserDevicesManagerConfig& config);
    ~TUserDevicesManager();

    virtual bool GetFullUserDevices(const TString& userId, TVector<TUserDevice>& devices, NDrive::TEntitySession& session) const override;
    virtual ENewDeviceStatus CheckDeviceId(const TString& userId, const TString& deviceId) const override;
    TMaybe<TUserDevice> StartDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, NDrive::TEntitySession& session, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes, bool dropVerifiedStatus = true) const override;
    bool StartPhoneVerification(TString& token, const TString& phoneNumber, const TDeviceIdVerificationContext& divc, IReplyContext::TPtr context, const EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const override;
    virtual bool FinishDeviceIdVerification(const TString& userId, const TString& deviceId, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes, NDrive::TEntitySession& session, TString* confirmedPhoneNumber = nullptr, bool skipVerifiedStatus = false) const override;
    virtual bool SetDevicesFeatures(const TString& userId, const TSet<TString>& devices, const TMaybe<bool> enabled, const TMaybe<bool> verified, const TString& historyUserId) const override;
    virtual bool RemoveDevices(const TString& userId, const TSet<TString>& devices, const TString& historyUserId) const override;
    virtual bool AddFirstDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context) const override;
    virtual bool GetUserDevices(const TString& userId, TVector<TShortUserDevice>& devices) const override;
    virtual bool GetUserDevices(const TSet<TString>& userIds, TSet<TShortUserDevice>& devices, NDrive::TEntitySession& session) const override;
    virtual bool GetUsersByDeviceIds(const TSet<TString>& deviceIds, TSet<TString>& userIds, NDrive::TEntitySession& session, bool verifiedOnly = false) const override;
    virtual bool GetFullHistoryByUser(const TString& userId, TVector<TObjectEvent<TUserDevice>>& events, NDrive::TEntitySession& session) const override;
    virtual TDuration GetMinPropositionsInterval() const override;
    virtual bool GetLastAdvertisingDevice(const TString& userId, TMaybe<TShortUserDevice>& result, const TInstant reqActuality, const TSet<EAdvertisingTokenType>& typeFilter = {}) const override;
    virtual bool RegisterAdvertisingDevice(const TString& userId, const TString& deviceId, IReplyContext::TPtr context, const TInstant activatingTime, const NOpenssl::IAbstractCipher* cipher = nullptr) const override;
    virtual TExpected<bool, EEventResult> RegisterEvent(const EEventGlobalTypes type, const TString& key, const TString& userId, const TInstant timestamp) const override;
    virtual bool GetLastDeviceEvent(const TSet<TString>& userIds, const TSet<EObjectHistoryAction>& includeActions, const TSet<EObjectHistoryAction>& excludeActions, NDrive::TEntitySession& session, TMaybe<TObjectEvent<TUserDevice>>& result) const override;

private:
    void AddAdvertisingInfo(TUserDevice& device, IReplyContext::TPtr context) const;
};

class TPassportUserDevicesManagerConfig
    : public TSimpleAsyncRequestSender::TConfig
    , public TUserDevicesManagerConfig
{
    R_FIELD(TString, Consumer, "carsharing");
    R_FIELD(TString, SubmitUri, "1/bundle/phone/confirm/submit/");
    R_FIELD(TString, CommitUri, "1/bundle/phone/confirm/commit/");
    R_FIELD(TString, ValidateUri, "1/bundle/validate/phone_number/");
    R_FIELD(TString, CreateTrackUri, "1/track/");

public:
    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;

    virtual THolder<IUserDevicesManager> BuildManager(const IServerBase& server) const override;
};

class TPassportUserDevicesManager: public TUserDevicesManager {
private:
    using TBase = TUserDevicesManager;

private:
    const TPassportUserDevicesManagerConfig& Config;
    TSimpleAsyncRequestSender Sender;

    struct LogData {
        TString SignalPrefix;
        TString EventName;
        NJson::TJsonValue LogReport;

        LogData(const TString& signalPrefix, const TString& eventName, const NJson::TJsonValue& logReport)
            : SignalPrefix(signalPrefix)
            , EventName(eventName)
            , LogReport(logReport)
        {
        }
    };

protected:
    virtual bool DoStartPhoneVerification(const TString& number, const TString& token, const TDeviceIdVerificationContext& divc, TInstant deadline, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const override;
    virtual bool ValidatePhoneNumber(const TString& number, const TString& token, const TString& clientIp, TInstant deadline, const TMaybe<IUserDevicesManager::EVerificationMethod> verificationMethod, TSet<TString>& errorCodes) const override;
    virtual TMaybe<TString> CreateTrack(const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const override;

private:
    TMaybe<NJson::TJsonValue> SendRequest(const TString& uri, const TString& postData, const TString& clientIp, const TInstant deadline, const LogData& logData, TSet<TString>& errorCodes) const;

public:
    TPassportUserDevicesManager(const IServerBase& server, const TPassportUserDevicesManagerConfig& config)
        : TBase(server, config)
        , Config(config)
        , Sender(Config, "passport_phone_verification")
    {
    }

    virtual bool FinishPhoneVerification(const TString& token, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const override;
};

class TFakeUserDevicesManagerConfig: public TUserDevicesManagerConfig {
public:
    virtual void Init(const TYandexConfig::Section* section) override {
        TUserDevicesManagerConfig::Init(section);
    }

    virtual void ToString(IOutputStream& os) const override {
        TUserDevicesManagerConfig::ToString(os);
    }

    virtual THolder<IUserDevicesManager> BuildManager(const IServerBase& server) const override;
};

class TFakeUserDevicesManager: public TUserDevicesManager {
private:
    using TBase = TUserDevicesManager;

private:
    const TFakeUserDevicesManagerConfig& Config;

protected:
    virtual bool DoStartPhoneVerification(const TString& /*number*/, const TString& token, const TDeviceIdVerificationContext& divc, TInstant deadline, const IUserDevicesManager::EVerificationMethod verificationMethod, TSet<TString>& errorCodes) const override;
    virtual bool ValidatePhoneNumber(const TString& number, const TString& token, const TString& clientIp, TInstant deadline, const TMaybe<IUserDevicesManager::EVerificationMethod> verificationMethod, TSet<TString>& errorCodes) const override;

    virtual TMaybe<TString> CreateTrack(const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const override;

public:
    TFakeUserDevicesManager(const IServerBase& server, const TFakeUserDevicesManagerConfig& config)
        : TBase(server, config)
        , Config(config)
    {
        Y_UNUSED(Config);
    }

    virtual bool FinishPhoneVerification(const TString& token, const TString& code, const TString& clientIp, TInstant deadline, TSet<TString>& errorCodes) const override;

    static const TMap<TString, TString>& GetCodes();
    static TString GetCodeByToken(const TString& token);
};
