#include <drive/backend/user_devices/manager.h>
#include <drive/backend/ut/library/helper.h>
#include <library/cpp/testing/unittest/registar.h>


Y_UNIT_TEST_SUITE(DevicesSuite) {
    class TLocalConfigGenerator: public NDrive::TServerConfigGenerator {
    private:
        void GetApiConfigString(IOutputStream& os) const override {
            Y_UNUSED(os);
        }
        void GetBackgroundConfigString(IOutputStream& os) const override {
            Y_UNUSED(os);
        }
    };

    const TString DefaultUserId = "12789938-63f9-406d-8407-c4f7cf5d741c";

    Y_UNIT_TEST(RegisterReplies) {
        DoInitGlobalLog("console", 6, false, false);
        TLocalConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetUserDevicesManager());
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const auto& manager = *server->GetUserDevicesManager();

        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 0);
        }

        TUserDevice firstDevice(DefaultUserId, "1");
        firstDevice.SetAdvertisingToken(Base64Encode("idfaToken")).SetAdvertisingTokenType(EAdvertisingTokenType::IDFA);

        UNIT_ASSERT(manager.AddFirstDevice(DefaultUserId, firstDevice.GetDeviceId(), nullptr));

        TUserDevice secondDevice(DefaultUserId, "2");
        secondDevice.SetAdvertisingToken(Base64Encode("adidToken")).SetAdvertisingTokenType(EAdvertisingTokenType::ADID);

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(!lastDevice);
        }

        IReplyContext::TPtr contextIDFA(new TReplyContextMock());
        contextIDFA->MutableRequestData().AddHeader("idfa", firstDevice.GetAdvertisingToken());

        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, firstDevice.GetDeviceId(), contextIDFA, ModelingNow()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 1);
            UNIT_ASSERT(!devices.front().GetAdvertisingToken());
        }

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(!lastDevice);
        }

        IReplyContext::TPtr contextOverrideIDFA(new TReplyContextMock());
        contextOverrideIDFA->MutableRequestData().AddHeader("new_idfa", firstDevice.GetAdvertisingToken());

        TInstant activateTime = ModelingNow();
        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, firstDevice.GetDeviceId(), contextOverrideIDFA, activateTime));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 1);
        }

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
        }

        Sleep(TDuration::Seconds(1));
        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, firstDevice.GetDeviceId(), contextOverrideIDFA, ModelingNow()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 1);
        }
        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
            UNIT_ASSERT_EQUAL(lastDevice->GetAdvertisingTokenTimestamp(), TInstant::Seconds(activateTime.Seconds()));
        }

        IReplyContext::TPtr contextADID(new TReplyContextMock());
        contextADID->MutableRequestData().AddHeader("adid", secondDevice.GetAdvertisingToken());

        Sleep(TDuration::Seconds(1));
        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, secondDevice.GetDeviceId(), contextADID, ModelingNow()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 1);
        }
        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
        }

        TSet<TString> errors;
        {
            TDeviceIdVerificationContext divc;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.StartDeviceIdVerification(DefaultUserId, secondDevice.GetDeviceId(), "", divc, nullptr, session, IUserDevicesManager::EVerificationMethod::Sms, errors).Defined());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, secondDevice.GetDeviceId(), contextADID, ModelingNow()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        TString code;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TUserDevice> devices;
            UNIT_ASSERT(manager.GetFullUserDevices(DefaultUserId, devices, session));
            UNIT_ASSERT_EQUAL(devices.size(), 2);
            for (auto&& device : devices) {
                if (device.GetDeviceId() == secondDevice.GetDeviceId()) {
                    code = TFakeUserDevicesManager::GetCodeByToken(device.GetToken());
                }
            }
        }
        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.FinishDeviceIdVerification(DefaultUserId, secondDevice.GetDeviceId(), code, "", TInstant::Zero(), errors, session, nullptr));
            UNIT_ASSERT(session.Commit());
        }
        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), secondDevice.GetDeviceId());
        }

        UNIT_ASSERT(manager.SetDevicesFeatures(DefaultUserId, { secondDevice.GetDeviceId() }, false, {}, DefaultUserId));

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
        }


        UNIT_ASSERT(manager.RemoveDevices(DefaultUserId, { firstDevice.GetDeviceId() }, DefaultUserId));
        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(!lastDevice);
        }

        UNIT_ASSERT(manager.SetDevicesFeatures(DefaultUserId, { secondDevice.GetDeviceId() }, true, {}, DefaultUserId));

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), secondDevice.GetDeviceId());
        }
    }

    Y_UNIT_TEST(EventRegistrator) {
        DoInitGlobalLog("console", 6, false, false);
        TLocalConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetUserDevicesManager());

        const auto& manager = *server->GetUserDevicesManager();

        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 0);
        }

        TUserDevice firstDevice(DefaultUserId, "1");
        firstDevice.SetAdvertisingToken(Base64Encode("idfaToken")).SetAdvertisingTokenType(EAdvertisingTokenType::IDFA);

        UNIT_ASSERT(manager.AddFirstDevice(DefaultUserId, firstDevice.GetDeviceId(), nullptr));

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(!lastDevice);
        }

        UNIT_ASSERT(!manager.RegisterEvent(IUserDevicesManager::EEventType::SuccessRegistration, DefaultUserId, ModelingNow()));
        UNIT_ASSERT(!manager.RegisterEvent(IUserDevicesManager::EEventType::SuccessFirstTrip, DefaultUserId, ModelingNow()));

        IReplyContext::TPtr contextOverrideIDFA(new TReplyContextMock());
        contextOverrideIDFA->MutableRequestData().AddHeader("new_idfa", firstDevice.GetAdvertisingToken());

        TInstant activateTime = ModelingNow();
        UNIT_ASSERT(manager.RegisterAdvertisingDevice(DefaultUserId, firstDevice.GetDeviceId(), contextOverrideIDFA, activateTime));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            TVector<TShortUserDevice> devices;
            UNIT_ASSERT(manager.GetUserDevices(DefaultUserId, devices));
            UNIT_ASSERT_EQUAL(devices.size(), 1);
        }

        {
            TMaybe<TShortUserDevice> lastDevice;
            UNIT_ASSERT(manager.GetLastAdvertisingDevice(DefaultUserId, lastDevice, ModelingNow()));
            UNIT_ASSERT(lastDevice);
            UNIT_ASSERT_EQUAL(lastDevice->GetDeviceId(), firstDevice.GetDeviceId());
        }

        UNIT_ASSERT(manager.RegisterEvent(IUserDevicesManager::EEventType::SuccessRegistration, DefaultUserId, ModelingNow()));
        UNIT_ASSERT(manager.RegisterEvent(IUserDevicesManager::EEventType::SuccessFirstTrip, DefaultUserId, ModelingNow()));
        UNIT_ASSERT(!manager.RegisterEvent(IUserDevicesManager::EEventType::SuccessFixFirstTrip, DefaultUserId, ModelingNow()));
    }
}
