#include "manager.h"

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/computer_vision/photo_recognizer_client.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/image_transformation/recognize.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/wait_guard/wait_guard.h>

class TDocumentPhotosUpdateCallbackWrapper : public TS3Client::IBaseCallback {
private:
    TAtomicSharedPtr<IDocumentMediaUpdateCallback> HighLevelCallback;
    const TString UserId;
    const TUserDocumentPhoto PhotoDBObject;
    const TUserDocumentVideo BvDBObject;
    const TUserDocumentPhotosDB& PhotoDB;
    const TUserDocumentVideoDB& VideoDB;
    const TString UpdateMeta;
    const ui32 ExpectedResponses;
    std::atomic<size_t> ResponsesReceived;
    std::atomic<size_t> FailedResponsesReceived;

public:
    TDocumentPhotosUpdateCallbackWrapper(
        TAtomicSharedPtr<IDocumentMediaUpdateCallback> highLevelCallback,
        const TString& userId,
        const TUserDocumentPhoto& photoDBObject,
        const TUserDocumentVideo& bvDBObject,
        const TUserDocumentPhotosDB& photoDB,
        const TUserDocumentVideoDB& videoDB,
        const TString& updateMeta,
        ui32 expectedResponses
    )
        : TS3Client::IBaseCallback(TS3Client::ERequestType::PutKey)
        , HighLevelCallback(highLevelCallback)
        , UserId(userId)
        , PhotoDBObject(photoDBObject)
        , BvDBObject(bvDBObject)
        , PhotoDB(photoDB)
        , VideoDB(videoDB)
        , UpdateMeta(updateMeta)
        , ExpectedResponses(expectedResponses)
        , ResponsesReceived(0)
        , FailedResponsesReceived(0)
    {
    }

private:
    void DoOnSuccess(const TString& /*path*/, const THttpReplyData<TString>& /*reply*/) override {
        if (++ResponsesReceived == ExpectedResponses) {
            auto session = PhotoDB.BuildTx<NSQL::Writable>();
            if (!PhotoDB.Insert(PhotoDBObject, session)) {
                if (HighLevelCallback) {
                    HighLevelCallback->OnFailure(MakeErrorMessage("cannot Insert into PhotoDB: " + session.GetStringReport()));
                }
                return;
            }
            if (!VideoDB.Insert(BvDBObject, session) || !session.Commit()) {
                if (HighLevelCallback) {
                    HighLevelCallback->OnFailure(MakeErrorMessage("cannot Insert into VideoDB or Commit: " + session.GetStringReport()));
                }
                return;
            }
            NDrive::TEventLog::Log("UpdateDocumentPhoto", NJson::TMapBuilder
                ("user_id", UserId)
                ("photo", NJson::ToJson(PhotoDBObject))
                ("bv", NJson::ToJson(BvDBObject))
                ("meta", UpdateMeta)
            );
            if (HighLevelCallback) {
                HighLevelCallback->OnSuccess(PhotoDBObject.GetId(), UpdateMeta);
            }
        }
    }

    void DoOnFailure(const TString& /*path*/, const THttpReplyData<TString>& reply) override {
        if (++FailedResponsesReceived == 1) {
            NDrive::TEventLog::Log("UpdateDocumentPhotoError", NJson::TMapBuilder
                ("user_id", UserId)
                ("photo", NJson::ToJson(PhotoDBObject))
                ("bv", NJson::ToJson(BvDBObject))
                ("meta", UpdateMeta)
                ("code", reply.GetHttpCode())
                ("report", NJson::ToJson(reply.GetReport()))
            );
            if (HighLevelCallback) {
                auto report = reply.HasReport() ? *reply.GetReport() : TString{};
                HighLevelCallback->OnFailure(MakeErrorMessage(
                    TStringBuilder() << "cannot UpdatePhotos: " << reply.GetHttpCode() << ' ' << report
                ));
            }
        }
    }

    template <class T>
    TString MakeErrorMessage(T&& error) const {
        if (UpdateMeta) {
            return UpdateMeta + ": " + error;
        } else {
            return error;
        }
    }
};

TExpected<TString> TDocumentPhotosManager::GetObjectPath(const TString& photoId, bool isBv, const NDrive::IServer& server, const TString& externalUid) const {
    TString path;
    auto photoDBFetchResult = PhotoDB.FetchInfo(photoId);
    auto photoPtr = photoDBFetchResult.GetResultPtr(photoId);
    if (!photoPtr) {
        return MakeUnexpected(yexception() << "No DB record about photo " << photoId);
    }

    auto ownerUserId = photoPtr->GetUserId();

    if (photoPtr->GetDocumentId() != "") {
        TString passportUid = externalUid;
        if (!passportUid) {
            const auto userDB = server.GetDriveDatabase().GetUsersData();
            Y_ENSURE(userDB);
            auto userFetchResult = userDB->FetchInfo(ownerUserId);
            auto userPtr = userFetchResult.GetResultPtr(ownerUserId);
            if (!userPtr) {
                return MakeUnexpected(yexception() << "No DB record about user " << ownerUserId);
            }
            passportUid = userPtr->GetUid();
        }
        if (!isBv) {
            path = passportUid + "/" + photoPtr->GetDocumentId() + "/" + photoId;
        } else {
            auto bvFetchResult = VideoDB.FetchInfo(photoId);
            auto bvPtr = bvFetchResult.GetResultPtr(photoId);
            if (!bvPtr) {
                return MakeUnexpected(yexception() << "No DB record about video for " << photoId);
            }
            path = passportUid + "/" + photoPtr->GetDocumentId() + "/" + bvPtr->GetId();
        }
    } else {
        path = ownerUserId + "/" + photoId;
        if (isBv) {
            path += "_bv";
        }
    }
    return path;
}

NThreading::TFuture<NS3::TFile> TDocumentPhotosManager::GetAbstractDocumentContent(const TString& photoId, bool isBv, const NDrive::IServer& server, const TString& externalUid) const {
    auto mdsClient = server.GetMDSClient();
    if (!mdsClient) {
        return NThreading::TExceptionFuture() << "TDocumentPhotosManager::GetAbstractDocumentContent: mds client undefined";
    }
    auto bucket = mdsClient->GetBucket(Config.GetBucketName());
    if (!bucket) {
        return NThreading::TExceptionFuture() << "TDocumentPhotosManager::GetAbstractDocumentContent: mds bucket is not found";
    }
    auto optionalPath = GetObjectPath(photoId, isBv, server, externalUid);
    if (!optionalPath) {
        return NThreading::TExceptionFuture() << optionalPath.GetError().what();
    }
    auto getFileFuture = bucket->GetFile(*optionalPath);
    auto getFileTagFuture = bucket->GetFile(*optionalPath + "_tag");
    auto futures = TVector{ getFileFuture, getFileTagFuture };
    auto state = NDrive::TEventLog::CaptureState();
    return NThreading::WaitExceptionOrAll(futures).Apply([
        getFileFuture = std::move(getFileFuture),
        getFileTagFuture = std::move(getFileTagFuture),
        cipher = Cipher,
        path = *optionalPath,
        state = std::move(state),
        photoId
    ] (const auto& waiter) mutable -> NThreading::TFuture<NS3::TFile> {
        NDrive::TEventLog::TStateGuard stateGuard(state);
        if (waiter.HasException() || !waiter.HasValue()) {
            return NThreading::TExceptionFuture() << "TDocumentPhotosManager::GetAbstractDocumentContent: cannot get file: " << NThreading::GetExceptionMessage(waiter);
        }
        auto fileReply = getFileFuture.ExtractValue();
        auto tagReply = getFileTagFuture.ExtractValue();
        if (auto decrypted = cipher.Decrypt(fileReply.GetContent(), path, tagReply.GetContent())) {
            auto contentType = NS3::DetectContentType(*decrypted);
            if (!contentType) {
                contentType = fileReply.GetContentType();
            }
            return NThreading::MakeFuture(NS3::TFile(photoId, std::move(*decrypted), contentType));
        } else {
            return NThreading::TExceptionFuture() << "TDocumentPhotosManager::GetAbstractDocumentContent: cannot decrypt file";
        }
    });
}

TDocumentPhotosManager::TDocumentPhotosManager(const TDocumentPhotosConfig& config, NStorage::IDatabase::TPtr database)
    : Config(config)
    , PhotoDB(database)
    , VideoDB(database)
    , DocumentVerificationAssignments(database)
    , Cipher(config.GetContentEncryptionKey())
{
}

NThreading::TFuture<void> TDocumentPhotosManager::UpdateBackgroundVideo(const TString& ownerUserId, const TString& photoId, const TString& content, const NDrive::IServer& server) const {
    if (!content) {
        return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "empty video data"));
    }

    auto videoFetchResult = VideoDB.FetchInfo(photoId);
    if (videoFetchResult.empty()) {
        return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "no db record for such bv"));
    }

    TString videoPath = ownerUserId + "/" + photoId + "_bv";
    auto videoEncrypted = Cipher.Encrypt(content, videoPath);
    if (!videoEncrypted) {
        return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "unable to encrypt video content"));
    }
    auto mdsClient = server.GetMDSClient();
    if (!mdsClient) {
        return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "mds client " << Config.GetBucketName() << " is not configured"));
    }
    auto bucket = mdsClient->GetBucket(Config.GetBucketName());
    if (!bucket) {
        return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "s3 bucket " << Config.GetBucketName() << " is not defined"));
    }

    NThreading::TFutures<NUtil::THttpReply> uploadResults;
    uploadResults.push_back(bucket->PutKey(videoPath, videoEncrypted->Data));
    uploadResults.push_back(bucket->PutKey(videoPath + "_tag", videoEncrypted->Tag));

    return NThreading::WaitExceptionOrAll(uploadResults);
}

TMaybe<NThreading::TFuture<TUserDocumentPhoto>> TDocumentPhotosManager::UploadPhoto(TString& photoId, const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const NDrive::IServer& server, TMessagesCollector& errors) const {
    if (!photoContent) {
        return NThreading::MakeFuture<TUserDocumentPhoto>();
    }

    TUserDocumentPhoto photoDBObject(ownerUserId, type);
    photoDBObject.SetOriginChat(originChat);

    bool checkPhotoFile = server.GetSettings().GetValue<bool>("document_photo_manager.check_photo_file").GetOrElse(true);
    if (checkPhotoFile && !NImageTransformation::TRecognize::CheckFileIsImage(photoContent)) {
        errors.AddMessage("UploadPhoto", "malformed photo data");
        return Nothing();
    }

    TString photoPath = ownerUserId + "/" + photoDBObject.GetId();
    auto photoEncrypted = Cipher.Encrypt(photoContent, photoPath);
    if (!photoEncrypted) {
        errors.AddMessage("UploadPhoto", "unable to encrypt photo content");
        return Nothing();
    }

    auto mdsClient = server.GetMDSClient();
    if (!mdsClient) {
        errors.AddMessage("UploadPhoto",  "MDS client " + Config.GetBucketName() + " is not configured");
        return Nothing();
    }

    auto bucket = mdsClient->GetBucket(Config.GetBucketName());
    if (!bucket) {
        errors.AddMessage("UploadPhoto", "S3 bucket " + Config.GetBucketName() + " is not defined");
        return Nothing();
    }

    NThreading::TFutures<void> futures;
    futures.push_back(bucket->CheckReply(bucket->PutKey(photoPath, photoEncrypted->Data)));
    futures.push_back(bucket->CheckReply(bucket->PutKey(photoPath + "_tag", photoEncrypted->Tag)));

    photoId = photoDBObject.GetId();

    return NThreading::WaitExceptionOrAll(futures).Apply([
        photoDBObject = std::move(photoDBObject)
    ](const NThreading::TFuture<void>& waiter) mutable -> TUserDocumentPhoto {
        if (waiter.HasException()) {
            throw yexception() << NThreading::GetExceptionMessage(waiter);
        }
        if (!waiter.HasValue()) {
            throw yexception() << "DocumentPhotosManager no value in mds future";
        }

        return photoDBObject;
    });
}

TMaybe<NThreading::TFuture<TUserDocumentVideo>> TDocumentPhotosManager::UploadVideo(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& videoContent, const NDrive::IServer& server, const TString& photoId, TMessagesCollector& errors) const {
    if (!videoContent) {
        return NThreading::MakeFuture<TUserDocumentVideo>();
    }

    auto videoDBObject = photoId ? TUserDocumentVideo(photoId, type) : TUserDocumentVideo(type);
    videoDBObject.SetOriginChat(originChat);
    videoDBObject.SetUserId(ownerUserId);

    TString videoPath = ownerUserId + "/" + videoDBObject.GetId() + "_bv";
    auto videoEncrypted = Cipher.Encrypt(videoContent, videoPath);
    if (!videoEncrypted) {
        errors.AddMessage("UploadVideo", "unable to encrypt video content");
        return Nothing();
    }

    auto mdsClient = server.GetMDSClient();
    if (!mdsClient) {
        errors.AddMessage("UploadVideo",  "MDS client " + Config.GetBucketName() + " is not configured");
        return Nothing();
    }

    auto bucket = mdsClient->GetBucket(Config.GetBucketName());
    if (!bucket) {
        errors.AddMessage("UploadVideo", "S3 bucket " + Config.GetBucketName() + " is not defined");
        return Nothing();
    }

    NThreading::TFutures<void> futures;
    futures.push_back(bucket->CheckReply(bucket->PutKey(videoPath, videoEncrypted->Data)));
    futures.push_back(bucket->CheckReply(bucket->PutKey(videoPath + "_tag", videoEncrypted->Tag)));

    return NThreading::WaitExceptionOrAll(futures).Apply([
        videoDBObject = std::move(videoDBObject)
    ](const NThreading::TFuture<void>& waiter) mutable -> TUserDocumentVideo {
        if (waiter.HasException()) {
            throw yexception() << NThreading::GetExceptionMessage(waiter);
        }
        if (!waiter.HasValue()) {
            throw yexception() << "DocumentPhotosManager no value in mds future";
        }

        return videoDBObject;
    });
}

NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> TDocumentPhotosManager::UploadMediaImpl(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent, const NDrive::IServer& server, bool shouldCommit, bool saveContent) const {
    TMessagesCollector errors;

    TMaybe<NThreading::TFuture<TUserDocumentPhoto>> optionalPhotoFuture;
    TString photoId = "";
    if (!(optionalPhotoFuture = UploadPhoto(photoId, ownerUserId, originChat, type, photoContent, server, errors))) {
        return NThreading::MakeErrorFuture<TDocumentPhotosManager::TAddDocumentResult>(std::make_exception_ptr(yexception() << errors.GetStringReport()));
    }

    TMaybe<NThreading::TFuture<TUserDocumentVideo>> optionalVideoFuture;
    if (!(optionalVideoFuture = UploadVideo(ownerUserId, originChat, type, videoContent, server, photoId, errors))) {
        return NThreading::MakeErrorFuture<TDocumentPhotosManager::TAddDocumentResult>(std::make_exception_ptr(yexception() << errors.GetStringReport()));
    }

    if (server.GetUserDevicesManager() && type != NUserDocument::EType::Selfie) {
        Y_UNUSED(server.GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::RegistrationPhotos, ::ToString(type), ownerUserId, ModelingNow()));
    }

    NThreading::TFutures<void> uploadResults = {
        optionalPhotoFuture->IgnoreResult(),
        optionalVideoFuture->IgnoreResult()
    };

    auto state = NDrive::TEventLog::CaptureState();
    return NThreading::WaitExceptionOrAll(uploadResults).Apply([
            photoFuture = std::move(*optionalPhotoFuture),
            videoFuture = std::move(*optionalVideoFuture),
            &photoDB = PhotoDB,
            &videoDB = VideoDB,
            photoContent = std::move(photoContent),
            videoContent = std::move(videoContent),
            state = std::move(state),
            userId = ownerUserId,
            shouldCommit, saveContent
        ] (const NThreading::TFuture<void>& waiter) mutable -> NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> {
            NDrive::TEventLog::TStateGuard stateGuard(state);
            if (waiter.HasException()) {
                return NThreading::TExceptionFuture() << NThreading::GetExceptionMessage(waiter);
            }
            if (!photoFuture.HasValue() || !videoFuture.HasValue()) {
                return NThreading::TExceptionFuture() << "DocumentPhotosManager no value in mds future";
            }
            auto photoDBObject = photoFuture.GetValue();
            auto videoDBObject = videoFuture.GetValue();
            TDocumentPhotosManager::TAddDocumentResult result(photoDBObject, videoDBObject);

            if (saveContent) {
                result.SetPhotoContent(std::move(photoContent));
                result.SetVideoContent(std::move(videoContent));
            }
            if (shouldCommit) {
                auto session = photoDB.BuildTx<NSQL::Writable>();
                if (photoDBObject.GetId()) {
                    videoDBObject.SetPhotoId(photoDBObject.GetId());
                    if (!photoDB.Insert(photoDBObject, session)) {
                        return NThreading::TExceptionFuture() << "cannot Insert into PhotoDB: " << session.GetStringReport();
                    }
                }
                if (videoDBObject.GetId()) {
                    if (!videoDB.Insert(videoDBObject, session)) {
                        return NThreading::TExceptionFuture() << "cannot Insert into VideoDB: " << session.GetStringReport();
                    }
                }
                if (!session.Commit()) {
                    return NThreading::TExceptionFuture() << session.GetStringReport();
                }
            }
            NDrive::TEventLog::Log("UpdateDocumentPhoto", NJson::TMapBuilder
                ("user_id", userId)
                ("photo", NJson::ToJson(photoDBObject))
                ("bv", NJson::ToJson(videoDBObject))
            );
            return NThreading::MakeFuture(result);
        }
    );
}

NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> TDocumentPhotosManager::AddDocumentPhoto(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent, const NDrive::IServer& server, bool shouldCommit, bool saveContent) const {
    return UploadMediaImpl(ownerUserId, originChat, type, photoContent, videoContent, server, shouldCommit, saveContent);
}

NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> TDocumentPhotosManager::AddDocumentVideo(const TString& ownerUserId, const TString &originChat, const NUserDocument::EType type, const TString &content, const NDrive::IServer &server, bool shouldCommit, bool saveContent) const {
    return UploadMediaImpl(ownerUserId, originChat, type, "", content, server, shouldCommit, saveContent);
}

void TDocumentPhotosManager::AddDocumentPhoto(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent, TAtomicSharedPtr<IDocumentMediaUpdateCallback> callback, const NDrive::IServer& server, const TString& updateMeta) const {
    auto makeErrorMessage = [&](auto&& error) -> TString {
        if (updateMeta) {
            return updateMeta + ": " + error;
        } else {
            return error;
        }
    };
    if (!photoContent) {
        callback->OnFailure(makeErrorMessage("empty photo data"));
        return;
    }

    bool checkPhotoFile = server.GetSettings().GetValue<bool>("document_photo_manager.check_photo_file").GetOrElse(true);
    if (checkPhotoFile && !NImageTransformation::TRecognize::CheckFileIsImage(photoContent)) {
        callback->OnFailure(makeErrorMessage("malformed photo data"));
        return;
    }

    TUserDocumentPhoto photoDBObject(ownerUserId, type);
    photoDBObject.SetOriginChat(originChat);
    auto photoId = photoDBObject.GetId();
    auto videoDBObject = TUserDocumentVideo(photoId, type);

    ui32 expectedResponses = 2 + (videoContent ? 2 : 0);
    auto callbackWrap = MakeAtomicShared<TDocumentPhotosUpdateCallbackWrapper>(
        callback,
        ownerUserId,
        photoDBObject,
        videoDBObject,
        PhotoDB,
        VideoDB,
        updateMeta,
        expectedResponses
    );

    // Encrypt photo
    TString photoPath = ownerUserId + "/" + photoId;
    auto photoEncrypted = Cipher.Encrypt(photoContent, photoPath);
    if (!photoEncrypted) {
        callback->OnFailure(makeErrorMessage("unable to encrypt photo content"));
        return;
    }

    auto mdsClient = server.GetMDSClient();
    if (!mdsClient) {
        callback->OnFailure(makeErrorMessage("MDS client " + Config.GetBucketName() + " is not configured"));
        return;
    }
    auto bucket = mdsClient->GetBucket(Config.GetBucketName());
    if (!bucket) {
        callback->OnFailure(makeErrorMessage("S3 bucket " + Config.GetBucketName() + " is not defined"));
        return;
    }
    bucket->PutKey(photoPath, photoEncrypted->Data, callbackWrap);
    bucket->PutKey(photoPath + "_tag", photoEncrypted->Tag, callbackWrap);

    if (videoContent) {
        // Encrypt background video
        TString videoPath = ownerUserId + "/" + photoId + "_bv";
        auto videoEncrypted = Cipher.Encrypt(videoContent, videoPath);
        if (!videoEncrypted) {
            callback->OnFailure(makeErrorMessage("unable to encrypt video content"));
            return;
        }

        // Upload to MDS
        bucket->PutKey(videoPath, videoEncrypted->Data, callbackWrap);
        bucket->PutKey(videoPath + "_tag", videoEncrypted->Tag, callbackWrap);
    }
    if (server.GetUserDevicesManager()) {
        Y_UNUSED(server.GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::RegistrationPhotos, ::ToString(type), ownerUserId, ModelingNow()));
    }
}

void TDocumentPhotosManager::GetDocumentPhoto(const TString& photoId, TAtomicSharedPtr<IDocumentMediaAcquisitionCallback> callback, const NDrive::IServer& server, const TString& externalUid) const {
    auto state = NDrive::TEventLog::CaptureState();
    GetDocumentPhoto(photoId, server, externalUid).Subscribe([
        photoId,
        state = std::move(state),
        callback
    ](const NThreading::TFuture<NS3::TFile>& f) {
        NDrive::TEventLog::TStateGuard stateGuard(state);
        try {
            auto file = f.GetValue();
            if (callback) {
                callback->OnSuccess(photoId, std::move(file.GetContent()));
            }
        } catch (const std::exception& e) {
            if (callback) {
                callback->OnFailure(photoId + " " + FormatExc(e));
            }
        }
    });
}

void TDocumentPhotosManager::GetDocumentBackgroundVideo(const TString& photoId, TAtomicSharedPtr<IDocumentMediaAcquisitionCallback> callback, const NDrive::IServer& server) const {
    auto state = NDrive::TEventLog::CaptureState();
    GetDocumentBackgroundVideo(photoId, server).Subscribe([
        photoId,
        state = std::move(state),
        callback
    ](const NThreading::TFuture<NS3::TFile>& f) {
        NDrive::TEventLog::TStateGuard stateGuard(state);
        try {
            auto file = f.GetValue();
            if (callback) {
                callback->OnSuccess(photoId, std::move(file.GetContent()));
            }
        } catch (const std::exception& e) {
            if (callback) {
                callback->OnFailure(photoId + " " + FormatExc(e));
            }
        }
    });
}

bool TDocumentPhotosManager::GetDocumentPhoto(const TString& photoId, TString& result, const NDrive::IServer& server, const TString& externalUid) const {
    auto future = GetDocumentPhoto(photoId, server, externalUid);
    future.Wait();
    if (!future.HasException() && future.HasValue()) {
        result = future.GetValue().GetContent();
        return true;
    }
    if (future.HasException()) {
        NDrive::TEventLog::Log("DocumentPhotosManagerError", NJson::TMapBuilder
            ("event", "GetDocumentPhoto")
            ("error", NThreading::GetExceptionMessage(future))
        );
    }
    return false;
}

NThreading::TFuture<NS3::TFile> TDocumentPhotosManager::GetDocumentPhoto(const TString& photoId, const NDrive::IServer& server, const TString& externalUid) const {
    return GetAbstractDocumentContent(photoId, false, server, externalUid);
}

bool TDocumentPhotosManager::GetDocumentBackgroundVideo(const TString& photoId, TString& result, const NDrive::IServer& server) const {
    auto future = GetDocumentBackgroundVideo(photoId, server);
    future.Wait();
    if (!future.HasException() && future.HasValue()) {
        result = future.GetValue().GetContent();
        return true;
    }
    if (future.HasException()) {
        NDrive::TEventLog::Log("DocumentPhotosManagerError", NJson::TMapBuilder
            ("event", "GetDocumentBackgroundVideo")
            ("error", NThreading::GetExceptionMessage(future))
        );
    }
    return false;
}

NThreading::TFuture<NS3::TFile> TDocumentPhotosManager::GetDocumentBackgroundVideo(const TString& photoId, const NDrive::IServer& server) const {
    return GetAbstractDocumentContent(photoId, true, server);
}

TString TDocumentPhotosManager::BuildPhotoAccessPath(const TString& photoId) const {
    return Config.GetPhotoAccessPathPrefix() + photoId + Config.GetPhotoAccessPathPostfix();
}
