#pragma once

#include "config.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/users/user.h>
#include <drive/backend/users/user_documents.h>
#include <drive/backend/users/yang.h>

#include <drive/library/cpp/aes/crypto.h>
#include <drive/library/cpp/mds/client.h>

#include <rtline/util/types/expected.h>

class TDriveAPI;

class IDocumentMediaUpdateCallback {
public:
    virtual void OnSuccess(const TString& id, const TString& updateMeta) = 0;
    virtual void OnFailure(const TString& updateMeta) = 0;
    virtual ~IDocumentMediaUpdateCallback() = default;
};

class IDocumentMediaAcquisitionCallback {
public:
    virtual void OnSuccess(const TString& id, const TString& content) = 0;
    virtual void OnFailure(const TString& id) = 0;
    virtual ~IDocumentMediaAcquisitionCallback() = default;
};

class TDocumentPhotosManager {
public:
    class TCryptoProcessor : public TCryptoGCMProcessor {
        using TBase = TCryptoGCMProcessor;
    public:
        TCryptoProcessor(const unsigned char* encryptionKey)
            : TBase(TBase::EMode::GCM_256, encryptionKey)
        {
        }
    };

    class TAddDocumentResult {
        R_READONLY(TUserDocumentPhoto, Photo);
        R_READONLY(TUserDocumentVideo, Video);
        R_FIELD(TString, PhotoContent);
        R_FIELD(TString, VideoContent);

    public:
        TAddDocumentResult(const TUserDocumentPhoto& photo, const TUserDocumentVideo& video)
            : Photo(photo)
            , Video(video)
        {
        }
    };

private:
    TDocumentPhotosConfig Config;

    TUserDocumentPhotosDB PhotoDB;
    TUserDocumentVideoDB VideoDB;
    TYangDocumentsAssignmentsDB DocumentVerificationAssignments;
    const TCryptoProcessor Cipher;

private:
    TMaybe<NThreading::TFuture<TUserDocumentPhoto>> UploadPhoto(TString& photoId, const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const NDrive::IServer& server, TMessagesCollector& errors) const;
    TMaybe<NThreading::TFuture<TUserDocumentVideo>> UploadVideo(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& videoContent, const NDrive::IServer& server, const TString& photoId, TMessagesCollector& errors) const;
    NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> UploadMediaImpl(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent, const NDrive::IServer& server, bool shouldCommit, bool saveContent = false) const;
    NThreading::TFuture<NS3::TFile> GetAbstractDocumentContent(const TString& photoId, bool isBv, const NDrive::IServer& server, const TString& externalUid = "") const;
    TExpected<TString> GetObjectPath(const TString& photoId, bool isBv, const NDrive::IServer& server, const TString& externalUid = "") const;

public:
    TDocumentPhotosManager(const TDocumentPhotosConfig& config, NStorage::IDatabase::TPtr database);

    NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> AddDocumentPhoto(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent,  const NDrive::IServer& server, bool shouldCommit, bool saveContent = false) const;
    void AddDocumentPhoto(const TString& ownerUserId, const TString& originChat, const NUserDocument::EType type, const TString& photoContent, const TString& videoContent, TAtomicSharedPtr<IDocumentMediaUpdateCallback> callback, const NDrive::IServer& server, const TString& updateMeta = "") const;
    NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> AddDocumentVideo(const TString& ownerUserId, const TString &originChat, const NUserDocument::EType type, const TString &content, const NDrive::IServer &server, bool shouldCommit, bool saveContent = false) const;
    NThreading::TFuture<void> UpdateBackgroundVideo(const TString& ownerUserId, const TString& photoId, const TString& content, const NDrive::IServer& server) const;

    void GetDocumentPhoto(const TString& photoId, TAtomicSharedPtr<IDocumentMediaAcquisitionCallback> callback, const NDrive::IServer& server, const TString& externalUid = "") const;
    bool GetDocumentPhoto(const TString& photoId, TString& result, const NDrive::IServer& server, const TString& externalUid = "") const;
    NThreading::TFuture<NS3::TFile> GetDocumentPhoto(const TString& photoId, const NDrive::IServer& server, const TString& externalUid = "") const;
    void GetDocumentBackgroundVideo(const TString& photoId, TAtomicSharedPtr<IDocumentMediaAcquisitionCallback> callback, const NDrive::IServer& server) const;
    bool GetDocumentBackgroundVideo(const TString& photoId, TString& result, const NDrive::IServer& server) const;
    NThreading::TFuture<NS3::TFile> GetDocumentBackgroundVideo(const TString& photoId, const NDrive::IServer& server) const;

    TString BuildPhotoAccessPath(const TString& photoId) const;

    const TUserDocumentPhotosDB& GetUserPhotosDB() const {
        return PhotoDB;
    }

    const TUserDocumentVideoDB& GetUserBackgroundVideosDB() const {
        return VideoDB;
    }

    const TYangDocumentsAssignmentsDB& GetDocumentVerificationAssignments() const {
        return DocumentVerificationAssignments;
    }
};
