#include "entities.h"

#include <drive/library/cpp/car/number.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/datetime.h>

#include <library/cpp/json/json_reader.h>


namespace {
    TMaybe<NDrive::NRenins::TUserMainData> ParseUserMainData(const NJson::TJsonValue& json) {
        NDrive::NRenins::TUserMainData user;
        if (json.Has("middle_name")) {
            TString name;
            if (!NJson::ParseField(json, "middle_name", name, /* required = */ false)) {
                return {};
            }
            user.SetMiddleName(name);
        }
        if (NJson::ParseField(json, "name", user.MutableName(), /* required = */ true)
            && NJson::ParseField(json, "last_name", user.MutableLastName(), /* required = */ true)
            && NJson::ParseField(json, "date_of_birth", user.MutableDateOfBirth(), /* required = */ true))
        {
            return user;
        };
        return {};
    }
}

NJson::TJsonValue NDrive::NRenins::TKaskoData::SerializeToJson() const {
    NJson::TJsonValue json;
    NJson::InsertNonNull(json, "id", TagId);
    NJson::InsertField(json, "package_name", PackageName);
    NJson::InsertField(json, "franchise", Franchise);
    NJson::InsertField(json, "order_id", OrderId);
    NJson::InsertNonNull(json, "start_date", InsuranceStart.Seconds());
    if (InsuranceEnd) {
        NJson::InsertField(json, "end_date", InsuranceEnd.Seconds());
    } else if (InsuranceStart) {
        NJson::InsertField(json, "end_date", AddMonths(InsuranceStart, 12).Seconds());
    }
    {
        auto& carReport = json.InsertValue("car", Car.GetReport());
        if (!carReport.Has("brand")) {
            NJson::InsertNonNull(carReport, "brand", CarReportData.GetBrand());
        }
        if (!carReport.Has("model")) {
            NJson::InsertNonNull(carReport, "model", CarReportData.GetModel());
        }
    }
    NJson::InsertField(json, "drivers", NJson::ToJson(Drivers));
    NJson::InsertField(json, "calc_final_price", CalcFinalPrice);
    NJson::InsertField(json, "calc_discount", Discount);
    if (Status) {
        NJson::InsertNonNull(json, "order_final_price", OrderFinalPrice);
        auto& status = json.InsertValue("renins_status", NJson::JSON_MAP);
        NJson::InsertField(status, "account_status", ToString(Status->GetAccountStatus()));
        NJson::InsertField(status, "pso_status", ToString(Status->GetPsoStatus()));
        NJson::InsertField(status, "payment_status", ToString(Status->GetPaymentStatus()));
        NJson::InsertField(json, "deal_id", Status->GetDealId());
    }
    {
        auto finalPrice = OrderFinalPrice
            ? OrderFinalPrice
            : CalcFinalPrice;
        auto& price = json.InsertValue("price", NJson::JSON_MAP);
        NJson::InsertNonNull(price, "final", finalPrice);
        auto discount = Discount;
        if (OrderFinalPrice && OrderFinalPrice != CalcFinalPrice) {
            discount = 0;
        }
        if (discount) {
            NJson::TJsonValue discounts = NJson::JSON_ARRAY;
            NJson::TJsonValue mainDiscount = NJson::TMapBuilder
                ("description_key", "scoring")
                ("value", discount)
            ;
            discounts.AppendValue(mainDiscount);
            NJson::InsertField(price, "discounts", discounts);
            if (discount && discount < 100) {
                const ui32 value = finalPrice * 100 / (100 - discount);
                NJson::InsertField(price, "without_discount", value);
            } else {
                NJson::InsertField(price, "without_discount", finalPrice);
            }
        }
    }
    NJson::InsertNonNull(json, "payment_link", PaymentLink);
    NJson::InsertNonNull(json, "document_path", DocumentPath);
    if (DocumentPath && Status) {
        NJson::InsertNonNull(json, "insurance_number", Status->GetPolicyNumber());
    }
    NJson::InsertField(json, "status", ToString(GetStatus()));
    NJson::InsertNonNull(json, "error_type", ErrorType);
    NJson::InsertNonNull(json, "error_text", ErrorText);
    return json;
}

NDrive::NRenins::EKaskoStatus NDrive::NRenins::TKaskoData::GetStatus() const {
    if (ErrorType) {
        return EKaskoStatus::Error;
    }
    EKaskoStatus status = CalcFinalPrice
        ? EKaskoStatus::Calculated
        : EKaskoStatus::NoData;
    if (OrderFinalPrice) {
        status = EKaskoStatus::Processing;
    }
    if (Status && Status->GetPsoStatus() != NDrive::NRenins::NKasko::EPsoStatus::Contract && Status->GetPsoStatus() != NDrive::NRenins::NKasko::EPsoStatus::Unknown) {
        status = EKaskoStatus::NeedPso;
    }
    if (PaymentLink) {
        status = EKaskoStatus::Ordered;
    }
    if (DocumentPath) {
        status = EKaskoStatus::Signed;
    }
    return status;
}

NDrive::NRenins::NProto::TKaskoOrderData NDrive::NRenins::TKaskoData::SerializeToProto() const {
    NDrive::NRenins::NProto::TKaskoOrderData proto;
    proto.SetPackName(PackageName);
    proto.SetFranchise(Franchise);
    if (OrderId) {
        proto.SetOrderId(OrderId);
    }
    if (InsuranceStart) {
        proto.SetStartDate(InsuranceStart.Seconds());
    }
    if (InsuranceEnd) {
        proto.SetEndDate(InsuranceEnd.Seconds());
    }
    {
        auto carProto = proto.MutableCar();
        carProto->SetVin(Car.GetVin());
        carProto->SetNumber(Car.GetNumber());
        carProto->SetBrand(CarReportData.GetBrand());
        carProto->SetModel(CarReportData.GetModel());
        auto sts = carProto->MutableSts();
        sts->SetNumber(Car.GetStsNumber());
        sts->SetSeries(Car.GetStsSeries());
        sts->SetIssueDate(Car.GetStsIssuedDate().Seconds());
    }
    for (const auto& driver : Drivers) {
        proto.AddDrivers(driver);
    }
    {
        auto priceProto = proto.MutableCalculatedPrice();
        priceProto->SetFinalPrice(CalcFinalPrice);
        priceProto->SetDiscount(Discount);
    }
    {
        auto price = proto.MutableCalculatedPrice();
        price->SetFinalPrice(CalcFinalPrice);
        price->SetDiscount(Discount);
    }
    if (Status) {
        auto price = proto.MutablePrice();
        price->SetFinalPrice(OrderFinalPrice);
        if (OrderFinalPrice == CalcFinalPrice) {
            price->SetDiscount(Discount);
        } else {
            price->SetDiscount(0);
        }
        auto statusProto = proto.MutableStatus();
        statusProto->SetAccountStatus(ToString(Status->GetAccountStatus()));
        statusProto->SetPaymentStatus(ToString(Status->GetPaymentStatus()));
        statusProto->SetPsoStatus(ToString(Status->GetPsoStatus()));
        statusProto->SetDealId(Status->GetDealId());
        statusProto->SetPolicyNumber(Status->GetPolicyNumber());
    }
    if (PaymentLink){
        proto.SetPaymentLink(PaymentLink);
    }
    if (DocumentPath) {
        proto.SetDocumentLink(DocumentPath);
    }
    if (Conditions.IsDefined()) {
        proto.SetConditions(Conditions.GetStringRobust());
    }
    if (ErrorType) {
        proto.SetErrorType(ErrorType);
    }
    if (ErrorText) {
        proto.SetErrorText(ErrorText);
    }
    return proto;
}

bool NDrive::NRenins::TKaskoData::DeserializeFromProto(const NProto::TKaskoOrderData& proto) {
    if (!proto.HasPackName()) {
        return false;
    }
    PackageName = proto.GetPackName();
    Franchise = proto.GetFranchise();
    if (proto.HasCar()) {
        const auto& car = proto.GetCar();
        Car.SetVin(car.GetVin());
        Car.SetNumber(car.GetNumber());
        const auto& sts = car.GetSts();
        Car.SetStsNumber(sts.GetNumber());
        Car.SetStsSeries(sts.GetSeries());
        Car.MutableStsIssuedDate().Seconds(sts.GetIssueDate());
        CarReportData.SetBrand(car.GetBrand());
        CarReportData.SetModel(car.GetModel());
    } else {
        return false;
    }
    if (proto.HasOrderId()) {
        OrderId = proto.GetOrderId();
    }
    if (proto.HasStartDate()) {
        InsuranceStart = TInstant::Seconds(proto.GetStartDate());
    }
    if (proto.HasEndDate()) {
        InsuranceEnd = TInstant::Seconds(proto.GetEndDate());
    }
    for (const auto& driver: proto.GetDrivers()) {
        Drivers.push_back(driver);
    }
    if (proto.HasCalculatedPrice()) {
        const auto& price = proto.GetCalculatedPrice();
        CalcFinalPrice = price.GetFinalPrice();
        Discount = price.GetDiscount();
    } else {
        return false;
    }
    if (proto.HasStatus()) {
        Status.ConstructInPlace();
        const auto protoStatus = proto.GetStatus();
        auto parseStatus = [](const TString& status, auto& object) {
            return TryFromString(status, object);
        };
        if (!parseStatus(protoStatus.GetAccountStatus(), Status->MutableAccountStatus())
            || !parseStatus(protoStatus.GetPaymentStatus(), Status->MutablePaymentStatus())
            || !parseStatus(protoStatus.GetPsoStatus(), Status->MutablePsoStatus()))
        {
            return false;
        }
        OrderFinalPrice = proto.GetPrice().GetFinalPrice();
        Status->SetDealId(protoStatus.GetDealId());
        Status->SetPolicyNumber(protoStatus.GetPolicyNumber());
    }
    if (proto.HasPaymentLink()) {
        PaymentLink = proto.GetPaymentLink();
    }
    if (proto.HasDocumentLink()) {
        DocumentPath = proto.GetDocumentLink();
    }
    if (proto.HasConditions() && !NJson::ReadJsonFastTree(proto.GetConditions(), &Conditions)) {
        return false;
    }
    if (proto.HasErrorType()) {
        ErrorType = proto.GetErrorType();
    }
    if (proto.HasErrorText()) {
        ErrorText = proto.GetErrorText();
    }
    return true;
}

TMaybe<NDrive::NRenins::NKasko::TAutoData> NDrive::NRenins::ParseKaskoCarData(const NJson::TJsonValue& json) {
    NKasko::TAutoData carData;
    TString grz;
    auto& mainCarData = carData.MutableMainAutoData();
    if (NJson::ParseField(json, "auto_cost", carData.MutableCost(), /* required = */ true)
        && NJson::ParseField(json, "mileage", carData.MutableMileage(), /* required = */ true)
        && NJson::ParseField(json, "in_credit", carData.MutableInCredit(), /* required = */ true)
        && NJson::ParseField(json, "auto_body_type_code", carData.MutableBodyTypeCode(), /* required = */ true)
        && NJson::ParseField(json, "registration_number", grz, /* required = */ true)
        && NJson::ParseField(json, "auto_brand_code", mainCarData.MutableBrandCode(), /* required = */ true)
        && NJson::ParseField(json, "auto_model_code", mainCarData.MutableModelCode(), /* required = */ true)
        && NJson::ParseField(json, "auto_year_of_issue_code", mainCarData.MutableYearOfIssueCode(), /* required = */ true)
        && NJson::ParseField(json, "engine_power_code", mainCarData.MutableEnginePowerCode(), /* required = */ true)
        && NJson::ParseField(json, "vin", carData.MutableVin(), /* required = */ true)
        && NJson::ParseField(json, "bank_of_credit", mainCarData.MutableBankOfCredit(), /* required = */ carData.IsInCredit()))
    {
        return carData;
    }
    mainCarData.SetNumber(NDrive::ToCyrillic(grz));
    return {};
}

TMaybe<NDrive::NRenins::NKasko::TCarOrderStartData> NDrive::NRenins::ParseCarKaskoOrderData(const NJson::TJsonValue& json) {
    NKasko::TCarOrderStartData carOrderData;
    TString sts;
    if (!NJson::ParseField(json, "sts_number", sts, /* required = */ true) || sts.size() < StsNumberSize) {
        return {};
    }
    carOrderData.SetStsNumber(sts.substr(0, StsNumberSize));
    carOrderData.SetStsSeries(sts.substr(StsNumberSize));
    TString grz;
    if (NJson::ParseField(json, "registration_number", grz, /* required = */ true)
        && NJson::ParseField(json, "sts_issue_date", carOrderData.MutableStsIssuedDate(), /* required = */ true)
        && NJson::ParseField(json, "vin", carOrderData.MutableVin(), /* required = */ true))
    {
        carOrderData.SetNumber(NDrive::ToCyrillic(grz));
        return carOrderData;
    }
    return {};
}

TMaybe<NDrive::NRenins::NKasko::TDriverOrderStartData> NDrive::NRenins::ParseDriverKaskoOrderData(const NJson::TJsonValue& json) {
    NDrive::NRenins::NKasko::TDriverOrderStartData result;
    {
        TString license;
        if (!NJson::ParseField(json, "driver_license", license, /* required = */ true) || license.size() < LicenseSeriesSize) {
            return {};
        }
        result.SetLicenseSeries(license.substr(0, LicenseSeriesSize));
        result.SetLicenseNumber(license.substr(LicenseSeriesSize));
    }
    {
        auto user = ParseUserMainData(json);
        if (!user) {
            return {};
        }
        result.SetUserData(std::move(*user));
    }
    if (NJson::ParseField(json, "driving_experience_date_start", result.MutableExperienceDateStart(), /* required = */ true)
        && NJson::ParseField(json, "maried", result.MutableMaried(), /* required = */ false)
        && NJson::ParseField(json, "sex_code", result.MutableSexCode(), /* required = */ false))
    {
        return result;
    }
    return {};
}
