#include "user_option.h"

#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

NJson::TJsonValue TCommonOption::DoGetPublicReport(ELocalization locale) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    NJson::TJsonValue result;
    result.InsertValue("id", GetOptionId());
    result.InsertValue("name", server.GetLocalization()->GetLocalString(locale, MetaInfo.GetTitle()));
    result.InsertValue("type", ::ToString("wide_variants"));  // refer to a deprecated scheme type TFSWideVariants
    result.InsertValue("value", GetValueOrDefault(TInstant::Zero()));
    NJson::TJsonValue& arrVariants = result.InsertValue("variants", NJson::JSON_ARRAY);

    for (auto&& i : MetaInfo.GetClientInfo()) {
        NJson::TJsonValue jsonValue = i.second;
        server.GetLocalization()->ApplyResourcesForJson(jsonValue, locale);
        arrVariants.AppendValue(std::move(jsonValue));
    }
    return result;
}

TMaybe<TString> IUserOption::GetValue(const TInstant reqActuality) const {
    if (!Context) {
        return {};
    }
    if (!CachedInstant || *CachedInstant < reqActuality) {
        CachedValue = DoGetValue(reqActuality);
        CachedInstant = reqActuality;
    }
    return CachedValue;
}

const TMaybe<TUserSetting> IUserOption::GetUserSetting() const {
    if (Context) {
        return Context->GetUserSetting();
    } else {
        return {};
    }
}

TMaybe<TString> TCommonOption::DoGetValue(const TInstant reqActuality) const {
    auto userSetting = GetUserSetting();
    if (!userSetting.Defined()) {
        return {};
    }
    NDrive::TInfoEntitySession session;
    TExpectedSettingString result;
    if (!TagName) {
        result = userSetting->GetValueFromCache(".*", MetaInfo.GetId(), session, reqActuality, true);
    } else {
        result = userSetting->GetValueFromCache(TagName, MetaInfo.GetId(), session, reqActuality);
    }
    if (!result) {
        ERROR_LOG << "TCommonOption::DoGetValue: can't get user setting: " << session.GetStringReport() << Endl;
        return {};
    }
    if (!HasValue(*result)) {
        return {};
    }
    return result.GetRef();
}

TVector<TString> TCommonOption::GetValues() const {
    TVector<TString> result;
    for (auto&& [id, value] : MetaInfo.GetClientInfo()) {
        result.push_back(id);
    }
    return result;
}

bool TCommonOption::HasValue(TStringBuf value) const {
    for (auto&& [id, _] : MetaInfo.GetClientInfo()) {
        if (id == value) {
            return true;
        }
    }
    return false;
}

NJson::TJsonValue IUserOption::GetPublicReport(ELocalization locale) const {
    if (Context) {
        return DoGetPublicReport(locale);
    } else {
        return NJson::JSON_NULL;
    }
}

NDrive::TScheme TUserOptionMeta::GetScheme(const NDrive::IServer* /*server*/) {
    NDrive::TScheme result;
    result.Add<TFSString>("id", "Идентификатор опции");
    result.Add<TFSString>("title", "Читаемый идентификатор опции");
    result.Add<TFSString>("default_value", "Значение по умолчанию");
    result.Add<TFSBoolean>("visible", "Видимость для пользователя");

    NDrive::TScheme elementScheme;
    elementScheme.Add<TFSString>("id", "Идентификатор значения");
    elementScheme.Add<TFSJson>("client_json", "Описание для клиента");
    result.Add<TFSArray>("variants", "Варианты значений с описаниями для клиента").SetElement(std::move(elementScheme));
    return result;
}

NJson::TJsonValue TUserOptionMeta::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    TJsonProcessor::Write(result, "id", Id);
    TJsonProcessor::Write(result, "title", Title);
    TJsonProcessor::Write(result, "default_value", DefaultValue);
    TJsonProcessor::Write(result, "visible", Visible);
    NJson::TJsonValue& jArray = result.InsertValue("variants", NJson::JSON_ARRAY);
    for (auto&& i : ClientInfo) {
        NJson::TJsonValue& clientInfo = jArray.AppendValue(NJson::JSON_MAP);
        clientInfo.InsertValue("id", i.first);
        clientInfo.InsertValue("client_json", i.second);
    }
    return result;
}

bool TUserOptionMeta::DeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TJsonProcessor::Read(jsonValue, "id", Id) || !Id) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "title", Title)) {
        return false;
    }
    if (!Title) {
        Title = Id;
    }
    if (!TJsonProcessor::Read(jsonValue, "default_value", DefaultValue)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "visible", Visible)) {
        return false;
    }
    const NJson::TJsonValue::TArray* variantsArray;
    if (!jsonValue["variants"].GetArrayPointer(&variantsArray)) {
        return false;
    }
    for (auto&& i : *variantsArray) {
        TString variantId;
        if (!TJsonProcessor::Read(i, "id", variantId)) {
            return false;
        }
        NJson::TJsonValue clientJson = i["client_json"];
        clientJson["value"] = variantId;
        ClientInfo.emplace_back(std::move(variantId), std::move(clientJson));
    }
    return true;
}
