#pragma once

#include "user_setting.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/tags/tags.h>

class TUserOptionMeta {
private:
    using TVariants = TVector<std::pair<TString, NJson::TJsonValue>>;

private:
    R_FIELD(TString, Id);
    R_FIELD(TString, Title);
    R_FIELD(TString, DefaultValue);
    R_FIELD(bool, Visible, false);
    R_FIELD(TVariants, ClientInfo);

public:
    static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/);

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonValue);
};

class IUserOption {
public:
    class TContext {
    private:
        TUserSetting UserSetting;

    public:
        TContext(const TString& userId)
            : UserSetting(userId)
        {
        }

        const TUserSetting& GetUserSetting() const {
            return UserSetting;
        }
    };

protected:
    const TContext* Context = nullptr;
    mutable TMaybe<TInstant> CachedInstant;
    mutable TMaybe<TString> CachedValue;
    virtual TMaybe<TString> DoGetValue(const TInstant reqActuality) const = 0;
    virtual NJson::TJsonValue DoGetPublicReport(ELocalization locale) const = 0;
    virtual TString GetDefaultValue() const {
        return "";
    }

public:
    virtual bool GetVisible() const {
        return true;
    }
    virtual NJson::TJsonValue GetPublicReport(ELocalization locale) const;
    virtual TString GetOptionId() const = 0;

    virtual ~IUserOption() = default;

    using TPtr = TAtomicSharedPtr<IUserOption>;

    IUserOption() = default;
    IUserOption(const TContext& context)
        : Context(&context)
    {
    }

    virtual TString GetValueOrDefault(TInstant actuality) const {
        return GetValue(actuality).GetOrElse(GetDefaultValue());
    }

    virtual TMaybe<TString> GetValue(const TInstant reqActuality) const;
    virtual TVector<TString> GetValues() const = 0;

    template <class T>
    TMaybe<T> GetValue(TInstant actuality) const {
        auto val = GetValue(actuality);
        if (!val) {
            return {};
        }
        T result;
        if (!TryFromString(*val, result)) {
            return {};
        }
        return result;
    }

    const TMaybe<TUserSetting> GetUserSetting() const;
};

class TFakeUserOption: public IUserOption {
protected:
    virtual TString GetOptionId() const override {
        return "undefined";
    }

    virtual NJson::TJsonValue DoGetPublicReport(ELocalization /*locale*/) const override {
        return NJson::JSON_NULL;
    }

    virtual TMaybe<TString> DoGetValue(const TInstant /*reqActuality*/) const override {
        return {};
    }

    virtual TVector<TString> GetValues() const override {
        return {};
    }
};

class TCommonOption: public IUserOption {
private:
    using TBase = IUserOption;

private:
    const TString TagName;

protected:
    const TUserOptionMeta MetaInfo;

protected:
    virtual TMaybe<TString> DoGetValue(const TInstant reqActuality) const override;
    virtual NJson::TJsonValue DoGetPublicReport(ELocalization locale) const override;
    virtual TString GetDefaultValue() const override {
        return MetaInfo.GetDefaultValue();
    }

public:
    virtual bool GetVisible() const override {
        return MetaInfo.GetVisible();
    }

    virtual TString GetOptionId() const override {
        return MetaInfo.GetId();
    }

    TCommonOption(const TContext& context, const TUserOptionMeta& metaInfo, const TString& tagName = "")
        : TBase(context)
        , TagName(tagName)
        , MetaInfo(metaInfo)
    {
    }

    virtual TVector<TString> GetValues() const override;
    bool HasValue(TStringBuf value) const;
};

class TExtendPriceOption: public TCommonOption {
private:
    using TBase = TCommonOption;

public:
    TExtendPriceOption(const TContext& context, const TUserOptionMeta& meta)
        : TBase(context, meta)
    {
    }
};
