#include "user_setting.h"

#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/tags/tags_manager.h>

TExpectedSettingString TUserSetting::GetDefaultValue(const TString& tagName, const TString& fieldName, NDrive::TInfoEntitySession& session) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    auto tagDescription = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
    if (!tagDescription) {
        session.SetErrorInfo("TUserSetting::GetDefaultValue", "Can't get description for tag " + tagName);
        return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::TagDescriptionError);
    }
    auto description = dynamic_cast<const TDictionaryTagDescription*>(tagDescription.Get());
    if (!description) {
        session.SetErrorInfo("TUserSetting::GetDefaultValue", "Tag is not dictionary tag " + tagName);
        return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::TagTypeError);
    }
    auto def = description->GetDefaults(fieldName);
    if (def) {
        return *def;
    } else {
        return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::NotFound);
    }
}

TExpectedSettingString TUserSetting::GetDefaultValue(const TRegExMatch& regMatch, const TString& fieldName) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    auto tagsDescriptions = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetTagsByType(TUserDictionaryTag::TypeName);
    for (auto&& description : tagsDescriptions) {
        if (regMatch.Match(description->GetName().data())) {
            auto dictDescription = dynamic_cast<const TDictionaryTagDescription*>(description.Get());
            if (dictDescription) {
                auto def = dictDescription->GetDefaults(fieldName);
                if (def) {
                    return def.GetRef();
                }
            }
        }
    }
    return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::NotFound);
}

TMaybe<TString> TUserSetting::GetFirstMatchingTagField(const TVector<TDBTag>& tags, const TString& fieldName, std::function<bool(const TDBTag& tag)> isMatching) const {
    for (auto&& tag : tags) {
        if (isMatching(tag)) {
            const IDictionaryTag* dictionaryTag = tag.GetTagAs<IDictionaryTag>();
            if (dictionaryTag) {
                auto fieldValue = dictionaryTag->GetField(fieldName);
                if (fieldValue.Defined()) {
                    return fieldValue;
                }
            }
        }
    }
    return {};
}

TExpectedSettingString TUserSetting::FindValueOrDefault(const TString& tagName, const TString& fieldName, const TVector<TDBTag>& tags, NDrive::TInfoEntitySession& session, const bool regex) const {
    TMaybe<TString> result;
    if (regex) {
        TRegExMatch regMatch;
        regMatch.Compile(tagName);
        if (!regMatch.IsCompiled()) {
            session.SetErrorInfo("TUserSetting::FindValueOrDefault", "Bad regular expression " + tagName);
            return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::RegexError);
        }
        result = GetFirstMatchingTagField(tags, fieldName, ([&regMatch](const TDBTag& tag) -> bool {
            return regMatch.Match(tag->GetName().data());
        }));
        if (!result) {
            return GetDefaultValue(regMatch, fieldName);
        }
    } else {
        result = GetFirstMatchingTagField(tags, fieldName, ([&tagName](const TDBTag& tag) -> bool {
            return tag->GetName() == tagName;
        }));
        if (!result) {
            return GetDefaultValue(tagName, fieldName, session);
        }
    }
    return result.GetRef();
}

TMaybe<TVector<TDBTag>> TUserSetting::GetTagsFromDB(const TVector<TString>& tagNames, NDrive::TEntitySession& session) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    TVector<TDBTag> tags;
    if (!server.GetDriveDatabase().GetTagsManager().GetUserTags().RestoreTags({ UserId }, tagNames, tags, session)) {
        return {};
    }
    return tags;
}

TMaybe<TVector<TDBTag>> TUserSetting::GetTagsFromCache(TInstant actuality) const {
    Y_UNUSED(actuality);
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    auto user = server.GetDriveDatabase().GetTagsManager().GetUserTags().GetCachedObject(UserId);
    return user->DetachTags();
}

TExpectedSettingString TUserSetting::GetValue(const TString& tagName, const TString& fieldName, NDrive::TEntitySession& session, const bool regex) const {
    TExpectedSettingString result;
    auto tags = regex ? GetTagsFromDB({}, session) : GetTagsFromDB({ tagName }, session);
    if (tags.Defined()) {
        return FindValueOrDefault(tagName, fieldName, *tags, session, regex);
    }
    return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::TagRetrievalError);
}

TExpectedSettingString TUserSetting::GetValueFromCache(const TString& tagName, const TString& fieldName, NDrive::TInfoEntitySession& session, TInstant actuality, bool regex) const {
    auto tags = GetTagsFromCache(actuality);
    if (tags.Defined()) {
        return FindValueOrDefault(tagName, fieldName, *tags, session, regex);
    } else {
        session.SetErrorInfo("TUserSetting::SetValue", "can't get tags from cache for user " + UserId);
        return MakeUnexpected<TUserSettingStatus>(TUserSettingStatus::TagRetrievalError);
    }
}

bool TUserSetting::SetValue(const TString tagName, const TString& fieldName, const TString& value, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    auto tags = GetTagsFromDB({ tagName }, session);
    if (!tags.Defined()) {
        return false;
    }
    if (tags->empty()) {
        const auto& server = NDrive::GetServerAs<NDrive::IServer>();
        auto tagPtr = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTag(tagName);
        if (!tagPtr) {
            session.SetErrorInfo("TUserSetting::SetDictionaryTagValue", "can't create tag " + tagName, EDriveSessionResult::InternalError);
            return false;
        }
        IDictionaryTag* dictTag = dynamic_cast<IDictionaryTag*>(tagPtr.Get());
        if (!dictTag) {
            session.SetErrorInfo("TUserSetting::SetDictionaryTagValue", "tag name " + tagName + " is not dictionary tag", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        dictTag->SetField(fieldName, value);
        return server.GetDriveDatabase().GetTagsManager().GetUserTags().AddTag(tagPtr, operatorUserId, UserId, &server, session).Defined();
    }

    TDBTag& tagWithField = tags->front();
    IDictionaryTag* tagWithFieldData = nullptr;
    for (auto&& tag : *tags) {
        IDictionaryTag* dictionaryTagData = tag.MutableTagAs<IDictionaryTag>();
        if (dictionaryTagData) {
            tagWithFieldData = dictionaryTagData;
            tagWithField = tag;
            auto fieldValue = dictionaryTagData->GetField(fieldName);
            if (fieldValue.Defined()) {
                break;
            }
        }
    }
    if (tagWithFieldData) {
        tagWithFieldData->SetField(fieldName, value);
        const auto& server = NDrive::GetServerAs<NDrive::IServer>();
        return server.GetDriveDatabase().GetTagsManager().GetUserTags().UpdateTagData(tagWithField, UserId, session);
    } else {
        session.SetErrorInfo("TUserSetting::SetDictionaryTagValue", "tag name " + tagName + " is not dictionary tag", EDriveSessionResult::IncorrectRequest);
        return false;
    }
}

IUserBaseSettings::IUserBaseSettings(const TString& userId)
    : UserSetting(userId)
{
}

TExpectedSettingString IUserBaseSettings::GetValueBySettingName(const TString& settingName, const TString& defaultFieldName, NDrive::TEntitySession& session) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    auto fieldName = server.GetSettings().GetValueDef<TString>(settingName, defaultFieldName);
    return UserSetting.GetValue(GetUserSettingsTagName(), fieldName, session);
}

TExpectedSettingString IUserBaseSettings::GetValueBySettingName(const TString& settingName, const TString& defaultFieldName, NDrive::TInfoEntitySession& session, const TInstant actuality) const {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    auto fieldName = server.GetSettings().GetValueDef<TString>(settingName, defaultFieldName);
    return UserSetting.GetValueFromCache(GetUserSettingsTagName(), fieldName, session, actuality);
}

TUserAppSettings::TUserAppSettings(const TString& userId)
    : TBase(userId)
{
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    UserAppSettingTagName = server.GetSettings().GetValueDef(NDrive::UserSettingsTagNameSetting, UserAppSettingTagName);
}

TString TUserAppSettings::GetUserSettingsTagName() const {
    return UserAppSettingTagName;
}

TExpectedSettingString TUserAppSettings::GetPushSubscription(NDrive::TEntitySession& session) const {
    return GetValueBySettingName(PushSubscriptionSetting, PushSubscriptionField, session);
}

TExpectedSettingString TUserAppSettings::GetEmailSubscription(NDrive::TEntitySession& session) const {
    return GetValueBySettingName(EmailSubscriptionSetting, EmailSubscriptionField, session);
}

TExpectedSettingString TUserAppSettings::GetLandingSubscription(NDrive::TEntitySession& session) const {
    return GetValueBySettingName(LandingSubscriptionSetting, LandingSubscriptionField, session);
}

TExpectedSettingString TUserAppSettings::GetCrossLogin(NDrive::TEntitySession& session) const {
    return GetValueBySettingName(DisableCrossLoginSetting, CrossLoginField, session);
}

TString TUserAppSettings::PushSubscriptionField = "notifications_push";
TString TUserAppSettings::EmailSubscriptionField = "notifications_email";
TString TUserAppSettings::LandingSubscriptionField = "notifications_landing";
TString TUserAppSettings::CrossLoginField = "disable_cross_login";
TString TUserAppSettings::PushSubscriptionSetting = "user_settings.f_notifications_push";
TString TUserAppSettings::EmailSubscriptionSetting = "user_settings.f_notifications_email";
TString TUserAppSettings::LandingSubscriptionSetting = "user_settings.f_notifications_landing";
TString TUserAppSettings::DisableCrossLoginSetting = "user_settings.f_disable_cross_login";


TUserAdminSettings::TUserAdminSettings(const TString& userId)
    : TBase(userId)
{
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    UserAdminSettingTagName = server.GetSettings().GetValueDef(NDrive::AdminSettingsTagNameSetting, UserAdminSettingTagName);
}

TString TUserAdminSettings::GetUserSettingsTagName() const {
    return UserAdminSettingTagName;
}

TExpectedSettingString TUserAdminSettings::GetInternalPhone(NDrive::TEntitySession& session) const {
    return GetValueBySettingName(InternalPhoneSetting, InternalPhoneField, session);
}

TString TUserAdminSettings::InternalPhoneField = "internal_phone";
TString TUserAdminSettings::InternalPhoneSetting = "admin_settings.f_internal_phone";
