#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_filter.h>

#include <rtline/util/types/expected.h>

enum TUserSettingStatus {
        NotFound /* "not_found" */,
        TagDescriptionError /* "tag_description_error" */,
        TagTypeError /* "tag_type_error" */,
        TagRetrievalError /* "tag_retrieval_error" */,
        RegexError /* "regex_error" */,
};

using TExpectedSettingString = TExpected<TString, TUserSettingStatus>;

class TUserSetting {
    R_FIELD(TString, UserId);

public:
    TUserSetting(const TString& userId)
        : UserId(userId)
    {
    }

    TExpectedSettingString GetValueFromCache(const TString& tagName, const TString& fieldName, NDrive::TInfoEntitySession& session, TInstant actuality = TInstant::Zero(), const bool regex = false) const;
    TExpectedSettingString GetValue(const TString& tagName, const TString& fieldName, NDrive::TEntitySession& session, const bool regex = false) const;
    bool SetValue(const TString tagName, const TString& fieldName, const TString& value, const TString& operatorUserId, NDrive::TEntitySession& session) const;

private:
    TMaybe<TVector<TDBTag>> GetTagsFromCache(TInstant actuality) const;
    TMaybe<TVector<TDBTag>> GetTagsFromDB(const TVector<TString>& tagNames, NDrive::TEntitySession& session) const;
    TMaybe<TString> GetFirstMatchingTagField(const TVector<TDBTag>& tags, const TString& fieldName, std::function<bool(const TDBTag& tag)> isMatching) const;
    TExpectedSettingString FindValueOrDefault(const TString& tagName, const TString& fieldName, const TVector<TDBTag>& tags, NDrive::TInfoEntitySession& session, const bool regex = false) const;
    TExpectedSettingString GetDefaultValue(const TRegExMatch& regMatch, const TString& fieldName) const;
    TExpectedSettingString GetDefaultValue(const TString& tagName, const TString& fieldName, NDrive::TInfoEntitySession& session) const;
};

class IUserBaseSettings {
private:
    TUserSetting UserSetting;

protected:
    TExpectedSettingString GetValueBySettingName(const TString& settingName, const TString& defaultFieldName, NDrive::TEntitySession& session) const;
    TExpectedSettingString GetValueBySettingName(const TString& settingName, const TString& defaultFieldName, NDrive::TInfoEntitySession& session, const TInstant actuality) const;

    virtual TString GetUserSettingsTagName() const = 0;

public:
    IUserBaseSettings(const TString& userId);
    virtual ~IUserBaseSettings() = default;
};

class TUserAppSettings : public IUserBaseSettings {
public:
    TUserAppSettings(const TString& userId);

    TExpectedSettingString GetPushSubscription(NDrive::TEntitySession& session) const;
    TExpectedSettingString GetEmailSubscription(NDrive::TEntitySession& session) const;
    TExpectedSettingString GetLandingSubscription(NDrive::TEntitySession& session) const;
    TExpectedSettingString GetCrossLogin(NDrive::TEntitySession& session) const;

private:
    using TBase = IUserBaseSettings;
    TString UserAppSettingTagName = "user_app_settings";

    static TString PushSubscriptionField;
    static TString EmailSubscriptionField;
    static TString LandingSubscriptionField;
    static TString CrossLoginField;

    static TString PushSubscriptionSetting;
    static TString EmailSubscriptionSetting;
    static TString LandingSubscriptionSetting;
    static TString DisableCrossLoginSetting;

protected:
    virtual TString GetUserSettingsTagName() const override;
};

class TUserAdminSettings : public IUserBaseSettings {
public:
    TUserAdminSettings(const TString& userId);

    TExpectedSettingString GetInternalPhone(NDrive::TEntitySession& session) const;

private:
    using TBase = IUserBaseSettings;
    TString UserAdminSettingTagName = "user_admin_settings";

    static TString InternalPhoneField;

    static TString InternalPhoneSetting;

protected:
    virtual TString GetUserSettingsTagName() const override;
};
