#include "driving_profile.h"

#include <drive/backend/roles/permissions.h>
#include <drive/backend/context_fetcher/aggression.h>

namespace NDrive {

    namespace {
        const double MaxProgress = 0.99; // Not 1.0 due to client rendering limitations.

        double CalcAggressionStageProgress(const TAggressionConfig::TStage& stage, double value) {
            return std::max(std::min((value - stage.BeginValue) / std::max(stage.EndValue - stage.BeginValue, 1e-9), MaxProgress), 0.0);
        }

        void SetAggressionReportFields(
            NJson::TJsonValue& drivingStyle,
            const TAggressionConfig::TStage& stage,
            const NDrive::TAggressionFetchContext& context,
            const TScoringUserTag* aggressionTag,
            double epsilon
        ) {
            TMessagesCollector errors;
            drivingStyle["title"] = NDrive::IAggressionContextFetcher::ProcessText(stage.Title, context, errors);
            if (stage.SubTitle) {
                drivingStyle["sub_title"] = NDrive::IAggressionContextFetcher::ProcessText(stage.SubTitle, context, errors);
            }
            drivingStyle["next_title"] = stage.NextTitle;
            drivingStyle["next_color"] = stage.NextColor;
            drivingStyle["prev_title"] = stage.PrevTitle;
            drivingStyle["prev_color"] = stage.PrevColor;
            NJson::TJsonValue descriptions = NJson::JSON_ARRAY;
            for (auto&& description : stage.Descriptions) {
                descriptions.AppendValue(description.GetReport());
            }
            drivingStyle["descriptions"] = descriptions;
            {
                NJson::TJsonValue photo = NJson::JSON_MAP;
                if (stage.Photo.Url) {
                    photo["url"] = NDrive::IAggressionContextFetcher::ProcessText(stage.Photo.Url, context, errors);
                }
                if (stage.Photo.Text) {
                    photo["text"] = NDrive::IAggressionContextFetcher::ProcessText(stage.Photo.Text, context, errors);
                }
                drivingStyle["photo"] = photo;
            }
            drivingStyle["button"] = stage.Button.GetReport();
            drivingStyle["last_change"]["title"] = stage.MenuTitle;
            if (aggressionTag) {
                drivingStyle["update_ts"] = aggressionTag->GetTimestamp().Seconds();
                if (aggressionTag->HasPreviousValue()) {
                    if (aggressionTag->GetValue() + epsilon < aggressionTag->GetPreviousValueRef()) {
                        drivingStyle["last_change"]["direction"] = 1;
                        if (stage.PositiveMenuTitle) {
                            drivingStyle["last_change"]["title"] = stage.PositiveMenuTitle;
                        }
                    } else if (aggressionTag->GetValue() - epsilon > aggressionTag->GetPreviousValueRef()) {
                        drivingStyle["last_change"]["direction"] = 0;
                        if (stage.NegativeMenuTitle) {
                            drivingStyle["last_change"]["title"] = stage.NegativeMenuTitle;
                        }
                    }
                }
            }
        }
    }

    i32 TDrivingProfile::TStageConfig::CalcPoints(double value, bool inverted) const {
        if (!BeginPoints || !EndPoints) {
            return 0;
        }
        auto progress = std::max(std::min((value - BeginValue) / std::max(EndValue - BeginValue, 1e-9), 1.), 0.);
        if (inverted) {
            progress = 1. - progress;
        }
        return progress * (*EndPoints - *BeginPoints) + *BeginPoints;
    }

    NJson::TJsonValue TDrivingProfile::GetPublicReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        if (permissions.GetSetting<bool>(server.GetSettings(), "driving_profile.v1_report").GetOrElse(true)) {
            return GetPublicReportV1(server, permissions, locale);
        }
        return GetPublicReportV2(server, permissions, locale);
    }

    NJson::TJsonValue TDrivingProfile::GetNotificationReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, const TString& tagName, bool isActiveStage, bool isActiveTag, const TString& stageName, TMaybe<TInstant> deadline) const {
        bool needFilter = !isActiveTag || !isActiveStage;
        auto notificationStages = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + ".stages").GetOrElse("");
        if (!notificationStages) {
            return {};
        }
        auto filteredStages = MakeSet(StringSplitter(notificationStages).SplitBySet(",").SkipEmpty().ToList<TString>());
        if (!needFilter || filteredStages.contains(stageName)) {
            TString notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName).GetOrElse("");
            bool isPlus = permissions.GetUserFeatures().GetIsPlusUser();
            notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + (isPlus ? ".plus" : ".no_plus")).GetOrElse(notificationString);
            if (isActiveTag) {
                notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + ".active").GetOrElse(notificationString);
                notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + (isPlus ? ".plus" : ".no_plus") + ".active").GetOrElse(notificationString);
            }
            if (!needFilter && !filteredStages.contains(stageName)) {
                notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + ".expires").GetOrElse(notificationString);
                notificationString = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.notification." + tagName + (isPlus ? ".plus" : ".no_plus") + ".expires").GetOrElse(notificationString);
            }
            if (notificationString) {
                auto notification = NJson::FromJson<TAggressionNotification>(NJson::ToJson(NJson::JsonString(notificationString))).GetReport(
                    locale,
                    server.GetLocalization(),
                    deadline
                );
                notification["type"] = tagName;
                return notification;
            }
        }
        return {};
    }

    NJson::TJsonValue TDrivingProfile::GetNotificationsReportV1(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        NJson::TJsonValue report = NJson::JSON_ARRAY;
        // TODO(iudovin@): DEPRECATED. We should remove this shit.
        // "high_cashback" notification.
        if (HighCashbackRole) {
            TString notificationKey = permissions.GetUserFeatures().GetIsPlusUser() ? "high_cashback" : "high_cashback.no_plus";
            if (auto notificationString = permissions.GetSetting<TString>(server.GetSettings(), "aggression.notification." + notificationKey)) {
                auto deadline = HighCashbackRole->GetDeadline() + TDuration::Hours(3);
                auto notification = NJson::FromJson<TAggressionNotification>(NJson::ToJson(NJson::JsonString(*notificationString))).GetReport(
                    locale,
                    server.GetLocalization(),
                    deadline
                );
                notification["type"] = "high_cashback";
                report.AppendValue(std::move(notification));
            }
        }
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetNotificationsReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, bool isActiveStage, const TString& stageName) const {
        NJson::TJsonValue report = NJson::JSON_ARRAY;
        if (!stageName) {
            TSet<TString> globalNotifications = StringSplitter(permissions.GetSetting<TString>("driving_profile.global_notifications").GetOrElse("")).Split(',').SkipEmpty();
            for (auto&& name : globalNotifications) {
                auto notification = GetNotificationReport(server, permissions, locale, name, /*isActiveStage*/ true, /*isActiveTag*/ true);
                if (notification.IsDefined()) {
                    report.AppendValue(std::move(notification));
                }
            }
            return report;
        }
        TSet<TString> notActiveTagNames = StringSplitter(permissions.GetSetting<TString>("driving_profile.cashback_notification_tag_names").GetOrElse("")).Split(',').SkipEmpty();
        for (const auto& tag : NotificationTags) {
            auto notification = GetNotificationReport(server, permissions, locale, tag->GetName(), isActiveStage, /*isActiveTag*/ true, stageName, tag->GetSLAInstant());
            if (notification.IsDefined()) {
                report.AppendValue(std::move(notification));
            }
            notActiveTagNames.erase(tag->GetName());
        }
        for (const auto& tag : notActiveTagNames) {
            auto notification = GetNotificationReport(server, permissions, locale, tag, isActiveStage, /*isActiveTag*/ false, stageName);
            if (notification.IsDefined()) {
                report.AppendValue(std::move(notification));
            }
        }
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetSessionsReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        NJson::TJsonValue report = NJson::JSON_MAP;
        if (auto local = server.GetLocalization()) {
            report["title"] = local->GetLocalString(locale, "driving_profile.sessions.title");
        }
        // Filter for sessions driving profile screen.
        report["filter"] = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.sessions.filter").GetOrElse("used_in_scoring");
        // Filter for main driving profile screen.
        report["profile_filter"] = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.sessions.profile_filter").GetOrElse("used_in_scoring,aggressive,hard_speeding,fined");
        return report;
    }

    TMaybe<TDrivingProfile::TStages> TDrivingProfile::GetVisibleStages(const NDrive::IServer& server, const TUserPermissions& permissions) const {
        if (UserStatus != NDrive::UserStatusActive) {
            if (const TUserProblemTag* tag = AggressiveBlockTag.GetTagAs<TUserProblemTag>()) {
                auto stageConfig = permissions.GetSetting<TStageConfig>(server.GetSettings(), "driving_profile.block_stage." + tag->GetName());
                TString stageId = tag->GetName();
                if (!stageConfig) {
                    stageConfig = permissions.GetSetting<TStageConfig>(server.GetSettings(), "driving_profile.block_stage");
                    stageId = "block_stage";
                }
                if (!stageConfig) {
                    return {};
                }
                TStages stages;
                TStage stage;
                stage.Id = stageId;
                stage.Config = *stageConfig;
                stages.CurrentStage = stage.Id;
                if (auto description = tag->GetDescriptionAs<TUserProblemTag::TDescription>(server)) {
                    if (auto totalDuration = description->GetBanDuration()) {
                        auto endTime = tag->GetSLAInstant();
                        auto beginTime = endTime - totalDuration;
                        auto duration = Now() - beginTime;
                        stages.CurrentPoints = duration.Days();
                        stages.BlockDuration = duration;
                        stages.BlockTotalDuration = totalDuration;
                        stages.BlockEndTime = endTime;
                    }
                }
                stages.Stages.push_back(std::move(stage));
                return stages;
            }
        }
        if (const TScoringUserTag* tag = AggressiveTrialTag.GetTagAs<TScoringUserTag>()) {
            auto stageConfig = permissions.GetSetting<TStageConfig>(server.GetSettings(), "driving_profile.trial_stage");
            if (!stageConfig) {
                return {};
            }
            TStages stages;
            TStage stage;
            stage.Id = "trial_stage";
            stage.Config = *stageConfig;
            stages.CurrentStage = stage.Id;
            stages.CurrentPoints = tag->GetValue();
            stages.Stages.push_back(std::move(stage));
            return stages;
        }
        auto config = permissions.GetSetting<TStagesConfig>(server.GetSettings(), "driving_profile.stages");
        if (!config) {
            return {};
        }
        TVector<size_t> perm(config->Stages.size());
        std::iota(perm.begin(), perm.end(), 0);
        std::sort(perm.begin(), perm.end(),
            [&stages = config->Stages](i32 lhs, i32 rhs) {
                return stages[lhs].FromValue < stages[rhs].FromValue;
            }
        );
        size_t stagePos = 0;
        TMaybe<double> stageValue;
        if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>()) {
            for (size_t i = 0; i < config->Stages.size(); i++) {
                if (config->Stages[perm[i]].FromValue > tag->GetValue()) {
                    break;
                }
                stagePos = perm[i];
            }
            stageValue = tag->GetValue();
        }
        TStages stages;
        if (auto stageConfig = permissions.GetSetting<TStageConfig>(server.GetSettings(), "driving_profile.block_stage")) {
            TStage stage;
            stage.Id = "block_stage";
            stage.Config = *stageConfig;
            stages.Stages.push_back(std::move(stage));
        }
        bool inverted = permissions.GetSetting<bool>(server.GetSettings(), "driving_profile.invert_points").GetOrElse(true);
        if (auto stageConfig = permissions.GetSetting<TStageConfig>(server.GetSettings(), "driving_profile.price_up_stage")) {
            TStage stage;
            stage.Id = "price_up_stage";
            stage.Config = *stageConfig;
            if (const ITag* tag = AggressivePriceUpTag.GetTagAs<ITag>()) {
                stages.CurrentStage = stage.Id;
                if (stageValue) {
                    stages.CurrentPoints = stageConfig->CalcPoints(*stageValue, inverted);
                }
                stagePos = -1;
            }
            stages.Stages.push_back(std::move(stage));
        }
        for (size_t i = 0; i < config->Stages.size(); i++) {
            auto&& stageConfig = config->Stages[i];
            TStage stage;
            stage.Id = "stage_" + ToString(i + 1);
            stage.Config = stageConfig;
            if (i == stagePos) {
                stages.CurrentStage = stage.Id;
                if (stageValue) {
                    stages.CurrentPoints = stageConfig.CalcPoints(*stageValue, inverted);
                }
            }
            stages.Stages.push_back(std::move(stage));
        }
        return stages;
    }

    NJson::TJsonValue TDrivingProfile::GetStagesReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, const TStages& stages) const {
        NJson::TJsonValue report = NJson::JSON_ARRAY;
        for (auto&& stage : stages.Stages) {
            auto stageReport = GetStageReport(server, permissions, locale, stage.Config, stage.Config.Id ? stage.Config.Id : stage.Id, stages);
            if (stages.CurrentStage == stage.Id) {
                stageReport["focused"] = true;
                if (stages.CurrentPoints) {
                    stageReport["value"] = *stages.CurrentPoints;
                }
            }
            report.AppendValue(std::move(stageReport));
        }
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetStageReport(
        const NDrive::IServer& server,
        const TUserPermissions& permissions,
        ELocalization locale,
        const TStageConfig& stageConfig,
        const TString& stageName,
        const TStages& stages
    ) const {
        NJson::TJsonValue report = NJson::JSON_MAP;
        report["id"] = stageName;
        report["prev_color"] = stageConfig.PrevColor;
        report["next_color"] = stageConfig.NextColor;
        if (stageConfig.BeginPoints) {
            report["prev_value"] = *stageConfig.BeginPoints;
        }
        if (stageConfig.EndPoints) {
            report["next_value"] = *stageConfig.EndPoints;
        }
        if (stageConfig.Parts) {
            report["parts"] = *stageConfig.Parts;
        }
        bool active = stageName == stages.CurrentStage;
        if (auto local = server.GetLocalization()) {
            auto stageLocalString = [
                local, locale,
                blockDuration = stages.BlockDuration,
                blockTotalDuration = stages.BlockTotalDuration,
                blockEndTime = stages.BlockEndTime,
                prefix = "driving_profile.stage." + stageName + ".",
                currentStage = stages.CurrentStage,
                active
            ](const TString& name) {
                auto result = local->GetLocalString(locale, prefix + name, "");
                if (active) {
                    result = local->GetLocalString(locale, prefix + name + ".active", result);
                } else {
                    result = local->GetLocalString(locale, prefix + name + ".on_active." + currentStage, result);
                }
                if (blockDuration) {
                    SubstGlobal(result, "_BlockDays_", local->DaysFormat(*blockDuration, locale));
                }
                if (blockTotalDuration) {
                    SubstGlobal(result, "_BlockTotalDays_", local->DaysFormat(*blockTotalDuration, locale));
                }
                if (blockDuration && blockTotalDuration) {
                    SubstGlobal(result, "_BlockRemainedDays_", local->DaysFormat(*blockTotalDuration - *blockDuration, locale));
                }
                if (blockEndTime) {
                    SubstGlobal(result, "_BlockEndDate_", local->FormatMonthDay(locale, *blockEndTime + TDuration::Hours(3)));
                }
                return result;
            };
            if (auto units = stageLocalString("units")) {
                report["units"] = units;
            }
            report["caption"] = stageLocalString("caption");
            report["sub_caption"] = stageLocalString("sub_caption");
            NJson::TJsonValue descriptions = NJson::JSON_MAP;
            descriptions["title"] = stageLocalString("descriptions_title");
            NJson::TJsonValue items = NJson::JSON_ARRAY;
            for (auto&& descriptionId : stageConfig.DescriptionIds) {
                NJson::TJsonValue item = NJson::JSON_MAP;
                item["icon"] = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.stage_description." + descriptionId + ".icon").GetOrElse("");
                item["text"] = local->GetLocalString(locale, "driving_profile.stage_description." + descriptionId + ".text", "");
                items.AppendValue(item);
            }
            descriptions["items"] = std::move(items);
            report["descriptions"] = std::move(descriptions);
            TString notificationTitle = local->GetLocalString(locale, "driving_profile.notification_title." + stageName + TString(active ? ".active" : "") + TString(permissions.GetUserFeatures().GetIsPlusUser() ? ".plus" : ".no_plus"), "");
            if (!notificationTitle) {
                notificationTitle = local->GetLocalString(locale, "driving_profile.notification_title" + TString(active ? ".active" : "") + TString(permissions.GetUserFeatures().GetIsPlusUser() ? ".plus" : ".no_plus"), "");
            }
            report["notification_title"] = notificationTitle;
            report["notifications"] = GetNotificationsReport(server, permissions, locale, active, stageName);
        }
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetLastChangeReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        NJson::TJsonValue report = NJson::JSON_MAP;
        if (auto local = server.GetLocalization()) {
            TString title = local->GetLocalString(locale, "driving_profile.last_change.title", "");
            if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>(); tag && tag->HasPreviousValue()) {
                const auto epsilon = permissions.GetSetting<double>(server.GetSettings(), "aggression.value_epsilon").GetOrElse(1e-9);
                if (tag->GetValue() + epsilon < tag->GetPreviousValueRef()) {
                    title = local->GetLocalString(locale, "driving_profile.last_change.positive_title", title);
                    report["direction"] = 1;
                } else if (tag->GetValue() - epsilon > tag->GetPreviousValueRef()) {
                    title = local->GetLocalString(locale, "driving_profile.last_change.negative_title", title);
                    report["direction"] = 0;
                }
            }
            report["title"] = title;
        }
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetButtonReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, const TStages& stages) const {
        NJson::TJsonValue report = NJson::JSON_MAP;
        if (auto local = server.GetLocalization()) {
            report["title"] = local->GetLocalString(locale, "driving_profile.stage." + stages.CurrentStage + ".button_title", "");
        }
        report["deeplink"] = permissions.GetSetting<TString>(server.GetSettings(), "driving_profile.stage." + stages.CurrentStage + ".button_deeplink").GetOrElse("");
        return report;
    }

    NJson::TJsonValue TDrivingProfile::GetPublicReportV2(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        NDrive::TAggressionEntry aggressionEntry;
        if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>()) {
            aggressionEntry.SetUserTag(*tag);
        }
        if (const TUserProblemTag* tag = AggressiveBlockTag.GetTagAs<TUserProblemTag>()) {
            aggressionEntry.SetBlockTag(*tag);
        }
        if (const TScoringUserTag* tag = AggressiveTrialTag.GetTagAs<TScoringUserTag>()) {
            aggressionEntry.SetTrialTag(*tag);
        }
        NDrive::TAggressionFetchContext aggressionContext(&server, aggressionEntry);
        NJson::TJsonValue drivingStyle = NJson::JSON_MAP;
        auto stages = GetVisibleStages(server, permissions);
        if (!stages) {
            return NJson::JSON_NULL;
        }
        drivingStyle["notifications"] = GetNotificationsReport(server, permissions, locale);
        drivingStyle["sessions"] = GetSessionsReport(server, permissions, locale);
        drivingStyle["items"] = GetStagesReport(server, permissions, locale, *stages);
        drivingStyle["last_change"] = GetLastChangeReport(server, permissions, locale);
        drivingStyle["button"] = GetButtonReport(server, permissions, locale, *stages);
        if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>()) {
            drivingStyle["update_ts"] = tag->GetTimestamp().Seconds();
        }
        return drivingStyle;
    }

    NJson::TJsonValue TDrivingProfile::GetPublicReportV1(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const {
        auto configString = permissions.GetSetting<TString>(server.GetSettings(), "aggression.stages_config");
        if (!configString) {
            return NJson::JSON_NULL;
        }
        auto configJson = NJson::ToJson(NJson::JsonString(*configString));
        auto localization = server.GetLocalization();
        if (localization) {
            localization->ApplyResourcesForJson(configJson, locale);
        }
        auto config = NJson::FromJson<TAggressionConfig>(configJson);
        NDrive::TAggressionEntry aggressionEntry;
        if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>()) {
            aggressionEntry.SetUserTag(*tag);
        }
        if (const TUserProblemTag* tag = AggressiveBlockTag.GetTagAs<TUserProblemTag>()) {
            aggressionEntry.SetBlockTag(*tag);
        }
        if (const TScoringUserTag* tag = AggressiveTrialTag.GetTagAs<TScoringUserTag>()) {
            aggressionEntry.SetTrialTag(*tag);
        }
        NDrive::TAggressionFetchContext aggressionContext(&server, aggressionEntry);
        NJson::TJsonValue drivingStyle = NJson::JSON_MAP;
        drivingStyle["notifications"] = GetNotificationsReportV1(server, permissions, locale);
        drivingStyle["sessions"] = GetSessionsReport(server, permissions, locale);
        auto epsilon = permissions.GetSetting<double>(server.GetSettings(), "aggression.value_epsilon").GetOrElse(1e-9);
        const TScoringUserTag* aggressionTag = AggressiveTag.GetTagAs<TScoringUserTag>();
        if (UserStatus != NDrive::UserStatusActive) {
            if (const TUserProblemTag* tag = AggressiveBlockTag.GetTagAs<TUserProblemTag>()) {
                const auto& stage = config.BlockStages[tag->GetName()];
                SetAggressionReportFields(drivingStyle, stage, aggressionContext, aggressionTag, epsilon);
                return drivingStyle;
            }
        }
        if (const TScoringUserTag* tag = AggressiveTrialTag.GetTagAs<TScoringUserTag>()) {
            SetAggressionReportFields(drivingStyle, config.TrialStage, aggressionContext, aggressionTag, epsilon);
            auto trialMileageLimit = permissions.GetSetting<double>(server.GetSettings(), "aggression.trial_mileage_limit").GetOrElse(100);
            auto progress = std::max(std::min(tag->GetValue() / std::max(trialMileageLimit, 1e-9), 1.0), 0.0);
            drivingStyle["progress"] = progress;
            return drivingStyle;
        }
        if (const ITag* tag = AggressivePriceUpTag.GetTagAs<ITag>()) {
            SetAggressionReportFields(drivingStyle, config.PriceUpStage, aggressionContext, aggressionTag, epsilon);
            drivingStyle["progress"] = CalcAggressionStageProgress(config.PriceUpStage, aggressionTag ? aggressionTag->GetValue() : 0);
            return drivingStyle;
        }
        if (!aggressionTag) {
            SetAggressionReportFields(drivingStyle, config.EmptyStage, aggressionContext, aggressionTag, epsilon);
            return drivingStyle;
        }
        size_t stagePos = 0;
        std::sort(
            config.Stages.begin(), config.Stages.end(),
            [](const TAggressionConfig::TStage& lhs, const TAggressionConfig::TStage& rhs) {
                return lhs.FromValue < rhs.FromValue;
            }
        );
        if (const TScoringUserTag* tag = AggressiveTag.GetTagAs<TScoringUserTag>()) {
            for (size_t i = 0; i < config.Stages.size(); i++) {
                if (config.Stages[i].FromValue > tag->GetValue()) {
                    break;
                }
                stagePos = i;
            }
        }
        if (stagePos >= config.Stages.size()) {
            return NJson::JSON_NULL;
        }
        auto&& stage = config.Stages[stagePos];
        SetAggressionReportFields(drivingStyle, stage, aggressionContext, aggressionTag, epsilon);
        Y_ENSURE_BT(aggressionTag);
        drivingStyle["progress"] = CalcAggressionStageProgress(stage, aggressionTag->GetValue());
        return drivingStyle;
    }
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrive::TDrivingProfile::TStageConfig& config) {
    return NJson::ParseField(json["id"], config.Id) &&
        NJson::ParseField(json["prev_color"], config.PrevColor) &&
        NJson::ParseField(json["next_color"], config.NextColor) &&
        NJson::ParseField(json["description_ids"], config.DescriptionIds) &&
        NJson::ParseField(json["from_value"], config.FromValue) &&
        NJson::ParseField(json["begin_value"], config.BeginValue) &&
        NJson::ParseField(json["end_value"], config.EndValue) &&
        NJson::ParseField(json["begin_points"], config.BeginPoints) &&
        NJson::ParseField(json["end_points"], config.EndPoints) &&
        NJson::ParseField(json["parts"], config.Parts);
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrive::TDrivingProfile::TStagesConfig& config) {
    return NJson::ParseField(json["stages"], config.Stages);
}

template<>
bool TryFromString(const TString& data, NDrive::TDrivingProfile::TStageConfig& config) {
    return NJson::TryFromJson(NJson::ToJson(NJson::JsonString(data)), config);
}

template<>
bool TryFromString(const TString& data, NDrive::TDrivingProfile::TStagesConfig& config) {
    return NJson::TryFromJson(NJson::ToJson(NJson::JsonString(data)), config);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionConfig& result) {
    return NJson::ParseField(value["stages"], result.Stages) &&
        NJson::ParseField(value["block_stages"], result.BlockStages) &&
        NJson::ParseField(value["price_up_stage"], result.PriceUpStage) &&
        NJson::ParseField(value["trial_stage"], result.TrialStage) &&
        NJson::ParseField(value["empty_stage"], result.EmptyStage) &&
        NJson::ParseField(value["sessions_title"], result.SessionsTitle) &&
        NJson::ParseField(value["sessions_filter"], result.SessionsFilter);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionConfig::TStage& result) {
    return NJson::ParseField(value["from_value"], result.FromValue) &&
        NJson::ParseField(value["begin_value"], result.BeginValue) &&
        NJson::ParseField(value["end_value"], result.EndValue) &&
        NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["sub_title"], result.SubTitle) &&
        NJson::ParseField(value["next_title"], result.NextTitle) &&
        NJson::ParseField(value["next_color"], result.NextColor) &&
        NJson::ParseField(value["prev_title"], result.PrevTitle) &&
        NJson::ParseField(value["prev_color"], result.PrevColor) &&
        NJson::ParseField(value["descriptions"], result.Descriptions) &&
        NJson::ParseField(value["button"], result.Button) &&
        NJson::ParseField(value["photo"], result.Photo) &&
        NJson::ParseField(value["menu_title"], result.MenuTitle) &&
        NJson::ParseField(value["positive_menu_title"], result.PositiveMenuTitle) &&
        NJson::ParseField(value["negative_menu_title"], result.NegativeMenuTitle);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionConfig::TStage::TButton& result) {
    return NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["deeplink"], result.Deeplink);
}

NJson::TJsonValue TAggressionConfig::TStage::TButton::GetReport() const {
    NJson::TJsonValue result;
    result.InsertValue("title", Title);
    result.InsertValue("deeplink", Deeplink);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionConfig::TStage::TPhoto& result) {
    return NJson::ParseField(value["url"], result.Url) &&
        NJson::ParseField(value["text"], result.Text);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionConfig::TStage::TDescription& result) {
    return NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["description"], result.Description);
}

NJson::TJsonValue TAggressionConfig::TStage::TDescription::GetReport() const {
    NJson::TJsonValue result;
    result.InsertValue("title", Title);
    result.InsertValue("description", Description);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionNotification& result) {
    return NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["description"], result.Description) &&
        NJson::ParseField(value["icon"], result.Icon) &&
        NJson::ParseField(value["bg_color"], result.BgColor) &&
        NJson::ParseField(value["bg_color_end"], result.BgColorEnd) &&
        NJson::ParseField(value["details"], result.Details);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionNotification::TDetails& result) {
    return NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["subtitle"], result.SubTitle) &&
        NJson::ParseField(value["description"], result.Description) &&
        NJson::ParseField(value["cashback"], result.Cashback) &&
        NJson::ParseField(value["image"], result.Image, false) &&
        NJson::ParseField(value["buttons"], result.Buttons);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAggressionNotification::TButton& result) {
    return NJson::ParseField(value["title"], result.Title) &&
        NJson::ParseField(value["deep_link"], result.DeepLink);
}

NJson::TJsonValue TAggressionNotification::GetReport(ELocalization locale, const ILocalization* localization, TMaybe<TInstant> deadline) const {
    NJson::TJsonValue result;
    result.InsertValue("title", Title);
    result.InsertValue("icon", Icon);
    result.InsertValue("bg_color", BgColor);
    if (BgColorEnd) {
        result.InsertValue("bg_color_end", BgColorEnd);
    }
    auto localStringWithDeadline = [
        localization = localization,
        locale = locale,
        deadline = deadline
    ](TString& stringWithDeadline) {
        if (localization && deadline) {
            SubstGlobal(stringWithDeadline, "_Deadline_", localization->FormatMonthDay(locale, *deadline));
        }
    };
    TString description = Description;
    localStringWithDeadline(description);
    result.InsertValue("description", description);
    if (Details) {
        NJson::TJsonValue details;
        details.InsertValue("title", Details->Title);
        TString subtitle = Details->SubTitle;
        localStringWithDeadline(subtitle);
        TString detailsDescription = Details->Description;
        localStringWithDeadline(detailsDescription);
        details.InsertValue("subtitle", subtitle);
        details.InsertValue("description", detailsDescription);
        if (Details->Cashback) {
            details.InsertValue("cashback", *Details->Cashback);
        }
        if (Details->Image) {
            details.InsertValue("image", Details->Image);
        }
        result.InsertValue("details", details);

        if (Details->Buttons) {
            NJson::TJsonValue buttons;
            for (const auto& button : Details->Buttons) {
                NJson::TJsonValue buttonJson;
                buttonJson.InsertValue("title", button.Title);
                buttonJson.InsertValue("link", button.DeepLink);
                buttons.AppendValue(buttonJson);
            }
            result.InsertValue("buttons", buttons);
        }
    }
    return result;
}
