#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/abstract/localization.h>
#include <drive/backend/roles/user.h>

namespace NDrive {

    // TDrivingProfile represents driving profile for specified user.
    class TDrivingProfile {
    public:
        struct TStageConfig {
            // Id overrides stage id.
            TString Id;
            TString PrevColor;
            TString NextColor;
            TVector<TString> DescriptionIds;
            double FromValue;
            double BeginValue;
            double EndValue;
            // Human readable values.
            TMaybe<i32> BeginPoints;
            TMaybe<i32> EndPoints;
            TMaybe<i32> Parts;

            // CalcPoints calculates amount of points in current range.
            //
            // Result is in range [BeginPoints, EndPoints].
            i32 CalcPoints(double value, bool inverted) const;
        };

        struct TStagesConfig {
            TVector<TStageConfig> Stages;
        };

        struct TStage {
            TString Id;
            TStageConfig Config;
        };

        struct TStages {
            TVector<TStage> Stages;
            TString CurrentStage;
            TMaybe<i32> CurrentPoints;
            // Additional information about block.
            TMaybe<TDuration> BlockDuration;
            TMaybe<TDuration> BlockTotalDuration;
            TMaybe<TInstant> BlockEndTime;
        };

        TString UserStatus;
        TDBTag AggressiveTag;
        TDBTag AggressiveBlockTag;
        TDBTag AggressiveTrialTag;
        TDBTag AggressivePriceUpTag;
        TVector<TDBTag> NotificationTags;
        TMaybe<TUserRoleInfo> HighCashbackRole;

        TDrivingProfile() = default;

        ~TDrivingProfile() = default;

        // GetVisibleStages returns currently visible stages for specified user by its permissions.
        TMaybe<TStages> GetVisibleStages(const NDrive::IServer& server, const TUserPermissions& permissions) const;

        // GetPublicReport returns public report for driving profile.
        NJson::TJsonValue GetPublicReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;

    private:
        NJson::TJsonValue GetNotificationsReportV1(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;

        NJson::TJsonValue GetNotificationsReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, bool isActiveStage = false, const TString& stageName = "") const;

        NJson::TJsonValue GetNotificationReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, const TString& tagName, bool isActiveStage, bool isActiveTag, const TString& stageName = "", TMaybe<TInstant> deadline = {}) const;

        NJson::TJsonValue GetSessionsReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;

        NJson::TJsonValue GetStagesReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale, const TStages& stages) const;

        NJson::TJsonValue GetStageReport(
            const NDrive::IServer& server,
            const TUserPermissions& permissions,
            ELocalization locale,
            const TStageConfig& stageConfig,
            const TString& stageName,
            const TStages& stages
        ) const;

        NJson::TJsonValue GetLastChangeReport(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;

        NJson::TJsonValue GetButtonReport(
            const NDrive::IServer& server,
            const TUserPermissions& permissions,
            ELocalization locale,
            const TStages& stages
        ) const;

        NJson::TJsonValue GetPublicReportV2(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;

        // deprecated.
        NJson::TJsonValue GetPublicReportV1(const NDrive::IServer& server, const TUserPermissions& permissions, ELocalization locale) const;
    };

}

struct TAggressionConfig {
    struct TStage {
        struct TButton {
            TString Title;
            TString Deeplink;

            NJson::TJsonValue GetReport() const;
        };

        struct TPhoto {
            TString Url;
            TString Text;
        };

        struct TDescription {
            TString Title;
            TString Description;

            NJson::TJsonValue GetReport() const;
        };

        double FromValue;
        double BeginValue;
        double EndValue;
        TString Title;
        TString SubTitle;
        TString NextTitle;
        TString NextColor;
        TString PrevTitle;
        TString PrevColor;
        TVector<TDescription> Descriptions;
        TButton Button;
        TPhoto Photo;
        TString MenuTitle;
        TString PositiveMenuTitle;
        TString NegativeMenuTitle;
    };

    TVector<TStage> Stages;
    TMap<TString, TStage> BlockStages;
    TStage PriceUpStage;
    TStage TrialStage;
    TStage EmptyStage;
    TString SessionsTitle;
    TString SessionsFilter;
};

struct TAggressionNotification {
    struct TButton {
        TString Title;
        TString DeepLink;
    };

    using TButtons = TVector<TButton>;

    struct TDetails {
        TString Title;
        TString SubTitle;
        TString Description;
        // For special types.
        TMaybe<TString> Cashback;
        TString Image;
        TButtons Buttons;
    };

    TString Title;
    TString Description;
    TString Icon;
    TString BgColor;
    TString BgColorEnd;
    TMaybe<TDetails> Details;

    NJson::TJsonValue GetReport(ELocalization locale, const ILocalization* localization, TMaybe<TInstant> deadline = {}) const;
};
