#pragma once
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/guid.h>

enum class ELoginType {
    Telegram = 0 /* "telegram" */,
};

class TUserLogin {
    R_FIELD(ELoginType, Type, ELoginType::Telegram);
    R_FIELD(TString, AltLogin);
    R_FIELD(TString, DeviceId);
    R_FIELD(TString, UserId);

public:
    TUserLogin() {
    }

    NJson::TJsonValue BuildJsonReport() const {
        return SerializeToTableRecord().SerializeToJson();
    }

    TString GetIdString() const {
        return ToString(Type) + "-" + AltLogin + (!!DeviceId ? ("-" + DeviceId) : "");
    }

    TUserLogin(const ELoginType type, const TString& altLogin)
        : Type(type)
        , AltLogin(altLogin)
    {
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord record;
        record.Set("login_type", ToString(Type)).Set("login_id", AltLogin).Set("user_id", UserId);
        record.Set("device_id", DeviceId);
        return record;
    }

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record) {
        if (!record.TryGet("login_type", Type) || !record.TryGet("login_id", AltLogin) || !record.TryGet("user_id", UserId)) {
            return false;
        }

        record.TryGet("device_id", DeviceId);
        return true;
    }
};

class TLoginsManager: public TDatabaseSessionConstructor {
private:
    mutable TCacheWithAge<TString, TString> Cache;

private:
    using ECachedValue = typename TCacheWithAge<TString, TString>::ECachedValue;

public:
    TLoginsManager(NStorage::IDatabase::TPtr database)
        : TDatabaseSessionConstructor(database)
    {
    }

    TString GetUserIdSafe(const ELoginType type, const TString& login, const TString& deviceId, const TDuration maxAge) const {
        return GetUserIdSafe(type, login, deviceId, Now() - maxAge);
    }

    TString GetUserIdSafe(const ELoginType type, const TString& login, const TString& deviceId, const TInstant reqActuality) const {
        TString result;
        CHECK_WITH_LOG(GetUserIdByLogin(type, login, deviceId, reqActuality, result));
        return result;
    }
    bool GetUserIdByLogin(const ELoginType type, const TString& login, const TString& deviceId, const TInstant reqActuality, TString& result) const;
    bool GetLogins(const TString& userId, TVector<TUserLogin>& result, NDrive::TEntitySession& session) const;
    bool AddLogin(const TUserLogin& login, NDrive::TEntitySession& session) const;
    bool RemoveLogin(const TUserLogin& login, NDrive::TEntitySession& session) const;
};
