#include "user.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/chat_robots/configuration/chat_script.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/database/entity/search_index.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/library/cpp/blackbox/client.h>
#include <drive/library/cpp/raw_text/datetime.h>
#include <drive/library/cpp/social_api/client.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/user_events_api/client.h>
#include <drive/library/cpp/user_events_api/realtime_user_data.h>

#include <library/cpp/blackbox2/src/responseimpl.h>
#include <library/cpp/blackbox2/src/xconfig.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/field.h>
#include <rtline/util/types/string_pool.h>

#include <util/string/join.h>
#include <util/system/tls.h>

static Y_THREAD(TStringPool) UserNamesPool;
static Y_THREAD(TStringPool) UserPNamesPool;
static Y_THREAD(TStringPool) UserStatusPool;
static Y_THREAD(TStringPool) UserRegionsPool;

bool NDrive::IsPreOnboarding(const TString& status) {
    return status == UserStatusReferrer;
}

namespace {
    template <class T>
    void SetPrivateDataResponse(NJson::TJsonValue& result, const TString& resultFieldName,
                                const NThreading::TFuture<T>& future, const TString& revision,
                                const NUserReport::TReportTraits& traits) {
        if (future.Initialized() && future.HasValue()) {
            auto passport = future.GetValue();
            NJson::TJsonValue serializedData(passport.SerializeToJson(traits));
            serializedData.InsertValue("revision", revision);
            result[resultFieldName].AppendValue(serializedData);
        } else if (future.HasException()) {
            ERROR_LOG << NThreading::GetExceptionMessage(future) << Endl;
        }
    }

    TString GetHRSocialNetworkName(const TString& internalCode) {
        if (internalCode == "vk") {
            return "VK";
        } else if (internalCode == "fb") {
            return "Facebook";
        } else if (internalCode == "tw") {
            return "Twitter";
        } else if (internalCode == "mr") {
            return "mail.ru";
        } else if (internalCode == "gg") {
            return "Google";
        } else if (internalCode == "ok") {
            return "Odnoklassniki";
        }
        return internalCode;
    }

    NJson::TJsonValue BuildBlackboxDataReport(const NDrive::TBlackboxClient::TResponsePtr& userBlackboxData) {
        // consider to rework using drive/library/cpp/blackbox/client.h (ctor and Parse method)

        if (!userBlackboxData) {
            return NJson::JSON_NULL;
        }

        NBlackbox2::TUid userId(userBlackboxData.Get());
        NBlackbox2::TAttributes attributes(userBlackboxData.Get());
        NBlackbox2::TLoginInfo loginInfo(userBlackboxData.Get());
        NBlackbox2::TPDDUserInfo pddUserInfo(userBlackboxData.Get());
        NBlackbox2::TAliasList aliases(userBlackboxData.Get());
        NBlackbox2::TDisplayNameInfo displayName(userBlackboxData.Get());

        NJson::TJsonValue result;

        result["plus"] = attributes.Get("1015") == "1";
        result["public_name"] = attributes.Get("1007");
        result["login"] = attributes.Get("1008");
        result["social"] = displayName.Social();
        result["hosted"]["is_hosted"] = userId.Hosted();
        result["hosted"]["domain"] = pddUserInfo.Domain();

        auto& phonesJson = result.InsertValue("bounded_phones", NJson::JSON_ARRAY);
        if (auto responseImpl = userBlackboxData->GetImpl()) {
            NBlackbox2::xmlConfig::Parts phones(responseImpl->GetParts("phones/phone"));
            for (int i = 0; i < phones.Size(); ++i) {
                NJson::TJsonValue phoneJson(NJson::JSON_MAP);
                NBlackbox2::xmlConfig::Parts phoneAttributes(phones[i].GetParts("attribute"));
                for (int j = 0; j < phoneAttributes.Size(); ++j) {
                    long int type = 0;
                    if (phoneAttributes[j].GetIfExists("@type", type)) {
                        switch (type) {
                        case 102:
                            phoneJson["phone"] = phoneAttributes[j].asString();
                            break;
                        default:
                            break;
                        }
                    }
                }
                phonesJson.AppendValue(phoneJson);
            }
        }

        if (displayName.Social()) {
            NJson::TJsonValue socialProviderData;
            socialProviderData["name"] = displayName.Name();
            socialProviderData["hr_soc_provider"] = GetHRSocialNetworkName(displayName.SocProvider());
            socialProviderData["soc_profile"] = displayName.SocProfile();
            socialProviderData["soc_provider"] = displayName.SocProvider();
            socialProviderData["soc_target"] = displayName.SocTarget();
            socialProviderData["default_avatar"] = displayName.DefaultAvatar();
            result["social_auth"] = std::move(socialProviderData);
        }

        bool isYandexoid = false;
        bool isPhonish = false;
        for (auto&& alias : aliases.GetAliases()) {
            if (alias.type() == NBlackbox2::TAliasList::TItem::EType::Yandexoid) {
                isYandexoid = true;
            }
            if (alias.type() == NBlackbox2::TAliasList::TItem::EType::Phonish || alias.type() == NBlackbox2::TAliasList::TItem::EType::NeoPhonish) {
                isPhonish = true;
            }
        }
        result["yandexoid"] = isYandexoid;
        result["phonish"] = isPhonish;

        return result;
    }

    TUnistatSignal<double> CacheHit { { "user-cache-hit" }, false };
    TUnistatSignal<double> CacheMiss { { "user-cache-miss" }, false };
    TUnistatSignal<double> CacheExpired { { "user-cache-expired" }, false };
    TUnistatSignal<double> CacheInvalidated { { "user-cache-invalidated" }, false };
}  // namespace

TUsersDB::TUsersDB(const ITagsHistoryContext& historyContext, const TUserTagsManager* userTagsManager, const TUsersDBConfig& config)
    : IAutoActualization("user-cache", TDuration::Seconds(1))
    , TBase(historyContext.GetDatabase())
    , Database(historyContext.GetDatabase())
    , HistoryManager(historyContext)
    , UserTagsManager(userTagsManager)
    , UserRolesDB(historyContext, TDuration::Seconds(1000))
    , AD(new TAsyncDelivery())
    , Config(config)
    , Index(config.GetIsSearchEnabled() && config.GetSearchViaDatabaseVector().empty()
        ? MakeHolder<TBasicSearchIndex>() : nullptr)
    , IndexUpdateThread(IThreadPool::TParams()
        .SetThreadName("user_index_update")
    )
    , ObjectCache(128 * 1024)
    , UidCache(256 * 1024)
{
}

TUsersDB::~TUsersDB() {
}

bool TUsersDB::DoStart() {
    if (Index) {
        IndexUpdateThread.Start(1);
        IndexUpdateThread.SafeAddFunc([this] {
            BuildIndex();
        });
    }
    if (!UserRolesDB.Start()) {
        return false;
    }
    AD->Start(Config.GetRequestConfig().GetThreadsStatusChecker(), Config.GetRequestConfig().GetThreadsSenders());
    return IAutoActualization::DoStart();
}

bool TUsersDB::DoStop() {
    if (!IAutoActualization::DoStop()) {
        return false;
    }
    AD->Stop();
    if (!UserRolesDB.Stop()) {
        return false;
    }
    if (Index) {
        IndexUpdateThread.Stop();
    }
    return true;
}

bool TUsersDB::BuildIndex() {
    if (!Index) {
        return false;
    }

    INFO_LOG << GetName() << ": building SearchIndex" << Endl;
    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalUsers = Fetch(session, {});
    if (!optionalUsers) {
        ERROR_LOG << GetName() << ": cannot BuildIndex: " << session.GetStringReport() << Endl;
        return false;
    }
    if (!session.Rollback()) {
        WARNING_LOG << GetName() << ": cannot Rollback: " << session.GetStringReport() << Endl;
    }
    for (auto&& user : *optionalUsers) {
        user.Index(*Index);
    }
    INFO_LOG << GetName() << ": SearchIndex built" << Endl;
    return true;
}

bool TUsersDB::Refresh() {
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!LastEventId) {
        LastEventId = HistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = HistoryManager.GetEvents(since, session, 1000);
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEventsSince " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    if (!session.Rollback()) {
        WARNING_LOG << GetName() << ": cannot Rollback: " << session.GetStringReport() << Endl;
    }
    for (auto&& ev : *optionalEvents) {
        LastEventId = std::max(LastEventId.GetOrElse(0), ev.GetHistoryEventId());
        bool erased = ObjectCache.erase(ev.GetUserId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << ev.GetUserId() << Endl;
        }
        if (Index) {
            if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
                IndexUpdateThread.SafeAddFunc([id = ev.GetUserId(), this] {
                    Index->Remove(id);
                });
            } else {
                IndexUpdateThread.SafeAddFunc([ev, this] {
                    ev.Index(*Index);
                });
            }
        }
    }
    return true;
}

bool TUsersDB::ActualizeRegistrationGeo(const TString& userId, const TString& operatorUserId, const TGeoCoord& userLocation, NDrive::TEntitySession& session) const {
    auto userFR = FetchInfo(userId, session);
    if (userFR.empty()) {
        ERROR_LOG << "Problem while updating registration geo for user " << userId << ": no such user" << Endl;
        return false;
    }
    TDriveUserData userData = std::move(userFR.MutableResult().begin()->second);
    TString newGeo = "moscow";
    double minDistance = 1e50;
    for (auto&& geo : Config.GetRegistrationGeoCenters()) {
        double currentDistance = geo.Center.GetLengthTo(userLocation);
        if (currentDistance < minDistance) {
            minDistance = currentDistance;
            newGeo = geo.LocationName;
        }
    }
    if (newGeo != userData.GetRegistrationGeo()) {
        userData.SetRegistrationGeo(newGeo);
        if (!UpdateUser(userData, operatorUserId, session)) {
            ERROR_LOG << "Problem while updating registration geo for user " << userId << ": unable to upsert" << Endl;
            return false;
        }
        return true;
    }
    return false;
}

bool TUsersDB::SelectUsers(const TString& fieldName, const TVector<TString>& keys, TVector<TString>& result, NDrive::TEntitySession& session, bool doGetDeleted) const {
    auto usersFR = FetchInfoByField(keys, fieldName, session);
    if (!usersFR) {
        return false;
    }

    for (auto&& it : usersFR) {
        if (!doGetDeleted && it.second.GetStatus() == NDrive::UserStatusDeleted) {
            continue;
        }
        result.push_back(it.second.GetUserId());
    }

    return true;
}

bool TUsersDB::CheckEmailExists(const TString& email) const {
    auto session = BuildTx<NSQL::ReadOnly>();
    TVector<TString> result;
    SelectUsers("email", {email}, result, session, false);
    return result.size() > 0;
}

TMaybe<TDriveUsers> TUsersDB::SelectUsers(const TString& fieldName, TConstArrayRef<TString> keys, NDrive::TEntitySession& session, bool withDeleted) const {
    auto timestamp = Now();
    auto fetchResult = FetchInfoByField(keys, fieldName, session);
    if (!fetchResult) {
        session.AddErrorMessage("UsersDB::SelectUsers", "cannot select users by field " + fieldName);
        return {};
    }

    TDriveUsers result;
    for (auto&& [id, user] : fetchResult) {
        if (!withDeleted && user.GetStatus() == NDrive::UserStatusDeleted) {
            continue;
        }
        result.push_back(std::move(user));
        result.back().SetTimestamp(timestamp);
    }
    return result;
}

TMaybe<TDriveUserData> TUsersDB::UpdateUser(TDriveUserData userData, const TString& operatorUserId, NDrive::TEntitySession& session, bool isNewUser) const {
    bool isNewAccount = userData.GetUserId() == "uuid_generate_v4()";

    userData.SetNewAccount(isNewAccount | isNewUser);
    userData.SetTimestamp(Now());
    NStorage::TObjectRecordsSet<TDriveUserData> upsertedUsers;

    if (!userData.GetNewAccount()) {
        if (!Upsert(userData, session, &upsertedUsers)) {
            session.AddErrorMessage("user", "unable to upsert user in db table");
            return {};
        }
    } else {
        if (!Insert(userData, session, &upsertedUsers)) {
            session.AddErrorMessage("user", "unable to insert user in db table");
            return {};
        }
    }

    if (upsertedUsers.size() != 1) {
        session.SetErrorInfo("user", "upsertedUsers.size() != 1", EDriveSessionResult::InternalError);
        return {};
    }

    if (!HistoryManager.AddHistory(*upsertedUsers.begin(), operatorUserId, isNewAccount ? EObjectHistoryAction::Add : EObjectHistoryAction::UpdateData, session)) {
        session.AddErrorMessage("user", "unable to write history");
        return {};
    }

    auto upsertedUser = std::move(upsertedUsers.front());
    session.Committed().Subscribe([upsertedUser](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log("UpdateUser", upsertedUser.GetReport(NUserReport::ReportAll));
        }
    });
    session.Committed().Subscribe([id = upsertedUser.GetUserId(), this](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            ObjectCache.erase(id);
        }
    });
    return upsertedUser;
}

TMaybe<TDriveUserData> TUsersDB::DeleteUser(const TString& userId, const TString& operatorUserId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(*Config.GetDeletedUserTags().begin(), "user deletion");
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, operatorUserId, userId, server, session)) {
        return {};
    }

    auto timestamp = Now();
    auto userFetchResult = FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    if (!userPtr) {
        session.AddErrorMessage("user", "not found");
        return {};
    }

    auto original = *userPtr;
    auto userData = *userPtr;
    userData.SetStatus(NDrive::UserStatusDeleted);
    userData.SetUid("");
    userData.SetTimestamp(timestamp);

    if (!Upsert(userData, session)) {
        return {};
    }

    if (!HistoryManager.AddHistory(userData, operatorUserId, EObjectHistoryAction::Remove, session)) {
        session.AddErrorMessage("user", "unable to write history");
        return {};
    }

    session.Committed().Subscribe([original](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log("DeleteUser", original.GetReport(NUserReport::ReportAll));
        }
    });

    return userData;
}

TUsersDB::TOptionalUserId TUsersDB::GetUserIdByUidDirect(const TString& uid, NDrive::TEntitySession& session) const {
    auto optionalUsers = SelectUsers("uid", { uid }, session);
    if (!optionalUsers) {
        return {};
    }
    for (auto&& user : *optionalUsers) {
        Y_ASSERT(uid == user.GetUid());
        const auto& userId = user.GetUserId();
        ObjectCache.update(userId, user);
        UidCache.update(uid, userId);
        return userId;
    }
    return TString{};
}

TString TUsersDB::GetUserIdByUid(const TString& uid) const {
    TString result;
    if (!result) {
        auto optionalId = UidCache.find(uid);
        if (optionalId) {
            auto expectedUser = GetCachedObject(*optionalId);
            if (expectedUser && expectedUser->GetStatus() != NDrive::UserStatusDeleted && expectedUser->GetUid() == uid) {
                result = std::move(*optionalId);
            }
        }
    }
    if (!result) {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto optionalResult = GetUserIdByUidDirect(uid, session);
        R_ENSURE(optionalResult, {}, "cannot GetUserIdByUidDirect", session);
        result = std::move(*optionalResult);
    }
    return result;
}

TVector<TString> TUsersDB::GetDocumentOwnersByHash(const TString& hash) const {
    if (!hash) {
        return {};
    }
    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalResult = GetDocumentOwnersByHash(hash, session);
    if (!optionalResult) {
        session.Check();
    }
    return std::move(*optionalResult);
}

namespace {
    const TString EmptyFieldHash = "05f0968218785ac4158eedde85b559bb7ffec349";
    const TString EmptyPassportNamesHash = "cba97f46a1ad17e0fa6af7f7e075685cc627b6a7"; // hash from "@@@"
}

TUsersDB::TOptionalUsers TUsersDB::GetSamePersons(const TString& userId, NDrive::TEntitySession& session) const {
    auto mainUser = RestoreUser(userId, session);
    if (!mainUser) {
        return {};
    }

    auto queryOptions = NSQL::TQueryOptions().SetGenericCondition("id", NSQL::Not(TSet<TString>{userId}));
    if (mainUser->GetPassportNamesHash() && mainUser->GetPassportNamesHash() != EmptyFieldHash && mainUser->GetPassportNamesHash() != EmptyPassportNamesHash) {
        queryOptions.AddGenericCondition("passport_names_hash", mainUser->GetPassportNamesHash());
    }
    if (mainUser->GetPassportNumberHash() && mainUser->GetPassportNumberHash() != EmptyFieldHash) {
        queryOptions.AddGenericCondition("passport_number_hash", mainUser->GetPassportNumberHash());
    }
    if (mainUser->GetDrivingLicenseNumberHash() && mainUser->GetDrivingLicenseNumberHash() != EmptyFieldHash) {
        queryOptions.AddGenericCondition("driving_license_number_hash", mainUser->GetDrivingLicenseNumberHash());
    }
    if (queryOptions.GetGenericConditions().size() < 2) {
        return TDriveUsers();
    }

    queryOptions.SetGenericCondition("status", NSQL::Not(TSet<TString>{NDrive::UserStatusDeleted}));

    auto optionalUsers = Fetch(session, queryOptions);
    if (!optionalUsers) {
        return {};
    }
    TDriveUsers result;
    for (auto&& user : *optionalUsers) {
        if (user.GetFullName() == mainUser->GetFullName()) {
            result.emplace_back(std::move(user));
        }
    }
    return result;
}

TMaybe<bool> TUsersDB::CheckDuplicates(const TVector<TString>& userIds, NDrive::TEntitySession& tx) const {
    if (userIds.empty()) {
        return false;
    }
    TSet<TString> dups;
    for (const auto& id : userIds) {
        auto dupUsers = GetSamePersons(id, tx);
        if (!dupUsers) {
            return Nothing();
        }
        for (auto&& user : *dupUsers) {
            if (user.GetUserId() != id) {
                dups.emplace(std::move(user.GetUserId()));
            }
        }
    }
    for (const auto& id : userIds) {
        if (!dups.contains(id)) {
            return false;
        }
    }
    return true;
}

bool TUsersDB::CompareUsersByFreshness(const TDriveUserData& left, const TDriveUserData& right) {
    if (left.GetStatus() != right.GetStatus()) {
        if (left.GetStatus() == "active") {
            return true;
        }
        if (right.GetStatus() == "active") {
            return false;
        }
    }
    const TInstant authDateFirst = Max(left.GetApprovedAt(), left.GetJoinedAt());
    const TInstant authDateSecond = Max(right.GetApprovedAt(), right.GetJoinedAt());
    if (authDateFirst == authDateSecond) {
        return true;
    }
    return authDateFirst > authDateSecond;
}

TMaybe<TDriveUsers> TUsersDB::GetDocumentOwnersObjectsByHash(const TString& hash, NDrive::TEntitySession& session) const {
    if (!hash) {
        return TDriveUsers();
    }
    auto queryOptions = NSQL::TQueryOptions().SetCustomCondition(TStringBuilder() << "False"
        << " OR passport_names_hash = " << session->Quote(hash)
        << " OR passport_number_hash = " << session->Quote(hash)
        << " OR driving_license_number_hash = " << session->Quote(hash)
    );
    return Fetch(session, queryOptions);
}

TMaybe<TVector<TString>> TUsersDB::GetDocumentOwnersByHash(const TString& hash, NDrive::TEntitySession& session) const {
    auto optionalUsers = GetDocumentOwnersObjectsByHash(hash, session);
    if (!optionalUsers) {
        return {};
    }
    TVector<TString> result;
    for (auto&& user : *optionalUsers) {
        result.push_back(user.GetUserId());
    }
    return result;
}

bool TUsersDB::SetChatToShow(const TString& userId, const TString& topicLink, const bool isClosable, const TString& operatorId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!UserTagsManager) {
        return false;
    }
    auto tag = new TUserChatShowTag(TUserChatShowTag::TypeName);
    tag->SetTopicLink(topicLink);
    tag->SetIsClosable(isClosable);
    auto optionalAddedTag = UserTagsManager->AddTag(tag, operatorId, userId, server, session);
    return optionalAddedTag.Defined();
}

bool TUsersDB::DropChatShow(const TString& userId, const TString& operatorId, NDrive::TEntitySession& session) const {
    if (!UserTagsManager) {
        return false;
    }

    TVector<TDBTag> tags;
    if (!UserTagsManager->RestoreTags({ userId }, { TUserChatShowTag::TypeName }, tags, session)) {
        return false;
    }

    return UserTagsManager->RemoveTagsSimple(tags, operatorId, session, false);
}

TUsersDB::TOptionalUser TUsersDB::GetCachedObject(const TString& userId, TInstant statementDeadline) const {
    auto eventLogger = NDrive::GetThreadEventLogger();
    auto lifetimeKey = "user.cache_lifetime";
    auto lifetime = NDrive::HasServer()
        ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
        : Nothing();
    auto now = Now();
    auto threshold = now - lifetime.GetOrElse(Config.GetDefaultCacheLifetime());
    auto optionalObject = ObjectCache.find(userId);
    if (optionalObject && optionalObject->GetTimestamp() > threshold) {
        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "UserCacheHit")
                ("user_id", userId)
                ("timestamp", NJson::ToJson(optionalObject->GetTimestamp()))
            );
        }
        CacheHit.Signal(1);
        return std::move(*optionalObject);
    }
    if (optionalObject) {
        CacheExpired.Signal(1);
    }

    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "UserCacheMiss")
            ("user_id", userId)
        );
    }

    auto lockTimeout = TDuration::Zero();
    auto statementTimeout = statementDeadline - now;
    auto session = BuildSession(true, false, lockTimeout, statementTimeout);
    auto fetchResult = FetchInfo(userId, session);
    if (!fetchResult) {
        session.Check();
    }
    auto restoredObject = fetchResult.MutableResult().FindPtr(userId);
    if (!restoredObject) {
        return {};
    }

    restoredObject->SetTimestamp(now);
    Y_ASSERT(userId == restoredObject->GetUserId());
    ObjectCache.update(userId, *restoredObject);
    const auto& uid = restoredObject->GetUid();
    if (uid) {
        UidCache.update(uid, userId);
    }
    CacheMiss.Signal(1);
    return std::move(*restoredObject);
}

TMaybe<TDriveUserData> TUsersDB::RestoreUser(const TString& userId, NDrive::TEntitySession& session) const {
    auto timestamp = Now();
    auto result = Fetch(userId, session);
    if (!result) {
        return {};
    }
    result->SetTimestamp(timestamp);
    return result;
}

TUsersDB::TOptionalUser TUsersDB::TryLinkToExistingUser(const TString& phoneNumber, const TString& email, NDrive::TEntitySession& tx) const {
    if (phoneNumber) {
        auto optionalUsers = SelectUsers("phone", { phoneNumber }, tx);
        if (!optionalUsers) {
            return {};
        }
        for (auto&& i : *optionalUsers) {
            if (i.GetUid()) {
                continue;
            }
            if (i.GetStatus() == NDrive::UserStatusExternal) {
                return std::move(i);
            }
        }
    }
    if (email) {
        auto optionalUsers = SelectUsers("email", { email }, tx);
        if (!optionalUsers) {
            return {};
        }
        for (auto&& i : *optionalUsers) {
            if (i.GetUid()) {
                continue;
            }
            if (i.GetStatus() == NDrive::UserStatusExternal) {
                return std::move(i);
            }
        }
    }
    return TDriveUserData{};
}

TMaybe<TUsersDB::TUserDeleteStatus> TUsersDB::CheckStatusBeforeDelete(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx, TVector<TString> deletionNames, TMaybe<TVector<TDBTag>> fetchedTags) {
    /* Check user existence */
    auto queryOptions = NSQL::TQueryOptions().SetGenericCondition("status", NSQL::Not(TSet<TString>{NDrive::UserStatusDeleted}));
    queryOptions.SetGenericCondition("id", TSet<TString>{userId});

    auto optionalUsers = Fetch(tx, queryOptions);
    if (!optionalUsers) {
        /* Transaction error */
        return {};
    } else if (optionalUsers->empty()) {
        /* There is no such user */
        return {TUsersDB::TUserDeleteStatus::Empty};
    }

    {
        /* Check if user never had 'user_deleted_finally' tag */
        auto historyQueryOptions = TTagEventsManager::TQueryOptions();
        historyQueryOptions.SetOrderBy({"history_event_id"}).SetDescending(true);
        historyQueryOptions.SetLimit(1);
        historyQueryOptions.SetGenericCondition("tag", TSet<TString>{"user_deleted_finally"});

        auto tagHistory = server.GetDriveAPI()->GetTagsManager().GetUserTags().GetHistoryManager().GetEventsByObject(userId, tx, 0, TInstant::Zero(), historyQueryOptions);
        if (!tagHistory) {
            /* Transaction error */
            return {};
        } else if (!tagHistory->empty()) {
            /* User has 'user_deleted_finally' tag */
            return {TUsersDB::TUserDeleteStatus::Empty};
        }
    }

    /* Check if user erasure in progress */
    if (fetchedTags.Empty()) {
        TVector<TDBTag> tags;
        const auto& userTagManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
        bool success = userTagManager.RestoreTags({userId}, deletionNames, tags, tx);

        if (!success) {
            return {};
        } else if (!tags.empty()) {
            return {TUsersDB::TUserDeleteStatus::DeleteInProgress};
        }
    } else {
        for (const auto& tag: *fetchedTags) {
            for (const TString& tagName: deletionNames) {
                if (tagName == tag->GetName()) {
                    return {TUsersDB::TUserDeleteStatus::DeleteInProgress};
                }
            }
        }
    }

    /* Check if user has active rides */
    {
        TVector<TAtomicSharedPtr<const ISession>> userSessions;
        bool success = server.GetDriveAPI()->GetCurrentUserSessions(userId, userSessions, TInstant::Zero());
        if (!success) {
            return {};
        } else if (!userSessions.empty()) {
            return {TUsersDB::TUserDeleteStatus::HasActiveSession};
        }
    }

    /* Check if user has debt */
    {
        bool inProc = false;
        TMaybe<ui32> debt = server.GetDriveAPI()->GetBillingManager().GetDebt(userId, tx, nullptr, {}, &inProc);
        if (debt.Empty()) {
            return {};
        } else if (*debt != 0) {
            return {TUsersDB::TUserDeleteStatus::HasDebt};
        }
    }

    /* Obtain sessions info */
    bool hasRides = !optionalUsers->front().IsFirstRiding();

    /* Check if user has duplicates */
    bool hasDuplicates = false;
    {
        auto allPersons = server.GetDriveAPI()->GetUsersData()->GetSamePersons(userId, tx);
        if (allPersons.Empty()) {
            return {};
        } else if (!allPersons->empty()) {
            for (auto&& person: *allPersons) {
                if (NDrive::UserStatusOnboarding != person.GetPublicStatus()) {
                    hasDuplicates = true;
                    break;
                }
            }
        }
    }

    if (hasRides || hasDuplicates) {
        return {TUsersDB::TUserDeleteStatus::NeedsHumanVerification};
    }

    return {TUsersDB::TUserDeleteStatus::ReadyToDelete};
}

TMaybe<TDriveUserData> TUsersDB::RegisterNewUser(const TString& operatorUserId, const TString& uid, const TString& userName, NDrive::TEntitySession& session, const TString& phoneNumber, const TString& email, bool phoneVerified, const TString& initialStatus, const TMaybe<TString>& onLinkStatus) const {
    auto optionalExistingUser = TryLinkToExistingUser(phoneNumber, email, session);
    if (!optionalExistingUser) {
        return {};
    }
    auto user = std::move(*optionalExistingUser);
    bool isLinked = false;
    if (!user) {
        user = TDriveUserData{
            uid,
            userName
        };
    } else {
        isLinked = true;
        user.SetUid(uid);
        user.SetLogin(userName);
    }
    user.SetJoinedAt(Now());

    if (isLinked) {
        user.SetStatus(onLinkStatus.GetOrElse(initialStatus));
    } else {
        user.SetStatus(initialStatus);
    }

    if (!user.GetEmail()) {
        user.SetEmail(email);
    }
    if (!user.GetPhone()) {
        user.SetPhone(phoneNumber);
    }
    if (user.GetPhone() == phoneNumber) {
        user.SetPhoneVerified(phoneVerified);
    }
    Y_ASSERT(user.GetUserId());
    auto upsertedUser = UpdateUser(user, operatorUserId, session);
    if (!upsertedUser) {
        return {};
    }
    Y_ASSERT(upsertedUser->GetUserId());
    session.Committed().Subscribe([upsertedUser](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log("RegisterNewUser", upsertedUser->GetReport(NUserReport::ReportAll));
        }
    });
    return upsertedUser;
}

TMaybe<TDriveUserData> TUsersDB::RegisterExternalUser(const TString& operatorUserId, NDrive::TEntitySession& tx, const NDrive::TExternalUser& externalUser) const {
    auto login = TString{};
    TDriveUserData newUser(externalUser.GetUid(), login);
    newUser
        .SetStatus(NDrive::UserStatusExternal)
        .SetFirstName(externalUser.GetFirstName())
        .SetLastName(externalUser.GetLastName())
        .SetPName(externalUser.GetPName())
        .SetPhone(externalUser.GetPhone())
        .SetEmail(externalUser.GetEmail())
    ;
    auto upsertedUser = UpdateUser(newUser, operatorUserId, tx);
    if (!upsertedUser) {
        return {};
    }

    tx.Committed().Subscribe([upsertedUser](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log("RegisterExternalUser", upsertedUser->GetReport(NUserReport::ReportAll));
        }
    });
    return upsertedUser;
}

TMaybe<TDriveUserData> TUsersDB::FindOrRegisterExternal(const TString& operatorUserId, NDrive::TEntitySession& tx, const NDrive::TExternalUser& externalUser) const {
    TFetchResult fetchResult;
    const auto& email = externalUser.GetEmail();
    const auto& phoneNumber = externalUser.GetPhone();
    const auto& uid = externalUser.GetUid();
    if (phoneNumber || email || uid) {
        NSQL::TQueryOptions options;
        if (phoneNumber) {
            options.AddGenericCondition("phone", phoneNumber);
        }
        if (email) {
            options.AddGenericCondition("email", email);
        }
        if (uid) {
            options.AddGenericCondition("uid", uid);
        }
        auto timestamp = Now();
        fetchResult = FetchInfo(tx, options);
        if (!fetchResult) {
            tx.AddErrorMessage("UsersDB::SelectUsers", "cannot select users with phone '" + phoneNumber + "' and email '" + email + "'");
            return {};
        }
        for (auto&& [id, user] : fetchResult.MutableResult()) {
            user.SetTimestamp(timestamp);
        }
    }
    if (!fetchResult.empty()) {
        for (auto&& [id, user] : fetchResult) {
            if (user.GetPublicStatus() == NDrive::UserStatusActive) {
                return user;
            }
        }
        return fetchResult.begin()->second;
    }
    return RegisterExternalUser(operatorUserId, tx, externalUser);
}

TMaybe<TVector<TDriveUserData>> TUsersDB::FindUsersByPhone(const TString& phoneNumber, const TSet<TString>& allowedStatuses, TMaybe<bool> phoneVerified, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.AddGenericCondition("phone", phoneNumber);
    if (!allowedStatuses.empty()) {
        options.SetGenericCondition("status", allowedStatuses);
    }
    if (phoneVerified) {
        options.SetGenericCondition("is_phone_verified", *phoneVerified);
    }
    TInstant timestamp = Now();
    TFetchResult fetchResult = FetchInfo(session, options);
    if (!fetchResult) {
        return {};
    }
    TVector<TDriveUserData> result;
    for (auto&& [id, user] : fetchResult.GetResult()) {
        result.emplace_back(user);
        result.back().SetTimestamp(timestamp);
    }
    return result;
}

TSet<TString> TUsersDB::GetUsersDeletedByTags() const {
    TSet<TString> deletedUserIds;
    TVector<TDBTag> tags;

    auto session = BuildTx<NSQL::ReadOnly>();
    if (!UserTagsManager || !UserTagsManager->RestoreTags(TSet<TString>(), Config.GetDeletedUserTags(), tags, session)) {
        ERROR_LOG << "Cannot restore users with deleted users tags" << Endl;
        return {};
    }

    for (auto&& tag : tags) {
        deletedUserIds.insert(tag.GetObjectId());
    }

    return deletedUserIds;
}

NBlackbox2::TOptions TUsersDB::ConstructAdminInfoOptions() const {
    TString attributesStr = JoinStrings(Config.GetBBAdminReportAttributes(), ",");
    NBlackbox2::TOptions options(NBlackbox2::TOption("attributes", attributesStr));
    options << NBlackbox2::OPT_GET_ALL_ALIASES;
    options << NBlackbox2::OPT_GET_ALL_EMAILS;
    options << NBlackbox2::OPT_REGNAME;
    options << NBlackbox2::TOption("getphones", "bound");
    options << NBlackbox2::TOption("phone_attributes", "102");
    return options;
}

TUsersDB::TOptionalUser TUsersDB::GetUserByPhone(const TString& phone, NDrive::TEntitySession& session) const {
    return GetUserByColumn("phone", phone, session);
}

TUsersDB::TOptionalUsers TUsersDB::GetUsersByPhone(const TString& phone, NDrive::TEntitySession& session) const {
    return GetUsersByColumn("phone", phone, session);
}
TUsersDB::TOptionalUsers TUsersDB::GetUsersByEmail(const TString& email, NDrive::TEntitySession& session) const {
    return GetUsersByColumn("email", email, session);
}

TUsersDB::TOptionalUser TUsersDB::GetUserByColumn(const TString& columnName, const TString& columnValue, NDrive::TEntitySession& session) const {
    TVector<TString> userIds;
    if (!SelectUsers(columnName, {columnValue}, userIds, session)) {
        return {};
    }

    TMaybe<TDriveUserData> activeInstance;
    TMaybe<TDriveUserData> blockedInstance;
    TMaybe<TDriveUserData> screeningInstance;
    TMaybe<TDriveUserData> onboardingInstance;
    TMaybe<TDriveUserData> externalInstance;

    auto timestamp = Now();
    auto usersMap = FetchInfo(userIds, session);
    if (!usersMap) {
        return {};
    }
    for (auto&& [userId, userObject] : usersMap.MutableResult()) {
        userObject.SetTimestamp(timestamp);
        if (userObject.IsPhoneVerified() && userObject.GetStatus() == NDrive::UserStatusActive) {
            return userObject;
        }
        if (userObject.GetStatus() == NDrive::UserStatusActive) {
            activeInstance = userObject;
        }
        if (userObject.GetStatus() == NDrive::UserStatusBlocked) {
            blockedInstance = userObject;
        }
        if (userObject.GetStatus() == NDrive::UserStatusScreening) {
            screeningInstance = userObject;
        }
        if (userObject.GetStatus() == NDrive::UserStatusOnboarding) {
            onboardingInstance = userObject;
        }
    }

    TVector<TMaybe<TDriveUserData>> maybeResults = {activeInstance, blockedInstance, screeningInstance, onboardingInstance};
    for (auto&& mr : maybeResults) {
        if (mr) {
            return mr;
        }
    }

    return TDriveUserData{};
}

TUsersDB::TOptionalUsers TUsersDB::GetUsersByColumn(const TString& columnName, const TString& columnValue, NDrive::TEntitySession& session) const {
    TVector<TString> userIds;
    if (!SelectUsers(columnName, {columnValue}, userIds, session)) {
        return {};
    }

    auto timestamp = Now();
    auto usersMap = FetchInfo(userIds, session);
    if (!usersMap) {
        return {};
    }

    TVector<TDriveUserData> users;
    for (auto&& [userId, userObject] : usersMap.MutableResult()) {
        if (NDrive::AcceptableUserStatuses.contains(userObject.GetStatus())) {
            userObject.SetTimestamp(timestamp);
            users.push_back(userObject);
        }
    }

    return users;
}

TUsersDB::TOptionalUserId TUsersDB::GetUserIdByLogin(const TString& login, NDrive::TEntitySession& session) const {
    auto userIds = TVector<TString>();
    if (!SelectUsers("username", {login}, userIds, session)) {
        return {};
    }
    if (userIds.empty()) {
        return TString();
    }
    if (userIds.size() > 1) {
        WARNING_LOG << "multiple user_ids for login " << login << Endl;
    }
    return std::move(userIds.front());
}

TUsersDB::TFetchResult TUsersDB::FetchUsersByPhone(const TSet<TString>& phones) const {
    auto session = BuildTx<NSQL::ReadOnly>();
    auto result = FetchInfoByField(phones, "phone", session);
    return result;
}

TMaybe<bool> TUsersDB::IsDeletedByTags(const TString& userId, NDrive::TEntitySession& session) const {
    auto optionalTags = UserTagsManager->RestoreTags(TVector<TString>{ userId }, { Config.GetDeletedUserTags() }, session);
    if (!optionalTags) {
        return {};
    }
    return !optionalTags->empty();
}

static const std::map<char, TSearchTraits> TraitToPrefix = {{
    // MUST be in sync with `user_tsvector` plsql function
    // all the strings are lower-ed to make search case insensitive,
    // so we'll free to use any uppercase prefixes
    {'L', NUserReport::ReportLogin},
    {'U', NUserReport::ReportPassportUid},
    {'E', NUserReport::ReportEmail},
    {'P', NUserReport::ReportPhone},
    {'N', NUserReport::ReportNames},
    {'S', NUserReport::ReportStatus},
}};

TString StringToPrefixTSQuery(const TString& str, const TSearchTraits& traits) {
    TStringBuilder result;
    bool empty = true;
    for (const auto& [prefix, trait] : TraitToPrefix) {
        if (!(traits & trait)) continue;
        if (empty) {
            result << '(';
            empty = false;
        } else {
            result << '|';
        }
        result << '\'' << prefix;
        for (const auto& chr : str) {
            if ((chr == '\'') || (chr == '\\')) {
                result << '\\';
            }
            result << chr;
        }
        result << "':*";
    }
    if (!empty) {
        result << ')';
    }
    return result;
}

TVector<TString> TUsersDB::GetMatchingIds(const TSearchRequest& searchRequest, const std::function<bool(const TString&)>& entityFilter) const {
    if (const auto& vector = Config.GetSearchViaDatabaseVector()) {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto queryOptions = NSQL::TQueryOptions();
        if ((!searchRequest.RequiredMatches.empty() || !searchRequest.OptionalMatches.empty()) && !searchRequest.HasEmptyOptionalMatch) {
            TStringBuilder query;
            bool empty = true;
            for (const auto& token : searchRequest.OptionalMatches) {
                if (empty) {
                    query << '(';
                    empty = false;
                } else {
                    query << '|';
                }
                query << StringToPrefixTSQuery(token, searchRequest.Traits);
            }
            if (!empty) {
                query << ')';
            }
            for (const auto& token : searchRequest.RequiredMatches) {
                if (!empty) {
                    query << '&';
                }
                query << StringToPrefixTSQuery(token, searchRequest.Traits);
                empty = false;
            }

            const auto& condition = vector + " @@ " + session->Quote(static_cast<const TString&>(query));
            queryOptions.SetCustomCondition(condition);
        }

        return FetchIds(session.GetTransaction(), queryOptions, entityFilter, searchRequest);
    }

    return Yensured(Index)->GetMatchingIds(searchRequest, entityFilter);
}

static const size_t FETCH_SIZE_INITIAL_NO_HARM = 1e3;
static const size_t FETCH_SIZE_FINAL_TOO_BIG = 1e9;
static const size_t FETCH_SIZE_X_STEP = 10;

template<typename T>
T GetSettingDef(const TString& key, const T& defaultValue) {
    return NDrive::HasServer()
        ? NDrive::GetServer().GetSettings().GetValueDef<T>(key, defaultValue)
        : defaultValue;
}

TVector<TString> TUsersDB::FetchIds(const NStorage::ITransaction::TPtr &transaction, NSQL::TQueryOptions& queryOptions, const std::function<bool(const TString&)>& entityFilter, const TSearchRequest& searchRequest) const {
    TVector<TString> result;
    // couldn't SQL LIMIT the SQL due to `entityFilter`-ing
    // using LIMIT on PostgreSQL 10 it refuses to user the index (workarounded by making the funciton too COST-ly)
    const auto& section = TString{"get-matching-ids.user.limit"};
    const auto& from = GetSettingDef<size_t>(section + ".from", FETCH_SIZE_INITIAL_NO_HARM);
    const auto& until = GetSettingDef<size_t>(section + ".until", FETCH_SIZE_FINAL_TOO_BIG);
    const auto& xstep = GetSettingDef<size_t>(section + ".xstep", FETCH_SIZE_X_STEP);
    size_t lastFetchSize = 0;
    for (auto limit = from; limit < until; limit *= xstep) {
        if (limit < searchRequest.Limit) {
            continue; // no reason to fetch less rows than we expect results
        }
        queryOptions.SetLimit(limit);
        const auto& query = queryOptions.PrintQuery(*Yensured(transaction), GetTableName(), {"id"});
        TRecordsSet records;
        const auto& queryResult = transaction->Exec(query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return {};
        }
        result.resize(0);
        lastFetchSize = records.size();
        for (const auto& record : records) {
            const auto& id = record.Get("id");
            if (entityFilter(id)) {
                result.push_back(id);
            }
            if (result.size() == searchRequest.Limit) {
                return result;
            }
        }
        if (records.size() < limit) {
            // checked all possible DB matches => nothing more to search
            return result;
        }
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "FetchIdsLimitHit")
            ("request.limit", searchRequest.Limit)
            ("fetched", lastFetchSize)
            ("result", result.size())
        );
    }

    // We kinda could return what we have
    // but I'm afraid there may be some logic relies on fetching/patching all the possible documents in the DB
    // So, let's signal we're in a trouble and let the man on duty fix the config e.g. `limit.until`
    throw yexception() << "not all rows were examined";
}

NStorage::IDatabase::TPtr TUsersDB::GetDatabase() const {
    return Database;
}

NJson::TJsonValue TUserDrivingLicenseData::SerializeToJson(const NUserReport::TReportTraits& traits) const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseNames) {
        NJson::InsertNonNull(result, "first_name", FirstName);
        NJson::InsertNonNull(result, "last_name", LastName);
        NJson::InsertNonNull(result, "middle_name", MiddleName);
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseNumbers) {
        NJson::InsertNonNull(result, "number_front", NumberFront);
        NJson::InsertNonNull(result, "number_back", NumberBack);
        NJson::InsertNonNull(result, "prev_licence_number_front", PrevLicenseNumberFront);
        NJson::InsertNonNull(result, "prev_licence_number", PrevLicenseNumber);
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseCategories) {
        NJson::InsertNonNull(result, "categories", Categories);
        NJson::InsertNonNull(result, "categories_back", CategoriesBack);
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseBirthDate) {
        if (BirthDate && !BirthDate->empty()) {
            NJson::InsertField(result, "birth_date", BirthDate);
        }
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseCountry) {
        NJson::InsertNonNull(result, "front_country", Country);
        NJson::InsertNonNull(result, "back_country", BackCountry);
    }

    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseExperienceFrom) {
        if (ExperienceFromStr && !ExperienceFromStr->empty()) {
            NJson::InsertField(result, "experience_from", ExperienceFromStr);
        }
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseIssueDates) {
        NJson::InsertNonNull(result, "issued_by", IssuedBy);
        if (IssueDate) {
            NJson::InsertField(result, "issue_date", NJson::IsoFormat(*IssueDate));
        }
        if (PrevLicenseIssueDateStrFront && !PrevLicenseIssueDateStrFront->empty()) {
            NJson::InsertField(result, "prev_licence_issue_date_front", PrevLicenseIssueDateStrFront);
        }
        if (PrevLicenseIssueDateStr && !PrevLicenseIssueDateStr->empty()) {
            NJson::InsertField(result, "prev_licence_issue_date", PrevLicenseIssueDateStr);
        }
    }
    if (traits & NUserReport::EReportTraits::ReportDrivingLicenseCategoriesBDates) {
        if (CategoriesBValidFromDateStr && !CategoriesBValidFromDateStr->empty()) {
            NJson::InsertField(result, "categories_b_valid_from_date", CategoriesBValidFromDateStr);
        }
        if (CategoriesBValidToDateFront) {
            NJson::InsertField(result, "categories_b_valid_to_date_front", NJson::IsoFormat(*CategoriesBValidToDateFront));
        }
        if (CategoriesBValidToDate) {
            NJson::InsertField(result, "categories_b_valid_to_date", NJson::IsoFormat(*CategoriesBValidToDate));
        }
        if (SpecialCategoryBDateStrFront && !SpecialCategoryBDateStrFront->empty()) {
            NJson::InsertField(result, "special_category_b_date_front", SpecialCategoryBDateStrFront);
        }
        if (SpecialCategoryBDateStrBack && !SpecialCategoryBDateStrBack->empty()) {
            NJson::InsertField(result, "special_category_b_date_back", SpecialCategoryBDateStrBack);
        }
    }
    return result;
}

TString TUserDrivingLicenseData::GetNumber() const {
    if (!!NumberFront) {
        return *NumberFront;
    }
    return NumberBack.GetOrElse("");
}

bool TUserDrivingLicenseData::ParseFromDatasync(const NJson::TJsonValue& report) {
    return NJson::TryFieldsFromJson(report, GetFields());
}

bool TUserDrivingLicenseData::IsDatasyncCompatible() const {
    // fields required by datasync
    if (!NumberFront || !NumberBack || !IssueDate || !FirstName || !LastName || !MiddleName || !BirthDate) {
        return false;
    }
    tm parsedTm;
    return
        (NUtil::ParseFomattedDatetime(*BirthDate, "%Y-%m-%d", parsedTm)) &&
        (!ExperienceFromStr || NUtil::ParseFomattedDatetime(*ExperienceFromStr, "%Y-%m-%d", parsedTm)) &&
        (!PrevLicenseIssueDateStrFront || NUtil::ParseFomattedDatetime(*PrevLicenseIssueDateStrFront, "%Y-%m-%d", parsedTm)) &&
        (!PrevLicenseIssueDateStr || NUtil::ParseFomattedDatetime(*PrevLicenseIssueDateStr, "%Y-%m-%d", parsedTm)) &&
        (!SpecialCategoryBDateStrFront || NUtil::ParseFomattedDatetime(*SpecialCategoryBDateStrFront, "%Y-%m-%d", parsedTm)) &&
        (!SpecialCategoryBDateStrBack || NUtil::ParseFomattedDatetime(*SpecialCategoryBDateStrBack, "%Y-%m-%d", parsedTm));
}

bool TUserDrivingLicenseData::IsForeign() const {
    const TString number = !!NumberFront.GetOrElse("")
        ? NumberFront.GetOrElse("")
        : NumberBack.GetOrElse("");
    if (!number)  {
        NDrive::TEventLog::Log("IsForeignError", NJson::TMapBuilder
            ("method", "TUserDrivingLicenseData::IsForeign")
            ("message", "Missing number")
            ("last_name", LastName.GetOrElse(""))
            ("first_name", FirstName.GetOrElse(""))
            ("middle_name", MiddleName.GetOrElse(""))
        );
        return false;
    }
    return
        !TRegExMatch("^[0-9]{10}$").Match(number.data()) &&
        !TRegExMatch("^[0-9]{2}[а-яА-Я]{2}[0-9]{6}$").Match(number.data());
}

NJson::TJsonValue TUserDrivingLicenseData::SerializeBackToYang(const bool isVerified) const {
    NJson::TJsonValue data;
    NJson::InsertField(data, "categories", CategoriesBack);
    NJson::InsertField(data, "categories_b_valid_from_date", CategoriesBValidFromDateStr);
    NJson::InsertField(data, "country", BackCountry);
    NJson::InsertField(data, "experience_from", ExperienceFromStr);
    NJson::InsertField(data, "number", NumberBack);
    NJson::InsertField(data, "prev_licence_issue_date", PrevLicenseIssueDateStr);
    NJson::InsertField(data, "prev_licence_number", PrevLicenseNumber);
    NJson::InsertField(data, "special_category_b_date", SpecialCategoryBDateStrBack);
    if (CategoriesBValidToDate) {
        NJson::InsertField(data, "categories_b_valid_to_date", NJson::IsoFormat(*CategoriesBValidToDate));
    } else {
        NJson::InsertField(data, "categories_b_valid_to_date", CategoriesBValidToDate);
    }
    NJson::TJsonValue result;
    result["data"] = std::move(data);
    result["is_verified"] = isVerified;
    return result;
}

NJson::TJsonValue TUserDrivingLicenseData::SerializeFrontToYang(const bool isVerified) const {
    NJson::TJsonValue data;
    NJson::InsertField(data, "number", NumberFront);
    NJson::InsertField(data, "prev_licence_number", PrevLicenseNumberFront);
    NJson::InsertField(data, "prev_licence_issue_date", PrevLicenseIssueDateStrFront);
    NJson::InsertField(data, "first_name", FirstName);
    NJson::InsertField(data, "last_name", LastName);
    NJson::InsertField(data, "middle_name", MiddleName);
    NJson::InsertField(data, "categories", Categories);
    NJson::InsertField(data, "country", Country);
    NJson::InsertField(data, "birth_date", BirthDate);
    NJson::InsertField(data, "experience_from", ExperienceFromStr);
    NJson::InsertField(data, "special_category_b_date", SpecialCategoryBDateStrFront);
    NJson::InsertField(data, "issued_by", IssuedBy);
    if (CategoriesBValidToDateFront) {
        NJson::InsertField(data, "categories_b_valid_to_date", NJson::IsoFormat(*CategoriesBValidToDateFront));
    } else {
        NJson::InsertField(data, "categories_b_valid_to_date", CategoriesBValidToDateFront);
    }
    if (IssueDate) {
        NJson::InsertField(data, "issue_date", NJson::IsoFormat(*IssueDate));
    } else {
        NJson::InsertField(data, "issue_date", IssueDate);
    }
    NJson::TJsonValue result;
    result["data"] = std::move(data);
    result["is_verified"] = isVerified;
    return result;
}

TInstant GetPossiblyEarlyInstant(const TString& value) {
    if (!value) {
        return TInstant::Zero();
    }
    if (value < "1970") {
        return TInstant::Seconds(1);
    }
    TInstant result;
    if (!TInstant::TryParseIso8601(value, result)) {
        return TInstant::Zero();
    }
    return result;
}

TInstant TUserDrivingLicenseData::GetExperienceFrom() const {
    return GetPossiblyEarlyInstant(ExperienceFromStr.GetOrElse(""));
}

TInstant TUserDrivingLicenseData::GetPrevLicenseIssueDate() const {
    return GetPossiblyEarlyInstant(PrevLicenseIssueDateStr.GetOrElse(""));
}

TInstant TUserDrivingLicenseData::GetCategoriesBValidFromDate() const {
    return GetPossiblyEarlyInstant(CategoriesBValidFromDateStr.GetOrElse(""));
}

TInstant TUserDrivingLicenseData::GetUserBirthDate() const {
    return GetPossiblyEarlyInstant(GetBirthDate());
}

bool TUserDrivingLicenseData::Parse(const NJson::TJsonValue& report) {
    return ParseFromDatasync(report);
}

NJson::TJsonValue TUserDrivingLicenseData::SerializeToDatasyncJson() const {
    return SerializeToJson(NUserReport::ReportAll);
}

void TUserDrivingLicenseData::Patch(const TUserDrivingLicenseData& other) {
    MergeFields(other);
}

bool TUserPassportRegistrationData::ParseFromDatasync(const NJson::TJsonValue& report) {
    return NJson::TryFieldsFromJson(report, GetFields());
}

TString TUserPassportData::GetNamesForHash() const {
    return TStringBuilder()
        << GetFirstName() << '@'
        << GetMiddleName() << '@'
        << GetLastName() << '@'
        << GetBirthDate()
    ;
}

TInstant TUserPassportData::GetUserBirthDate() const {
    return GetPossiblyEarlyInstant(GetBirthDate());
}

bool TUserPassportData::HasNamesForHash() const {
    return
        FirstName ||
        MiddleName ||
        LastName ||
        BirthDate;
}

bool TUserPassportData::ParseFromDatasync(const NJson::TJsonValue& report) {
    return
        NJson::TryFieldsFromJson(report, GetFields()) &&
        Registration.ParseFromDatasync(report);
}

bool TUserPassportData::Parse(const NJson::TJsonValue& report) {
    return ParseFromDatasync(report);
}

bool TUserPassportData::IsDatasyncCompatible() const {
    // fields required by datasync
    if (!FirstName || !LastName || !MiddleName || !Number || !BirthDate || !Gender || !Citizenship) {
        return false;
    }
    tm parsedTm;
    return NUtil::ParseFomattedDatetime(*BirthDate, "%Y-%m-%d", parsedTm);
}

void TUserPassportData::Patch(const TUserPassportData& other) {
    MergeFields(other);
    Registration.MergeFields(other.GetRegistration());
}

NJson::TJsonValue TUserPassportData::SerializeToDatasyncJson() const {
    return SerializeToJson(NUserReport::ReportAll);
}

NJson::TJsonValue TUserPassportData::SerializeToJson(const NUserReport::TReportTraits& traits) const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "doc_type", TStringBuf("id"));

    if (traits & NUserReport::EReportTraits::ReportPassportNumber) {
        NJson::InsertNonNull(result, "doc_value", Number);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportNames) {
        NJson::InsertNonNull(result, "first_name", FirstName);
        NJson::InsertNonNull(result, "last_name", LastName);
        NJson::InsertNonNull(result, "middle_name", MiddleName);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportBirthPlace) {
        NJson::InsertNonNull(result, "birth_place", BirthPlace);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportCitizenship) {
        NJson::InsertNonNull(result, "citizenship", Citizenship);
        NJson::InsertNonNull(result, "biographical_country", BiographicalCountry);
        NJson::InsertNonNull(result, "registration_country", RegistrationCountry);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportGender) {
        NJson::InsertNonNull(result, "gender", Gender);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportSubdivisionCode) {
        NJson::InsertNonNull(result, "subdivision_code", SubdivisionCode);
    }
    if (traits & NUserReport::EReportTraits::ReportPassportBirthDate) {
        if (BirthDate && !BirthDate->empty()) {
            NJson::InsertField(result, "birth_date", BirthDate);
        }
    }

    // Either use ISO format or field absence (null causes 400 in DS)
    if (traits & NUserReport::EReportTraits::ReportPassportValidityDates) {
        NJson::InsertNonNull(result, "issued_by", IssuedBy);
        if (IssueDate) {
            NJson::InsertField(result, "issue_date", NJson::IsoFormat(*IssueDate));
        }
        if (ExpirationDate) {
            NJson::InsertField(result, "expiration_date", NJson::IsoFormat(*ExpirationDate));
        }
    }

    // Registration
    if (traits & NUserReport::EReportTraits::ReportPassportRegistration) {
        NJson::InsertNonNull(result, "registration_apartment", Registration.OptionalApartment());
        NJson::InsertNonNull(result, "registration_housing", Registration.OptionalHousing());
        NJson::InsertNonNull(result, "registration_letter", Registration.OptionalLetter());
        NJson::InsertNonNull(result, "registration_house", Registration.OptionalHouse());
        NJson::InsertNonNull(result, "registration_street", Registration.OptionalStreet());
        NJson::InsertNonNull(result, "registration_locality", Registration.OptionalLocality());
        NJson::InsertNonNull(result, "registration_area", Registration.OptionalArea());
        NJson::InsertNonNull(result, "registration_region", Registration.OptionalRegion());
        NJson::InsertNonNull(result, "registration_type", Registration.OptionalType());
        if (Registration.HasExpirationDate()) {
            NJson::InsertField(result, "registration_expiration_date", NJson::IsoFormat(Registration.GetExpirationDateRef()));
        }
    }

    return result;
}

bool TUserPassportRegistrationData::ParseFromYang(const NJson::TJsonValue& registration, TMessagesCollector& errors) {
    return
        NJson::ParseField(registration, "apartment", Apartment, errors) &&
        NJson::ParseField(registration, "housing", Housing, errors) &&
        NJson::ParseField(registration, "registration_letter", Letter, errors) &&
        NJson::ParseField(registration, "house", House, errors) &&
        NJson::ParseField(registration, "street", Street, errors) &&
        NJson::ParseField(registration, "area", Area, errors) &&
        NJson::ParseField(registration, "locality", Locality, errors) &&
        NJson::ParseField(registration, "region", Region, errors) &&
        NJson::ParseField(registration, "registration_expiration_date", ExpirationDate, errors) &&
        NJson::ParseField(registration, "type", Type, errors);
}

NJson::TJsonValue TUserPassportData::SerializeBioToYang(const bool isVerified) const {
    NJson::TJsonValue data;
    NJson::InsertField(data, "first_name", FirstName);
    NJson::InsertField(data, "last_name", LastName);
    NJson::InsertField(data, "middle_name", MiddleName);
    NJson::InsertField(data, "birth_date", BirthDate);
    NJson::InsertField(data, "gender", Gender);
    NJson::InsertField(data, "citizenship", Citizenship);
    NJson::InsertField(data, "number", Number);
    NJson::InsertField(data, "birth_place", BirthPlace);
    NJson::InsertField(data, "subdivision_code", SubdivisionCode);
    NJson::InsertField(data, "country", BiographicalCountry);
    NJson::InsertField(data, "issued_by", IssuedBy);
    if (IssueDate) {
        NJson::InsertField(data, "issue_date", NJson::IsoFormat(*IssueDate));
    } else {
        NJson::InsertField(data, "issue_date", IssueDate);
    }
    if (ExpirationDate) {
        NJson::InsertField(data, "expiration_date", NJson::IsoFormat(*ExpirationDate));
    } else {
        NJson::InsertField(data, "expiration_date", ExpirationDate);
    }
    NJson::TJsonValue result;
    result["data"] = std::move(data);
    result["is_verified"] = isVerified;
    return result;
}

NJson::TJsonValue TUserPassportData::SerializeRegToYang(const bool isVerified) const {
    NJson::TJsonValue data;
    NJson::InsertField(data, "region", Registration.OptionalRegion());
    NJson::InsertField(data, "area", Registration.OptionalArea());
    NJson::InsertField(data, "locality", Registration.OptionalLocality());
    NJson::InsertField(data, "street", Registration.OptionalStreet());
    NJson::InsertField(data, "house", Registration.OptionalHouse());
    NJson::InsertField(data, "housing", Registration.OptionalHousing());
    NJson::InsertField(data, "apartment", Registration.OptionalApartment());
    NJson::InsertField(data, "registration_letter", Registration.OptionalLetter());
    NJson::InsertField(data, "type", Registration.OptionalType());
    NJson::InsertField(data, "country", RegistrationCountry);
    if (Registration.HasExpirationDate()) {
        NJson::InsertField(data, "registration_expiration_date", NJson::IsoFormat(Registration.GetExpirationDateRef()));
    } else {
        NJson::InsertField(data, "registration_expiration_date", Registration.OptionalExpirationDate());
    }
    NJson::TJsonValue result;
    result["data"] = std::move(data);
    result["is_verified"] = isVerified;
    return result;
}

bool TUserPassportData::ParseFromYang(const NJson::TJsonValue& biographical, const NJson::TJsonValue& registration, TMessagesCollector& errors, const bool isPatch) {
    bool parsed =
        NJson::ParseField(biographical, "first_name", FirstName, errors) &&
        NJson::ParseField(biographical, "last_name", LastName, errors) &&
        NJson::ParseField(biographical, "middle_name", MiddleName, errors) &&
        NJson::ParseField(biographical, "birth_date", BirthDate, errors) &&
        NJson::ParseField(biographical, "gender", Gender, errors) &&
        NJson::ParseField(biographical, "citizenship", Citizenship, errors) &&
        NJson::ParseField(biographical, "number", Number, errors) &&
        NJson::ParseField(biographical, "birth_place", BirthPlace, errors) &&
        NJson::ParseField(biographical, "subdivision_code", SubdivisionCode, errors) &&
        NJson::ParseField(biographical, "country", BiographicalCountry, errors) &&
        NJson::ParseField(biographical, "issued_by", IssuedBy, errors) &&
        NJson::ParseField(biographical, "issue_date", IssueDate, errors) &&
        NJson::ParseField(biographical, "expiration_date", ExpirationDate, errors) &&
        NJson::ParseField(registration, "country", RegistrationCountry, errors) &&
        Registration.ParseFromYang(registration, errors);
    if (!parsed) {
        return false;
    }
    if (BirthDate && BirthDate->size() == 10) {
        BirthDate->append("T00:00:00.000Z");
    }
    if (!isPatch && !IsDatasyncCompatible()) {
        errors.AddMessage("ParseFromYang", "fields are not compatible with datasync");
        return false;
    }
    return true;
}

bool TUserDrivingLicenseData::ParseFromYang(const NJson::TJsonValue& back, const NJson::TJsonValue& front, TMessagesCollector& errors, const bool isPatch) {
    bool parsed =
        NJson::ParseField(front, "number", NumberFront, errors) &&
        NJson::ParseField(front, "prev_license_number", PrevLicenseNumberFront, errors) &&
        NJson::ParseField(front, "prev_licence_issue_date", PrevLicenseIssueDateStrFront, errors) &&
        NJson::ParseField(front, "first_name", FirstName, errors) &&
        NJson::ParseField(front, "last_name", LastName, errors) &&
        NJson::ParseField(front, "middle_name", MiddleName, errors) &&
        NJson::ParseField(front, "birth_date", BirthDate, errors) &&
        NJson::ParseField(front, "categories", Categories, errors) &&
        NJson::ParseField(front, "country", Country, errors) &&
        NJson::ParseField(front, "experience_from", ExperienceFromStr, errors) &&
        NJson::ParseField(front, "issued_by", IssuedBy, errors) &&
        NJson::ParseField(front, "issue_date", IssueDate, errors) &&
        NJson::ParseField(front, "categories_b_valid_to_date", CategoriesBValidToDateFront, errors) &&
        NJson::ParseField(front, "special_category_b_date", SpecialCategoryBDateStrFront, errors) &&
        NJson::ParseField(back, "number", NumberBack, errors) &&
        NJson::ParseField(back, "prev_licence_number", PrevLicenseNumber, errors) &&
        NJson::ParseField(back, "categories", CategoriesBack, errors) &&
        NJson::ParseField(back, "country", BackCountry, errors) &&
        NJson::ParseField(back, "experience_from", ExperienceFromStr, errors) &&
        NJson::ParseField(back, "prev_licence_issue_date", PrevLicenseIssueDateStr, errors) &&
        NJson::ParseField(back, "special_category_b_date", SpecialCategoryBDateStrBack, errors) &&
        NJson::ParseField(back, "categories_b_valid_from_date", CategoriesBValidFromDateStr, errors) &&
        NJson::ParseField(back, "categories_b_valid_to_date", CategoriesBValidToDate, errors);
    if (!parsed) {
        return false;
    }
    auto normalize = [](TMaybe<TString>& date) {
        if (date && date->size() == 10) {
            date->append("T00:00:00.000Z");
        }
    };
    normalize(BirthDate);
    normalize(ExperienceFromStr);
    normalize(PrevLicenseIssueDateStrFront);
    normalize(PrevLicenseIssueDateStr);
    normalize(CategoriesBValidFromDateStr);
    normalize(SpecialCategoryBDateStrFront);
    normalize(SpecialCategoryBDateStrBack);
    if (!isPatch && !IsDatasyncCompatible()) {
        errors.AddMessage("ParseFromYang", "fields are not compatible with datasync");
        return false;
    }
    return true;
}

TDriveUserData::TFieldMask TDriveUserData::GetFilledFields() const {
    TDriveUserData::TFieldMask result = 0;
    result |= (ui32)TDriveUserData::EField::FirstName * !!FirstName;
    result |= (ui32)TDriveUserData::EField::LastName * !!LastName;
    result |= (ui32)TDriveUserData::EField::PName * !!PName;
    result |= (ui32)TDriveUserData::EField::Phone * !!Phone;
    return result;
}

TString TDriveUserData::GetShortName() const {
    if (!FirstName) {
        return Login;
    }
    if (!LastName) {
        return FirstName;
    }

    TStringBuf buf = LastName;
    size_t charLen;
    if (GetUTF8CharLen(charLen, (const unsigned char*)buf.begin(), (const unsigned char*)buf.end()) != RECODE_OK) {
        return FirstName;
    }
    return FirstName + " " + buf.SubStr(0, charLen) + ".";
}

TDriveUserData::TDriveUserDataDecoder::TDriveUserDataDecoder(const TMap<TString, ui32>& decoderBase) {
    UserId = GetFieldDecodeIndex("id", decoderBase);
    Address = GetFieldDecodeIndex("address", decoderBase);
    Phone = GetFieldDecodeIndex("phone", decoderBase);
    FirstName = GetFieldDecodeIndex("first_name", decoderBase);
    LastName = GetFieldDecodeIndex("last_name", decoderBase);
    PName = GetFieldDecodeIndex("patronymic_name", decoderBase);
    Status = GetFieldDecodeIndex("status", decoderBase);
    FirstRiding = GetFieldDecodeIndex("is_first_riding", decoderBase);
    Login = GetFieldDecodeIndex("username", decoderBase);
    Uid = GetFieldDecodeIndex("uid", decoderBase);
    PhoneVerified = GetFieldDecodeIndex("is_phone_verified", decoderBase);
    EMailVerified = GetFieldDecodeIndex("is_email_verified", decoderBase);
    Email = GetFieldDecodeIndex("email", decoderBase);
    RegistrationGeo = GetFieldDecodeIndex("registration_geo", decoderBase);
    JoinedAt = GetFieldDecodeIndex("date_joined", decoderBase);
    ApprovedAt = GetFieldDecodeIndex("registered_at", decoderBase);
    PassportDatasyncRevision = GetFieldDecodeIndex("passport_ds_revision", decoderBase);
    DrivingLicenseDatasyncRevision = GetFieldDecodeIndex("driving_license_ds_revision", decoderBase);
    PassportNamesHash = GetFieldDecodeIndex("passport_names_hash", decoderBase);
    PassportNumberHash = GetFieldDecodeIndex("passport_number_hash", decoderBase);
    DrivingLicenseNumberHash = GetFieldDecodeIndex("driving_license_number_hash", decoderBase);
    Environment = GetFieldDecodeIndex("environment", decoderBase);
    HasATMark = GetFieldDecodeIndex("has_at_mark", decoderBase);
}

bool TDriveUserData::DeserializeWithDecoder(const TDriveUserDataDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, Phone);
    READ_DECODER_VALUE(decoder, values, FirstName);
    READ_DECODER_VALUE(decoder, values, LastName);
    READ_DECODER_VALUE(decoder, values, PName);
    READ_DECODER_VALUE(decoder, values, Status);
    READ_DECODER_VALUE(decoder, values, FirstRiding);
    READ_DECODER_VALUE(decoder, values, Login);
    READ_DECODER_VALUE(decoder, values, Uid);
    READ_DECODER_VALUE_DEF(decoder, values, PhoneVerified, false);
    READ_DECODER_VALUE_DEF(decoder, values, EMailVerified, false);
    READ_DECODER_VALUE(decoder, values, Email);
    READ_DECODER_VALUE_DEF(decoder, values, RegistrationGeo, "moscow");
    if (!RegistrationGeo) {
        RegistrationGeo = "moscow";
    }
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, JoinedAt);
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ApprovedAt);
    READ_DECODER_VALUE(decoder, values, PassportDatasyncRevision);
    READ_DECODER_VALUE(decoder, values, DrivingLicenseDatasyncRevision);
    READ_DECODER_VALUE(decoder, values, PassportNamesHash);
    READ_DECODER_VALUE(decoder, values, PassportNumberHash);
    READ_DECODER_VALUE(decoder, values, DrivingLicenseNumberHash);
    READ_DECODER_VALUE_DEF(decoder, values, HasATMark, false);
    READ_DECODER_VALUE(decoder, values, Environment);
    if (decoder.GetAddress() >= 0) {
        READ_DECODER_VALUE(decoder, values, Address);
    }

    FirstName = TlsRef(UserNamesPool).Get(FirstName);
    PName = TlsRef(UserPNamesPool).Get(PName);
    Status = TlsRef(UserStatusPool).Get(Status);
    RegistrationGeo = TlsRef(UserRegionsPool).Get(RegistrationGeo);
    return true;
}

void TDriveUserData::DoBuildReportItem(NJson::TJsonValue& item) const {
    item.InsertValue("object", GetReport(NUserReport::ReportAll));
}

bool TDriveUserData::Parse(const NStorage::TTableRecord& row) {
    UserId = row.Get("id");
    Address = row.Get("address");
    Login = row.Get("username");
    Email = row.Get("email");
    Phone = row.Get("phone");
    if (!row.TryGet("is_phone_verified", PhoneVerified)) {
        PhoneVerified = false;
    }
    if (!row.TryGet("is_email_verified", EMailVerified)) {
        EMailVerified = false;
    }
    if (!Phone.StartsWith("+") && !!Phone) {
        Phone = "+" + Phone;
    }

    FirstName = row.Get("first_name");
    LastName = row.Get("last_name");
    PName = row.Get("patronymic_name");
    Status = row.Get("status");
    Uid = row.Get("uid");
    row.TryGetDefault("is_first_riding", FirstRiding, false);
    PassportDatasyncRevision = row.Get("passport_ds_revision");
    DrivingLicenseDatasyncRevision = row.Get("driving_license_ds_revision");

    PassportNamesHash = row.Get("passport_names_hash");
    PassportNumberHash = row.Get("passport_number_hash");
    DrivingLicenseNumberHash = row.Get("driving_license_number_hash");

    if (row.Get("registration_geo")) {
        RegistrationGeo = row.Get("registration_geo");
    } else {
        RegistrationGeo = "moscow";
    }

    TString dateJoined = row.Get("date_joined");
    if (!!dateJoined && !TInstant::TryParseIso8601(dateJoined, JoinedAt)) {
        ui32 dateJoinedSecondsAd;
        if (!TryFromString<ui32>(dateJoined, dateJoinedSecondsAd)) {
            return false;
        } else {
            JoinedAt = TInstant::Seconds(dateJoinedSecondsAd);
        }
    }
    TString registeredAt = row.Get("registered_at");
    if (!!registeredAt && !TInstant::TryParseIso8601(registeredAt, ApprovedAt)) {
        ui32 registeredAtSecondsAd;
        if (!TryFromString<ui32>(registeredAt, registeredAtSecondsAd)) {
            return false;
        } else {
            ApprovedAt = TInstant::Seconds(registeredAtSecondsAd);
        }
    }

    if (row.Get("has_at_mark")) {
        HasATMark = IsTrue(row.Get("has_at_mark"));  // no AT-only mark => can drive on MT
    }

    Environment = row.Get("environment");

    return true;
}

NStorage::TTableRecord TDriveUserData::SerializeToTableRecord() const {
    NStorage::TTableRecord result;

    result.Set("id", UserId);
    result.Set("username", Login);
    result.Set("email", Email);
    result.Set("phone", Phone);
    result.Set("is_phone_verified", PhoneVerified);
    result.Set("is_email_verified", EMailVerified);
    result.Set("first_name", FirstName);
    result.Set("last_name", LastName);
    result.Set("patronymic_name", PName);
    result.Set("status", Status);
    if (Uid) {
        result.Set("uid", Uid);
    } else {
        result.Set("uid", "get_null()");
    }
    result.Set("passport_ds_revision", PassportDatasyncRevision);
    result.Set("driving_license_ds_revision", DrivingLicenseDatasyncRevision);
    result.Set("has_at_mark", HasATMark);
    result.Set("passport_number_hash", PassportNumberHash);
    result.Set("driving_license_number_hash", DrivingLicenseNumberHash);
    result.Set("is_first_riding", FirstRiding);
    result.Set("registration_geo", RegistrationGeo);
    if (ApprovedAt != TInstant::Zero()) {
        result.Set("registered_at", ApprovedAt.ToString());
    }
    result.Set("date_joined", JoinedAt.ToString());
    if (Address) {
        result.Set("address", Address);
    }
    if (Environment) {
        result.Set("environment", Environment);
    }
    if (PassportNamesHash) {
        result.Set("passport_names_hash", PassportNamesHash);
    }

    return result;
}

bool TDriveUserData::Patch(const NJson::TJsonValue& info, const bool isNewUser, const NUserReport::TReportTraits& traits) {
    JREAD_STRING_OPT(info, "id", UserId);
    if (!NJson::ParseField(info["address"], Address)) {
        return false;
    }
    if (traits & NUserReport::EReportTraits::ReportPhone) {
        JREAD_STRING_OPT(info, "phone", Phone);
        if (!NJson::ParseField(info["is_phone_verified"], PhoneVerified)) {
            return false;
        }
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportFirstName)) {
        JREAD_STRING_OPT(info, "first_name", FirstName);
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportLastName)) {
        JREAD_STRING_OPT(info, "last_name", LastName);
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportMiddleName)) {
        JREAD_STRING_OPT(info, "pn", PName);
    }

    if (traits & NUserReport::EReportTraits::ReportLogin) {
        JREAD_STRING_OPT(info, "username", Login);
    }
    if (traits & NUserReport::EReportTraits::ReportEmail) {
        JREAD_STRING_OPT(info, "email", Email);
        if (!NJson::ParseField(info["is_email_verified"], EMailVerified)) {
            return false;
        }
    }
    if (traits & NUserReport::EReportTraits::ReportAllowedTransmissionTypes) {
        bool isMTAllowed = !HasATMark;
        JREAD_BOOL_OPT(info, "is_mechanic_transmission_allowed", isMTAllowed);
        HasATMark = !isMTAllowed;
    }
    if (traits & NUserReport::EReportTraits::ReportIsFirstRiding) {
        if (!NJson::ParseField(info["is_first_riding"], FirstRiding)) {
            return false;
        }
    }
    if (isNewUser) {
        Status = NDrive::UserStatusOnboarding;
    }
    const auto& uid = info["uid"];
    if (uid.IsUInteger()) {
        Uid = ToString(uid.GetUInteger());
    } else if (!NJson::ParseField(uid, Uid)) {
        return false;
    }
    if (Uid == "0") {
        Uid.clear();
    }
    JREAD_STRING_OPT(info, "status", Status);
    return true;
}

void TDriveUserData::Index(TBasicSearchIndex& index) const {
    TVector<TBasicSearchIndex::TSearchEntityProp> result(
        {
            TBasicSearchIndex::TSearchEntityProp(Login).SetTraits(NUserReport::ReportLogin),
            TBasicSearchIndex::TSearchEntityProp(Uid).SetTraits(NUserReport::ReportPassportUid),
            TBasicSearchIndex::TSearchEntityProp(Email).SetTraits(NUserReport::ReportEmail)
        }
    );
    if (Phone != "") {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(Phone.substr(1, Phone.length() - 1)).SetTraits(NUserReport::ReportPhone));
        if (Phone.length() > 2) {
            result.push_back(TBasicSearchIndex::TSearchEntityProp(Phone.substr(2, Phone.length() - 2)).SetTraits(NUserReport::ReportPhone)); // start with operator code
            result.push_back(TBasicSearchIndex::TSearchEntityProp("8" + Phone.substr(2, Phone.length() - 2)).SetTraits(NUserReport::ReportPhone)); // start with 8
        }
    }
    if (FirstName != "") {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(FirstName).SetTraits(NUserReport::ReportNames));
    }
    if (LastName != "") {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(LastName).SetTraits(NUserReport::ReportNames));
    }
    if (PName != "") {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(PName).SetTraits(NUserReport::ReportNames));
    }
    result.push_back(TBasicSearchIndex::TSearchEntityProp(Status).SetTraits(NUserReport::ReportStatus));

    index.Refresh(UserId, result);
}

NJson::TJsonValue TDriveUserData::GetReport(const NUserReport::TReportTraits traits /*= NUserReport::ReportPublicId*/, bool reportPublicStatus) const {
    NJson::TJsonValue report = NJson::JSON_MAP;
    if (traits & NUserReport::EReportTraits::ReportId) {
        report["id"] = UserId;
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportFirstName)) {
        report["first_name"] = FirstName;
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportLastName)) {
        report["last_name"] = LastName;
    }
    if (traits & (NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportMiddleName)) {
        report["pn"] = PName;
    }
    if (traits & NUserReport::EReportTraits::ReportStatus) {
        report["status"] = reportPublicStatus ? GetPublicStatus() : Status;
    }
    if (traits & NUserReport::EReportTraits::ReportIsFirstRiding) {
        report["is_first_riding"] = FirstRiding;
    }
    if (traits & NUserReport::EReportTraits::ReportLogin) {
        report["username"] = Login;
    }
    if (traits & NUserReport::EReportTraits::ReportPassportUid) {
        report["uid"] = Uid;
    }
    if (traits & (NUserReport::EReportTraits::ReportEmail | NUserReport::EReportTraits::ReportPhone)) {
        auto& jsonSetup = report.InsertValue("setup", NJson::JSON_MAP);
        if (traits & NUserReport::EReportTraits::ReportPhone) {
            auto& jsonPhone = jsonSetup.InsertValue("phone", NJson::JSON_MAP);
            jsonPhone.InsertValue("verified", PhoneVerified);
            jsonPhone.InsertValue("number", Phone);
        }
        if (traits & NUserReport::EReportTraits::ReportEmail) {
            auto& jsonEMail = jsonSetup.InsertValue("email", NJson::JSON_MAP);
            jsonEMail.InsertValue("address", Email);
            jsonEMail.InsertValue("verified", (Status != NDrive::UserStatusOnboarding) || (Status == NDrive::UserStatusOnboarding && EMailVerified));
        }
        if (Address) {
            report["address"] = Address;
        }
    }
    if (traits & NUserReport::ReportPassport) {
        report["passport_revision"] = PassportDatasyncRevision;
    }
    if (traits & NUserReport::ReportDrivingLicense) {
        report["driving_license_revision"] = DrivingLicenseDatasyncRevision;
    }
    if (traits & NUserReport::EReportTraits::ReportPreliminaryPayments) {
        auto& jsonPrePayments = report.InsertValue("preliminary_payments", NJson::JSON_MAP);
        jsonPrePayments.InsertValue("enabled", false);
        jsonPrePayments.InsertValue("amount", 0);
    }
    if (traits & NUserReport::EReportTraits::ReportRegistrationDates) {
        auto& regData = report.InsertValue("registration", NJson::JSON_MAP);
        regData.InsertValue("joined_at", JoinedAt.Seconds());
        regData.InsertValue("approved_at", ApprovedAt.Seconds());
    }
    if (traits & NUserReport::EReportTraits::ReportAllowedTransmissionTypes) {
        report["is_mechanic_transmission_allowed"] = IsMTAllowed();
    }
    return report;
}

TAtomicSharedPtr<NDrive::TBlackboxClient> TDriveUserData::GetBlackboxClient(const NDrive::IServer& server) const {
    ui64 tvmId = server.GetDriveAPI()->GetUsersData()->GetConfig().GetSelfTvmId();
    TString blackboxUrl = server.GetDriveAPI()->GetUsersData()->GetConfig().GetBlackboxUrl();
    if (Environment == "testing") {
        tvmId = server.GetDriveAPI()->GetUsersData()->GetConfig().GetTestSelfTvmId();
        blackboxUrl = server.GetDriveAPI()->GetUsersData()->GetConfig().GetTestBlackboxUrl();
    }
    auto tvmClient = server.GetTvmClient(tvmId);
    TAtomicSharedPtr<NDrive::TBlackboxClient> blackboxClient;
    if (tvmClient) {
        blackboxClient.Reset(new NDrive::TBlackboxClient(blackboxUrl, tvmClient, server.GetDriveAPI()->GetUsersData()->GetAsyncDelivery()));
        blackboxClient->SetExternalOptions(server.GetDriveAPI()->GetUsersData()->ConstructAdminInfoOptions());
    }
    return blackboxClient;
}

NThreading::TFuture<NJson::TJsonValue> TDriveUserData::GetFullReport(const NUserReport::TReportTraits traits, const NDrive::IServer& server) const {
    TVector<NThreading::TFuture<void>> subrequests;
    NThreading::TFuture<NDrive::TRealtimeUserData> realtimeDataFuture;
    if (traits & NUserReport::ReportRealtimeData && server.GetUserEventsApi()) {
        const NDrive::TUserEventsApi* userEventsApi = server.GetUserEventsApi();
        realtimeDataFuture = userEventsApi->GetRealtimeUserInfo(
                UserId, traits & NUserReport::ReportRealtimeLocation,
                traits & NUserReport::ReportRealtimeClientAppInfo);
        subrequests.push_back(realtimeDataFuture.IgnoreResult());
    }

    const TDriveAPI& driveApi = *server.GetDriveAPI();
    NThreading::TFuture<TUserPassportData> passportFuture;
    NThreading::TFuture<TUserDrivingLicenseData> licenseFuture;
    auto passportRevision = GetPassportDatasyncRevision();
    auto licenseRevision = GetDrivingLicenseDatasyncRevision();
    {
        if ((traits & NUserReport::ReportPassport) && passportRevision) {
            passportFuture = driveApi.GetPrivateDataClient().GetPassport(*this, passportRevision);
            subrequests.push_back(passportFuture.IgnoreResult());
        }
        if ((traits & NUserReport::ReportDrivingLicense) && licenseRevision) {
            licenseFuture = driveApi.GetPrivateDataClient().GetDrivingLicense(*this, licenseRevision);
            subrequests.push_back(licenseFuture.IgnoreResult());
        }
    }

    NJson::TJsonValue result = GetReport(traits);
    if (traits & NUserReport::EReportTraits::ReportBillingBase && driveApi.HasBillingManager()) {
        driveApi.GetBillingManager().FillUserReport(UserId, result["billing"]);
    }

    if (traits & NUserReport::EReportTraits::ReportPhotos && driveApi.HasDocumentPhotosManager()) {
        driveApi.GetDocumentPhotosManager().GetUserPhotosDB().FillUserReport(UserId, result["photos"]);
    }

    if (traits & NUserReport::EReportTraits::ReportYang && driveApi.HasDocumentPhotosManager()) {
        driveApi.GetDocumentPhotosManager().GetDocumentVerificationAssignments().FillUserReport(UserId, result["yang"]);
    }

    if (traits & NUserReport::EReportTraits::ReportStatus) {
        auto registrationRobot = server.GetChatRobot("registration");
        TChatRobotScriptItem currentScriptItem;
        if (!!registrationRobot && registrationRobot->IsExistsForUser(UserId, "") && registrationRobot->GetCurrentScriptItem(UserId, "", currentScriptItem, TInstant::Zero())) {
            result["registration_chat"]["stage"] = currentScriptItem.GetId();
        }
    }

    NThreading::TFuture<TSocialProfilesData> socialDataFuture;
    if (traits & NUserReport::EReportTraits::ReportSocialData && server.GetSocialAPIClient()) {
        auto passportUid = GetPassportUid();
        socialDataFuture = server.GetSocialAPIClient()->GetSocialData(passportUid);
        subrequests.push_back(socialDataFuture.IgnoreResult());
    }

    if (server.GetSupportCenterManager() && server.GetSupportCenterManager()->GetLegacyPriorityManager()) {
        result["call_priority_user"] = server.GetSupportCenterManager()->GetLegacyPriorityManager()->IsPrioritized(UserId);
    }

    NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> blackboxDataFuture;
    if (auto client = GetBlackboxClient(server); client && (traits & NUserReport::EReportTraits::ReportYPassport)) {
        blackboxDataFuture = client->UidInfoRequest(Uid, "8.8.8.8");
        subrequests.push_back(blackboxDataFuture.IgnoreResult());
    }

    return NThreading::WaitExceptionOrAll(subrequests)
            .Apply([result, traits, realtimeDataFuture, passportFuture, passportRevision, licenseFuture, licenseRevision, socialDataFuture, blackboxDataFuture](const NThreading::TFuture<void>& waiter) mutable {
                waiter.Wait();

                if (realtimeDataFuture.Initialized() && realtimeDataFuture.HasValue()) {
                    auto userEventsData = realtimeDataFuture.GetValue();
                    result["realtime_data"] = userEventsData.SerializeToJson();
                } else if (realtimeDataFuture.HasException()) {
                    ERROR_LOG << NThreading::GetExceptionMessage(realtimeDataFuture) << Endl;
                }

                NJson::TJsonValue privateData(NJson::JSON_MAP);
                SetPrivateDataResponse(privateData, "passport", passportFuture, passportRevision, traits);
                SetPrivateDataResponse(privateData, "driving_license", licenseFuture, licenseRevision, traits);
                if (!privateData.GetMap().empty()) {
                    result["documents"] = privateData;
                }

                if (socialDataFuture.Initialized() && socialDataFuture.HasValue()) {
                    auto userSocialData = socialDataFuture.GetValue();
                    if (userSocialData.IsBroken()) {
                        ERROR_LOG << "could not correctly acquire social data for user" << Endl;
                    }
                    result["social"] = userSocialData.SerializeToJson();
                } else if (socialDataFuture.HasException()) {
                    ERROR_LOG << NThreading::GetExceptionMessage(socialDataFuture) << Endl;
                }

                if (blackboxDataFuture.Initialized() && blackboxDataFuture.HasValue()) {
                    auto userBlackboxData = blackboxDataFuture.GetValue();
                    result["blackbox"] = BuildBlackboxDataReport(userBlackboxData);
                } else if (blackboxDataFuture.HasException()) {
                    ERROR_LOG << NThreading::GetExceptionMessage(blackboxDataFuture) << Endl;
                }

                return result;
            });
}

NJson::TJsonValue TDriveUserData::GetFullReportSync(const NUserReport::TReportTraits traits, const NDrive::IServer& server) const {
    auto reportFuture = GetFullReport(traits, server);
    return reportFuture.GetValue(TDuration::Max());
}

bool TDriveUserData::IsMTAllowed() const {
    return !HasATMark;
}

bool TDriveUserData::HasPassportNamesHashData() const {
    return FirstName || LastName || PName;
}

NJson::TJsonValue TDriveUserData::GetPublicReport() const {
    return GetReport(NUserReport::ReportPublic, /*reportPublicStatus=*/true);
}

NJson::TJsonValue TDriveUserData::GetSearchReport() const {
    return GetReport();
}

NJson::TJsonValue TDriveUserData::GetChatReport() const {
    return GetReport(NUserReport::ReportChat);
}

TString TDriveUserData::GetHRReport() const {
    TString name;
    if (!name) {
        name = GetFullName();
    }
    if (!name) {
        name = Login;
    }
    return Sprintf("<a href=\"%s\">%s (%s)</a>", TCarsharingUrl().ClientInfo(UserId).data(), name.data(), Phone.data());
}

TString TDriveUserData::GetDisplayName() const {
    return !!FirstName ? FirstName : Login;
}

TString TDriveUserData::GetFullName() const {
    return Strip(LastName + " " + FirstName + " " + PName);
}

TString TDriveUserData::GetFullNameOrLogin() const {
    auto name = GetFullName();
    return name ? name : Login;
}

TString TDriveUserData::GetObjectId(const TDriveUserData& object) const {
    return object.GetUserId();
}

TString TDriveUserData::GetPublicStatus() const {
    if (Status == NDrive::UserStatusFastRegistered && FirstRiding) {
        return NDrive::UserStatusActive;
    }
    return Status;
}

bool TDriveUserData::IsStaffAccount() const {
    /*
        According to https://wiki.yandex-team.ru/passport/uids/
    */
    return (Uid >= "1120000000000000" && Uid < "1130000000000000" && Uid.size() == 16);
}

TString TDriveUserData::GetObfuscatedLogin() const {
    if (Login.empty()) {
        return Login;
    }
    TSet<ui32> showAt;
    showAt.emplace(0);
    showAt.emplace(Login.size() - 1);
    if (Login.size() >= 5) {
        showAt.emplace(1);
    }
    if (Login.size() >= 8) {
        for (size_t i = 2; i < Login.size(); ++i) {
            if (!IsAlnum(Login.at(i)) && !IsAlpha(Login.at(i))) {
                showAt.emplace(i);
                showAt.emplace(i-1);
            }
        }
    }
    TStringBuilder sb;
    for (size_t i = 0; i < Login.size(); ++i) {
        if (showAt.contains(i)) {
            sb << Login.at(i);
        } else {
            sb << '*';
        }
    }
    return sb;
}

TDriveUserData& TDriveUserData::UpdateStatus(const TString& newStatus) {
    SetStatus(newStatus);
    if (newStatus == NDrive::UserStatusActive && GetApprovedAt() == TInstant::Zero()) {
        SetApprovedAt(Now());
    }
    return *this;
}

TString TDriveUserData::GetPaymethodsUid(const ISettings& settings) const {
    TString uid = GetUid();
    TString userEnv = GetEnvironment();

    bool needTestPassport = settings.GetValueDef<bool>("billing.testing.use_personal_passport", false);
    needTestPassport &= userEnv == "testing";
    if (!needTestPassport) {
        settings.GetValue("billing.sponsor_id", uid);
    }
    return uid;
}

void TUsersDBConfig::Init(const TYandexConfig::Section* section) {
    IsSearchEnabled = section->GetDirectives().Value<bool>("IsSearchEnabled", IsSearchEnabled);
    SearchViaDatabaseVector = section->GetDirectives().Value<TString>("SearchViaDatabaseVector", SearchViaDatabaseVector);
    BlackboxUrl = section->GetDirectives().Value<TString>("BlackboxUrl", BlackboxUrl);
    SelfTvmId = section->GetDirectives().Value<ui32>("SelfTvmId", SelfTvmId);
    TestBlackboxUrl = section->GetDirectives().Value<TString>("TestBlackboxUrl", TestBlackboxUrl);
    TestSelfTvmId = section->GetDirectives().Value<ui32>("TestSelfTvmId", TestSelfTvmId);
    auto allChildren = section->GetAllChildren();
    {
        auto p = allChildren.find("RegistrationGeo");
        if (p != allChildren.end()) {
            const TYandexConfig::TSectionsMap children = p->second->GetAllChildren();
            for (auto&& itSection : children) {
                TRegistrationGeoData location;
                location.Center.Y = itSection.second->GetDirectives().Value<double>("Lat", location.Center.Y);
                location.Center.X = itSection.second->GetDirectives().Value<double>("Lon", location.Center.X);
                location.LocationName = itSection.second->GetDirectives().Value<TString>("LocationName", location.LocationName);
                RegistrationGeoCenters.push_back(std::move(location));
            }
        }
    }
    {
        TString tmpAttrs;
        tmpAttrs = section->GetDirectives().Value<TString>("BBAdminReportAttributes", tmpAttrs);
        if (tmpAttrs) {
            BBAdminReportAttributes = SplitString(tmpAttrs, ",");
        }
    }
}

void TUsersDBConfig::ToString(IOutputStream& os) const {
    os << "IsSearchEnabled: " << IsSearchEnabled << Endl;
    os << "SearchViaDatabaseVector: " << SearchViaDatabaseVector << Endl;
    os << "<RegistrationGeo>" << Endl;
    for (auto&& geo : RegistrationGeoCenters) {
        os << "<" << geo.LocationName << ">" << Endl;
        os << "Lat: " << geo.Center.Y << Endl;
        os << "Lon: " << geo.Center.X << Endl;
        os << "LocationName: " << geo.LocationName << Endl;
        os << "</" << geo.LocationName << ">" << Endl;
    }
    os << "</RegistrationGeo>" << Endl;
    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
    os << "BBAdminReportAttributes: " << JoinStrings(BBAdminReportAttributes, ",") << Endl;
    os << "BlackboxUrl: " << BlackboxUrl << Endl;
    os << "SelfTvmId: " << SelfTvmId << Endl;
    os << "TestBlackboxUrl: " << TestBlackboxUrl << Endl;
    os << "TestSelfTvmId: " << TestSelfTvmId << Endl;
}
