#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/entity/search_request.h>
#include <drive/backend/roles/user.h>

#include <drive/library/cpp/threading/concurrent_cache.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/metasearch/simple/config.h>
#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

class TAreasDB;
class TBasicSearchIndex;
class IPrivateDataJsonCallback;

namespace NBlackbox2 {
    class TOptions;
}

namespace NDrive {
    class TBlackboxClient;
}

namespace NUserReport {
    using TReportTraits = ui64;
    static const TReportTraits NoTraits = 0;
    enum EReportTraits : ui64 {
        ReportId = 1 << 0,
        ReportPhone = 1 << 1,
        ReportNames = 1 << 2,
        ReportStatus = 1 << 3,
        ReportIsFirstRiding = 1 << 4,
        ReportLogin = 1 << 5,
        ReportPassportUid = 1 << 6,
        ReportEmail = 1 << 8,
        ReportYang = 1 << 9,
        ReportRegistrationDates = 1 << 10,
        ReportAllowedTransmissionTypes = 1 << 11,
        ReportPreliminaryPayments = 1 << 12,
        ReportBillingBase = 1 << 13,

        ReportPassportNames = 1 << 14,
        ReportPassportBirthPlace = 1 << 15,
        ReportPassportCitizenship = 1 << 16,
        ReportPassportGender = 1 << 17,
        ReportPassportNumber = 1 << 18,
        ReportPassportSubdivisionCode = 1 << 19,
        ReportPassportBirthDate = 1 << 20,
        ReportPassportRegistration = 1 << 21,
        ReportPassportValidityDates = 1 << 22,

        ReportDrivingLicenseNames = 1 << 23,
        ReportDrivingLicenseNumbers = 1 << 24,
        ReportDrivingLicenseCategories = 1 << 25,
        ReportDrivingLicenseBirthDate = 1 << 26,
        ReportDrivingLicenseExperienceFrom = 1 << 27,
        ReportDrivingLicenseIssueDates = 1 << 28,
        ReportDrivingLicenseCategoriesBDates = 1 << 29,
        ReportDrivingLicenseCountry = 1 << 30,

        ReportPhotos = (ui64)1 << 31,

        ReportRealtimeLocation = (ui64)1 << 32,
        ReportRealtimeClientAppInfo = (ui64)1 << 33,
        ReportRequestsHistory = (ui64)1 << 34,

        ReportHideNoTags = (ui64)1 << 35,
        ReportSocialData = (ui64)1 << 36,
        ReportYPassport = (ui64)1 << 37,

        ReportDeletionStatus = (ui64)1 << 38,

        ReportFirstName = (ui64)1 << 39,
        ReportLastName = (ui64)1 << 40,
        ReportMiddleName = (ui64)1 << 41
    };
    static constexpr TReportTraits ReportNone = Min<TReportTraits>();
    static constexpr TReportTraits ReportAll = Max<TReportTraits>();
    static constexpr TReportTraits ReportPublic = EReportTraits::ReportStatus | EReportTraits::ReportPreliminaryPayments
        | EReportTraits::ReportId | EReportTraits::ReportLogin | EReportTraits::ReportPhone | EReportTraits::ReportEmail
        | EReportTraits::ReportNames | EReportTraits::ReportFirstName | EReportTraits::ReportLastName | EReportTraits::ReportMiddleName;
    static constexpr TReportTraits ReportPublicId = ReportPublic | EReportTraits::ReportId;
    static constexpr TReportTraits ReportPassport = EReportTraits::ReportPassportNames | EReportTraits::ReportPassportBirthPlace | EReportTraits::ReportPassportCitizenship |
        EReportTraits::ReportPassportGender | EReportTraits::ReportPassportNumber | EReportTraits::ReportPassportSubdivisionCode | EReportTraits::ReportPassportBirthDate |
        EReportTraits::ReportPassportRegistration | EReportTraits::ReportPassportValidityDates;
    static constexpr TReportTraits ReportDrivingLicense = EReportTraits::ReportDrivingLicenseNames | EReportTraits::ReportDrivingLicenseNumbers |
        EReportTraits::ReportDrivingLicenseCategories | EReportTraits::ReportDrivingLicenseBirthDate | EReportTraits::ReportDrivingLicenseExperienceFrom |
        EReportTraits::ReportDrivingLicenseIssueDates | EReportTraits::ReportDrivingLicenseCategoriesBDates | ReportDrivingLicenseCountry;
    static constexpr TReportTraits ReportGDPR = ReportNames | ReportEmail | ReportPhone | ReportRegistrationDates;
    static constexpr TReportTraits ReportRealtimeData = ReportRealtimeLocation | ReportRealtimeClientAppInfo;
    static constexpr TReportTraits ReportChat = ReportId | ReportNames | ReportLogin;
    static constexpr TReportTraits ReportUserSignal = EReportTraits::ReportNames | EReportTraits::ReportStatus;
}

class TUserDrivingLicenseData {
    R_OPTIONAL(TString, FirstName);
    R_OPTIONAL(TString, LastName);
    R_OPTIONAL(TString, MiddleName);
    R_OPTIONAL(TString, NumberFront);
    R_OPTIONAL(TString, NumberBack);
    R_OPTIONAL(TString, PrevLicenseNumberFront);
    R_OPTIONAL(TString, PrevLicenseNumber);
    R_OPTIONAL(TString, Categories);
    R_OPTIONAL(TString, CategoriesBack);
    R_OPTIONAL(TString, BirthDate);
    R_OPTIONAL(TString, Country);
    R_OPTIONAL(TString, BackCountry);
    R_OPTIONAL(TString, IssuedBy);

    R_OPTIONAL(TString, ExperienceFromStr);
    R_OPTIONAL(TString, PrevLicenseIssueDateStrFront);
    R_OPTIONAL(TString, PrevLicenseIssueDateStr);
    R_OPTIONAL(TString, CategoriesBValidFromDateStr);
    R_OPTIONAL(TString, SpecialCategoryBDateStrFront);
    R_OPTIONAL(TString, SpecialCategoryBDateStrBack);

    R_OPTIONAL(TInstant, IssueDate);
    R_OPTIONAL(TInstant, CategoriesBValidToDateFront);
    R_OPTIONAL(TInstant, CategoriesBValidToDate);

public:
    const TString& GetBirthDate() const {
        return BirthDate.GetOrElse(Default<TString>());
    }
    const TString& GetCountry() const {
        return Country.GetOrElse(Default<TString>());
    }
    const TString& GetBackCountry() const {
        return BackCountry.GetOrElse(Default<TString>());
    }
    const TString& GetFirstName() const {
        return FirstName.GetOrElse(Default<TString>());
    }
    const TString& GetLastName() const {
        return LastName.GetOrElse(Default<TString>());
    }
    const TString& GetMiddleName() const {
        return MiddleName.GetOrElse(Default<TString>());
    }
    TInstant GetCategoriesBValidToDate() const {
        return CategoriesBValidToDate.GetOrElse({});
    }
    TInstant GetIssueDate() const {
        return IssueDate.GetOrElse({});
    }

    NJson::TJsonValue SerializeToJson(const NUserReport::TReportTraits& traits) const;
    NJson::TJsonValue SerializeToDatasyncJson() const;
    NJson::TJsonValue SerializeBackToYang(const bool isVerified) const;
    NJson::TJsonValue SerializeFrontToYang(const bool isVerified) const;

    TString GetNumber() const;
    TInstant GetExperienceFrom() const;
    TInstant GetPrevLicenseIssueDate() const;
    TInstant GetCategoriesBValidFromDate() const;
    TInstant GetUserBirthDate() const;

    bool ParseFromDatasync(const NJson::TJsonValue& report);
    bool ParseFromYang(const NJson::TJsonValue& back, const NJson::TJsonValue& front, TMessagesCollector& errors, const bool isPatch = false);
    bool Parse(const NJson::TJsonValue& report);
    bool IsDatasyncCompatible() const;
    bool IsForeign() const;

    void Patch(const TUserDrivingLicenseData& other);

public:
    DECLARE_FIELDS(
        Field(FirstName, "first_name"),
        Field(LastName, "last_name"),
        Field(MiddleName, "middle_name"),
        Field(NumberFront, "number_front"),
        Field(NumberBack, "number_back"),
        Field(PrevLicenseNumberFront, "prev_license_number_front"),
        Field(PrevLicenseNumber, "prev_licence_number"),
        Field(Categories, "categories"),
        Field(CategoriesBack, "categories_back"),
        Field(BirthDate, "birth_date"),
        Field(Country, "front_country"),
        Field(BackCountry, "back_country"),
        Field(IssuedBy, "issued_by"),
        Field(IssueDate, "issue_date"),
        Field(ExperienceFromStr, "experience_from"),
        Field(PrevLicenseIssueDateStrFront, "prev_licence_issue_date_front"),
        Field(PrevLicenseIssueDateStr, "prev_licence_issue_date"),
        Field(SpecialCategoryBDateStrFront, "special_category_b_date_front"),
        Field(SpecialCategoryBDateStrBack, "special_category_b_date_back"),
        Field(CategoriesBValidFromDateStr, "categories_b_valid_from_date"),
        Field(CategoriesBValidToDateFront, "categories_b_valid_to_date_front"),
        Field(CategoriesBValidToDate, "categories_b_valid_to_date")
    );
};

class TUserPassportRegistrationData {
    R_OPTIONAL(TString, Apartment);
    R_OPTIONAL(TString, Housing);
    R_OPTIONAL(TString, Letter);
    R_OPTIONAL(TString, House);
    R_OPTIONAL(TString, Street);
    R_OPTIONAL(TString, Area);
    R_OPTIONAL(TString, Locality);
    R_OPTIONAL(TString, Region);
    R_OPTIONAL(TInstant, ExpirationDate);
    R_OPTIONAL(TString, Type);

public:
    bool ParseFromDatasync(const NJson::TJsonValue& report);
    bool ParseFromYang(const NJson::TJsonValue& registration, TMessagesCollector& errors);

public:
    DECLARE_FIELDS(
        Field(Apartment, "registration_apartment"),
        Field(Housing, "registration_housing"),
        Field(Letter, "registration_letter"),
        Field(House, "registration_house"),
        Field(Street, "registration_street"),
        Field(Area, "registration_area"),
        Field(Locality, "registration_locality"),
        Field(Region, "registration_region"),
        Field(ExpirationDate, "registration_expiration_date"),
        Field(Type, "registration_type")
    );
};

class TUserPassportData {
    R_OPTIONAL(TString, FirstName);
    R_OPTIONAL(TString, LastName);
    R_OPTIONAL(TString, MiddleName);
    R_OPTIONAL(TString, BirthPlace);
    R_OPTIONAL(TString, Citizenship);
    R_OPTIONAL(TString, Gender);
    R_OPTIONAL(TString, Number);
    R_OPTIONAL(TString, SubdivisionCode);
    R_OPTIONAL(TString, BirthDate);
    R_OPTIONAL(TString, BiographicalCountry);
    R_OPTIONAL(TString, RegistrationCountry);
    R_OPTIONAL(TString, IssuedBy);
    R_OPTIONAL(TInstant, IssueDate);
    R_OPTIONAL(TInstant, ExpirationDate);

    R_FIELD(TUserPassportRegistrationData, Registration);

public:
    const TString& GetBiographicalCountry() const {
        return BiographicalCountry.GetOrElse(Default<TString>());
    }
    const TString& GetRegistrationCountry() const {
        return RegistrationCountry.GetOrElse(Default<TString>());
    }
    const TString& GetBirthDate() const {
        return BirthDate.GetOrElse(Default<TString>());
    }
    const TString& GetFirstName() const {
        return FirstName.GetOrElse(Default<TString>());
    }
    const TString& GetLastName() const {
        return LastName.GetOrElse(Default<TString>());
    }
    const TString& GetMiddleName() const {
        return MiddleName.GetOrElse(Default<TString>());
    }
    const TString& GetNumber() const {
        return Number.GetOrElse(Default<TString>());
    }
    TInstant GetIssueDate() const {
        return IssueDate.GetOrElse({});
    }
    TInstant GetUserBirthDate() const;

    TString GetNamesForHash() const;
    bool HasNamesForHash() const;

    bool ParseFromDatasync(const NJson::TJsonValue& report);
    bool ParseFromYang(const NJson::TJsonValue& biographical, const NJson::TJsonValue& registration, TMessagesCollector& errors, const bool isPatch = false);
    bool Parse(const NJson::TJsonValue& report);
    bool IsDatasyncCompatible() const;

    void Patch(const TUserPassportData& other);

    NJson::TJsonValue SerializeToDatasyncJson() const;
    NJson::TJsonValue SerializeToJson(const NUserReport::TReportTraits& traits) const;
    NJson::TJsonValue SerializeBioToYang(const bool isVerified) const;
    NJson::TJsonValue SerializeRegToYang(const bool isVerified) const;

public:
    DECLARE_FIELDS(
        Field(FirstName, "first_name"),
        Field(LastName, "last_name"),
        Field(MiddleName, "middle_name"),
        Field(BirthPlace, "birth_place"),
        Field(Citizenship, "citizenship"),
        Field(Gender, "gender"),
        Field(Number, "doc_value"),
        Field(SubdivisionCode, "subdivision_code"),
        Field(BirthDate, "birth_date"),
        Field(BiographicalCountry, "biographical_country"),
        Field(RegistrationCountry, "registration_country"),
        Field(IssuedBy, "issued_by"),
        Field(IssueDate, "issue_date"),
        Field(ExpirationDate, "expiration_date")
    );
};

namespace NDrive {
    class TExternalUser: public TUserContacts {
    private:
        using TBase = TUserContacts;

    public:
        R_FIELD(TString, FirstName);
        R_FIELD(TString, LastName);
        R_FIELD(TString, PName);

    public:
        using TBase::TBase;
    };

    static const TString UserStatusActive       = "active";
    static const TString UserStatusBlocked      = "blocked";
    static const TString UserStatusDeleted      = "deleted";
    static const TString UserStatusExternal     = "external";
    static const TString UserStatusFastRegistered   = "fastregistered";
    static const TString UserStatusOnboarding   = "onboarding";
    static const TString UserStatusPassive      = "passive";
    static const TString UserStatusRejected     = "rejected";
    static const TString UserStatusScreening    = "screening";
    static const TString UserStatusStaff        = "staff";
    static const TString UserStatusReferrer     = "referrer";

    bool IsPreOnboarding(const TString& status);

    static const TSet<TString> AcceptableUserStatuses = {
        NDrive::UserStatusActive,
        NDrive::UserStatusBlocked,
        NDrive::UserStatusExternal,
        NDrive::UserStatusFastRegistered,
        NDrive::UserStatusOnboarding,
        NDrive::UserStatusPassive,
        NDrive::UserStatusRejected,
        NDrive::UserStatusScreening,
        NDrive::UserStatusStaff,
        NDrive::UserStatusReferrer
    };
}

class TDriveUserData: public NDrive::TExternalUser {
private:
    using TBase = NDrive::TExternalUser;

public:
    using TFieldMask = ui32;
    using TFieldValues = TVector<TString>;

    enum class EField: ui32 {
        FirstName = 1 << 0,
        LastName = 1 << 1,
        PName = 1 << 2,
        Phone = 1 << 3
    };

private:
    R_FIELD(TString, FirstName);
    R_FIELD(TString, LastName);
    R_FIELD(TString, PName);
    R_FIELD(TString, Address);
    R_FIELD(TString, Environment);
    R_FIELD(TString, Status, NDrive::UserStatusOnboarding);
    R_FIELD(TString, Login);
    R_FIELD(TString, RegistrationGeo, "moscow");
    R_FIELD(TInstant, JoinedAt, TInstant::Zero());
    R_FIELD(TInstant, ApprovedAt, TInstant::Zero());
    R_FIELD(TInstant, Timestamp);
    R_FIELD(TString, PassportDatasyncRevision);
    R_FIELD(TString, DrivingLicenseDatasyncRevision);
    R_FIELD(TString, PassportNamesHash);
    R_FIELD(TString, PassportNumberHash);
    R_FIELD(TString, DrivingLicenseNumberHash);
    R_FIELD(bool, HasATMark, false);
    R_FIELD(bool, EMailVerified, false);
    R_FIELD(bool, FirstRiding, true);
    R_FIELD(bool, PhoneVerified, false);
    R_FIELD(bool, NewAccount, false);

public:
    class TDriveUserDataDecoder: public TBaseDecoder {
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Address, -1);
        R_FIELD(i32, Phone, -1);
        R_FIELD(i32, FirstName, -1);
        R_FIELD(i32, LastName, -1);
        R_FIELD(i32, PName, -1);
        R_FIELD(i32, Status, -1);
        R_FIELD(i32, FirstRiding, -1);
        R_FIELD(i32, Login, -1);
        R_FIELD(i32, Uid, -1);
        R_FIELD(i32, PhoneVerified, -1);
        R_FIELD(i32, EMailVerified, -1);
        R_FIELD(i32, Email, -1);
        R_FIELD(i32, RegistrationGeo, -1);
        R_FIELD(i32, JoinedAt, -1);
        R_FIELD(i32, ApprovedAt, -1);
        R_FIELD(i32, PassportDatasyncRevision, -1);
        R_FIELD(i32, DrivingLicenseDatasyncRevision, -1);
        R_FIELD(i32, PassportNamesHash, -1);
        R_FIELD(i32, PassportNumberHash, -1);
        R_FIELD(i32, DrivingLicenseNumberHash, -1);
        R_FIELD(i32, HasATMark, -1);
        R_FIELD(i32, Environment, -1);

    public:
        TDriveUserDataDecoder() = default;
        TDriveUserDataDecoder(const TMap<TString, ui32>& decoderBase);
    };
    using TDecoder = TDriveUserDataDecoder;

public:
    bool Patch(const NJson::TJsonValue& info, const bool isNewUser, const NUserReport::TReportTraits& traits);

public:
    TDriveUserData() = default;
    TDriveUserData(const TString& uid, const TString& login)
        : TBase(uid)
        , Login(login)
        , NewAccount(true)
    {
        SetUserId("uuid_generate_v4()");
    }

    explicit operator bool() const {
        return !UserId.empty();
    }

    TDriveUserData& UpdateStatus(const TString& newStatus);

    bool IsMTAllowed() const;
    bool HasPassportNamesHashData() const;

    void Index(TBasicSearchIndex& index) const;

    bool DeserializeWithDecoder(const TDriveUserDataDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    void DoBuildReportItem(NJson::TJsonValue& item) const;

    bool Parse(const NStorage::TTableRecord& row);
    NStorage::TTableRecord SerializeToTableRecord() const;

    NJson::TJsonValue GetReport(const NUserReport::TReportTraits traits = NUserReport::ReportPublicId, bool reportPublicStatus = false) const;
    NJson::TJsonValue GetPublicReport() const;
    NJson::TJsonValue GetSearchReport() const;
    NJson::TJsonValue GetChatReport() const;

    TFieldMask GetFilledFields() const;

    TString GetHRReport() const;
    TString GetFullName() const;
    TString GetFullNameOrLogin() const;
    TString GetShortName() const;
    TString GetDisplayName() const;
    TString GetObjectId(const TDriveUserData& object) const;
    TString GetPublicStatus() const;

    bool IsStaffAccount() const;

    TString GetObfuscatedLogin() const;

    NThreading::TFuture<NJson::TJsonValue> GetFullReport(const NUserReport::TReportTraits traits, const NDrive::IServer& server) const;
    NJson::TJsonValue GetFullReportSync(const NUserReport::TReportTraits traits, const NDrive::IServer& server) const;
    TString GetPaymethodsUid(const ISettings& settings) const;
    TAtomicSharedPtr<NDrive::TBlackboxClient> GetBlackboxClient(const NDrive::IServer& server) const;
};
using TDriveUsers = TVector<TDriveUserData>;

class TDriveUserDataHistoryManager: public TDatabaseHistoryManager<TDriveUserData> {
private:
    using TBase = TDatabaseHistoryManager<TDriveUserData>;

public:
    TDriveUserDataHistoryManager(const IHistoryContext& context)
        : TBase(context, "drive_user_data_history")
    {
    }
};

class TUsersDBConfig {
public:
    struct TRegistrationGeoData {
        TGeoCoord Center;
        TString LocationName;
    };

public:
    R_READONLY(TDuration, DefaultCacheLifetime, TDuration::Seconds(1000));
    R_READONLY(TVector<TRegistrationGeoData>, RegistrationGeoCenters);
    R_READONLY(bool, IsSearchEnabled, false);
    R_READONLY(TString, SearchViaDatabaseVector);
    R_READONLY(TString, BlackboxUrl, "https://blackbox.yandex.net:443/blackbox");
    R_READONLY(ui32, SelfTvmId, 2000615);
    R_READONLY(TString, TestBlackboxUrl, "http://pass-test.yandex.ru/blackbox");
    R_READONLY(ui32, TestSelfTvmId, 2010062);
    R_READONLY(TVector<TString>, DeletedUserTags, {"user_deleted_finally"});
    R_READONLY(NSimpleMeta::TConfig, RequestConfig);

    R_READONLY(TVector<TString>, BBAdminReportAttributes, TVector<TString>({
        "1007",
        "1008",
        "1009",
        "1015"
    }));

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TUsersDB
    : public IAutoActualization
    , public TDatabaseEntityManager<TDriveUserData, true, TString, TUuidFilter>
{
private:
    using TBase = TDatabaseEntityManager<TDriveUserData, true, TString, TUuidFilter>;

public:
    using THistoryManager = TDriveUserDataHistoryManager;
    using TRecordType = TDriveUserData;
    using TOptionalUser = TMaybe<TDriveUserData>;
    using TOptionalUsers = TMaybe<TDriveUsers>;
    using TOptionalUserId = TMaybe<TString>;

public:
    enum class TUserDeleteStatus {
        Unknown /* "internal_error" */,
        HasDebt,
        HasActiveSession,
        NeedsHumanVerification,
        DeleteInProgress,
        ReadyToDelete,
        Empty
    };

public:
    TUsersDB(const ITagsHistoryContext& context, const TUserTagsManager* userTagsManager = nullptr, const TUsersDBConfig& config = TUsersDBConfig());
    ~TUsersDB();

    virtual bool DoStart() override;
    virtual bool DoStop() override;

    TAsyncDelivery::TPtr GetAsyncDelivery() const {
        return AD;
    }

    const THistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }

    const TUserRolesDB& GetRoles() const {
        return UserRolesDB;
    }

    TUserRolesDB& GetRoles() {
        return UserRolesDB;
    }

    bool IsSearchEnabled() const {
        return Config.GetIsSearchEnabled();
    }

    const TUsersDBConfig& GetConfig() const {
        return Config;
    }

    NBlackbox2::TOptions ConstructAdminInfoOptions() const;

    bool ActualizeRegistrationGeo(const TString& userId, const TString& operatorUserId, const TGeoCoord& userLocation, NDrive::TEntitySession& session) const;
    bool SelectUsers(const TString& fieldName, const TVector<TString>& keys, TVector<TString>& result, NDrive::TEntitySession& session, bool doGetDeleted = true) const;
    bool CheckEmailExists(const TString& email) const;

    TOptionalUserId GetUserIdByLogin(const TString& login, NDrive::TEntitySession& session) const;
    TOptionalUserId GetUserIdByUidDirect(const TString& uid, NDrive::TEntitySession& session) const;
    TString GetUserIdByUid(const TString& uid) const;
    TFetchResult FetchUsersByPhone(const TSet<TString>& phones) const;
    TVector<TString> GetDocumentOwnersByHash(const TString& hash) const;
    TMaybe<TVector<TString>> GetDocumentOwnersByHash(const TString& hash, NDrive::TEntitySession& session) const;
    TMaybe<TDriveUsers> GetDocumentOwnersObjectsByHash(const TString& hash, NDrive::TEntitySession& session) const;
    TOptionalUsers GetSamePersons(const TString& userId, NDrive::TEntitySession& session) const;
    TMaybe<bool> CheckDuplicates(const TVector<TString>& userIds, NDrive::TEntitySession& tx) const;
    static bool CompareUsersByFreshness(const TDriveUserData& left, const TDriveUserData& right);

    TOptionalUser GetCachedObject(const TString& userId, TInstant statementDeadline = TInstant::Zero()) const noexcept(false);
    TOptionalUser GetUserByPhone(const TString& phone, NDrive::TEntitySession& session) const;
    TOptionalUsers GetUsersByPhone(const TString& phone, NDrive::TEntitySession& session) const;
    TOptionalUsers GetUsersByEmail(const TString& email, NDrive::TEntitySession& session) const;
    TOptionalUser RestoreUser(const TString& userId, NDrive::TEntitySession& session) const;
    TOptionalUser RegisterNewUser(const TString& operatorUserId, const TString& uid, const TString& userName, NDrive::TEntitySession& session, const TString& phoneNumber = "", const TString& email = "", bool phoneVerified = false, const TString& initialStatus = NDrive::UserStatusOnboarding, const TMaybe<TString>& onLinkStatus = {}) const;
    TOptionalUser RegisterExternalUser(const TString& operatorUserId, NDrive::TEntitySession& tx, const NDrive::TExternalUser& externalUser) const;
    TOptionalUser FindOrRegisterExternal(const TString& operatorUserId, NDrive::TEntitySession& tx, const NDrive::TExternalUser& externalUser) const;
    TOptionalUsers FindUsersByPhone(const TString& phoneNumber, const TSet<TString>& allowedStatuses, TMaybe<bool> phoneVerified, NDrive::TEntitySession& session) const;
    TOptionalUsers SelectUsers(const TString& fieldName, TConstArrayRef<TString> keys, NDrive::TEntitySession& session, bool withDeleted = false) const;
    TOptionalUser UpdateUser(TDriveUserData user, const TString& operatorUserId, NDrive::TEntitySession& session, bool isNewUser = false) const;
    TOptionalUser DeleteUser(const TString& userId, const TString& operatorUserId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    TOptionalUser TryLinkToExistingUser(const TString& phoneNumber, const TString& email, NDrive::TEntitySession& tx) const;
    TMaybe<TUserDeleteStatus> CheckStatusBeforeDelete(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx, TVector<TString> deletionNames, TMaybe<TVector<TDBTag>> fetchedTags = {});

    TMaybe<bool> IsDeletedByTags(const TString& userId, NDrive::TEntitySession& session) const;
    TSet<TString> GetUsersDeletedByTags() const;

    bool SetChatToShow(const TString& userId, const TString& topicLink, const bool isClosable, const TString& operatorId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    bool DropChatShow(const TString& userId, const TString& operatorId, NDrive::TEntitySession& session) const;

    NStorage::IDatabase::TPtr GetDatabase() const;
    TVector<TString> GetMatchingIds(const TSearchRequest& searchRequest, const std::function<bool(const TString&)>& entityFilter) const;

private:
    TOptionalUser GetUserByColumn(const TString& columnName, const TString& columnValue, NDrive::TEntitySession& session) const;
    TOptionalUsers GetUsersByColumn(const TString& columnName, const TString& columnValue, NDrive::TEntitySession& session) const;
    TVector<TString> FetchIds(const NStorage::ITransaction::TPtr &transaction, NSQL::TQueryOptions& queryOptions, const std::function<bool(const TString&)>& entityFilter, const TSearchRequest& searchRequest) const;

    TString GetMainId(const TDriveUserData& e) const override {
        return e.GetUserId();
    }
    TString GetTableName() const override {
        return "\"user\"";
    }
    bool GetStartFailIsProblem() const override {
        return false;
    }
    bool BuildIndex();
    bool Refresh() override;

private:
    NStorage::IDatabase::TPtr Database;
    THistoryManager HistoryManager;
    const TUserTagsManager* UserTagsManager;
    TUserRolesDB UserRolesDB;
    TAtomicSharedPtr<TAsyncDelivery> AD;
    const TUsersDBConfig Config;
    THolder<TBasicSearchIndex> Index;
    TThreadPool IndexUpdateThread;

    TOptionalObjectEventId<TUserRole> LastEventId;

    mutable NUtil::TConcurrentCache<TString, TDriveUserData> ObjectCache;
    mutable NUtil::TConcurrentCache<TString, TString> UidCache;
};
