#include "user_documents.h"

#include <drive/backend/abstract/base.h>

#include <rtline/library/json/parse.h>

#include <rtline/util/json_processing.h>

template <>
NJson::TJsonValue NJson::ToJson(const NUserDocument::EType& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NUserDocument::EType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
NJson::TJsonValue NJson::ToJson(const TRecognitionConfidenceData::EDocumentField& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRecognitionConfidenceData::EDocumentField& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

void TRecognitionConfidenceData::AddField(const EDocumentField field, const double confidence) {
    Confidences[field] = confidence;
}

NJson::TJsonValue TRecognitionConfidenceData::SerializeToJson() const {
    NJson::TJsonValue fields = NJson::JSON_ARRAY;
    for (auto&& it : Confidences) {
        NJson::TJsonValue singleField;
        singleField["field"] = ToString(it.first);
        singleField["confidence"] = it.second;
        fields.AppendValue(std::move(singleField));
    }
    NJson::TJsonValue result;
    result["confidences"] = std::move(fields);
    NJson::InsertField(result, "recognition_failed", RecognitionFailed);
    NJson::InsertNonNull(result, "quasi_gibdd_score", OptionalQuasiGibddScore());
    NJson::InsertNonNull(result, "from_screen_model_score", OptionalFromScreenScore());
    NJson::InsertNonNull(result, "quasi_fms_score", OptionalQuasiFmsScore());
    NJson::InsertNonNull(result, "bad_format_model_score", OptionalBadFormatScore());
    return result;
}

bool TRecognitionConfidenceData::DeserializeFromJson(const NJson::TJsonValue& json) {
    Confidences.clear();
    if (json.Has("confidences")) {
        if (!json["confidences"].IsArray()) {
            return false;
        }
        for (auto&& elem : json["confidences"].GetArray()) {
            EDocumentField field;
            double confidence;
            JREAD_FROM_STRING(elem, "field", field);
            JREAD_DOUBLE(elem, "confidence", confidence);
            Confidences[field] = confidence;
        }
    }
    return
        NJson::ParseField(json, "recognition_failed", RecognitionFailed, false) &&
        NJson::ParseField(json, "quasi_fms_score", QuasiFmsScore, false) &&
        NJson::ParseField(json, "quasi_gibdd_score", QuasiGibddScore, false) &&
        NJson::ParseField(json, "from_screen_model_score", FromScreenScore, false) &&
        NJson::ParseField(json, "bad_format_model_score", BadFormatScore, false);
}

double TRecognitionConfidenceData::GetField(const EDocumentField field) const {
    auto it = Confidences.find(field);
    if (it != Confidences.end()) {
        return it->second;
    }
    return FIELD_DOES_NOT_EXIST;
}

NJson::TJsonValue TRecognitionConfidenceData::SerializePassportBiographical() const {
    NJson::TJsonValue result;
    JWRITE_DEF_NULL(result, "first_name", GetField(EDocumentField::FirstName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "last_name", GetField(EDocumentField::LastName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "middle_name", GetField(EDocumentField::MiddleName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "birth_date", GetField(EDocumentField::BirthDate), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "gender", GetField(EDocumentField::Gender), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "citizenship", GetField(EDocumentField::Citizenship), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "number", GetField(EDocumentField::PassportNumber), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "birth_place", GetField(EDocumentField::BirthPlace), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "subdivision_code", GetField(EDocumentField::PassportSubdivisionCode), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "country", GetField(EDocumentField::Country), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "issue_date", GetField(EDocumentField::IssueDate), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "expiration_date", GetField(EDocumentField::ExpirationDate), FIELD_DOES_NOT_EXIST);
    return result;
}

NJson::TJsonValue TRecognitionConfidenceData::SerializeDrivingLicenseFront() const {
    NJson::TJsonValue result;
    JWRITE_DEF_NULL(result, "number", GetField(EDocumentField::LicenseNumberFront), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "prev_licence_number", GetField(EDocumentField::LicensePrevNumberFront), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "prev_licence_issue_date", GetField(EDocumentField::LicensePrevIssueDateFront), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "first_name", GetField(EDocumentField::FirstName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "last_name", GetField(EDocumentField::LastName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "middle_name", GetField(EDocumentField::MiddleName), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "categories", GetField(EDocumentField::LicenseCategories), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "categories_b_valid_to_date", GetField(EDocumentField::LicenseCategoriesBValidToFront), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "country", GetField(EDocumentField::Country), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "birth_date", GetField(EDocumentField::BirthDate), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "experience_from", GetField(EDocumentField::LicenseExperienceFrom), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "issue_date", GetField(EDocumentField::IssueDate), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "special_category_b_date", GetField(EDocumentField::SpecialCategoryBDateFront), FIELD_DOES_NOT_EXIST);
    return result;
}

NJson::TJsonValue TRecognitionConfidenceData::SerializeDrivingLicenseBack() const {
    NJson::TJsonValue result;
    JWRITE_DEF_NULL(result, "number", GetField(EDocumentField::LicenseNumberBack), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "prev_licence_number", GetField(EDocumentField::LicensePrevNumber), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "categories", GetField(EDocumentField::LicenseCategoriesBack), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "country", GetField(EDocumentField::LicenseBackCountry), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "categories_b_valid_from_date", GetField(EDocumentField::LicenseCategoriesBValidFrom), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "categories_b_valid_to_date", GetField(EDocumentField::LicenseCategoriesBValidTo), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "prev_licence_issue_date", GetField(EDocumentField::LicensePrevIssueDate), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "experience_from", GetField(EDocumentField::LicenseExperienceFrom), FIELD_DOES_NOT_EXIST);
    JWRITE_DEF_NULL(result, "special_category_b_date", GetField(EDocumentField::SpecialCategoryBDateBack), FIELD_DOES_NOT_EXIST);
    return result;
}

TUserDocumentPhoto::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    Type = GetFieldDecodeIndex("type", decoderBase);
    UserId = GetFieldDecodeIndex("user_id", decoderBase);
    DocumentId = GetFieldDecodeIndex("document_id", decoderBase);
    VerificationStatus = GetFieldDecodeIndex("verification_status", decoderBase);
    SubmittedAt = GetFieldDecodeIndex("submitted_at", decoderBase);
    VerifiedAt = GetFieldDecodeIndex("verified_at", decoderBase);
    OriginChat = GetFieldDecodeIndex("origin_chat", decoderBase);
    RecognizerMeta = GetFieldDecodeIndex("recognizer_meta", decoderBase);
}

bool TUserDocumentPhoto::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE_DEF(decoder, values, DocumentId, "");
    READ_DECODER_VALUE_DEF(decoder, values, OriginChat, "");

    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, VerifiedAt);
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, SubmittedAt);

    READ_DECODER_VALUE_DEF(decoder, values, VerificationStatus, NUserDocument::EVerificationStatus::NotYetProcessed);
    READ_DECODER_VALUE_DEF(decoder, values, Type, NUserDocument::EType::Unknown);

    NJson::TJsonValue jsonMeta;
    READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, RecognizerMeta);
    if (jsonMeta.IsDefined() && jsonMeta.Has("confidences")) {
        TRecognitionConfidenceData conf;
        if (!conf.DeserializeFromJson(jsonMeta)) {
            return false;
        }
        SetRecognizerMeta(conf);
    }
    return true;
}

TUserDocumentPhoto::TUserDocumentPhoto(const TString& userId, NUserDocument::EType type)
    : Id(NUtil::CreateUUID())
    , Type(type)
    , SubmittedAt(Now())
    , UserId(userId)
{
}

bool TUserDocumentPhoto::Parse(const NStorage::TTableRecord& row) {
    Id = row.Get("id");

    if (!TryFromString(row.Get("type"), Type)) {
        return false;
    }

    UserId = row.Get("user_id");
    DocumentId = row.Get("document_id");

    TString verificationStatus = row.Get("verification_status");
    if (!!verificationStatus && !TryFromString(verificationStatus, VerificationStatus)) {
        return false;
    }

    if (row.Get("submitted_at") && !TInstant::TryParseIso8601(row.Get("submitted_at"), SubmittedAt)) {
        return false;
    }
    if (row.Get("verified_at") && !TInstant::TryParseIso8601(row.Get("verified_at"), VerifiedAt)) {
        return false;
    }

    {
        auto originChat = row.Get("origin_chat");
        if (originChat) {
            OriginChat = std::move(originChat);
        }
    }

    if (row.Get("recognizer_meta")) {
        NJson::TJsonValue json;
        if (!NJson::ReadJsonFastTree(row.Get("recognizer_meta"), &json)) {
            ERROR_LOG << "Could not parse json in 'recognizer_meta'" << Endl;
            return false;
        }
        TRecognitionConfidenceData conf;
        if (!conf.DeserializeFromJson(json)) {
            ERROR_LOG << "Could not parse recognizer meta object" << Endl;
            return false;
        }
        SetRecognizerMeta(std::move(conf));
    }

    return true;
}

NStorage::TTableRecord TUserDocumentPhoto::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("id", Id);
    result.Set("type", ToString(Type));
    result.Set("user_id", UserId);
    result.Set("submitted_at", SubmittedAt.ToString());

    if (DocumentId) {
        result.Set("document_id", DocumentId);
    }

    if (VerificationStatus != NUserDocument::EVerificationStatus::NotYetProcessed) {
        result.Set("verification_status", ToString(VerificationStatus));
    }

    if (VerifiedAt != TInstant::Zero()) {
        result.Set("verified_at", VerifiedAt.ToString());
    }

    result.Set("origin_chat", OriginChat);

    if (HasRecognizerMeta()) {
        result.Set("recognizer_meta", RecognizerMeta->SerializeToJson().GetStringRobust());
    }

    return result;
}

NJson::TJsonValue TUserDocumentPhoto::GetReport() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["submitted_at"] = SubmittedAt.Seconds();
    result["type"] = ToString(Type);

    if (VerificationStatus != NUserDocument::EVerificationStatus::NotYetProcessed) {
        result["verification_status"] = ToString(VerificationStatus);
    } else {
        result["verification_status"] = NJson::JSON_NULL;
    }

    if (VerifiedAt != TInstant::Zero()) {
        result["verified_at"] = VerifiedAt.Seconds();
    } else {
        result["verified_at"] = NJson::JSON_NULL;
    }

    bool newHandler = IsNewHandlerRequired();
    result["uri"] = BuildUri(newHandler);
    result["background_video_uri"] = BuildBackgroundVideoUri(newHandler);
    result["origin_chat"] = OriginChat;

    if (HasRecognizerMeta()) {
        result["recognizer_meta"] = RecognizerMeta->SerializeToJson();
    }

    return result;
}

TString TUserDocumentPhoto::BuildUri(bool newHandler) const {
    TString documentId = "null";
    if (DocumentId) {
        documentId = DocumentId;
    }
    if (newHandler) {
        return "/api/staff/user/document_data?user_id=" + UserId + "&photo_id=" + Id;
    } else {
        return "/api/admin/v1/users/" + UserId + "/documents/" + documentId + "/photos/" + Id + "/content/";
    }
}

TString TUserDocumentPhoto::BuildBackgroundVideoUri(bool newHandler) const {
    TString documentId = "null";
    if (DocumentId) {
        documentId = DocumentId;
    }
    if (newHandler) {
        return "/api/staff/user/document_data?user_id=" + UserId + "&photo_id=" + Id + "&background_video=true";
    } else {
        return "/api/admin/v1/users/" + UserId + "/documents/" + documentId + "/photos/" + Id + "/background-video/";
    }
}

bool TUserDocumentPhoto::IsNewHandlerRequired() const {
    if (!NDrive::HasServer() || !NDrive::GetServer().HasSettings()) {
        return false;
    }
    const auto& server = NDrive::GetServer();
    return server.GetSettings().GetValueDef<bool>("document_photo.new_handler", false);
}

bool TUserDocumentPhoto::IsResubmitNeeded() const {
    return (
        VerificationStatus == NUserDocument::EVerificationStatus::NeedInfo ||
        VerificationStatus == NUserDocument::EVerificationStatus::Unrecognizable ||
        VerificationStatus == NUserDocument::EVerificationStatus::VideoError
    );
}

TMap<TString, NUserDocument::EVerificationStatus> TUserDocumentPhoto::GetVerificationStatusMap() {
    auto values = GetEnumAllValues<NUserDocument::EVerificationStatus>();
    auto names = GetEnumAllCppNames<NUserDocument::EVerificationStatus>();
    if (values.size() != names.size()) {
        return {};
    }
    TMap<TString, NUserDocument::EVerificationStatus> result;
    for (size_t i = 0; i < values.size(); ++i) {
        TString key = names[i];
        if (auto parts = StringSplitter(key).SplitByString("::").ToList<TString>(); !parts.empty()) {
            key = parts.back();
        }
        result.emplace(key, values[i]);
    }
    return result;
}

TVector<TUserDocumentPhoto> TUserDocumentPhotosDB::GetPhotosOfUsersWhichAddedPhotosRecently(const TInstant& since) const {
    return GetAllForUsers(GetRecentlyActiveUsers(since));
}

TMaybe<TMap<TString, TVector<TUserDocumentPhoto>>> TUserDocumentPhotosDB::GetUsersWithPhotoIds(const TSet<TString>& photoIds, NDrive::TEntitySession& session) const {
    if (photoIds.empty()) {
        return TMap<TString, TVector<TUserDocumentPhoto>>();
    }
    auto options = TQueryOptions()
                    .SetGenericCondition("id", photoIds)
                    .SetGenericCondition("type", NSQL::Not(TSet<TString>({"xx"})));
    auto optionalPhotos = Fetch(session, options);
    if (!optionalPhotos) {
        return Nothing();
    }
    TMap<TString, TVector<TUserDocumentPhoto>> result;
    for (auto&& photo : *optionalPhotos) {
        auto userId = photo.GetUserId();
        result[userId].emplace_back(std::move(photo));
    }
    return result;
}

TVector<TUserDocumentPhoto> TUserDocumentPhotosDB::GetUnverifiedPhotosOfType(const NUserDocument::EType type) const {
    auto fetchResult = FetchWithCustomQuery("SELECT * FROM " + GetTableName() + " WHERE type = '" + ToString(type) + "' AND verification_status is NULL");
    return MakeVector(NContainer::Values(fetchResult.GetResult()));
}

TMap<NUserDocument::EType, TUserDocumentPhoto> TUserDocumentPhotosDB::GetTypeToRecentPhotoMapping(const TString& userId, const TVector<NUserDocument::EType>& soughtTypes, const TSet<NUserDocument::EVerificationStatus>& verificationStatuses) const {
    TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> result = GetTypeToRecentPhotoMapping(TSet<TString>( { userId } ), soughtTypes, verificationStatuses);
    if (result.size() == 1) {
        return result.begin()->second;
    } else {
        return TMap<NUserDocument::EType, TUserDocumentPhoto>();
    }
}

TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> TUserDocumentPhotosDB::GetTypeToRecentPhotoMapping(const TSet<TString>& userIds, const TVector<NUserDocument::EType>& soughtTypes, const TSet<NUserDocument::EVerificationStatus>& verificationStatuses) const {
    TString allowedTypesStr = "";
    for (auto&& type : soughtTypes) {
        TString typeStr = "'" + ToString(type) + "'";
        if (allowedTypesStr) {
            allowedTypesStr += ",";
        }
        allowedTypesStr += typeStr;
    }
    allowedTypesStr = "(" + allowedTypesStr + ")";

    TString userIdsStr = "";
    for (auto userId : userIds) {
        if (userIdsStr) {
            userIdsStr += ",";
        }
        userIdsStr += "'" + userId + "'";
    }
    userIdsStr = "(" + userIdsStr + ")";
    TString verificationCond;
    if (!verificationStatuses.empty()) {
        for (auto i : verificationStatuses) {
            if (verificationCond) {
                verificationCond += ",";
            }
            verificationCond += "'" + ToString(i) + "'";
        }
        verificationCond = " AND verification_status IN (" + verificationCond + ")";
    }

    auto fetchResult = FetchWithCustomQuery("SELECT * FROM " + GetTableName() + " WHERE user_id IN " + userIdsStr + " AND type IN " + allowedTypesStr + verificationCond);

    TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> result;
    for (auto&& photoIt : fetchResult) {
        auto userId = photoIt.second.GetUserId();
        auto type = photoIt.second.GetType();
        auto submittedAt = photoIt.second.GetSubmittedAt();
        if (!result[userId].contains(type) || result[userId][type].GetSubmittedAt() < submittedAt)  {
            result[userId][type] = std::move(photoIt.second);
        }
    }

    return result;
}

TUserDocumentPhotosDB::TFetchResult TUserDocumentPhotosDB::GetPhotosForUsers(const TSet<TString>& userIds, const NUserDocument::EType& type) const {
    TString userIdsStr = "";
    for (auto&& userId : userIds) {
        if (userIdsStr) {
            userIdsStr += ",";
        }
        userIdsStr += "'" + userId + "'";
    }
    userIdsStr = "(" + userIdsStr + ")";
    auto fetchResult = FetchWithCustomQuery("SELECT * FROM " + GetTableName() + " WHERE user_id IN " + userIdsStr + " AND type = '" + ToString(type) + "'");
    return fetchResult;
}

void TUserDocumentPhotosDB::FillUserReport(const TString& userId, NJson::TJsonValue& result) const {
    auto allObjects = GetAllForUser(userId);
    Sort(
        allObjects.begin(),
        allObjects.end(),
        [] (const TUserDocumentPhoto& lhs, const TUserDocumentPhoto& rhs) {
            return lhs.GetSubmittedAt() < rhs.GetSubmittedAt();
        }
    );

    TMap<NUserDocument::EType, NJson::TJsonValue*> reportParts;
    reportParts[NUserDocument::EType::LicenseFront] = &result["driving_license"].InsertValue("front", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::LicenseBack] = &result["driving_license"].InsertValue("back", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::LicenseSelfie] = &result["driving_license"].InsertValue("selfie", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::PassportBiographical] = &result["passport"].InsertValue("biographical", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::PassportRegistration] = &result["passport"].InsertValue("registration", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::PassportSelfie] = &result["passport"].InsertValue("selfie", NJson::JSON_ARRAY);
    reportParts[NUserDocument::EType::Selfie] = &result.InsertValue("selfie", NJson::JSON_ARRAY);

    for (auto&& photo : allObjects) {
        auto report = photo.GetReport();
        auto reportPartIt = reportParts.find(photo.GetType());
        if (reportPartIt != reportParts.end()) {
            reportPartIt->second->AppendValue(std::move(report));
        }
    }
}

TUserDocumentVideo::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    Type = GetFieldDecodeIndex("type", decoderBase);
    UserId = GetFieldDecodeIndex("user_id", decoderBase);
    SubmittedAt = GetFieldDecodeIndex("submitted_at", decoderBase);
    OriginChat = GetFieldDecodeIndex("origin_chat", decoderBase);
    MimeType = GetFieldDecodeIndex("mime_type", decoderBase);
    PhotoId = GetFieldDecodeIndex("photo_id", decoderBase);
}

bool TUserDocumentVideo::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, PhotoId);
    READ_DECODER_VALUE(decoder, values, MimeType);
    READ_DECODER_VALUE_DEF(decoder, values, OriginChat, "");
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, SubmittedAt);
    READ_DECODER_VALUE_DEF(decoder, values, Type, NUserDocument::EType::Unknown);
    return true;
}

bool TUserDocumentVideo::Parse(const NStorage::TTableRecord& row) {
    Id = row.Get("id");
    PhotoId = row.Get("photo_id");
    MimeType = row.Get("mime_type");
    OriginChat = row.Get("origin_chat");
    UserId = row.Get("user_id");
    if (row.Get("submitted_at") && !TInstant::TryParseIso8601(row.Get("submitted_at"), SubmittedAt)) {
        WARNING_LOG << "unable to parse submitted_at timestamp: " << row.Get("submitted_at") << Endl;
    }
    if (!TryFromString(row.Get("type"), Type)) {
        WARNING_LOG << "unable to parse type" << Endl;
    }
    return true;
}

NStorage::TTableRecord TUserDocumentVideo::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("id", Id);
    result.Set("photo_id", PhotoId);
    result.Set("mime_type", MimeType);
    result.Set("user_id", UserId);
    result.Set("type", ::ToString(Type));
    result.Set("origin_chat", OriginChat);
    if (SubmittedAt != TInstant::Zero()) {
        result.Set("submitted_at", SubmittedAt.ToString());
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserDocumentPhoto& object) {
    return object.GetReport();
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserDocumentVideo& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    result["photo_id"] = object.GetPhotoId();
    result["mime_type"] = object.GetMimeType();
    result["submitted_at"] = object.GetSubmittedAt().Seconds();
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NUserDocument::EVerificationStatus& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TDocumentResubmitOverride& resubmitOverride) {
    return
        NJson::ParseField(value, "localization", resubmitOverride.Localization, true) &&
        NJson::ParseField(value, "node", resubmitOverride.Node) &&
        NJson::ParseField(value, "resubmit_item", NJson::Stringify(resubmitOverride.ResubmitItem), true);
}

template<>
NJson::TJsonValue NJson::ToJson(const TDocumentResubmitOverride& resubmitOverride) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "localization", resubmitOverride.Localization);
    NJson::InsertField(result, "node", resubmitOverride.Node);
    NJson::InsertField(result, "resubmit_item", NJson::Stringify(resubmitOverride.ResubmitItem));
    return result;
}
