#pragma once

#include <drive/backend/database/entity/manager.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/uuid.h>

#include <util/datetime/base.h>

namespace NUserDocument {
    enum EType: ui32 {
        Unknown = 0                     /* "xx", "unknown" */,
        LicenseFront = 1 << 0           /* "lf", "license_front" */,
        LicenseBack = 1 << 1            /* "lb", "license_back" */,
        PassportBiographical = 1 << 2   /* "pb", "passport_biographical" */,
        PassportRegistration = 1 << 3   /* "pr", "passport_registration" */,
        PassportSelfie = 1 << 4         /* "ps", "passport_selfie" */,
        Selfie = 1 << 5                 /* "ss", "selfie" */,
        LicenseSelfie = 1 << 6          /* "ls", "license_selfie" */,
        LicenseVideo = 1 << 7           /* "lv", "license_video" */,
        PassportVideo = 1 << 8          /* "pv", "passport_video" */,
        VideoSelfie = 1 << 9            /* "vs", "video_selfie" */
    };

    static constexpr ui32 Video = LicenseVideo | PassportVideo | VideoSelfie;
    static constexpr ui32 InvalidResubmitMaskFlag = 1 << 30;

    static const EType AllTypes[] = {
        LicenseFront,
        LicenseBack,
        LicenseSelfie,
        PassportBiographical,
        PassportRegistration,
        PassportSelfie
    };

    enum EVerificationStatus {
        NonLatin /* "n", "NON_LATIN" */,
        Foreign /* "f", "FOREIGN" */,
        NeedInfo /* "i", "NEED_INFO" */,
        Ok /* "o", "OK" */,
        Unrecognizable /* "u", "UNRECOGNIZABLE" */,
        Discarded /* "d", "DISCARDED" */,
        VideoScreencap /* "v", "VIDEO_SCREENCAP" */,
        VideoAnotherPerson /* "a", "VIDEO_ANOTHER_PERSON" */,
        VideoError /* "e", "VIDEO_ERROR" */,
        NotYetProcessed /* "-" */,
    };
}

class TRecognitionConfidenceData {
public:
    static constexpr double FIELD_DOES_NOT_EXIST = -1.0;
    enum class EDocumentField {
        FirstName /* "first_name" */,
        LastName /* "last_name" */,
        MiddleName /* "middle_name" */,
        BirthDate /* "birth_date" */,
        Country /* "country" */,
        IssuedBy /* "issued_by" */,
        IssueDate /* "issue_date" */,
        BirthPlace /* "birth_place" */,
        Citizenship /* "citizenship" */,
        Gender /* "gender" */,
        ExpirationDate /* "expiration_date" */,

        LicenseNumberFront /* "license_number_front" */,
        LicenseNumberBack /* "license_number_back" */,
        LicensePrevNumberFront /* "license_prev_number_front" */,
        LicensePrevNumber /* "license_prev_number" */,
        LicenseCategories /* "license_categories" */,
        LicenseCategoriesBack /* "license_categories_back" */,
        LicenseBackCountry /* "license_back_country" */,
        LicenseExperienceFrom /* "license_experince_from" */,
        LicensePrevIssueDateFront /* "license_prev_issue_date_front" */,
        LicensePrevIssueDate /* "license_prev_issue_date" */,
        LicenseCategoriesBValidFrom /* "license_categories_b_valid_from" */,
        LicenseCategoriesBValidToFront /* "license_categories_b_valid_to_front" */,
        LicenseCategoriesBValidTo /* "license_categories_b_valid_to" */,
        LicenseExpirationDate /* "license_expiration_date" */,

        PassportSubdivisionCode /* "passport_subdivision_code" */,
        PassportBiographicalCountry /* "passport_biographical_country" */,
        PassportRegistrationCountry /* "passport_biographical_country" */,
        PassportNumber /* "passport_number" */,

        CountryFront /* "country_front" */,
        CountryBack /* "country_back" */,

        SpecialCategoryBDateFront /* "special_category_b_date_front" */,
        SpecialCategoryBDateBack /* "special_category_b_date_back" */,
    };

private:
    R_OPTIONAL(double, QuasiGibddScore);
    R_OPTIONAL(double, QuasiFmsScore);
    R_OPTIONAL(double, FromScreenScore);
    R_OPTIONAL(double, BadFormatScore);
    R_FIELD(bool, RecognitionFailed, false);

public:
    void AddField(const EDocumentField field, const double condifence);
    double GetField(const EDocumentField field) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& json);

    NJson::TJsonValue SerializePassportBiographical() const;
    NJson::TJsonValue SerializeDrivingLicenseFront() const;
    NJson::TJsonValue SerializeDrivingLicenseBack() const;

private:
    TMap<EDocumentField, double> Confidences;
};

class TUserDocumentPhoto {
public:
    class TDecoder: public TBaseDecoder {
    private:
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, SubmittedAt, -1);
        R_FIELD(i32, VerifiedAt, -1);
        R_FIELD(i32, VerificationStatus, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, DocumentId, -1);
        R_FIELD(i32, OriginChat, -1);
        R_FIELD(i32, RecognizerMeta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

private:
    R_READONLY(TString, Id);
    R_FIELD(NUserDocument::EType, Type, NUserDocument::EType::Unknown);
    R_READONLY(TInstant, SubmittedAt, TInstant::Zero());
    R_FIELD(TInstant, VerifiedAt, TInstant::Zero());
    R_FIELD(NUserDocument::EVerificationStatus, VerificationStatus, NUserDocument::EVerificationStatus::NotYetProcessed);
    R_READONLY(TString, UserId);
    R_READONLY(TString, DocumentId);
    R_FIELD(TString, OriginChat, "registration");
    R_OPTIONAL(TRecognitionConfidenceData, RecognizerMeta);

public:
    TUserDocumentPhoto() = default;
    TUserDocumentPhoto(const TString& userId, NUserDocument::EType type);

    explicit operator bool() const {
        return !Id.empty();
    }

    bool Parse(const NStorage::TTableRecord& row);
    NStorage::TTableRecord SerializeToTableRecord() const;

    TString BuildTakeoutFilename() const {
        return ToString(Type) + "-" + ToString((ui32)SubmittedAt.Seconds()) + ".jpg";
    }

    constexpr static bool HasVerificationStatus() {
        return true;
    }

    bool IsResubmitNeeded() const;
    NJson::TJsonValue GetReport() const;
    TString BuildUri(bool newHandler = false) const;
    TString BuildBackgroundVideoUri(bool newHandler = false) const;

    static TMap<TString, NUserDocument::EVerificationStatus> GetVerificationStatusMap();

private:
    bool IsNewHandlerRequired() const;
};

template <class TEntity, class TFetchResult, class TStorage>
class IDocumentMediaStorage {
public:
    virtual TFetchResult GetRecentlyAdded(const TInstant& since) const {
        const auto* thisStorage = static_cast<const TStorage*>(this);
        TStringStream queryStream;
        queryStream << "SELECT * FROM " << thisStorage->GetTableName();
        queryStream << " WHERE submitted_at >= '" << since.ToString() << "' AND type != 'xx'";
        if (TEntity::HasVerificationStatus()) {
            queryStream << " AND verification_status is null";
        }
        return thisStorage->FetchWithCustomQuery(queryStream.Str());
    }

    virtual TSet<TString> GetRecentlyActiveUsers(const TInstant& since = TInstant::Now() - TDuration::Hours(1)) const {
        auto newMedia = GetRecentlyAdded(since);
        TSet<TString> userIds;
        for (auto&& it : newMedia) {
            const auto& userId = it.second.GetUserId();
            if (userId && !GetUuid(userId).IsEmpty()) {
                userIds.insert(userId);
            }
        }
        return userIds;
    }

    virtual TVector<TEntity> GetAllForUser(const TString& userId) const {
        return GetAllForUsers({userId});
    }

    virtual TVector<TEntity> GetAllForUsers(const TSet<TString>& userIds) const {
        if (userIds.empty()) {
            return {};
        }
        const auto* thisStorage = static_cast<const TStorage*>(this);
        typename TStorage::TQueryOptions queryOptions;
        queryOptions.SetGenericCondition("user_id", userIds);
        queryOptions.AddCustomCondition("type != 'xx'");

        auto session = thisStorage->template BuildTx<NSQL::ReadOnly>();
        auto fetchResult = thisStorage->Fetch(session, queryOptions);
        if (!fetchResult) {
            ERROR_LOG << "IDocumentMediaStorage::GetAllForUsers: " << session.GetStringReport() << Endl;
            return {};
        }
        return *fetchResult;
    }
};

class TUserDocumentPhotosDB
    : public TCachedEntityManager<TUserDocumentPhoto, true, TString, TUuidFilter>
    , public IDocumentMediaStorage<TUserDocumentPhoto, TCachedEntityManager<TUserDocumentPhoto, true, TString, TUuidFilter>::TFetchResult, TUserDocumentPhotosDB>
{
private:
    using TBase = TCachedEntityManager<TUserDocumentPhoto, true, TString, TUuidFilter>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "user_document_photo";
    }

    virtual TString GetMainId(const TUserDocumentPhoto& e) const override {
        return e.GetId();
    }

    TFetchResult GetRecentlyAddedUnverifiedPhotos(const TInstant& since = TInstant::Now() - TDuration::Hours(1)) const;

    TVector<TUserDocumentPhoto> GetPhotosOfUsersWhichAddedPhotosRecently(const TInstant& since = TInstant::Now() - TDuration::Hours(1)) const;

    TVector<TUserDocumentPhoto> GetUnverifiedPhotosOfType(const NUserDocument::EType type) const;
    TMaybe<TMap<TString, TVector<TUserDocumentPhoto>>> GetUsersWithPhotoIds(const TSet<TString>& photoIds, NDrive::TEntitySession& session) const;

    TMap<NUserDocument::EType, TUserDocumentPhoto> GetTypeToRecentPhotoMapping(const TString& userId, const TVector<NUserDocument::EType>& soughtTypes, const TSet<NUserDocument::EVerificationStatus>& verificationStatuses = {}) const;
    TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> GetTypeToRecentPhotoMapping(const TSet<TString>& userIds, const TVector<NUserDocument::EType>& soughtTypes, const TSet<NUserDocument::EVerificationStatus>& verificationStatuses = {}) const;

    TFetchResult GetPhotosForUsers(const TSet<TString>& userIds, const NUserDocument::EType& type) const;

    void FillUserReport(const TString& userId, NJson::TJsonValue& result) const;
};

class TUserDocumentVideo {
public:
    class TDecoder: public TBaseDecoder {
    private:
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, PhotoId, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, SubmittedAt, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, OriginChat, -1);
        R_FIELD(i32, MimeType, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

private:
    R_READONLY(TString, Id);
    R_FIELD(TString, PhotoId);
    R_READONLY(TString, MimeType);
    R_READONLY(TInstant, SubmittedAt, TInstant::Zero());

    R_FIELD(NUserDocument::EType, Type, NUserDocument::Unknown);
    R_FIELD(TString, UserId);
    R_FIELD(TString, OriginChat);

public:
    TUserDocumentVideo() = default;

    TUserDocumentVideo(NUserDocument::EType type)
        : Id(NUtil::CreateUUID())
        , SubmittedAt(Now())
        , Type(type)
    {
    }

    TUserDocumentVideo(const TString& id, NUserDocument::EType type)
        : Id(id)
        , PhotoId(id)
        , SubmittedAt(Now())
        , Type(type)
    {
    }

    explicit operator bool() const {
        return !Id.empty();
    }

    bool Parse(const NStorage::TTableRecord& row);
    NStorage::TTableRecord SerializeToTableRecord() const;

    constexpr static bool HasVerificationStatus() {
        return false;
    }

    bool IsBackground() const {
        return !PhotoId.Empty();
    }
};

class TUserDocumentVideoDB
    : public TDBEntities<TUserDocumentVideo>
    , public IDocumentMediaStorage<TUserDocumentVideo, TDBEntities<TUserDocumentVideo>::TFetchResult, TUserDocumentVideoDB>
{
private:
    using TBase = TDBEntities<TUserDocumentVideo>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "user_document_background_video";
    }

    TString GetColumnName() const override {
        return "photo_id";
    }

    virtual TString GetMainId(const TUserDocumentVideo& e) const override {
        return e.GetPhotoId();
    }
};

struct TDocumentResubmitOverride {
    TString Localization;
    TString Node;
    NUserDocument::EType ResubmitItem;

    template<class TProto>
    static void AddToProto(const TVector<TDocumentResubmitOverride>& resubmitOverrides, TProto& proto) {
        for (auto&& resubmitOverride : resubmitOverrides) {
            auto* overrideProto = proto.AddResubmitOverrides();
            overrideProto->SetResubmitItem(ToString(resubmitOverride.ResubmitItem));
            overrideProto->SetNode(resubmitOverride.Node);
            overrideProto->SetLocalization(resubmitOverride.Localization);
        }
    }

    template<class TProto>
    static TMaybe<TVector<TDocumentResubmitOverride>> GetFromProto(const TProto& proto) {
        TVector<TDocumentResubmitOverride> resubmitOverrides;
        for (const auto& overrideProto : proto.GetResubmitOverrides()) {
            auto& resubmitOverride = resubmitOverrides.emplace_back();
            resubmitOverride.Node = overrideProto.GetNode();
            resubmitOverride.Localization = overrideProto.GetLocalization();
            NUserDocument::EType resubmitItem;
            if (!TryFromString(overrideProto.GetResubmitItem(), resubmitItem)) {
                return {};
            }
            resubmitOverride.ResubmitItem = resubmitItem;
        }
        return resubmitOverrides;
    }
};
