#include "user_documents_check.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/processors/yang_proxy/callback_persdata.h>
#include <drive/backend/user_document_photos/manager.h>

#include <rtline/library/storage/sql/query.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserDocumentsChecksManager::TPhotoChecksPriority& priority) {
    if (!value.IsArray()) {
        return false;
    }

    TSet<TString> mentionedVerdicts;

    ui64 verdictPrioroty = 0;
    for (auto&& checkConfiguration: value.GetArray()) {
        if (!checkConfiguration.IsMap() || checkConfiguration.GetMap().size() != 1) {
            return false;
        }

        const auto& front = *checkConfiguration.GetMap().begin();
        if (!front.second.IsArray()) {
            return false;
        }

        TString photoVerdict = front.first;
        for (const auto& checkVerdict: front.second.GetArray()) {
            if (!checkVerdict.IsString() || mentionedVerdicts.contains(checkVerdict.GetString())) {
                return false;
            }
            const TString verdict = checkVerdict.GetString();
            mentionedVerdicts.insert(verdict);

            priority[verdict] = {photoVerdict, verdictPrioroty + 1};
            verdictPrioroty++;
        }
    }
    return true;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserDocumentsCheck& result) {
    return
        NJson::ParseField(value, "user_id", result.MutableUserId(), true) &&
        NJson::ParseField(value, "status", result.MutableStatus(), true) &&
        NJson::ParseField(value, "type", result.MutableType(), true) &&
        NJson::ParseField(value, "secret_id", result.MutableSecretId()) &&
        NJson::ParseField(value, "worker_id", result.MutableWorkerId()) &&
        NJson::ParseField(value, "pool_id", result.MutablePoolId()) &&
        NJson::ParseField(value, "project_id", result.MutableProjectId()) &&
        NJson::ParseField(value, "assignment_id", result.MutableAssignmentId()) &&
        NJson::ParseField(value, "login", result.MutableLogin()) &&
        NJson::ParseField(value, "submit_ts", result.MutableTimestamp(), false) &&
        NJson::ParseField(value, "comment", result.MutableComment());
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserDocumentsCheck& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "user_id", object.GetUserId());
    object.DoBuildReportItem(result);
    return result;
}

NUserDocument::EVerificationStatus NYangAssignment::ParsePhotoStatus(const NJson::TJsonValue& yangStatusJson) {
    if (!yangStatusJson.IsString()) {
        return NUserDocument::EVerificationStatus::NotYetProcessed;
    }
    auto yangStatus = yangStatusJson.GetString();
    NUserDocument::EVerificationStatus status;
    if (!TryFromString(yangStatus, status)) {
        return NUserDocument::EVerificationStatus::NotYetProcessed;
    }
    return status;
}

NStorage::TTableRecord TUserDocumentsCheck::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("user_id", UserId);
    row.Set("status", Status);
    row.Set("type", Type);
    row.Set("meta", SerializeMeta());
    return row;
}

void TUserDocumentsCheck::DoBuildReportItem(NJson::TJsonValue& result) const {
    NJson::InsertField(result, "status", Status);
    NJson::InsertField(result, "type", Type);
    NJson::InsertField(result, "secret_id", SecretId);
    NJson::InsertField(result, "worker_id", WorkerId);
    NJson::InsertField(result, "pool_id", PoolId);
    NJson::InsertField(result, "project_id", ProjectId);
    NJson::InsertField(result, "assignment_id", AssignmentId);
    NJson::InsertField(result, "login", Login);
    NJson::InsertField(result, "submit_ts", Timestamp);
    NJson::InsertField(result, "comment", Comment);
}

bool TUserDocumentsCheck::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

bool TUserDocumentsCheck::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, Status);
    READ_DECODER_VALUE(decoder, values, Type);

    return DeserializeMeta(decoder, values, nullptr);
}

bool TUserDocumentsCheck::DeserializeMeta(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    TString meta;
    READ_DECODER_VALUE_TEMP(decoder, values, meta, Meta);
    NJson::TJsonValue metaJson;
    if (!NJson::ReadJsonTree(meta, &metaJson)) {
        return false;
    }
    return
        NJson::ParseField(metaJson, "secret_id", SecretId) &&
        NJson::ParseField(metaJson, "worker_id", WorkerId) &&
        NJson::ParseField(metaJson, "pool_id", PoolId) &&
        NJson::ParseField(metaJson, "project_id", ProjectId) &&
        NJson::ParseField(metaJson, "assignment_id", AssignmentId) &&
        NJson::ParseField(metaJson, "login", Login) &&
        NJson::ParseField<unsigned long &>(metaJson, "submit_ts", Timestamp) &&
        NJson::ParseField(metaJson, "comment", Comment);
}

TString TUserDocumentsCheck::SerializeMeta() const {
    NJson::TJsonValue metaJson;
    NJson::InsertField(metaJson, "secret_id", SecretId);
    NJson::InsertField(metaJson, "worker_id", WorkerId);
    NJson::InsertField(metaJson, "pool_id", PoolId);
    NJson::InsertField(metaJson, "project_id", ProjectId);
    NJson::InsertField(metaJson, "assignment_id", AssignmentId);
    NJson::InsertField(metaJson, "login", Login);
    NJson::InsertField(metaJson, "submit_ts", Timestamp);
    NJson::InsertField(metaJson, "comment", Comment);
    return metaJson.GetStringRobust();
}

bool TUserDocumentsChecksDB::Upsert(const TUserDocumentsCheck& entity, const TString& historyUserId, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable(GetTableName());
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::TTableRecord record = entity.SerializeToTableRecord();
    NStorage::TTableRecord uniqueRecord;
    uniqueRecord.Set("user_id", entity.GetUserId());
    uniqueRecord.Set("type", entity.GetType());
    bool isUpdate = false;
    NStorage::TObjectRecordsSet<TUserDocumentsCheck> upsertedData;
    if (!table->Upsert(record, transaction, uniqueRecord, &isUpdate, &upsertedData)) {
        session.SetErrorInfo("user_documents_checks", "cannot upsert", EDriveSessionResult::InternalError);
        return false;
    }
    if (upsertedData.size() != 1) {
        session.SetErrorInfo("user_documents_checks", "expected 1 upserted rows, got " + ToString(upsertedData.size()), EDriveSessionResult::InternalError);
        return false;
    }
    TRecordsSet addedHistory;
    if (!HistoryWriter.AddHistory<NStorage::TObjectRecordsSet<TUserDocumentsCheck>>(upsertedData, historyUserId, EObjectHistoryAction::UpdateData, session, &addedHistory)) {
        session.AddErrorMessage("user_documents_checks_history", "cannot add history");
        return false;
    }
    if (addedHistory.size() != 1) {
        session.SetErrorInfo("user_documents_checks_history", "expected 1 added history rows, got " + ToString(addedHistory.size()), EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

TVector<TUserDocumentsChecksHistoryManager::TCheckHistoryEvent> TUserDocumentsChecksDB::GetEvents(const TString& type, const TString& userId, const TInstant& since, NDrive::TEntitySession& session) const {
    TUserDocumentsChecksHistoryManager::TQueryOptions options;
    if (userId) {
        options.AddGenericCondition("user_id", userId);
    }
    if (type) {
        options.AddGenericCondition("type", type);
    }
    options.SetGenericCondition("history_timestamp", TRange<ui64>(since.Seconds()));

    auto events = HistoryWriter.GetEvents(0, session, options);
    if (!events) {
        session.Check();
        return {};
    }

    return *events;
}

bool TUserDocumentsChecksManager::SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const TChecksSettings& settings, NDrive::TEntitySession& session) const {
    auto typeIt = settings.find(check.GetType());
    if (typeIt == settings.end() || !typeIt->second.contains(check.GetStatus())) {
        session.SetErrorInfo("TUserDocumentsChecksManager::SetStatus", "cannot set check type " + check.GetType() + " to status " + check.GetStatus());
        return false;
    }
    return UserDocumentsChecksDb.Upsert(check, historyUserId, session);
}

bool TUserDocumentsChecksManager::SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    auto allowedStatuses = GetAllowedStatusesByTypes(server);
    return SetStatus(check, historyUserId, allowedStatuses, session);
}

bool TUserDocumentsChecksManager::SetStatus(const TString& userId, const TString& type, const TString& status, const TString& historyUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    return SetStatus({userId, type, status}, historyUserId, server, session);
}

bool TUserDocumentsChecksManager::SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const TVector<TString>& photoIds, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const {
    return
        SetStatus(check, historyUserId, checksConfig.Settings, chatSession) &&
        SetPhotosStatus(check.GetUserId(), photoIds, checksConfig, server, chatSession, tagSession) &&
        ContinueToChat(check.GetUserId(), checksConfig, server, chatSession, tagSession);
}

bool TUserDocumentsChecksManager::ContinueToChat(const TString& userId, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const {
    const auto& chatIdsSet = checksConfig.ChatsToMove;

    auto permissions = Yensured(server.GetDriveAPI())->GetUserPermissions(userId, TUserPermissionsFeatures());
    IChatUserContext::TPtr context = MakeAtomicShared<TChatUserContext>(permissions, &server);

    for (auto&& chatId : chatIdsSet) {
        auto chatRobot = server.GetChatRobot(chatId);
        if (!chatRobot) {
            continue;
        }
        for (auto&& chat : chatRobot->GetTopics(context, true)) {
            context->SetChatId(chatId);
            context->SetChatTopic(chat.GetTopic());
            TChatContext stateContext(&tagSession, &chatSession);
            if (!chatRobot->MaybeContinueChat(context, permissions->GetUserId(), stateContext)) {
                return false;
            }
        }
    }

    return true;
}

TMaybe<TString> TUserDocumentsChecksManager::GetStatus(const TString& userId, const TString& type, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.AddGenericCondition("user_id", userId);
    options.AddGenericCondition("type", type);
    auto fetchResult = UserDocumentsChecksDb.FetchInfo(session, options);
    if (!fetchResult) {
        return Nothing();
    }
    if (fetchResult.size() == 0) {
        return "";
    }
    if (fetchResult.size() > 1) {
        session.SetErrorInfo("TUserDocumentsChecksManager::GetCheckStatus", "expected unique type for each user id, got " + ToString(fetchResult.size()) + " for user " + userId + " and type " + type);
        return Nothing();
    }
    auto documentCheck = fetchResult.GetResultPtr(UserDocumentsChecksDb.GetMainId(userId, type));
    if (!documentCheck) {
        session.SetErrorInfo("TUserDocumentsChecksManager::GetCheckStatus", "wrong user in the result");
        return Nothing();
    }
    return documentCheck->GetStatus();
}

bool TUserDocumentsChecksManager::SetPhotosStatus(const TString& userId, const TVector<TString>& photoIds, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const {
    if (!photoIds) {
        return true;
    }

    if (!server.GetDriveAPI() || !server.GetDriveAPI()->HasDocumentPhotosManager()) {
        chatSession.SetErrorInfo(
            "TUserDocumentsChecksManager::SetPhotosStatus",
            "Document photo manager is not configured"
        );
        return false;
    }
    const auto& photoManager = server.GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB();

    auto photos = photoManager.FetchInfo(photoIds, tagSession);
    if (!photos) {
        tagSession.SetErrorInfo(
            "TUserDocumentsChecksManager::SetPhotosStatus",
            "Failed to fetch photos"
        );
        return false;
    }

    for (auto&& [_, photo] : photos.MutableResult()) {
        TString photoStatus;
        ui64 currentPriority = 0;

        const auto& photoType = photo.GetType();

        const auto checks = GetChecks(
            userId,
            checksConfig.GetCheckTypes(photoType),
             chatSession
        );
        if (!checks) {
            return false;
        }

        for (const auto& check : *checks) {
            const auto& type = check.GetType();
            const auto& status = check.GetStatus();

            if (!checksConfig.Priority.contains(status))
            {
                chatSession.SetErrorInfo(
                    "TUserDocumentsChecksManager::SetPhotosStatus",
                    "Encountered invalid check type (" + type + ")"
                );
                return false;
            }

            const auto& actualStatus = checksConfig.Priority.at(status).first;
            const auto verdictPriority = checksConfig.Priority.at(status).second;
            if (verdictPriority > currentPriority) {
                photoStatus = actualStatus;
                currentPriority = verdictPriority;
            }
        }

        if (!photoStatus) {
            chatSession.SetErrorInfo(
                "TUserDocumentsChecksManager::SetPhotosStatus",
                "Failed to determine photo status for some reason"
            );
            return false;
        }

        auto status = NYangAssignment::ParsePhotoStatus(photoStatus);
        if (status != photo.GetVerificationStatus()) {
            photo.SetVerifiedAt(Now());
            photo.SetVerificationStatus(status);
            if (!photoManager.Upsert(photo, tagSession)) {
                return false;
            }
        }
    }

    return true;
}

TMaybe<TVector<TUserDocumentsCheck>> TUserDocumentsChecksManager::GetChecks(const TString& userId, const TSet<TString>& types, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions options;
    options.AddGenericCondition("user_id", userId);
    if (types.size()) {
        options.SetGenericCondition("type", types);
    }
    auto fetchResult = UserDocumentsChecksDb.FetchInfo(session, options);
    if (!fetchResult) {
        return Nothing();
    }
    TVector<TUserDocumentsCheck> result;
    for (auto&& [_, check] : fetchResult) {
        result.push_back(std::move(check));
    }
    return result;
}

TVector<TUserDocumentsChecksHistoryManager::TCheckHistoryEvent> TUserDocumentsChecksManager::GetChecksHistory(const TString& userId, const TString& type, const TInstant& since, NDrive::TEntitySession& session) const {
    return UserDocumentsChecksDb.GetEvents(type, userId, since, session);
}

TUserDocumentsChecksManager::TChecksSettings TUserDocumentsChecksManager::GetAllowedStatusesByTypes(const NDrive::IServer& server) const {
    TChecksSettings result;
    if (auto settings = GetChecksSettings(Config.GetChecksSettingsKey(), server)) {
        result = std::move(*settings);
    }
    return result;
}

void TUserDocumentsCheckConfig::Init(const TYandexConfig::Section* section) {
    TDBEntitiesManagerConfig::Init(section);
    ChecksSettingsKey = section->GetDirectives().Value("ChecksSettingsKey", ChecksSettingsKey);
    PhotoChecksSettingsKey = section->GetDirectives().Value("PhotoChecksSettingsKey", PhotoChecksSettingsKey);
}

void TUserDocumentsCheckConfig::ToString(IOutputStream& os) const {
    TDBEntitiesManagerConfig::ToString(os);
    os << "ChecksSettingsKey: " << ChecksSettingsKey << Endl;
    os << "PhotoChecksSettingsKey: " << PhotoChecksSettingsKey << Endl;
}

TMaybe<TUserDocumentsChecksManager::TChecksSettings> TUserDocumentsChecksManager::GetChecksSettings(const TString& varName, const NDrive::IServer& server) {
    auto complaintConfigs = server.GetSettings().GetJsonValue(varName);
    TChecksSettings result;
    for (auto&& [type, statusesJson] : complaintConfigs.GetMap()) {
        TSet<TString> statuses;
        if (!NJson::ParseField(statusesJson, statuses)) {
            return Nothing();
        }
        result[type] = std::move(statuses);
    }
    return result;
}

TMaybe<TUserDocumentsChecksManager::TChecksGenericMap> TUserDocumentsChecksManager::GetChecksGenericMap(const TString& varName, const NDrive::IServer& server) {
    auto rawTypeNames = server.GetSettings().GetValueDef<TString>(varName,
        "{\"passport_biographical\":[\"passport_biographical\"],"
        "\"passport_registration\":[\"passport_registration\"],"
        "\"license_front\":[\"license_front\"],"
        "\"license_back\":[\"license_back\"]}"
    );
    NJson::TJsonValue typeNames;
    if (!NJson::ReadJsonTree(rawTypeNames, &typeNames) || !typeNames.IsMap()) {
        ERROR_LOG << "Failed to read json " << rawTypeNames << Endl;
        return Nothing();
    }

    TChecksGenericMap result;
    for (const auto& [name, _] : typeNames.GetMap()) {
        if (!TJsonProcessor::ReadContainer(typeNames, name, result[name])) {
            ERROR_LOG << "Failed to read json " << rawTypeNames << Endl;
            return Nothing();
        }
    }
    return result;
}

TMaybe<TUserDocumentsChecksManager::TPhotoChecksPriority> TUserDocumentsChecksManager::GetPhotoChecksPriority(const TString& varName, const NDrive::IServer& server) {
    TPhotoChecksPriority checksPriority;
    const auto settingsJson = server.GetSettings().GetJsonValue(varName);
    if (!NJson::TryFromJson(settingsJson, checksPriority)) {
            return Nothing();
    }
    return checksPriority;
}

TString TUserDocumentsChecksManager::GetPhotoIdByType(const TYangDocumentVerificationAssignment& assignment, NUserDocument::EType type) {
    if (type & NUserDocument::EType::LicenseFront) {
        return assignment.GetLicenseFrontId();
    } else if (type & NUserDocument::EType::LicenseBack) {
        return assignment.GetLicenseBackId();
    } else if (type & NUserDocument::EType::LicenseSelfie) {
        return assignment.GetLicenseSelfieId();
    } else if (type & NUserDocument::EType::PassportBiographical) {
        return assignment.GetPassportBiographicalId();
    } else if (type & NUserDocument::EType::PassportRegistration) {
        return assignment.GetPassportRegistrationId();
    } else if (type & NUserDocument::EType::PassportSelfie) {
        return assignment.GetPassportSelfieId();
    } else {
        throw yexception() << "Unknown document type";
    }
}

TMaybe<TVector<TString>> TUserDocumentsChecksManager::GetPhotoIds(const TString& checkType, const TString& secretId, const TUserDocumentsChecksManager::TChecksGenericMap& genericChecks, const NDrive::IServer& server, NDrive::TEntitySession& session) {
        auto DriveApi = server.GetDriveAPI();
        const auto& photoManager = DriveApi->GetDocumentPhotosManager();
        auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
        const TYangDocumentVerificationAssignment* assignment;
        if (!(assignment = fetchAssignmentResult.GetResultPtr(secretId))) {
            throw yexception() << "Failed to restore assignment by secret id";
        }

        if (!genericChecks.contains(checkType)) {
            return Nothing();
        }

        TVector<TString> photoIds;
        const auto& photoTypes = genericChecks.at(checkType);
        for (auto&& type : photoTypes) {
            if (auto photoId = GetPhotoIdByType(*assignment, type)) {
                photoIds.push_back(std::move(photoId));
            }
        }

        return photoIds;
}

bool TUserDocumentsChecksManager::TChecksConfiguration::Init(
    const TString& checksSettingsVar,
    const TString& checksGenericMapVar,
    const TString& photoChecksPriority,
    const TString& moveChatIds,
    const NDrive::IServer &server,
    TMessagesCollector& errors
) {
    if (auto checksSettings = TUserDocumentsChecksManager::GetChecksSettings(checksSettingsVar, server)) {
        Settings = std::move(*checksSettings);
    } else {
        errors.AddMessage(
            "TUserDocumentsChecksManager::TChecksConfiguration::Init",
            "Failed to get checks settings " + checksSettingsVar
        );
        return false;
    }

    if (auto checksGenericMap = TUserDocumentsChecksManager::GetChecksGenericMap(checksGenericMapVar, server)) {
        GenericMap = std::move(*checksGenericMap);
    } else {
        errors.AddMessage(
            "TUserDocumentsChecksManager::TChecksConfiguration::Init",
            "Failed to get checks generic map " + checksGenericMapVar
        );
        return false;
    }

    if (auto checksPriority = TUserDocumentsChecksManager::GetPhotoChecksPriority(photoChecksPriority, server)) {
        Priority = std::move(*checksPriority);
    } else {
        errors.AddMessage(
            "TUserDocumentsChecksManager::TChecksConfiguration::Init",
            "Failed to get checks priority " + photoChecksPriority
        );
        return false;
    }

    TString chatsRaw = server.GetSettings().GetValue<TString>(moveChatIds).GetOrElse("");
    ChatsToMove = MakeSet(SplitString(chatsRaw, ","));

    return true;
}

TSet<TString> TUserDocumentsChecksManager::TChecksConfiguration::GetCheckTypes(const NUserDocument::EType& photoType) const {
    TSet<TString> checks;
    for (auto&& [checkType, photoTypes] : GenericMap) {
        if (photoTypes.contains(photoType)) {
            checks.insert(checkType);
        }
    }
    return checks;
}
