#pragma once

#include "user_documents.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/common.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>

#include <drive/backend/proto/snapshot.pb.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

class TYangDocumentVerificationAssignment;

namespace NYangAssignment {
    NUserDocument::EVerificationStatus ParsePhotoStatus(const NJson::TJsonValue& yangStatusJson);
}

struct TRegistrationTagInfo {
    TString RegistrationTagName;
    TMap<TString, TString> RegistrationTagMap;
};

class TUserDocumentsCheck {
    R_FIELD(TString, UserId);
    R_FIELD(TString, Type);
    R_FIELD(TString, Status);

    R_FIELD(TString, SecretId);
    R_FIELD(TString, WorkerId);
    R_FIELD(TString, PoolId);
    R_FIELD(TString, ProjectId);
    R_FIELD(TString, AssignmentId);
    R_FIELD(TString, Login);
    R_FIELD(ui64, Timestamp, 0);
    R_FIELD(TString, Comment);

public:
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Status, -1);

        R_FIELD(i32, Meta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            Type = GetFieldDecodeIndex("type", decoderBase);
            Status = GetFieldDecodeIndex("status", decoderBase);

            Meta = GetFieldDecodeIndex("meta", decoderBase);
        }
    };

    TUserDocumentsCheck() = default;

    TUserDocumentsCheck(const TString& userId, const TString& type, const TString& status)
        : UserId(userId)
        , Type(type)
        , Status(status)
    {
    }

    TUserDocumentsCheck(
        const TString& userId, const TString& type,
        const TString& status, const TString& secretId,
        const TString& workerId, const TString& poolId,
        const TString& projectId, const TString& assignmentId,
        const TString& login, const ui64 timestamp, const TString& comment)
        : UserId(userId), Type(type)
        , Status(status), SecretId(secretId)
        , WorkerId(workerId), PoolId(poolId)
        , ProjectId(projectId), AssignmentId(assignmentId)
        , Login(login), Timestamp(timestamp), Comment(comment)
        {
        }

    bool Parse(const NStorage::TTableRecord& row);

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;

    void DoBuildReportItem(NJson::TJsonValue& result) const;

    bool DeserializeMeta(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    TString SerializeMeta() const;
};

class TUserDocumentsChecksHistoryManager : public TDatabaseHistoryManager<TUserDocumentsCheck> {
public:
    using TCheckHistoryEvent = TObjectEvent<TUserDocumentsCheck>;

private:
    using TBase = TDatabaseHistoryManager<TUserDocumentsCheck>;

public:
    TUserDocumentsChecksHistoryManager(const IHistoryContext& context)
        : TBase(context, "user_documents_checks_history")
    {
    }
};

class TUserDocumentsChecksDB: public TDBEntities<TUserDocumentsCheck> {
private:
    using TBase = TDBEntities<TUserDocumentsCheck>;

public:
    TUserDocumentsChecksDB(const IHistoryContext& context)
        : TBase(context.GetDatabase())
        , HistoryWriter(context)
    {
    }

    TString GetTableName() const override {
        return "user_documents_checks";
    }

    TString GetMainId(const TString& userId, const TString& type) const {
        return userId + "-" + type;
    }

    TString GetMainId(const TUserDocumentsCheck& entity) const override {
        return GetMainId(entity.GetUserId(), entity.GetType());
    }

    bool Upsert(const TUserDocumentsCheck& entity, const TString& historyUserId, NDrive::TEntitySession& session) const;
    TVector<TUserDocumentsChecksHistoryManager::TCheckHistoryEvent> GetEvents(const TString& type, const TString& userId, const TInstant& since, NDrive::TEntitySession& session) const;

private:
    TUserDocumentsChecksHistoryManager HistoryWriter;
};

class TUserDocumentsCheckConfig: public TDBEntitiesManagerConfig {
    R_FIELD(TString, ChecksSettingsKey, "user_documents_checks.settings");
    R_FIELD(TString, PhotoChecksSettingsKey, "user_documents_checks.photo_settings");
public:
    using TDBEntitiesManagerConfig::TDBEntitiesManagerConfig;

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TUserDocumentsChecksManager {
public:
    using TChecksSettings = TMap<TString, TSet<TString>>;
    using TChecksGenericMap = TMap<TString, TSet<NUserDocument::EType>>;
    using TPhotoChecksPriority = TMap<TString, std::pair<TString, ui64>>;

    struct TChecksConfiguration {
    public:
        bool Init(
            const TString& checksSettingsVar,
            const TString& checksGenericMapVar,
            const TString& photoChecksPriority,
            const TString& moveChatIds,
            const NDrive::IServer& server,
            TMessagesCollector& errors
        );

        TSet<TString> GetCheckTypes(const NUserDocument::EType& photoType) const;

    public:
        TSet<TString> ChatsToMove;
        TChecksSettings Settings;
        TChecksGenericMap GenericMap;
        TPhotoChecksPriority Priority;
    };

public:
    TUserDocumentsChecksManager(const IHistoryContext& context, const TUserDocumentsCheckConfig& config)
        : UserDocumentsChecksDb(context)
        , Config(config)
    {
    }

    static TString GetPhotoIdByType(const TYangDocumentVerificationAssignment& assignment, NUserDocument::EType type);
    static TMaybe<TVector<TString>> GetPhotoIds(const TString& checkType, const TString& secretId, const TUserDocumentsChecksManager::TChecksGenericMap& genericChecks, const NDrive::IServer& server, NDrive::TEntitySession& session);

    static TMaybe<TChecksSettings> GetChecksSettings(const TString& varName, const NDrive::IServer& server);
    static TMaybe<TChecksGenericMap> GetChecksGenericMap(const TString& varName, const NDrive::IServer& server);
    static TMaybe<TPhotoChecksPriority> GetPhotoChecksPriority(const TString& varName, const NDrive::IServer& server);

    bool SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const TChecksSettings& settings, NDrive::TEntitySession& session) const;
    bool SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    bool SetStatus(const TString& userId, const TString& type, const TString& status, const TString& historyUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    bool SetStatus(const TUserDocumentsCheck& check, const TString& historyUserId, const TVector<TString>& photoIds, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const;
    TMaybe<TString> GetStatus(const TString& userId, const TString& type, NDrive::TEntitySession& session) const;
    TMaybe<TVector<TUserDocumentsCheck>> GetChecks(const TString& userId, const TSet<TString>& types, NDrive::TEntitySession& session) const;
    TChecksSettings GetAllowedStatusesByTypes(const NDrive::IServer& server) const;
    TVector<TUserDocumentsChecksHistoryManager::TCheckHistoryEvent> GetChecksHistory(const TString& userId, const TString& type, const TInstant& since, NDrive::TEntitySession& session) const;

private:
    bool SetPhotosStatus(const TString& userId, const TVector<TString>& photoIds, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const;
    bool ContinueToChat(const TString& userId, const TChecksConfiguration& checksConfig, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) const;

private:
    const TUserDocumentsChecksDB UserDocumentsChecksDb;
    const TUserDocumentsCheckConfig Config;
};
