#include "yang.h"

#include <drive/backend/yang/task.h>

#include <drive/library/cpp/yt/node/cast.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/types/uuid.h>

#include <util/digest/multi.h>

#define YREAD_COMMON(source, name, value, type)\
{\
    if (!(source).HasKey(name) || !((source)[name].Is ## type())) {\
        return false;\
    } else {\
        value = (source)[name].As ## type();\
    }\
}

#define YREAD_COMMON_OPT(source, name, value, type)\
{\
    if ((source).HasKey(name) && (source)[name].IsList()) {\
        YREAD_COMMON(source, name, value, type)\
    }\
}

#define YREAD_ARRAY_COMMON(source, name, value, type)\
{\
    if (!(source).HasKey(name) || !((source)[name].IsList())) {\
        return false;\
    } else {\
        value.clear();\
        for (auto&& item : (source)[name].AsList()) {\
            if ((item).IsNull()) {\
                continue;\
            }\
            if (!(item.Is ## type())) {\
                return false;\
            }\
            value.push_back(item.As ## type());\
        }\
    }\
}

#define YREAD_ARRAY_COMMON_OPT(source, name, value, type)\
{\
    if ((source).HasKey(name) && (source)[name].IsList()) {\
        YREAD_ARRAY_COMMON(source, name, value, type)\
    } else {\
        if (!((source)[name].IsNull())) {\
            return false;\
        }\
    }\
}

#define YREAD_STRING(source, name, value) YREAD_COMMON(source, name, value, String);
#define YREAD_STRING_OPT(source, name, value) YREAD_COMMON_OPT(source, name, value, String);
#define YREAD_STRING_ARR_OPT(source, name, value) YREAD_ARRAY_COMMON_OPT(source, name, value, String);

TYangDocumentVerificationAssignment::TYangDocumentVerificationAssignment(
    const TString& licenseBackId,
    const TString& licenseFrontId,
    const TString& licenseSelfieId,
    const TString& passportBiographicalId,
    const TString& passportRegistrationId,
    const TString& passportSelfieId,
    const TString& selfieId,
    const TString& licenseVideoId,
    const TString& passportVideoId,
    const TString& videoSelfieId,
    const TString& userId
)
    : LicenseBackId(licenseBackId)
    , LicenseFrontId(licenseFrontId)
    , LicenseSelfieId(licenseSelfieId)
    , PassportBiographicalId(passportBiographicalId)
    , PassportRegistrationId(passportRegistrationId)
    , PassportSelfieId(passportSelfieId)
    , SelfieId(selfieId)
    , LicenseVideoId(licenseVideoId)
    , PassportVideoId(passportVideoId)
    , VideoSelfieId(videoSelfieId)
    , UserId(userId)
    , CreatedAt(ModelingNow())
{
    Id = NUtil::CreateUUID();
}

TYangDocumentVerificationAssignment::TYangDocumentVerificationAssignment(TMap<NUserDocument::EType, TString> typeIds, const TString& userId)
    : TYangDocumentVerificationAssignment(
        typeIds[NUserDocument::EType::LicenseBack],
        typeIds[NUserDocument::EType::LicenseFront],
        typeIds[NUserDocument::EType::LicenseSelfie],
        typeIds[NUserDocument::EType::PassportBiographical],
        typeIds[NUserDocument::EType::PassportRegistration],
        typeIds[NUserDocument::EType::PassportSelfie],
        typeIds[NUserDocument::EType::Selfie],
        typeIds[NUserDocument::EType::LicenseVideo],
        typeIds[NUserDocument::EType::PassportVideo],
        typeIds[NUserDocument::EType::VideoSelfie],
        userId
    )
{
}

ui64 TYangDocumentVerificationAssignment::CalcHash() const {
    return MultiHash(
        LicenseBackId,
        LicenseFrontId,
        LicenseSelfieId,
        PassportBiographicalId,
        PassportRegistrationId,
        PassportSelfieId,
        SelfieId,
        UserId,
        LicenseVideoId,
        PassportVideoId,
        VideoSelfieId
    );
}

void TYangDocumentVerificationAssignment::SerializeToYtNode(NYT::TNode& row) const {
    row["secret_id"] = Id;
    row["user_id"] = UserId;
    row["meta"] = Meta.GetStringRobust();
    if (LicenseBackConfidence) {
        row["license_back_confidence"] = NYT::ToNode(LicenseBackConfidence->SerializeDrivingLicenseBack());
    }
    if (LicenseFrontConfidence) {
        row["license_front_confidence"] = NYT::ToNode(LicenseFrontConfidence->SerializeDrivingLicenseFront());
    }
    if (PassportBiographicalConfidence) {
        row["passport_biographical_confidence"] = NYT::ToNode(PassportBiographicalConfidence->SerializePassportBiographical());
    }
}

bool TYangDocumentVerificationAssignment::DeserializeFromYtNode(const NYT::TNode& row) {
    YREAD_STRING(row, "secretId", Id);

    if (!row.HasKey("result") || !row["result"].IsMap()) {
        return false;
    }
    {
        TString isFraudStr;
        YREAD_STRING(row["result"], "is_fraud", isFraudStr);
        isFraudStr = ToLowerUTF8(isFraudStr);
        if (!TryFromString(isFraudStr, IsFraud)) {
            IsFraud = TYangDocumentVerificationAssignment::EFraudStatus::Unverified;
        }
    }
    YREAD_STRING(row["result"], "license_back_status", LicenseBackStatus);
    YREAD_STRING(row["result"], "license_front_status", LicenseFrontStatus);
    YREAD_STRING(row["result"], "passport_biographical_status", PassportBiographicalStatus);
    YREAD_STRING(row["result"], "passport_registration_status", PassportRegistrationStatus);
    YREAD_STRING(row["result"], "passport_selfie_status", PassportSelfieStatus);
    YREAD_STRING_OPT(row["result"], "license_video_status", LicenseVideoStatus);
    YREAD_STRING_OPT(row["result"], "passport_video_status", PassportVideoStatus);
    YREAD_STRING_OPT(row["result"], "video_selfie_status", VideoSelfieStatus);

    if (row.HasKey("history")) {
        NJson::TJsonValue historyJson = NYT::FromNode<NJson::TJsonValue>(row["history"]);
        History = historyJson.GetStringRobust();
    }

    YREAD_STRING_ARR_OPT(row, "assignmentIds", AssignmentIds);
    YREAD_STRING_ARR_OPT(row, "comments", Comments);
    YREAD_STRING_ARR_OPT(row, "workersIds", Workers);

    return true;
}

bool TYangDocumentVerificationAssignment::Parse(const NStorage::TTableRecord& record) {
    Id = record.Get("id");
    LicenseBackId = record.Get("license_back_id");
    LicenseFrontId = record.Get("license_front_id");
    LicenseSelfieId = record.Get("license_selfie_id");
    PassportBiographicalId = record.Get("passport_biographical_id");
    PassportRegistrationId = record.Get("passport_registration_id");
    PassportSelfieId = record.Get("passport_selfie_id");
    SelfieId = record.Get("selfie_id");
    UserId = record.Get("user_id");
    LicenseVideoId = record.Get("license_video_id");
    PassportVideoId = record.Get("passport_video_id");
    VideoSelfieId = record.Get("video_selfie_id");
    History = record.Get("history");

    TString isFraudStr = record.Get("is_fraud");
    if (!isFraudStr || !TryFromString(isFraudStr, IsFraud)) {
        IsFraud = TYangDocumentVerificationAssignment::EFraudStatus::Unverified;
    }

    ParseArrayFromDB(record.Get("assignment_ids"), AssignmentIds);
    ParseArrayFromDB(record.Get("fraud_reasons"), FraudReasons);
    ParseArrayFromDB(record.Get("comments"), Comments);
    ParseArrayFromDB(record.Get("workers"), Workers);

    if (record.Get("created_at") && !TInstant::TryParseIso8601(record.Get("created_at"), CreatedAt)) {
        return false;
    }
    if (record.Get("processed_at") && !TInstant::TryParseIso8601(record.Get("processed_at"), ProcessedAt)) {
        return false;
    }
    if (record.Get("ingested_at") && !TInstant::TryParseIso8601(record.Get("ingested_at"), IngestedAt)) {
        return false;
    }
    if (record.Get("finalized_at") && !TInstant::TryParseIso8601(record.Get("finalized_at"), FinalizedAt)) {
        return false;
    }
    if (record.Get("is_experimental") && !record.TryGetDefault("is_experimental", IsExperimental, false)) {
        return false;
    }
    return true;
}

NJson::TJsonValue TYangDocumentVerificationAssignment::BuildReport() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["license_back_id"] = LicenseBackId;
    result["license_front_id"] = LicenseFrontId;
    result["license_selfie_id"] = LicenseSelfieId;
    result["passport_biographical_id"] = PassportBiographicalId;
    result["passport_registration_id"] = PassportRegistrationId;
    result["passport_selfie_id"] = PassportSelfieId;
    result["selfie_id"] = SelfieId;
    result["license_video_id"] = LicenseVideoId;
    result["passport_video_id"] = PassportVideoId;
    result["video_selfie_id"] = VideoSelfieId;
    if (History) {
        result["history"] = History;
    }
    if (IsFraud != TYangDocumentVerificationAssignment::EFraudStatus::Unverified) {
        result["is_fraud"] = ToString(IsFraud);
    } else {
        result["is_fraud"] = NJson::JSON_NULL;
    }
    result["assignment_ids"] = NJson::JSON_ARRAY;
    for (auto&& assignmentId : AssignmentIds) {
        result["assignment_ids"].AppendValue(assignmentId);
    }
    result["fraud_reasons"] = NJson::JSON_ARRAY;
    for (auto&& fraudReason : FraudReasons) {
        result["fraud_reasons"].AppendValue(fraudReason);
    }
    result["comments"] = NJson::JSON_ARRAY;
    for (auto&& comment : Comments) {
        result["comments"].AppendValue(comment);
    }
    result["created_at"] = ToString(CreatedAt);
    if (ProcessedAt != TInstant::Zero()) {
        result["processed_at"] = ToString(ProcessedAt);
    }
    if (FinalizedAt != TInstant::Zero()) {
        result["finalized_at"] = ToString(FinalizedAt);
    }
    result["is_experemental"] = IsExperimental;
    return result;
}

TString TYangDocumentsAssignmentsDB::MakeQuery(const TSet<TString>& userIds, const TSet<NUserDocument::EType>& documents, NStorage::ITransaction::TPtr transaction) const {
    auto query = TStringBuilder() << "SELECT * FROM " << GetTableName() << " WHERE True";
    if (userIds) {
        query << " AND (user_id in (" << transaction->Quote(userIds) << "))";
    }
    auto adjust = [&](NUserDocument::EType type, TStringBuf field) {
        if (documents.contains(type)) {
            query << " AND " << field << " IS NOT null";
        }
    };
    adjust(NUserDocument::LicenseBack, "license_back_id");
    adjust(NUserDocument::LicenseFront, "license_front_id");
    adjust(NUserDocument::LicenseSelfie, "license_selfie_id");
    adjust(NUserDocument::PassportBiographical, "passport_biographical_id");
    adjust(NUserDocument::PassportRegistration, "passport_registration_id");
    adjust(NUserDocument::PassportSelfie, "passport_selfie_id");
    adjust(NUserDocument::Selfie, "selfie_id");
    adjust(NUserDocument::LicenseVideo, "license_video_id");
    adjust(NUserDocument::PassportVideo, "passport_video_id");
    adjust(NUserDocument::VideoSelfie, "video_selfie_id");
    return query;
}

TYangDocumentsAssignmentsDB::TFetchResult TYangDocumentsAssignmentsDB::GetNotVerifiedCustomAssignments(const TSet<NUserDocument::EType>& documents) const {
    auto transaction = Database->CreateTransaction(true);
    auto query = MakeQuery({}, documents, transaction) + " AND processed_at IS null";
    return FetchWithCustomQuery(query, transaction);
}

TVector<TYangDocumentVerificationAssignment> TYangDocumentsAssignmentsDB::GetAllAssignments(const TSet<TString>& userIds, const TSet<NUserDocument::EType>& documents) const {
    auto transaction = Database->CreateTransaction(true);
    auto query = MakeQuery(userIds, documents, transaction);
    auto fetchResult = FetchWithCustomQuery(query, transaction);
    auto result = MakeVector(NContainer::Values(fetchResult.GetResult()));
    std::sort(result.begin(), result.end(), [](const TYangDocumentVerificationAssignment& lhs, const TYangDocumentVerificationAssignment& rhs) {
        return lhs.GetCreatedAt() > rhs.GetCreatedAt();
    });
    return result;
}

TVector<TYangDocumentVerificationAssignment> TYangDocumentsAssignmentsDB::GetAllAssignmentsForUsers(const TSet<TString>& userIds) const {
    auto transaction = Database->CreateTransaction(true);
    if (userIds.empty()) {
        return {};
    }

    TStringStream queryStream;
    queryStream << "SELECT * FROM " + GetTableName() + " WHERE (user_id in (" << transaction->Quote(userIds) << "))";

    auto fetchTasksResult = FetchWithCustomQuery(queryStream.Str(), transaction);
    TVector<TYangDocumentVerificationAssignment> verificationTasks;
    verificationTasks.reserve(fetchTasksResult.size());
    for (auto&& it : fetchTasksResult) {
        verificationTasks.push_back(std::move(it.second));
    }

    Sort(
        verificationTasks.begin(),
        verificationTasks.end(),
        [](const TYangDocumentVerificationAssignment& lhs, const TYangDocumentVerificationAssignment& rhs) {
            return lhs.GetCreatedAt() > rhs.GetCreatedAt();
        }
    );

    return verificationTasks;
}

TVector<TYangDocumentVerificationAssignment> TYangDocumentsAssignmentsDB::GetAllAssignmentsForUser(const TString& userId) const {
    auto result = GetAllAssignmentsForUsers({userId});
    std::reverse(result.begin(), result.end());
    return result;
}

bool TYangDocumentsAssignmentsDB::GetLastAssignmentForUser(const TString& userId, TYangDocumentVerificationAssignment& assignment) const {
    auto queryResult = GetLastAssignmentsForUsers({userId});
    if (queryResult.empty()) {
        return false;
    }
    assignment = std::move(queryResult.front());
    return true;
}

TVector<TYangDocumentVerificationAssignment> TYangDocumentsAssignmentsDB::GetLastAssignmentsForUsers(const TSet<TString>& userIds) const {
    auto verificationTasks = GetAllAssignmentsForUsers(userIds);

    TSet<TString> processedUsers;
    TVector<TYangDocumentVerificationAssignment> result;
    for (auto&& task : verificationTasks) {
        TString userId = task.GetUserId();
        if (!processedUsers.contains(userId)) {
            processedUsers.insert(userId);
            result.push_back(std::move(task));
        }
    }

    return result;
}

TSet<TString> TYangDocumentsAssignmentsDB::GetQueuedRegisteringUsers() const {
    auto activeAssignments = GetNotVerifiedRegistrationAssignments();

    TSet<TString> userIds;
    for (auto&& assignmentIt : activeAssignments) {
        userIds.insert(assignmentIt.second.GetUserId());
    }

    return userIds;
}

void TYangDocumentsAssignmentsDB::FillUserReport(const TString& userId, NJson::TJsonValue& result) const {
    auto assignments = GetAllAssignmentsForUser(userId);
    result = NJson::JSON_ARRAY;
    for (auto&& assignment : assignments) {
        result.AppendValue(assignment.BuildReport());
    }
}

TYangDocumentVerificationAssignment TYangDocumentVerificationAssignment::Clone(const TYangDocumentVerificationAssignment& assignment) {
    return TYangDocumentVerificationAssignment(
        assignment.GetLicenseBackId(),
        assignment.GetLicenseFrontId(),
        assignment.GetLicenseSelfieId(),
        assignment.GetPassportBiographicalId(),
        assignment.GetPassportRegistrationId(),
        assignment.GetPassportSelfieId(),
        assignment.GetSelfieId(),
        assignment.GetLicenseVideoId(),
        assignment.GetPassportVideoId(),
        assignment.GetVideoSelfieId(),
        assignment.GetUserId()
    );
}

NStorage::TTableRecord TYangDocumentVerificationAssignment::SerializeToTableRecord() const {
    NStorage::TTableRecord result;

    result.Set("id", Id);
    if (LicenseBackId) {
        result.Set("license_back_id", LicenseBackId);
    }
    if (LicenseFrontId) {
        result.Set("license_front_id", LicenseFrontId);
    }
    if (LicenseSelfieId) {
        result.Set("license_selfie_id", LicenseSelfieId);
    }
    if (PassportBiographicalId) {
        result.Set("passport_biographical_id", PassportBiographicalId);
    }
    if (PassportRegistrationId) {
        result.Set("passport_registration_id", PassportRegistrationId);
    }
    if (PassportSelfieId) {
        result.Set("passport_selfie_id", PassportSelfieId);
    }
    if (SelfieId) {
        result.Set("selfie_id", SelfieId);
    }
    if (UserId) {
        result.Set("user_id", UserId);
    }

    if (LicenseVideoId) {
        result.Set("license_video_id", LicenseVideoId);
    }
    if (PassportVideoId) {
        result.Set("passport_video_id", PassportVideoId);
    }
    if (VideoSelfieId) {
        result.Set("video_selfie_id", VideoSelfieId);
    }

    if (!!History) {
        result.Set("history", History);
    }

    if (IsFraud != TYangDocumentVerificationAssignment::EFraudStatus::Unverified) {
        result.Set("is_fraud", ToString(IsFraud));
    }
    if (AssignmentIds.size()) {
        result.Set("assignment_ids", ToDBArray(AssignmentIds));
    }
    if (FraudReasons.size()) {
        result.Set("fraud_reasons", ToDBArray(FraudReasons));
    }
    if (Workers.size()) {
        result.Set("workers", ToDBArray(Workers));
    }
    if (Comments.size()) {
        result.Set("comments", ToDBArray(Comments, true));
    }
    if (CreatedAt != TInstant::Zero()) {
        result.Set("created_at", CreatedAt.ToString());
    }
    if (ProcessedAt != TInstant::Zero()) {
        result.Set("processed_at", ProcessedAt.ToString());
    }

    {
        if (IngestedAt != TInstant::Zero()) {
            result.Set("ingested_at", IngestedAt.ToString());
        }
        if (FinalizedAt != TInstant::Zero()) {
            result.Set("finalized_at", FinalizedAt.ToString());
        }
        if (NeedReprocessing) {
            result.ForceSet("ingested_at", "get_null()");
            result.ForceSet("finalized_at", "get_null()");
        }
    }

    if (IsExperimental) {
        result.Set("is_experimental", true);
    }

    return result;
}

void TYangDocumentVerificationAssignment::ParseArrayFromDB(const TString& str, TVector<TString>& result) const {
    result.clear();

    if (str.size() <= 2) {
        return;
    }

    bool isQuoteOpened = false;
    TString currentToken = "";
    for (size_t i = 1; i < str.size() - 1; ++i) {
        if (str[i] == '"') {
            isQuoteOpened ^= true;
            continue;
        }
        if (!isQuoteOpened && str[i] == ',' && currentToken) {
            result.push_back(currentToken);
            currentToken = "";
            continue;
        }
        if (str[i] == '\\' && str[i + 1] == '"') {
            currentToken += "\"";
            i += 1;
        } else if (str[i] == '\\' && str[i + 1] == '\'') {
            currentToken += "'";
            i += 1;
        } else if (str[i] == '\\' && str[i + 1] == '\\') {
            currentToken += "\\";
            i += 1;
        } else {
            currentToken += str[i];
        }
    }

    if (currentToken) {
        result.push_back(currentToken);
    }
}

TString TYangDocumentVerificationAssignment::ToDBArray(TVector<TString> vec, bool escape) const {
    if (escape) {
        for (auto& element : vec) {
            SubstGlobal(element, "\\", "\\\\");
            SubstGlobal(element, "\"", "\\\"");
            SubstGlobal(element, "\'", "\\\'");
        }
    }

    TString result = "{";
    for (auto&& element : vec) {
        if (result.size() > 1) {
            result += ",";
        }
        result += "\"" + element + "\"";
    }
    result += "}";

    return result;
}

TString TYangDocumentVerificationAssignment::GetDocumentId(NUserDocument::EType documentType) const {
    switch (documentType) {
        case NUserDocument::LicenseFront:
            return LicenseFrontId;
        case NUserDocument::LicenseBack:
            return LicenseBackId;
        case NUserDocument::PassportBiographical:
            return PassportBiographicalId;
        case NUserDocument::PassportRegistration:
            return PassportRegistrationId;
        case NUserDocument::PassportSelfie:
            return PassportSelfieId;
        case NUserDocument::Selfie:
            return SelfieId;
        case NUserDocument::LicenseSelfie:
            return LicenseSelfieId;
        case NUserDocument::LicenseVideo:
            return LicenseVideoId;
        case NUserDocument::PassportVideo:
            return PassportVideoId;
        case NUserDocument::VideoSelfie:
            return VideoSelfieId;
        default:
            return "";
    }
}

TVector<TYangDocumentVerificationAssignment> TYangDocumentsAssignmentsDB::GetUnfinalizedAssignments(const bool onlyExperimental) const {
    TStringStream queryStream;
    queryStream << "SELECT * FROM " + GetTableName() + " WHERE (finalized_at is null) AND (processed_at is not null)";
    if (onlyExperimental) {
        queryStream << " AND (is_experimental = '1')";
    }

    auto fetchResult = FetchWithCustomQuery(queryStream.Str());

    TVector<TYangDocumentVerificationAssignment> result;
    result.reserve(fetchResult.GetResult().size());

    for (auto&& assignmentIt : fetchResult) {
        result.emplace_back(std::move(assignmentIt.second));
    }

    return result;
}

void TYangVideoScreencapAssignment::SerializeToYtNode(NYT::TNode& /*row*/) const {
    Y_UNREACHABLE();
    // not needed
}

bool TYangVideoScreencapAssignment::DeserializeFromYtNode(const NYT::TNode& row) {
    YREAD_STRING(row, "secret_id", Id);

    YREAD_STRING(row["result"], "license_back", LicenseBackStatus);
    YREAD_STRING(row["result"], "license_front", LicenseFrontStatus);
    YREAD_STRING(row["result"], "passport_biographical", PassportBiographicalStatus);
    YREAD_STRING(row["result"], "passport_registration", PassportRegistrationStatus);
    YREAD_STRING(row["result"], "passport_selfie", PassportSelfieStatus);
    return true;
}

bool TYangVideoScreencapAssignment::IsFraudlent() const {
    return (
        LicenseBackStatus == "SCREENCAP" ||
        LicenseFrontStatus == "SCREENCAP" ||
        LicenseSelfieStatus == "SCREENCAP" ||
        PassportBiographicalStatus == "SCREENCAP" ||
        PassportRegistrationStatus == "SCREENCAP" ||
        PassportSelfieStatus == "SCREENCAP"
    );
}

template <>
NJson::TJsonValue NJson::ToJson(const TYangDocumentVerificationAssignment& object) {
    return object.BuildReport();
}
