#pragma once

#include "user_documents.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/yang/task.h>

#include <rtline/util/types/accessor.h>

#include <util/string/split.h>
#include <util/string/vector.h>

class TYangDocumentVerificationAssignment : public IYangAssignment {
public:
    enum EFraudStatus {
        Unverified /* "unverified" */,
        NotFraud /* "not_fraud" */,
        MaybeFraud /* "maybe_fraud" */,
        DefinitelyFraud /* "definitely_fraud" */
    };

    R_READONLY(TString, Id);

    // photos for verification
    R_FIELD(TString, LicenseBackId);
    R_FIELD(TString, LicenseFrontId);
    R_FIELD(TString, LicenseSelfieId);
    R_FIELD(TString, PassportBiographicalId);
    R_FIELD(TString, PassportRegistrationId);
    R_FIELD(TString, PassportSelfieId);
    R_FIELD(TString, SelfieId);

    // videos for verification
    R_FIELD(TString, LicenseVideoId);
    R_FIELD(TString, PassportVideoId);
    R_FIELD(TString, VideoSelfieId);

    R_FIELD(TString, UserId);

    // results
    R_FIELD(EFraudStatus, IsFraud, EFraudStatus::Unverified);
    R_FIELD(TString, LicenseBackStatus);
    R_FIELD(TString, LicenseFrontStatus);
    R_FIELD(TString, LicenseSelfieStatus);
    R_FIELD(TString, PassportBiographicalStatus);
    R_FIELD(TString, PassportRegistrationStatus);
    R_FIELD(TString, PassportSelfieStatus);
    R_FIELD(TString, SelfieStatus);
    R_FIELD(TString, LicenseVideoStatus);
    R_FIELD(TString, PassportVideoStatus);
    R_FIELD(TString, VideoSelfieStatus);

    R_FIELD(TVector<TString>, AssignmentIds);
    R_FIELD(TVector<TString>, Comments);
    R_FIELD(TVector<TString>, Workers);
    R_FIELD(TVector<TString>, FraudReasons);

    R_FIELD(TString, History);

    // timestamps
    R_READONLY(TInstant, CreatedAt, TInstant::Zero());
    R_FIELD(TInstant, ProcessedAt, TInstant::Zero());       // when fetched back from Yang

    R_FIELD(TInstant, IngestedAt, TInstant::Zero());       // when handled by registration daemon

    R_FIELD(TInstant, FinalizedAt, TInstant::Zero());

    // migration
    R_FIELD(bool, IsExperimental, false);

    // special
    R_FIELD(bool, NeedReprocessing, false);

    R_FIELD(NJson::TJsonValue, Meta);
    R_OPTIONAL(TRecognitionConfidenceData, LicenseBackConfidence);
    R_OPTIONAL(TRecognitionConfidenceData, LicenseFrontConfidence);
    R_OPTIONAL(TRecognitionConfidenceData, PassportBiographicalConfidence);

public:
    TYangDocumentVerificationAssignment() = default;
    TYangDocumentVerificationAssignment(const TYangDocumentVerificationAssignment& other) = default;
    TYangDocumentVerificationAssignment(const TString& id)
        : Id(id)
    {
    }

    TYangDocumentVerificationAssignment(
        const TString& licenseBackId,
        const TString& licenseFrontId,
        const TString& licenseSelfieId,
        const TString& passportBiographicalId,
        const TString& passportRegistrationId,
        const TString& passportSelfieId,
        const TString& selfieId,
        const TString& licenseVideoId,
        const TString& passportVideoId,
        const TString& videoSelfieId,
        const TString& userId
    );
    TYangDocumentVerificationAssignment(TMap<NUserDocument::EType, TString> typeIds, const TString& userId);

    ui64 CalcHash() const;

    virtual void SerializeToYtNode(NYT::TNode& row) const override;
    virtual bool DeserializeFromYtNode(const NYT::TNode& row) override;

    bool Parse(const NStorage::TTableRecord& record);
    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue BuildReport() const;

    TString GetDocumentId(NUserDocument::EType documentType) const;

public:
    static TYangDocumentVerificationAssignment Clone(const TYangDocumentVerificationAssignment& assignment);

private:
    void ParseArrayFromDB(const TString& str, TVector<TString>& result) const;

    TString ToDBArray(TVector<TString> vec, bool escape=false) const;
};

class TYangDocumentsAssignmentsDB: public TDBEntities<TYangDocumentVerificationAssignment> {
private:
    using TBase = TDBEntities<TYangDocumentVerificationAssignment>;

public:
    using TBase::TBase;

    TYangDocumentsAssignmentsDB(const NStorage::IDatabase::TPtr database)
        : TBase(database)
    {
    }

    virtual TString GetTableName() const override {
        return "yang_assignments";
    }

    virtual TString GetMainId(const TYangDocumentVerificationAssignment& e) const override {
        return e.GetId();
    }

    TFetchResult GetNotVerifiedCustomAssignments(const TSet<NUserDocument::EType>& documents) const;

    TFetchResult GetNotVerifiedRegistrationAssignments() const {
        return FetchWithCustomQuery("SELECT * FROM " + GetTableName() + " WHERE processed_at IS null AND passport_biographical_id IS NOT null");
    }

    TFetchResult GetNotVerifiedSelfieAssignments() const {
        return FetchWithCustomQuery("SELECT * FROM " + GetTableName() + " WHERE processed_at IS null AND selfie_id IS NOT null");
    }

    void FillUserReport(const TString& userId, NJson::TJsonValue& result) const;

    TVector<TYangDocumentVerificationAssignment> GetUnfinalizedAssignments(const bool onlyExperimental) const;

    TSet<TString> GetQueuedRegisteringUsers() const;

    TVector<TYangDocumentVerificationAssignment> GetAllAssignments(const TSet<TString>& userIds, const TSet<NUserDocument::EType>& documents) const;
    TVector<TYangDocumentVerificationAssignment> GetAllAssignmentsForUser(const TString& userId) const;
    TVector<TYangDocumentVerificationAssignment> GetAllAssignmentsForUsers(const TSet<TString>& userIds) const;
    TVector<TYangDocumentVerificationAssignment> GetLastAssignmentsForUsers(const TSet<TString>& userIds) const;
    bool GetLastAssignmentForUser(const TString& userId, TYangDocumentVerificationAssignment& assignment) const;

private:
    using TBase::MakeQuery;
    TString MakeQuery(const TSet<TString>& userIds, const TSet<NUserDocument::EType>& documents, NStorage::ITransaction::TPtr transaction) const;
};

class TYangVideoScreencapAssignment : public IYangAssignment {
    R_READONLY(TString, Id);

    R_READONLY(TString, LicenseBackStatus);
    R_READONLY(TString, LicenseFrontStatus);
    R_READONLY(TString, LicenseSelfieStatus);
    R_READONLY(TString, PassportBiographicalStatus);
    R_READONLY(TString, PassportRegistrationStatus);
    R_READONLY(TString, PassportSelfieStatus);

public:
    virtual void SerializeToYtNode(NYT::TNode& row) const override;
    virtual bool DeserializeFromYtNode(const NYT::TNode& row) override;

    bool IsFraudlent() const;
};
