#include "ucontroller.h"

#include <drive/library/cpp/maps_router/router.h>

#include <library/cpp/string_utils/parse_vector/vector_parser.h>

#include <rtline/library/geometry/polyline.h>
#include <rtline/library/graph/geometry_graph/common/types.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/protos/proto_helper.h>

#include <util/string/join.h>

void TUsersControllerConfig::Init(const TYandexConfig::Section* section) {
    SurgeServiceName = section->GetDirectives().Value("SurgeServiceName", SurgeServiceName);
    AssertCorrectConfig(!!SurgeServiceName, "incorrect SurgeServiceName configuration field");
}

void TUsersControllerConfig::ToString(IOutputStream& os) const {
    os << "SurgeServiceName: " << SurgeServiceName << Endl;
}

bool TUsersController::GetAreas(const TDuration duration, const TGeoCoord& startPosition, const ui32 numSegments, TVector<TWalkingArea>& areas, bool useMapsRouter) const {
    if (!numSegments) {
        return true;
    }
    const TDuration timeout = Server->GetSettings().GetValueDef<TDuration>("scanner.areas.timeout", TDuration::Seconds(1));
    TVector<std::pair<TGeoCoord, TGeoCoord>> startFinish;
    if (useMapsRouter) {
        auto router = Server->GetPedestrianRouter();
        Y_ENSURE(router);
        auto isochrone = router->GetIsochrone(startPosition, duration);
        if (isochrone.Initialized() && isochrone.Wait(timeout) && isochrone.HasValue()) {
            for (const auto& coord : isochrone.GetValue()) {
                startFinish.emplace_back(std::make_pair(startPosition, coord));
            }
        } else {
            return false;
        }
    } else {
        NRTLine::TQuery query;
        query.AddExtraParam("component", "Graph");
        query.AddExtraParam("meta_search", "first_found");
        query.AddExtraParam("ms", "proto");
        TVector<NRTLine::TSearchReply> contexts;

        query.SetText("type:routing_area;border:" + ToString(duration.Seconds() / numSegments) + ";start:" + startPosition.ToString() + ";perm:ptPedestrian;d:500");
        query.SetTimeout(timeout);
        contexts.emplace_back(RTLineAPIController->GetSearchClient().SendAsyncQuery(query, timeout));

        query.SetText("type:routing_area;border:" + ToString(duration.Seconds()) + ";start:" + startPosition.ToString() + ";perm:ptPedestrian;d:500");
        query.SetTimeout(timeout);
        contexts.emplace_back(RTLineAPIController->GetSearchClient().SendAsyncQuery(query, timeout));

        for (ui32 i = 0; i < contexts.size(); ++i) {
            const NRTLine::TSearchReply& reply = contexts[i];
            if (!reply.IsSucceeded()) {
                return false;
            }
            TWalkingArea walkingArea;
            if (!walkingArea.Parse(reply.GetReport())) {
                return false;
            }
            areas.emplace_back(std::move(walkingArea));
        }

        const TPolyLine<TGeoCoord> minArea(areas.front().GetCoords());
        const TPolyLine<TGeoCoord> maxArea(areas.back().GetCoords());
        TGeoCoord minStartPos;
        TPolyLine<TGeoCoord>::TPosition startPos;
        if (!minArea.IsPointInternal(startPosition)) {
            if (!minArea.Project(startPosition, startPos, &minStartPos)) {
                return false;
            }
        } else {
            minStartPos = startPosition;
        }
        for (auto&& i : maxArea.GetCoords()) {
            TGeoCoord c;
            if (minArea.CrossCheck(minStartPos, i, &c)) {
                startFinish.emplace_back(std::make_pair(c, i));
            } else {
                startFinish.emplace_back(std::make_pair(minStartPos, i));
            }
        }
        areas.resize(areas.size() - 1);
    }

    for (ui32 i = 0; i < numSegments; ++i) {
        TVector<TGeoCoord> coords;
        double alpha = 1.0 * (i + 1) / numSegments;
        for (auto&& i : startFinish) {
            coords.emplace_back(i.first * (1 - alpha) + i.second * (alpha));
        }
        TWalkingArea area;
        area.SetDuration(TDuration::Seconds(duration.Seconds() / numSegments * (i + 1)));
        area.SetCoords(std::move(coords));
        areas.emplace_back(std::move(area));
    }
    return true;
}

bool TWalkingArea::Parse(const NMetaProtocol::TReport& report) {
    for (auto&& grouping : report.GetGrouping()) {
        for (auto&& group : grouping.GetGroup()) {
            for (auto&& d : group.GetDocument()) {
                TReadSearchProtoHelper helper(d);
                TString coordsString;
                if (helper.GetProperty("routing_area", coordsString)) {
                    return TGeoCoord::DeserializeVector(coordsString, Coords);
                }
            }
        }
    }
    return false;
}
