#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/util/types/accessor.h>

class TWalkingArea {
    R_FIELD(TDuration, Duration, TDuration::Max());
    R_FIELD(TVector<TGeoCoord>, Coords);

public:
    TWalkingArea() = default;

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result;
        result.InsertValue("coords", TGeoCoord::SerializeVectorToJsonIFace(Coords));
        result.InsertValue("duration", Duration.Seconds());
        return result;
    }

    bool Parse(const NMetaProtocol::TReport& report);
};

class IUsersController {
public:
    virtual ~IUsersController() = default;
    virtual bool GetAreas(const TDuration duration, const TGeoCoord& startPosition, const ui32 numSegments, TVector<TWalkingArea>& area, bool useMapsRouter = false) const = 0;
};

class TUsersControllerConfig {
private:
    R_READONLY(TString, SurgeServiceName);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TUsersController: public IUsersController {
private:
    const NDrive::IServer* Server = nullptr;
    const TRTLineAPI* RTLineAPIController;
    const TUsersControllerConfig Config;

public:
    TUsersController(const TUsersControllerConfig& config, const NDrive::IServer* server)
        : Server(server)
        , RTLineAPIController(Server ? Server->GetRTLineAPI(config.GetSurgeServiceName()) : nullptr)
        , Config(config)
    {
        AssertCorrectConfig(!!RTLineAPIController, "incorrect SurgeServiceName: %s", config.GetSurgeServiceName().c_str());
    }

    virtual bool GetAreas(const TDuration duration, const TGeoCoord& startPosition, const ui32 numSegments, TVector<TWalkingArea>& area, bool useMapsRouter = false) const override;
};
