#include <drive/backend/ut/library/drivematics.h>

#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/database/drive_api.h>

#include <library/cpp/testing/unittest/registar.h>

namespace {
    const TSet<TString> RequiredTagsInArea{"allow_drop_car"};

    using TAreaIdToArea = TMap<TString, TArea>;

    TAreaIdToArea CheckCarWithTaggedZone(const NDrive::TZoneIds carInZonesIds, const TSet<TString>& requiredTags, const TVector<NDrivematics::TZone>& zoneStorage, const NDrive::IServer& server) {
        TAreaIdToArea areaToReport;
        auto action = [&areaToReport, &requiredTags](const TString& key, const TArea& entity) -> void {
            if (requiredTags.empty() || MakeIntersection(requiredTags, entity.GetTags()).size() == requiredTags.size()) {
                areaToReport[key] = entity;
            }
        };

        for (const auto& zone : zoneStorage) {
            if (carInZonesIds.contains(zone.GetInternalId())) {
                R_ENSURE(
                    server.GetDriveAPI()->GetAreasDB()->ForObjectsMap(action, {}, zone.GetAreaIdsPtr())
                    , {}
                    , "cannot get objects from cache"
                    , NDrive::MakeError("area.object_not_found")
                );
            }
        }
        return areaToReport;
    }

    const TString addZone = R"({
        "object": [
            {
                "area_title": "title_zone_allow_drop",
                "area_coords": [
                    [
                        38.80394376370033,
                        53.9203403411307
                    ],
                    [
                        38.956712419639224,
                        53.91407334063559
                    ],
                    [
                        38.93383955917397,
                        53.818194269414676
                    ],
                    [
                        38.77162282508495,
                        53.83119267419434
                    ],
                    [
                        38.80394376370033,
                        53.9203403411307
                    ]
                ],
                "area_type": "zone",
                "revision": 100,
                "area_details": {
                    "group": "zone allow drop",
                    "attributed_entity": {
                        "groupping_attributes": []
                    }
                },
                "area_tags": "allow_drop_car"
            }
        ],
        "for_mobile": true,
        "revision": 1
    })";
    const TString removeZone = R"({"ids": ["zone allow drop"]})";

    TGeoCoord CarLocation {38.862047, 53.880881};
}

Y_UNIT_TEST_SUITE(ACL) {
    Y_UNIT_TEST(ZoneProcess) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment((ui32)EEnvironmentFeatures::Root);

        auto userId = TString{USER_ROOT_DEFAULT};

        auto reply = configGenerator.ZoneRequest(userId, "get", {});
        UNIT_ASSERT(!reply.IsSuccessReply());
        reply = configGenerator.ZoneRequest(userId, "add", addZone);
        UNIT_ASSERT(!reply.IsSuccessReply());

        UNIT_ASSERT(AddCompanyTag(userId, *server));

        TVector<TDBTag> dbTags;
        {
            auto tx = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"my company"}, dbTags, tx));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
        dbTags.clear();

        reply = configGenerator.ZoneRequest(userId, "get", {});
        UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());

        reply = configGenerator.ZoneRequest(userId, "add", addZone);
        UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());
        Sleep(TDuration::Seconds(1));

        {
            auto tx = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"my company"}, dbTags, tx));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        reply = configGenerator.ZoneRequest(userId, "get", {});
        UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());

        NJson::TJsonValue response;
        NJson::TJsonValue zone;
        UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &response));
        UNIT_ASSERT(response.GetValueByPath("objects.company_zones", zone));
        UNIT_ASSERT_C(zone.GetMap().size(), TStringBuilder() << "zone: " << zone.GetStringRobust() << " response: " << response.GetStringRobust());

        reply = configGenerator.ZoneRequest(userId, "remove", removeZone);
        UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());
    }

    Y_UNIT_TEST(CarInZone) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        const auto& server =  env.GetServer();
        auto driveApi =  server->GetDriveAPI();
        const auto& areaManager = driveApi->GetAreaManager();

        auto userId = TString{USER_ROOT_DEFAULT};

        UNIT_ASSERT(AddCompanyTag(userId, *server));

        auto reply = env->Request(USER_ROOT_DEFAULT, "/api/leasing/zone/add", {}, addZone);
        Sleep(TDuration::Seconds(1));

        reply = env->Request(USER_ROOT_DEFAULT, "/api/leasing/zone/get", "new=true", addZone);
        TString uidNewZone;
        for (const auto& zoneReport : reply["company_zones"].GetMap()) {
            if (zoneReport.first == "zone allow drop") {
                auto areas = zoneReport.second.GetValueByPath("meta.areas")->GetArray();
                uidNewZone = areas[0]["area_id"].GetString();
                break;
            }
        }
        UNIT_ASSERT(uidNewZone);

        auto tx = driveApi->BuildTx<NSQL::ReadOnly>();
        auto affiliatedCompanyTagDescription = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(userId, *server, tx);

        TZoneStorage::TOptions options(affiliatedCompanyTagDescription);
        options.SetForMobile(true);
        TMaybe<TVector<NDrivematics::TZone>> zoneObjects = driveApi->GetZoneDB()->GetObjects(options, Now());

        NDrive::TZoneIds carInZonesIds;
        {
            NDrive::TZoneIds zoneIds;
            ForEach(zoneObjects->begin(), zoneObjects->end(), [&zoneIds](const NDrivematics::TZone& zone) {
                zoneIds.insert(zone.GetInternalId());
            });

            UNIT_ASSERT(driveApi->GetZoneDB()->GetZoneIdsInPoint(zoneIds, CarLocation, std::ref(carInZonesIds), areaManager));
        }
        auto areas = CheckCarWithTaggedZone(carInZonesIds, RequiredTagsInArea, *zoneObjects, *server);
        UNIT_ASSERT(!areas.empty() && areas.contains(uidNewZone));
    }

    Y_UNIT_TEST(CreateCarModel) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        const auto& server =  env.GetServer();
        auto driveApi =  server->GetDriveAPI();

        auto userId = TString{USER_ROOT_DEFAULT};

        UNIT_ASSERT(AddCompanyTag(userId, *server));

        NJson::TJsonValue payload = NJson::TMapBuilder("code", "some_code")("name", "some_name")("manufacturer", "some_manufacturer");
        NJson::TJsonValue jBody;
        jBody["models"].AppendValue(payload);

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);

        const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
        auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
        auto res = driveApi->GetModelsData()->Fetch("some_code", tx);
        UNIT_ASSERT(res);
        UNIT_ASSERT("some_code" == res->GetCode());
        auto& tserver =  *server->GetAsPtrSafe<NDrive::IServer>();
        auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, tserver);
        auto aclCompany = acl->GetCompanyTagDescription(USER_ROOT_DEFAULT, tserver);
        auto objects = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::Models, nullptr, true);
        UNIT_ASSERT(objects.contains("some_code"));
    }

    Y_UNIT_TEST(ListModels) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        auto userId = TString{USER_ROOT_DEFAULT};

        UNIT_ASSERT(AddCompanyTag(userId, *env.GetServer()));

        NJson::TJsonValue payload = NJson::TMapBuilder("code", "some_code")("name", "some_name")("manufacturer", "some_manufacturer");
        NJson::TJsonValue jBody;
        jBody["models"].AppendValue(payload);
        payload["code"] = "some_code2";
        jBody["models"].AppendValue(payload);

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);

        reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/list", {}, {});
        UNIT_ASSERT(reply.Has("models"));
        UNIT_ASSERT(reply["models"].IsArray());
        auto jArr = reply["models"].GetArray();
        UNIT_ASSERT(jArr.size() == 2);
    }

    Y_UNIT_TEST(UpdateCarModel) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        auto userId = TString{USER_ROOT_DEFAULT};

        UNIT_ASSERT(AddCompanyTag(userId, *env.GetServer()));

        {
            NJson::TJsonValue payload = NJson::TMapBuilder("code", "some_code")("name", "some_name")("manufacturer", "some_manufacturer");
            NJson::TJsonValue jBody;
            jBody["models"].AppendValue(payload);

            UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);
        }

        {
            NJson::TJsonValue payload = NJson::TMapBuilder("code", "some_code")("name", "some_name")("manufacturer", "changed_manufacturer");
            NJson::TJsonValue jBody;
            jBody["models"].AppendValue(payload);

            UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/update", {}, jBody);
        }

        {
            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/list", {}, {});
            UNIT_ASSERT(reply.Has("models"));
            UNIT_ASSERT(reply["models"].IsArray());
            auto jArr = reply["models"].GetArray();
            UNIT_ASSERT(jArr.size() == 1);
            auto strReply = ToString(reply);
            UNIT_ASSERT(strReply.find("changed_manufacturer") != std::string::npos);
        }
    }

    Y_UNIT_TEST(DeleteCarModel) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        auto userId = TString{USER_ROOT_DEFAULT};

        {
            NJson::TJsonValue payload = NJson::TMapBuilder("code", "some_code")("name", "some_name")("manufacturer", "some_manufacturer");
            NJson::TJsonValue jBody;
            jBody["models"].AppendValue(payload);
            payload["code"] = "some_code2";
            jBody["models"].AppendValue(payload);

            UNIT_ASSERT(AddCompanyTag(userId, *env.GetServer()));

            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);
        }

        {
            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/delete", "id=some_code", {});
        }

        {
            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/list", {}, {});
            UNIT_ASSERT(reply.Has("models"));
            UNIT_ASSERT(reply["models"].IsArray());
            auto jArr = reply["models"].GetArray();
            UNIT_ASSERT(jArr.size() == 1);

            const auto& modelsDb = env.GetServer()->GetDriveAPI()->GetModelsDB();
            auto tx = modelsDb.BuildTx<NSQL::ReadOnly>();
            auto model = modelsDb.Fetch("some_code", tx);
            UNIT_ASSERT(model);
            UNIT_ASSERT(model->HasDeprecated());
            UNIT_ASSERT(model->GetDeprecatedRef());
        }

        {
            auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/list", "show_deleted=true", {});
            UNIT_ASSERT(reply.Has("models"));
            UNIT_ASSERT(reply["models"].IsArray());
            auto jArr = reply["models"].GetArray();
            UNIT_ASSERT(jArr.size() == 2);
        }
    }

    Y_UNIT_TEST(CreateCarModelGenerateCode) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        auto userId = TString{USER_ROOT_DEFAULT};

        NJson::TJsonValue payload = NJson::TMapBuilder("name", "some_name")("manufacturer", "some_manufacturer");
        NJson::TJsonValue jBody;
        jBody["models"].AppendValue(payload);

        UNIT_ASSERT(AddCompanyTag(userId, *env.GetServer()));

        auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);
        reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/list", {}, {});
        UNIT_ASSERT(reply.Has("models"));
        UNIT_ASSERT(reply["models"].IsArray());
        auto jArr = reply["models"].GetArray();
        UNIT_ASSERT(jArr.size() == 1);
        UNIT_ASSERT(jArr[0].Has("deprecated"));
        UNIT_ASSERT(!jArr[0]["deprecated"].GetBoolean());
    }
}
