#include <drive/backend/ut/library/script.h>

#include <drive/backend/alerts/condition.h>
#include <drive/backend/alerts/container.h>
#include <drive/backend/alerts/actions/base.h>
#include <drive/backend/alerts/fetchers/car_tags.h>
#include <drive/backend/alerts/fetchers/user_sessions.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/markers.h>
#include <drive/backend/user_options/user_setting.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>


namespace NDrive::NTest {
    class TRunAlertContainer : public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, StateName);
        R_FIELD(TString, Config);
        R_FIELD(NAlerts::EAlertEntityType, EntityType, NAlerts::EAlertEntityType::User);

    public:
        TRunAlertContainer(const TInstant startInstant, const TString& stateName)
            : TBase(startInstant)
            , StateName(stateName)
        {}

    protected:
        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue jsonConfig;
            UNIT_ASSERT(ReadJsonFastTree(Config, &jsonConfig));
            NAlerts::TAlertContainer alertImpl(EntityType, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), nullptr, StateName));
        }
    public:
        using TBase::TBase;
    };

    class TCleanAllTags : public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, TagName);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::User);

    public:
        TCleanAllTags(const TInstant startInstant, const TString& tagName)
            : TBase(startInstant)
            , TagName(tagName)
        {}

    protected:
        void DoExecute(TRTContext& context) override {
            const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(EntityType);
            TVector<TDBTag> taggedObjects;
            auto session =  context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(entityTagsManager.RestoreTags({}, { TagName }, taggedObjects, session));
            UNIT_ASSERT(entityTagsManager.RemoveTags(taggedObjects, USER_ROOT_DEFAULT, context.GetServer().Get(), session, true));
            UNIT_ASSERT(session.Commit());
        }
    public:
        using TBase::TBase;
    };

    class TCheckAlertNodeContainer : public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, StateTag);
        R_OPTIONAL(TString, LastState);
        R_FIELD(TString, ObjectId);
        R_OPTIONAL(ui32, StatesCount);
        R_OPTIONAL(ui32, ObjectsCount);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::User);

    public:
        TCheckAlertNodeContainer(const TInstant startInstant, const TString& objectId, const TString& stateTag)
            : TBase(startInstant)
            , StateTag(stateTag)
            , ObjectId(objectId)
        {}

    protected:
        void DoExecute(TRTContext& context) override {
            const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(EntityType);
            TVector<TDBTag> taggedObjects;
            {
                auto session =  context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(entityTagsManager.RestoreTags({}, { StateTag }, taggedObjects, session));
            }
            if (HasObjectsCount()) {
                UNIT_ASSERT_VALUES_EQUAL(taggedObjects.size(), GetObjectsCountUnsafe());
            }

            for (auto&& object : taggedObjects) {
                if (object.GetObjectId() != ObjectId) {
                    continue;
                }
                TAlertTag* tagData = dynamic_cast<TAlertTag*>(object.GetData().Get());
                UNIT_ASSERT(tagData);
                if (HasStatesCount()) {
                    UNIT_ASSERT_VALUES_EQUAL(tagData->GetStatesHistory().size(), GetStatesCountUnsafe());
                }
                if (HasLastState()) {
                    UNIT_ASSERT_VALUES_EQUAL(tagData->GetStatesHistory().back(), GetLastStateUnsafe());
                }
            }
        }
    public:
        using TBase::TBase;
    };
}

Y_UNIT_TEST_SUITE(AlertsSuite) {
    using namespace NDrive::NTest;
    using namespace NAlerts;

    class TTestDataFetcher : public IServiceDataFetcher {
        class TIterator : public IFetchedIterator {
        public:
            TIterator(const TMap<TString, ui64>& items, EFetchedItems field, IDataIntervalChecker::TPtr checker)
                : IFetchedIterator(EDataFetcherType::ETestFetcher, EAlertEntityType::Undefined)
                , Current(items.begin())
                , End(items.end())
                , Field(field)
                , Checker(checker)
            {}

            bool Init(const IServiceDataFetcher& fetcher) override {
                auto impl = dynamic_cast<const TTestDataFetcher*>(&fetcher);
                if (impl) {
                    return true;
                }
                return false;
            }

            TString GetSubstitutionName() const override {
                return ToString(Field);
            }

            EFetchedItems GetField() const override {
                return Field;
            }

            TMaybe<TFetchedValue> Get() override {
                return Current->second;
            }

            TStringBuf GetObjectId() const override {
                return Current->first;
            }

            TStringBuf GetObjectId(const NAlerts::EAlertEntityType) const override {
                return Current->first;
            }

            bool IsFinished() const override {
                return Current == End;
            }

            bool Next() override {
                ++Current;
                return !IsFinished();
            }

            IDataIntervalChecker::TPtr GetChecker() const override {
                return Checker;
            }

        private:
            TMap<TString, ui64>::const_iterator Current;
            TMap<TString, ui64>::const_iterator End;
            EFetchedItems Field;
            IDataIntervalChecker::TPtr Checker;
        };


    private:
        static TMap<NAlerts::EFetchedItems, IDataIntervalChecker::TPtr> TestCheckers;
        static TMap<NAlerts::EFetchedItems, TMap<TString, ui64>> Items;

    public:
        using TBase = IServiceDataFetcher;
        using TBase::TBase;

        TMaybe<TVector<IFetchedIterator::TPtr>> BuildIterators(TFetcherContext& context) const override {
            Y_UNUSED(context);
            TVector<IFetchedIterator::TPtr> result;
            for (auto && [iteratorType, items] : Items) {
                auto checkerIt = TestCheckers.find(iteratorType);
                if (checkerIt != TestCheckers.end()) {
                    result.push_back(new TIterator(items, iteratorType, checkerIt->second));
                }
            }
            return result;
        }

        static void AddItem(const NAlerts::EFetchedItems field, const TString& key, ui64 value) {
            Items[field][key] = value;
        }

        static void AddChecker(const NAlerts::EFetchedItems field, const IDataIntervalChecker::TPtr& checker) {
            TestCheckers[field] = checker;
        }

    protected:
        bool DoFetch(const TFetcherContext& /*context*/) override {
            return true;
        }
    };

    TMap<NAlerts::EFetchedItems, IDataIntervalChecker::TPtr> TTestDataFetcher::TestCheckers;
    TMap<NAlerts::EFetchedItems, TMap<TString, ui64>> TTestDataFetcher::Items;


    class TTestDataFetcherConfig : public IServiceDataFetcherConfig {
    public:
        bool DoDeserializeFromJson(const NJson::TJsonValue& /*json*/) override { return true; }

        NJson::TJsonValue DoSerializeToJson() const override {
            return NJson::JSON_NULL;
        }

        NDrive::TScheme DoGetScheme(const IServerBase& /*server*/) const override {
            return NDrive::TScheme();
        }

        TString GetSchemeDescription() const override {
            return "";
        }

        EDataFetcherType GetFetcherType() const override {
            return EDataFetcherType::ETestFetcher;
        }

        NAlerts::EAlertEntityType GetEntityType() const override {
            return NAlerts::EAlertEntityType::Car;
        }

        IServiceDataFetcher::TPtr BuildFetcher() const override {
            return new TTestDataFetcher(*this);
        }

        virtual TSet<NAlerts::EFetchedItems> GetIteratorFields() const {
            return { NAlerts::EFetchedItems::Main, NAlerts::EFetchedItems::Sensor};
        }
    };


    class TTestActor : public ICheckerAction {
        mutable TSet<TString> Objects;
    public:
        bool DoFinish(const TSet<TString>& objectIds, TFetcherContext& /*context*/) override {
            Objects = objectIds;
            return true;
        }
        bool DeserializeFromJson(const NJson::TJsonValue& /*json*/) override {
            return true;
        }
        NJson::TJsonValue SerializeToJson() const override {
            return NJson::JSON_NULL;
        }
        NDrive::TScheme GetScheme(const IServerBase&) const override {
            return NDrive::TScheme();
        }

        const TSet<TString>& GetObjectIds() const {
            return Objects;
        }
    };

    Y_UNIT_TEST(MetaInfo) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(7);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            INFO_LOG << "META INFO " << NAlerts::TAlertContainer::GetMetaInfo(*context.GetServer().Get());
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(TestTemplates) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(7);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& rtContext) {
            TFetcherContext context(rtContext.GetServer().Get(), NAlerts::EAlertEntityType::User);
            context.RegisterIterator(NAlerts::EFetchedItems::LastActionDelta, ToString(NAlerts::EFetchedItems::LastActionDelta));
            context.RegisterIterator(NAlerts::EFetchedItems::BillingTagSum, ToString(NAlerts::EFetchedItems::BillingTagSum));
            context.SaveData(USER_ROOT_DEFAULT, "test_key", "test_value");
            context.SaveIteratorData(USER_ROOT_DEFAULT, ToString(NAlerts::EFetchedItems::LastActionDelta), 300);
            context.SaveIteratorData(USER_ROOT_DEFAULT, ToString(NAlerts::EFetchedItems::BillingTagSum), 300);
            TInstantGuard g(TInstant::Seconds(1571045152));
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, "_TsNow_", context, 0), "1571045152");
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, "_TsToday_", context, 0), "1571011200");
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, "_TsToday+10_", context, 0), "1571875200");
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, "<test_key>", context, 0), "test_value");
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, "<last_action_time>", context, 0), "5 мин");

            const TString dataStr = R"({"data": {"action_type": "<billing_sum>"})";
            NJson::TJsonValue jsonData;
            UNIT_ASSERT(ReadJsonFastTree(dataStr, &jsonData));

            const TString controlataStr = R"({"action_type": "3"})";
            NJson::TJsonValue controlData;
            UNIT_ASSERT(ReadJsonFastTree(controlataStr, &controlData));
            UNIT_ASSERT_VALUES_EQUAL(INotifierActionBase::PatchData(USER_ROOT_DEFAULT, jsonData["data"].GetStringRobust(), context, 0), controlData.GetStringRobust());
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Simple) {
        TTestDataFetcher::AddItem(NAlerts::EFetchedItems::Main, "1", 45);
        TTestDataFetcher::AddItem(NAlerts::EFetchedItems::Main, "2", 20);
        TTestDataFetcher::AddItem(NAlerts::EFetchedItems::Sensor, "1", 25);
        TTestDataFetcher::AddItem(NAlerts::EFetchedItems::Sensor, "2", 25);

        TCheckersSet checkers;
        {
            IDataIntervalChecker::TPtr rule = IDataIntervalChecker::Construct("default");
            TString checkerStr = R"({"min_value": "10", "max_value":"30", "skip_empty":true})";
            NJson::TJsonValue json = NJson::JSON_NULL;
            UNIT_ASSERT(NJson::ReadJsonTree(checkerStr, &json));
            UNIT_ASSERT(rule->DeserializeFromJson(json));
            TTestDataFetcher::AddChecker(NAlerts::EFetchedItems::Main, rule);
        }
        {
            IDataIntervalChecker::TPtr rule = IDataIntervalChecker::Construct("default");
            TString checkerStr = R"({"min_value": "20", "max_value":"50", "skip_empty":true})";
            NJson::TJsonValue json = NJson::JSON_NULL;
            UNIT_ASSERT(NJson::ReadJsonTree(checkerStr, &json));
            UNIT_ASSERT(rule->DeserializeFromJson(json));
            TTestDataFetcher::AddChecker(NAlerts::EFetchedItems::Sensor, rule);
        }
        TMap<EDataFetcherType, IServiceDataFetcherConfig::TPtr> fetchers;
        fetchers.insert({ EDataFetcherType::ETestFetcher, new TTestDataFetcherConfig() });

        TTestActor actor;
        TFetcherContext context(nullptr, NAlerts::EAlertEntityType::Car);
        UNIT_ASSERT(checkers.ApplyFilters(fetchers, context, actor));
        actor.Finish(context);
        UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(*actor.GetObjectIds().begin(), "2");
    }

    Y_UNIT_TEST(UserBySession) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        const TString configStr = R"({"action": {"action_type":"user_notify","notifier_name":"nsofya_tests","hr_name":"Fake"},
                                        "fetchers":[{"config":{"type":"sessions","states_filter":["old_state_riding"], "offers_filter": ["fixpoint_offer_constructor"]}}],
                                        "checkers":[{"max_value":3000,"min_value":0,"fetcher":"sessions.time","skip_empty":false}]
                                        })";
        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::User, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT(!actor.GetObjectIds().contains(context.GetUserId()));
        });

        script.Add<NDrive::NTest::TRide>(TDuration::Zero());
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::User, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT(actor.GetObjectIds().contains(context.GetUserId()));
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(CarTagsHistory) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        const TString configStr = R"({
    "action": {
        "action_type": "user_notify",
        "notifier_name": "nsofya_tests",
        "hr_name": "Fake"
    },
    "fetchers": [{
        "config": {
            "type": "car_tags",
            "filter": {
                "undefined_position_policy": "AcceptUndefined",
                "exclude_areas": [],
                "include_areas": [],
                "use_LBS_for_area_tags": false,
                "include_cars": ["65e60f94-29f7-4aa6-97e0-35f9786829cc"],
                "exclude_models": [],
                "exclude_cars": "",
                "skip_without_location": false,
                "area_tags": [],
                "include_models": []
            }
        }
    }],
    "checkers": [{
        "max_value": 300,
        "min_value": 0,
        "fetcher": "car_tags.last_action_time",
        "skip_empty": false,
        "meta": {"history_tags": ["comment"],
        "history_actions": ["add"]}
    }]
})";
        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT(actor.GetObjectIds().empty());
        });

        NJson::TJsonValue tagData(NJson::JSON_MAP);
        script.Add<NDrive::NTest::TAddTagActions>("comment", OBJECT_ID_DEFAULT).SetCustomData(tagData).SetEntityType(NEntityTagsManager::EEntityType::Car).SetComment("test");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 1);
            UNIT_ASSERT(actor.GetObjectIds().contains(OBJECT_ID_DEFAULT));
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(TestChain) {
        /*  3 node chain for users in resevation */
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        const TString node0Str = R"({
            "action": {
                "state_tag":"simple_alert",
                "additional_tags":[
                    {"comment":"Автораздача","tag":"test_push_smile"}
                ],
                "save_fetched":true,
                "node_type":"control",
                "action_type":"chain_node",
                "acceptable_states":["none"]
            },
            "fetchers": [{
                "config": {
                    "type":"sessions",
                    "models_filter":[],
                    "offers_filter":[],
                    "states_filter":["old_state_reservation"]
                }
            }],
            "checkers": [{
                "max_value": 300,
                "min_value": 0,
                "fetcher": "sessions.time",
                "skip_empty": false,
            }]
        })";

        const TString node1Str = R"({
            "action": {
                "state_tag":"simple_alert",
                "additional_tags":[
                    {"comment":"Автораздача","tag":"test_push_smile"}
                ],
                "save_fetched":true,
                "action_type":"chain_node",
                "acceptable_states":["simple_chain0"]
            },
            "fetchers": [{
                "config": {
                    "filter": {"has_one_of":["simple_alert"],"has_all_of":[],"limit":100,"has_none_of":[],"performer":""},
                    "entity":"user",
                    "type":"object_tags"
                }
            }],
            "checkers": [{
                "max_value": 300,
                "min_value": 0,
                "fetcher": "object_tags.last_action_time",
                "skip_empty": false,
                "meta": {"history_tags":["simple_alert"],"history_actions":["add"],"history_deep":2592000}
            }]
        })";
        const TString node2Str = R"({
            "action": {
                "state_tag":"simple_alert",
                "additional_tags":[
                    {"comment":"Автораздача","tag":"test_push_smile"}
                ],
                "save_fetched":true,
                "action_type":"chain_node",
                "acceptable_states":["simple_chain1"]
            },
            "fetchers": [{
                "config": {
                    "filter": {"has_one_of":["simple_alert"],"has_all_of":[],"limit":100,"has_none_of":[],"performer":""},
                    "entity":"user",
                    "type":"object_tags"
                }
            }],
            "checkers": [{
                "max_value": 300,
                "min_value": 0,
                "fetcher": "object_tags.last_action_time",
                "skip_empty": false,
                "meta": {"history_tags":["simple_alert"],"history_actions":["add"],"history_deep":2592000}
            }]
        })";

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);

        // Make a decoy session
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("decoy", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TCreateCar>().SetModel("kia_rio").SetPosition(from).SetNamedCar("decoy");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("decoy");
        script.Add<TAccept>();

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCleanAllTags>("simple_alert");
        script.Add<NDrive::NTest::TCheckAlertNodeContainer>(USER_ID_DEFAULT, "simple_alert").SetObjectsCount(0);

        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferName("standart_offer_constructor");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TRunAlertContainer>("simple_chain0").SetConfig(node0Str);
        script.Add<NDrive::NTest::TCheckAlertNodeContainer>(USER_ID_DEFAULT, "simple_alert")
            .SetLastState("simple_chain0")
            .SetObjectsCount(1)
            .SetStatesCount(1);

        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TRunAlertContainer>("simple_chain1").SetConfig(node1Str);
        script.Add<NDrive::NTest::TCheckAlertNodeContainer>(USER_ID_DEFAULT, "simple_alert")
            .SetLastState("simple_chain1")
            .SetObjectsCount(1)
            .SetStatesCount(2);

        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TRunAlertContainer>("simple_chain2").SetConfig(node2Str);
        script.Add<NDrive::NTest::TCheckAlertNodeContainer>(USER_ID_DEFAULT, "simple_alert")
            .SetLastState("simple_chain2")
            .SetObjectsCount(1)
            .SetStatesCount(3);

        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Minutes(1));
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(2));

        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TRunAlertContainer>("simple_chain0").SetConfig(node0Str);
        script.Add<NDrive::NTest::TCheckAlertNodeContainer>(USER_ID_DEFAULT, "simple_alert").SetObjectsCount(0);
        UNIT_ASSERT(script.Execute());
    }

    TVector<TString> SetupFetchedUsers(NDrive::NTest::TScript& script, const TGeoCoord& from, const TGeoCoord& to) {
        TVector<TString> users;

        // user 1:
        //  audi_q3, first
        //  standart_offer_constructor
        //  riding
        //  user_problem_tag_minor, support_chat_2_line
        // assign user_problem_tag_major
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("standart_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TUserProblemTag("user_problem_tag_minor"));
        script.Add<TAddUserTag>(new TSupportChatTag("support_chat_2_line"));
        script.Add<TCreateCar>().SetModel("audi_q3").SetPosition(from).SetNamedCar("first");
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from).SetNamedCar("first");
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Minutes(1));

        // user 2:
        //  ford_transit, second
        //  fixpoint_offer_constructor
        //  riding
        //  user_problem_tag_minor
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_2", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TUserProblemTag("user_problem_tag_minor"));
        script.Add<TAddUserTag>(new TSupportChatTag("support_chat_2_line"));
        script.Add<TCreateCar>().SetModel("ford_transit").SetPosition(from).SetNamedCar("second");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("second");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Minutes(10));

        // user 3: - removed by sessions
        //  audi_q3, third
        //  fixpoint_offer_constructor
        //  parking
        //  user_problem_tag_minor
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_3", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TUserProblemTag("user_problem_tag_minor"));
        script.Add<TCreateCar>().SetModel("audi_q3").SetPosition(from).SetNamedCar("third");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("third");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Minutes(10));
        script.Add<TPark>();

        // user 4: - removed by car_tags
        //  kia_rio, fourth
        //  fixpoint_offer_constructor
        //  parking
        //  user_problem_tag_minor
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_4", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TUserProblemTag("user_problem_tag_minor"));
        script.Add<TCreateCar>().SetModel("kia_rio").SetPosition(from).SetNamedCar("fourth");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("fourth");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Minutes(10));

        // user 5:
        //  ford_transit, fifth
        //  fixpoint_offer_constructor
        //  parking
        //  user_problem_tag_minor
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_5", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TUserProblemTag("user_problem_tag_minor"));
        script.Add<TAddUserTag>(new TSupportChatTag("support_chat_2_line"));
        script.Add<TCreateCar>().SetModel("ford_transit").SetPosition(from).SetNamedCar("fifth");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("fifth");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Minutes(10));

        // user 6: - removed by user tags
        //  ford_transit, fifth
        //  fixpoint_offer_constructor
        //  parking
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("fetcher_user_6", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
            TUserRole userRole;
            userRole.SetRoleId("fixpoint_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TAddUserTag>(new TSupportChatTag("support_chat_2_line"));
        script.Add<TCreateCar>().SetModel("ford_transit").SetPosition(from).SetNamedCar("sixth");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetNamedCar("sixth");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Minutes(10));
        return users;
    }

    Y_UNIT_TEST(OrderedFetchersCarToUser) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        auto users = SetupFetchedUsers(script, from, to);

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"user tags fetcher"},

            "checkers":[
                {"fetcher":"car_tags.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"user_data.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"object_tags.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"sessions.main","max_value":2,"min_value":0,"skip_empty":false}
                ],

            "fetchers":[
                {"config":{"filter":{"undefined_position_policy":"AcceptUndefined","use_LBS_for_area_tags":false,"skip_without_location":true,"area_tags":[],"include_models":["audi_q3","ford_transit"]},"index":0,"type":"car_tags"}},
                {"config":{"index":1,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"user","states_filter":["old_state_riding"]}},
                {"config":{"state_tag":"","filter":{"has_one_of":[],"has_all_of":["user_problem_tag_minor"],"limit":100,"has_none_of":[],"performer":""},"index":2,"entity":"user","type":"object_tags","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":3,"type":"user_data","without_ridings":false,"user_roles":["fixpoint_access"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}]
            })";

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig,&users](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("common_alerts.use_chained_fetchers", "1", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 2);
            UNIT_ASSERT(actor.GetObjectIds().contains(users[1]));
            UNIT_ASSERT(actor.GetObjectIds().contains(users[4]));
        });

        script.Execute();
    }

    Y_UNIT_TEST(OrderedFetchersEmptyFilter) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        auto users = SetupFetchedUsers(script, from, to);

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"user tags fetcher"},

            "checkers":[
                {"fetcher":"user_data.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"object_tags.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"sessions.main","max_value":2,"min_value":0,"skip_empty":false}
                ],

            "fetchers":[
                {"config":{"index":0,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"user","states_filter":["old_state_riding"]}},
                {"config":{"state_tag":"","filter":{"has_one_of":[],"has_all_of":["user_problem_tag_minor","bad_tag_name"],"limit":100,"has_none_of":[],"performer":""},"index":1,"entity":"user","type":"object_tags","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":2,"type":"user_data","without_ridings":false,"user_roles":[],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}]
            })";

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("common_alerts.use_chained_fetchers", "1", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 0);
        });

        script.Execute();
    }

    Y_UNIT_TEST(OrderedFetchersUserToCar) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        auto users = SetupFetchedUsers(script, from, to);

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"user tags fetcher"},

            "checkers":[
                {"fetcher":"car_tags.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"user_data.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"object_tags.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"sessions.main","max_value":2,"min_value":0,"skip_empty":false}
                ],

            "fetchers":[
                {"config":{"filter":{"undefined_position_policy":"AcceptUndefined","use_LBS_for_area_tags":false,"skip_without_location":true,"area_tags":[],"include_models":["audi_q3","ford_transit"]},"index":3,"type":"car_tags"}},
                {"config":{"index":2,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"car","states_filter":["old_state_riding"]}},
                {"config":{"state_tag":"","filter":{"has_one_of":[],"has_all_of":["user_problem_tag_minor"],"limit":100,"has_none_of":[],"performer":""},"index":1,"entity":"user","type":"object_tags","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":0,"type":"user_data","without_ridings":false,"user_roles":["fixpoint_access"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}]
            })";

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("common_alerts.use_chained_fetchers", "1", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 2);
            auto car1 = context.GetNamedCar("second");
            auto car2 = context.GetNamedCar("fifth");
            UNIT_ASSERT(car1 && car2);
            UNIT_ASSERT(actor.GetObjectIds().contains(car1->Id));
            UNIT_ASSERT(actor.GetObjectIds().contains(car2->Id));
        });

        script.Execute();
    }

    Y_UNIT_TEST(MultipleIterators) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        auto users = SetupFetchedUsers(script, from, to);

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"fetcher":"user_data.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"sessions.main","max_value":2,"min_value":0,"skip_empty":false},
                {"fetcher":"sessions.sensor","max_value":101,"min_value":100,"meta":{"aggregation":"current","sensor_id":2103,"max_sensor_age":1000,"min_sensor_age":0},"skip_empty":false},
                {"fetcher":"sessions.last_action_time","max_value":10000,"min_value":1,"meta":{"history_tags":["support_chat_2_line"],"history_actions":["add"],"history_deep":360},"skip_empty":false}
                ],

            "fetchers":[
                {"config":{"index":0,"type":"user_data","without_ridings":false,"user_roles":["fixpoint_access"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}},
                {"config":{"index":1,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"user","states_filter":["old_state_riding","old_state_parking"]}}]
            })";

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.SetCar(*context.GetNamedCar("second"));
            context.SetMileage(100);
            context.SetCar(*context.GetNamedCar("third"));
            context.SetMileage(100);
            context.SetCar(*context.GetNamedCar("fifth"));
            context.SetMileage(100);
        });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig,&users](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("common_alerts.use_chained_fetchers", "1", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 1);
            UNIT_ASSERT(actor.GetObjectIds().contains(users[1]));
        });

        script.Execute();
    }

    Y_UNIT_TEST(IndexedFetchersDeserialization) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());

        // correct 1
        const TString configStr1 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"sessions.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":2,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"user","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":0,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":4,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":1,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"user","states_filter":["old_state_riding"]}},
                {"config":{"index":3,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig1;
        UNIT_ASSERT(ReadJsonFastTree(configStr1, &jsonConfig1));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig1](NDrive::NTest::TRTContext& /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig1));
        });

        // correct 2
        const TString configStr2 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"sessions.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":4,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"car","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":3,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":1,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":2,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"car","states_filter":["old_state_riding"]}},
                {"config":{"index":0,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig2;
        UNIT_ASSERT(ReadJsonFastTree(configStr2, &jsonConfig2));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig2](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig2));
        });

        // bad indexes
        const TString configStr3 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"sessions.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":2,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"user","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":0,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":0,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":0,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"session","states_filter":["old_state_riding"]}},
                {"config":{"index":3,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig3;
        UNIT_ASSERT(ReadJsonFastTree(configStr3, &jsonConfig3));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig3](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(!alertImpl.DeserializeFromJson(jsonConfig3));
        });

        // bad chain
        const TString configStr5 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"sessions.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":0,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"user","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":1,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":2,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":3,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"session","states_filter":["old_state_riding"]}},
                {"config":{"index":4,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig5;
        UNIT_ASSERT(ReadJsonFastTree(configStr5, &jsonConfig5));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig5](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(!alertImpl.DeserializeFromJson(jsonConfig5));
        });

        // missing iterator
        const TString configStr6 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":4,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"car","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":3,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":1,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":2,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"car","states_filter":["old_state_riding"]}},
                {"config":{"index":0,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig6;
        UNIT_ASSERT(ReadJsonFastTree(configStr6, &jsonConfig6));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig6](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(!alertImpl.DeserializeFromJson(jsonConfig6));
        });

        // bad indexes
        const TString configStr7 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"sessions.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":4,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"car","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":3,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":1,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":2,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"car","states_filter":["old_state_riding"]}},
                {"config":{"index":2,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig7;
        UNIT_ASSERT(ReadJsonFastTree(configStr7, &jsonConfig7));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig7](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(!alertImpl.DeserializeFromJson(jsonConfig7));
        });

        // bad indexes
        const TString configStr8 = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {"max_value":2,"min_value":0,"fetcher":"car_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"chat_state.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"object_tags.main","skip_empty":false},
                {"max_value":2,"min_value":0,"fetcher":"user_data.main","skip_empty":false}],

            "fetchers":[
                {"config":{"index":4,"type":"object_tags","state_tag":"","filter":{"has_one_of":[],"has_all_of":["support_chat_2_line"],"limit":100,"has_none_of":[],"performer":""},"entity":"car","state_interval":0,"acceptable_states":[]}},
                {"config":{"index":3,"type":"car_tags","filter":{"use_LBS_for_area_tags":false,"skip_without_location":true,"include_models":["audi_q3","bmw_520i_w"],"undefined_position_policy":"AcceptUndefined","area_tags":[]}}},
                {"config":{"index":10,"type":"chat_state","chat_topic":"","chat_id":"234","node_names":[],"exclude_active_twins":false,"base_status":"onboarding","cv_filter_blocks":[]}},
                {"config":{"index":2,"type":"sessions","models_filter":[],"offer_tags_filter":"","offers_filter":[],"entity_type":"car","states_filter":["old_state_riding"]}},
                {"config":{"index":0,"type":"user_data","user_actions":[],"without_ridings":false,"user_roles":["GR_Cleaner_1"],"check_ridings":false,"reg_geo":[],"special_users":[],"user_status":[]}}
                ]
            })";

        NJson::TJsonValue jsonConfig8;
        UNIT_ASSERT(ReadJsonFastTree(configStr8, &jsonConfig8));

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig8](NDrive::NTest::TRTContext&  /*context*/) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);
            UNIT_ASSERT(!alertImpl.DeserializeFromJson(jsonConfig8));
        });

        script.Execute();
    }

    void AddSupportChatTag(NDrive::NTest::TScript& script, const TString& comment, const TString& title, const bool muted) {
        THolder<TSupportChatTag> tag(new TSupportChatTag("support_chat_2_line"));
        tag->SetComment(comment);
        tag->SetMuted(muted);
        tag->SetChatTitle(title);
        script.Add<TAddUserTag>(tag.Release());
    }

    void AddEventTag(NDrive::NTest::TScript& script, const TVector<TString>& eventNames) {
        THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag"));
        for (auto&& eventName : eventNames) {
            tag->AddEvent(eventName, TInstant::Now());
        }
        script.Add<TAddUserTag>(tag.Release());
    }

    Y_UNIT_TEST(OneTypeIterators) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());

        script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            TDictionaryTagDescription description1;
            description1.SetName("alerts_dictionary_tag_1").SetType(TUserDictionaryTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(description1), USER_ROOT_DEFAULT, session), session.GetStringReport());
            TDictionaryTagDescription description2;
            description2.SetName("alerts_dictionary_tag_2").SetType(TUserDictionaryTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(description2), USER_ROOT_DEFAULT, session), session.GetStringReport());
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        });

        NJson::TJsonValue descJson;
        descJson["type"] = "user_event_tag";
        descJson["name"] = "alerts_test_event_tag";
        descJson["display_name"] = "event tag";
        descJson["comment"] = "";
        script.Add<NDrive::NTest::TRegisterTag>(descJson);

        TVector<TString> users;

        // user 1:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", false);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2"});

        // user 2:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value3").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", false);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2"});

        // user 3:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2", "event2"});

        // user 4:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "more titles", true);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2", "event2"});

        // user 5:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value0").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "more titles", false);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2", "event2"});

        // user 6:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value0").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2", "event2"});

        // user 7:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", false);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "a title", false);
        AddEventTag(script, {"event0", "event2", "event2", "event2", "event2", "event2"});

        // user 8:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "a title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddEventTag(script, {"event1", "event2", "event2"});

        // user 9:
        script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("alerts_user_1", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            users.push_back(userId);
        });
        script.Add<TAddSettingTag>("alerts_dictionary_tag_1", "name", "value1").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TAddSettingTag>("alerts_dictionary_tag_2", "name", "value2").SetActionUserId(USER_ROOT_DEFAULT);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "another title", false);
        AddSupportChatTag(script, "Обращение из theChat", "another title", true);
        AddSupportChatTag(script, "Обращение из theChat", "a title", false);
        AddEventTag(script, {"event1", "event2", "event2", "event2", "event2", "event2"});

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {
                    "fetcher": "user_data.dictionary_tag", "fetcher_type": "user_data", "iterator_type": "dictionary_tag", "skip_empty": false,
                    "max_value": "inf", "min_value": "1",
                    "meta": { "field_name": "name", "tag_name": "alerts_dictionary_tag_1", "field_values": [ "value1" ] }
                },
                {
                    "fetcher": "user_data.dictionary_tag", "fetcher_type": "user_data", "iterator_type": "dictionary_tag", "skip_empty": false,
                    "max_value": "inf", "min_value": "1",
                    "meta": { "field_name": "name", "tag_name": "alerts_dictionary_tag_2", "field_values": [ "value2" ] }
                },
                {
                    "fetcher_type": "user_data", "iterator_type": "tag_field",
                    "max_value": "inf", "min_value": "3",
                    "meta": { "tag_name": "support_chat_2_line", "field_path": "comment", "field_values": [ "Обращение из theChat" ] },
                    "skip_empty": false
                },
                {
                    "fetcher_type": "user_data", "iterator_type": "tag_field",
                    "max_value": "4", "min_value": "2",
                    "meta": { "tag_name": "support_chat_2_line", "field_path": "muted", "field_values": [ "true" ] },
                    "skip_empty": false
                },
                {
                    "fetcher_type": "user_data", "iterator_type": "tag_field",
                    "max_value": "10", "min_value": "1",
                    "meta": { "tag_name": "support_chat_2_line", "field_path": "chat_title", "field_values": [ "a title" ] },
                    "skip_empty": false
                },
                {
                    "fetcher_type": "user_data", "iterator_type":"event_tag_event_count",
                    "max_value":"2", "min_value":"1",
                    "meta":{"event_names":["event1"],"tag_name":"alerts_test_event_tag","check_order":false},
                    "skip_empty":false
                },
                {
                    "fetcher_type": "user_data", "iterator_type":"event_tag_event_count",
                    "max_value":"6", "min_value":"3",
                    "meta":{"event_names":["event2"],"tag_name":"alerts_test_event_tag","check_order":false},
                    "skip_empty":false
                }
            ],

            "fetchers":[ {
                "config": { "user_actions": [], "index": 0, "type": "user_data", "without_ridings": false, "user_roles": [], "check_ridings": false, "reg_geo": [], "user_status": []
                    }}]
        })";

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig,&users](NDrive::NTest::TRTContext& context) {
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 3);
            UNIT_ASSERT(actor.GetObjectIds().contains(users[0]));
            UNIT_ASSERT(actor.GetObjectIds().contains(users[2]));
            UNIT_ASSERT(actor.GetObjectIds().contains(users[8]));
        });
        script.Execute();
    }

    Y_UNIT_TEST(OneTypeIteratorsMultipleFetchers) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        auto users = SetupFetchedUsers(script, from, to);

        const TString configStr = R"({
            "action":{"action_type":"user_notify","report_type":"simple","timezone":3,"multi_line":true,"save_fetched":false,"notifier_name":"crucian_tests","hr_name":"testiki"},

            "checkers":[
                {
                    "max_value": "2", "min_value": "1",
                    "meta": { "event_names": [ "event1" ], "tag_name": "alerts_test_event_tag_2" },
                    "skip_empty": false,
                    "fetcher_type": "user_data", "iterator_type": "event_tag_event_count"
                },
                {
                    "max_value": "3", "min_value": "1",
                    "meta": { "event_names": [ "event2" ], "tag_name": "alerts_test_event_tag_2" },
                    "skip_empty": false,
                    "fetcher_type": "user_data", "iterator_type": "event_tag_event_count"
                },
                {
                    "max_value": "50", "min_value": "1",
                    "meta": { "sensor_id": 2107, "min_sensor_age": 0 },
                    "skip_empty": false,
                    "fetcher_type": "sessions", "iterator_type": "sensor"
                },
                {
                    "max_value": "11", "min_value": "10",
                    "meta": { "sensor_id": 2103, "min_sensor_age": 0 },
                    "skip_empty": false,
                    "fetcher_type": "sessions", "iterator_type": "sensor"
                }
                ],

            "fetchers":[
                {
                    "config": {
                        "user_actions": [], "index": 0, "type": "user_data", "without_ridings": false,
                        "user_roles": [ "fixpoint_access" ],
                        "check_ridings": false, "reg_geo": [], "special_users": [], "user_status": []
                    }
                },
                {
                    "config": {
                        "index": 1, "type": "sessions", "states_filter": [ "old_state_parking", "old_state_riding" ]
                    }
                }
                ]
            })";

        NJson::TJsonValue descJson;
        descJson["type"] = "user_event_tag";
        descJson["name"] = "alerts_test_event_tag_2";
        descJson["display_name"] = "event tag";
        descJson["comment"] = "";
        script.Add<NDrive::NTest::TRegisterTag>(descJson);

        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.SetCar(*context.GetNamedCar("second"));
            context.SetFuelLevel(30);
            context.SetCar(*context.GetNamedCar("fourth"));
            context.SetFuelLevel(40);
            context.SetCar(*context.GetNamedCar("fifth"));
            context.SetMileage(100);
            context.SetFuelLevel(20);
            context.SetCar(*context.GetNamedCar("sixth"));
            context.SetFuelLevel(45);
        });
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event1", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[0], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event1", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[1], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event0", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[2], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event1", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[3], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event1", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[4], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
            {
                THolder<TUserEventTag> tag(new TUserEventTag("alerts_test_event_tag_2"));
                tag->AddEvent("event1", TInstant::Now());
                tag->AddEvent("event2", TInstant::Now());
                UNIT_ASSERT(context.GetConfigGenerator().AddTag(tag.Release(), users[5], USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            }
        });

        script.Add<NDrive::NTest::TCommonChecker>([&jsonConfig,&users](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("common_alerts.use_chained_fetchers", "1", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            NAlerts::TAlertContainer alertImpl(NAlerts::EAlertEntityType::Car, USER_ROOT_DEFAULT);

            UNIT_ASSERT(alertImpl.DeserializeFromJson(jsonConfig));
            TTestActor actor;
            UNIT_ASSERT(alertImpl.Execute(context.GetServer().Get(), TInstant::Now(), &actor));
            UNIT_ASSERT_VALUES_EQUAL(actor.GetObjectIds().size(), 2);
            UNIT_ASSERT(actor.GetObjectIds().contains(users[3]));
            UNIT_ASSERT(actor.GetObjectIds().contains(users[5]));
        });
        script.Execute();
    }
}
