#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/markers.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(Areas) {
    Y_UNIT_TEST(UserAreaInfo) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        THolder<TAreaInfoForUser> tagUserInfo(new TAreaInfoForUser("common_area_user_info"));
        const TString testTagName = "user_strange_info_" + ::ToString(RandomNumber<ui32>());
        tagUserInfo->MutableInfos().emplace(testTagName, "phone");

        const TGeoCoord userPosition(37.58753866611312, 55.73411259048583);
        {
            NJson::TJsonValue userSessionReport = gServer.GetCurrentSession(USER_ID_DEFAULT, &userPosition);
            const NJson::TJsonValue& areaUserInfo = userSessionReport["area_common_user_info"];
            UNIT_ASSERT_C(areaUserInfo.IsMap(), TStringBuilder() << userSessionReport << Endl);
            UNIT_ASSERT_C(!areaUserInfo.Has(testTagName), TStringBuilder() << userSessionReport << Endl);
        }

        UNIT_ASSERT(gServer.AddTag(tagUserInfo.Release(), "moscow_oblast", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            NJson::TJsonValue userSessionReport = gServer.GetCurrentSession(USER_ID_DEFAULT, &userPosition);
            const NJson::TJsonValue& areaUserInfo = userSessionReport["area_common_user_info"];
            UNIT_ASSERT_C(areaUserInfo.IsMap(), TStringBuilder() << userSessionReport << Endl);
            UNIT_ASSERT_VALUES_EQUAL(areaUserInfo[testTagName].GetString(), "phone");
        }
    }

    Y_UNIT_TEST(CarsSpecialForArea) {
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        auto car1 = eg.CreateCar();
        auto car2 = eg.CreateCar();
        TGeoCoord someCoord1(12.58643489401084, 45.73761541977004);
        TGeoCoord someCoord2(13.58643489401084, 45.73761541977004);
        auto emulator1 = tmBuilder.BuildEmulator(car1.IMEI);
        auto emulator2 = tmBuilder.BuildEmulator(car2.IMEI);
        UNIT_ASSERT(configGenerator.WaitCar(car1.Id));
        UNIT_ASSERT(configGenerator.WaitCar(car2.Id));

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        driver.RelocateCar(car1.IMEI, someCoord1);
        driver.RelocateCar(car2.IMEI, someCoord2);

        UNIT_ASSERT(configGenerator.WaitLocation(car1.Id, someCoord1));
        UNIT_ASSERT(configGenerator.WaitLocation(car2.Id, someCoord2));

        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("12.48643489401084 45.63761541977004 12.48643489401084 45.83761541977004 12.68643489401084 45.63761541977004 12.68643489401084 45.83761541977004", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("zone_1", USER_ROOT_DEFAULT, coords, {"global_area"}));
        }
        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("13.48643489401084 45.63761541977004 13.48643489401084 45.83761541977004 13.68643489401084 45.63761541977004 13.68643489401084 45.83761541977004", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("zone_2", USER_ROOT_DEFAULT, coords, {"global_area"}));
        }

        {
            NJson::TJsonValue report1 = configGenerator.GetServiceCarsList({}, USER_ID_DEFAULT, "", &someCoord1);
            {
                ui32 count = 0;
                for (auto&& i : report1["cars"].GetArraySafe()) {
                    if (i["location"].IsDefined()) {
                        UNIT_ASSERT_VALUES_EQUAL(i["id"].GetStringSafe(), car1.Id);
                        ++count;
                    }
                }
                UNIT_ASSERT_VALUES_EQUAL(count, 1);
            }
        }

        {
            NJson::TJsonValue report1 = configGenerator.GetServiceCarsList({}, USER_ID_DEFAULT, "", &someCoord2);
            {
                ui32 count = 0;
                for (auto&& i : report1["cars"].GetArraySafe()) {
                    if (i["location"].IsDefined()) {
                        UNIT_ASSERT_VALUES_EQUAL(i["id"].GetStringSafe(), car2.Id);
                        ++count;
                    }
                }
                UNIT_ASSERT_VALUES_EQUAL(count, 1);
            }
        }

        TGeoCoord moscowCoord(37.58643489401084, 55.73761541977004);
        NJson::TJsonValue report3 = configGenerator.GetServiceCarsList({}, USER_ID_DEFAULT, "", &moscowCoord);
        UNIT_ASSERT_C(report3["cars"].GetArraySafe().size() > 10, TStringBuilder() << report3 << Endl);
    }

    Y_UNIT_TEST(Clusters) {
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        auto car = eg.CreateCar();
        TGeoCoord someCoord(12.58643489401084, 45.73761541977004);
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("12.48 45.63 12.48 45.83 12.68 45.83 12.68 45.83", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("cluster_0", USER_ROOT_DEFAULT, coords, { "clstr" }, "cluster"));
            TMap<TString, TArea> areas;
            UNIT_ASSERT(server->GetDriveAPI()->GetAreasDB()->GetAllObjectsFromCache(areas, Now()));
            UNIT_ASSERT(areas.contains("cluster_0"));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsInPoint(someCoord).contains("clstr"));
        }
        emulator->GetContext().SetCurrentPosition(someCoord);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id, someCoord, "clstr"));
        {
            NJson::TJsonValue report = configGenerator.GetUserCarsList({ car.Id }, USER_ID_DEFAULT);
            const NJson::TJsonValue& cars = report["cars"];
            UNIT_ASSERT(cars.IsArray());
            UNIT_ASSERT_VALUES_EQUAL(cars.GetArraySafe().size(), 1);
            const NJson::TJsonValue& car = cars.GetArraySafe().at(0);
            const ui64 clusterId = car["cluster"].GetUIntegerSafe();
            UNIT_ASSERT(clusterId != 0);

            const NJson::TJsonValue& clusters = report["clusters"];
            UNIT_ASSERT(clusters.IsArray());
            UNIT_ASSERT(!clusters.GetArraySafe().empty());
            bool appropriateClusterExists = false;
            for (const NJson::TJsonValue& cluster : clusters.GetArraySafe()) {
                if (cluster["id"].GetUIntegerSafe() != clusterId)
                    continue;
                UNIT_ASSERT_VALUES_EQUAL(cluster["count"].GetUIntegerSafe(), 1);
                appropriateClusterExists = true;
                break;
            }
            UNIT_ASSERT(appropriateClusterExists);
        }
        {
            const TDriveAPI* api = server->GetDriveAPI();
            UNIT_ASSERT(api);
            auto& areaTagsManager = api->GetAreasDB()->GetTagsManager();
            auto tag = MakeAtomicShared<TClusterTag>();
            tag->SetName(TClusterTag::TypeName);
            tag->SetThreshold(1000);
            auto session = api->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(areaTagsManager.AddTag(tag, USER_ROOT_DEFAULT, "cluster_0", server.Get(), session));
            UNIT_ASSERT(session.Commit());
            TVector<TTaggedArea> allAreas;
            UNIT_ASSERT(areaTagsManager.GetAllObjectsFromCache(allAreas, Now()));
        }
        {
            NJson::TJsonValue report = configGenerator.GetUserCarsList({ car.Id }, USER_ID_DEFAULT);
            const NJson::TJsonValue& cars = report["cars"];
            UNIT_ASSERT(cars.IsArray());
            UNIT_ASSERT_VALUES_EQUAL(cars.GetArraySafe().size(), 1);
            const NJson::TJsonValue& car = cars.GetArraySafe().at(0);
            const ui64 clusterId = car["cluster"].GetUInteger();
            UNIT_ASSERT_VALUES_EQUAL(clusterId, 0);

            const NJson::TJsonValue& clusters = report["clusters"];
            UNIT_ASSERT(clusters.IsArray());
            bool appropriateClusterExists = false;
            for (const NJson::TJsonValue& cluster : clusters.GetArraySafe()) {
                if (cluster["id"].GetUIntegerSafe() == clusterId) {
                    appropriateClusterExists = true;
                    break;
                }
            }
            UNIT_ASSERT(!appropriateClusterExists);
        }
    }

    Y_UNIT_TEST(AreaDropCarPrecision) {
        using namespace NDrive::NTest;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        const TString coordsDenyArea = "37.57861227451155 55.73412469769679 37.583440250738356 55.736291827707184 37.587130970342834 55.7336282989543 37.58133739887069 55.7318242598975 37.57861227451155 55.73412469769679";
        TGeoCoord guaranteeDenyArea(37.58222789226299, 55.73386439157935);
        TGeoCoord possibleDenyArea(37.57874102054332, 55.73411259009151);
        TGeoCoord possibleAllowArea(37.578488892896864, 55.734142858506196);
        TGeoCoord guaranteeAllowArea(37.57664353309461, 55.73430327862065);
        TGeoCoord guaranteeAllowRidingOnly(38.353825, 55.896188);
        TGeoCoord guaranteeDenyRiding(43.861751, 56.799523);

        script.Add<TCreateArea>().SetId("test-deny-1").SetCoords(coordsDenyArea).SetAreaTags({"deny_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        const auto sessionCheckerDenyDropCar = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json["notification"] << Endl;
            if (json.Has("notification") && json["notification"].GetStringSafe() == "ALLOW_PARKING") {
                return true;
            }
            return false;
        };
        const auto sessionCheckerAllowDropCar = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json["notification"] << Endl;
            if (json.Has("notification") && json["notification"].GetStringSafe() == "ALLOW_DROP_CAR") {
                return true;
            }
            return false;
        };
        const auto sessionCheckerDenyRidingCar = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json["notification"] << Endl;
            if (json.Has("notification") && json["notification"].GetStringSafe() == "DENY_RIDING") {
                return true;
            }
            return false;
        };
        script.Add<TRelocateCar>().SetPosition(guaranteeDenyArea);
        script.Add<TDrop>().SetExpectOK(false);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerDenyDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TRelocateCar>().SetPosition(possibleDenyArea);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerAllowDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TRelocateCar>().SetPosition(possibleAllowArea);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerAllowDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TRelocateCar>().SetPosition(guaranteeAllowArea);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerAllowDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TRelocateCar>().SetPosition(guaranteeDenyRiding);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerDenyRidingCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TDrop>().SetExpectOK(false);

        script.Add<TRelocateCar>().SetPosition(guaranteeAllowRidingOnly);
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerDenyDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TDrop>().SetExpectOK(false);

        script.Add<TDrop>().SetCarPosition(possibleAllowArea);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(AreaBasedPromo) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eg.CreateUser("skulik-was-there", true, "onboarding");

        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("0 0 0 90 90 90 90 0", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("zone_1", USER_ROOT_DEFAULT, coords, {"global_area"}));
        }

        {
            THolder<TStandRegistrationPromoTag> tagData(new TStandRegistrationPromoTag());
            tagData->SetName("test_area_reg_promo");
            tagData->SetStartInstant(Now() - TDuration::Hours(10));
            tagData->SetFinishInstant(Now() + TDuration::Hours(10));
            UNIT_ASSERT(configGenerator.AddTag(tagData.Release(), "zone_1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
        }

        Sleep(TDuration::Seconds(10));

        for (auto it = 0; it < 3; ++it) {
            {
                UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 45.63761541977004, 12.48643489401084));
            }

            {
                TVector<TDBTag> dbTags;
                {
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"promo_markup"}, dbTags, session));
                    UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
                }
            }
        }
    }

    Y_UNIT_TEST(AreaBasedPromoOutsideLocation) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("0 0 0 90 90 90 90 0", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("zone_1", USER_ROOT_DEFAULT, coords, {"global_area"}));
        }

        {
            THolder<TStandRegistrationPromoTag> tagData(new TStandRegistrationPromoTag());
            tagData->SetName("test_area_reg_promo");
            tagData->SetStartInstant(Now() - TDuration::Hours(10));
            tagData->SetFinishInstant(Now() + TDuration::Hours(10));
            UNIT_ASSERT(configGenerator.AddTag(tagData.Release(), "zone_1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
        }

        Sleep(TDuration::Seconds(10));

        for (auto it = 0; it < 3; ++it) {
            {
                UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, -1, 12.48643489401084));
            }

            {
                TVector<TDBTag> dbTags;
                {
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"promo_markup"}, dbTags, session));
                    UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
                }
            }
        }
    }

    Y_UNIT_TEST(AreaBasedPromoEnded) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("0 0 0 90 90 90 90 0", coords));
            UNIT_ASSERT(configGenerator.UpsertArea("zone_1", USER_ROOT_DEFAULT, coords, {"global_area"}));
        }

        {
            THolder<TStandRegistrationPromoTag> tagData(new TStandRegistrationPromoTag());
            tagData->SetName("test_area_reg_promo");
            tagData->SetStartInstant(Now() - TDuration::Hours(10));
            tagData->SetFinishInstant(Now() - TDuration::Hours(8));
            UNIT_ASSERT(configGenerator.AddTag(tagData.Release(), "zone_1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
        }

        Sleep(TDuration::Seconds(10));

        for (auto it = 0; it < 3; ++it) {
            {
                UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 45.63761541977004, 12.48643489401084));
            }

            {
                TVector<TDBTag> dbTags;
                {
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"promo_markup"}, dbTags, session));
                    UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
                }
            }
        }
    }
}
