#include <drive/backend/auth/external/database.h>

#include <drive/tests/library/database.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/sqlite/structured.h>

#include <util/folder/path.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>

Y_UNIT_TEST_SUITE(ExternalAccessTokensManager) {

    THolder<IHistoryContext> CreateContext() {
        NSQL::IDatabase::TPtr db = NDrive::CreateDatabase(JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/ut/auth/db_config"));
        auto session = NDrive::TEntitySession(db->CreateTransaction());
        NStorage::IQueryResult::TPtr result = session->Exec(R"(DROP TABLE IF EXISTS external_access_tokens;)");
        UNIT_ASSERT(result->IsSucceed());
        result = session->Exec(R"(DROP TABLE IF EXISTS external_access_tokens_history;)");
        UNIT_ASSERT(result->IsSucceed());
        result = session->Exec(R"(CREATE TABLE external_access_tokens (token TEXT PRIMARY KEY,
                                    ip_whitelist JSON,
                                    user_id UUID,
                                    scope text
                                );)");
        UNIT_ASSERT(result->IsSucceed());
        result = session->Exec(R"(CREATE TABLE external_access_tokens_history (
                                    history_event_id BIGSERIAL PRIMARY KEY,
                                    history_user_id text NOT NULL,
                                    history_originator_id text,
                                    history_action text NOT NULL,
                                    history_timestamp integer NOT NULL,
                                    history_comment text,
                                    token TEXT,
                                    ip_whitelist JSON,
                                    user_id UUID,
                                    scope text
                                );)");
        UNIT_ASSERT(result->IsSucceed());
        UNIT_ASSERT(session.Commit());
        // Singleton<NSQLite::TUniqueFields>()->RegisterRowIdColumn("external_access_tokens", "token");
        return MakeHolder<THistoryContext>(db);
    }

    TExternalAccessTokensManager CreateManager(const IHistoryContext& context) {
        return TExternalAccessTokensManager(context);
    }

    Y_UNIT_TEST(SuccessfulValidation) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetValue() == "user1");
    }

    Y_UNIT_TEST(NotWhiteIp) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.0.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
    }

    Y_UNIT_TEST(InvalidToken) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token2", "208.1.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
    }

    Y_UNIT_TEST(InvalidTokenAndNotWhiteIp) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token2", "208.0.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
    }

    Y_UNIT_TEST(NoSuchToken) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
    }

    Y_UNIT_TEST(TokenInvalidation) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetValue() == "user1");
        UNIT_ASSERT(manager.Invalidate("token1", "admin2"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
    }

    Y_UNIT_TEST(ChangeOfIpWhiteList) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetValue() == "user1");
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.64.0.0/11"}, "user2", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetError() == IExternalAccessTokensManager::EValidationError::NotAllowed);
        UNIT_ASSERT(manager.Validate("token1", "208.64.123.123").GetValue() == "user2");
    }

    Y_UNIT_TEST(SeveralIpRegions) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11", "208.160.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetValue() == "user1");
        UNIT_ASSERT(manager.Validate("token1", "208.160.123.123").GetValue() == "user1");
    }

    Y_UNIT_TEST(IncorrectIp) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11", "208.160.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "288.128.123.123").GetError() == IExternalAccessTokensManager::EValidationError::IncorrectIp);
    }

    Y_UNIT_TEST(History) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());

        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11", "208.160.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Invalidate("token1", "admin2"));

        NStorage::TObjectRecordsSet<TObjectEvent<TExternalAccessToken>> history;
        NStorage::IQueryResult::TPtr result = contextHolder->GetDatabase()->CreateTransaction()->Exec(R"(SELECT * FROM external_access_tokens_history)", &history);
        UNIT_ASSERT(result->IsSucceed());
        UNIT_ASSERT(history.size() == 2);
        {
            const TObjectEvent<TExternalAccessToken>& historyRecord = history.front();
            UNIT_ASSERT(historyRecord.GetToken() == "token1");
            TVector<TString> expectedIpWhiteList{"208.128.0.0/11", "208.160.0.0/11"};
            UNIT_ASSERT(historyRecord.GetIpWhiteList() == expectedIpWhiteList);
            UNIT_ASSERT(historyRecord.GetUserId() == "user1");
            UNIT_ASSERT(historyRecord.GetHistoryUserId() == "admin");
            UNIT_ASSERT(historyRecord.GetHistoryAction() == EObjectHistoryAction::UpdateData);
        }
        {
            const TObjectEvent<TExternalAccessToken>& historyRecord = history.back();
            UNIT_ASSERT(historyRecord.GetToken() == "token1");
            TVector<TString> expectedIpWhiteList{"208.128.0.0/11", "208.160.0.0/11"};
            UNIT_ASSERT(historyRecord.GetIpWhiteList() == expectedIpWhiteList);
            UNIT_ASSERT(historyRecord.GetUserId() == "user1");
            UNIT_ASSERT(historyRecord.GetHistoryUserId() == "admin2");
            UNIT_ASSERT(historyRecord.GetHistoryAction() == EObjectHistoryAction::Remove);
        }
    }

    Y_UNIT_TEST(Scope) {
        auto contextHolder = CreateContext();
        auto manager = CreateManager(*contextHolder.Get());
        UNIT_ASSERT(manager.Upsert("token1", TVector<TString>{"208.128.0.0/11"}, "user1", "", "admin"));
        UNIT_ASSERT(manager.Upsert("token2", TVector<TString>{"208.128.0.0/11"}, "user2", "scope2", "admin"));
        UNIT_ASSERT(manager.Upsert("token3", TVector<TString>{"208.128.0.0/11"}, "user3", "scope3", "admin"));
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123").GetValue() == "user1");
        UNIT_ASSERT(manager.Validate("token1", "208.128.123.123", "scope2").GetValue() == "user1");
        UNIT_ASSERT(manager.Validate("token2", "208.128.123.123", "scope2").GetValue() == "user2");
        UNIT_ASSERT(manager.Validate("token3", "208.128.123.123", "scope3").GetValue() == "user3");
        UNIT_ASSERT(manager.Validate("token3", "208.128.123.123", "scope2").GetError() == IExternalAccessTokensManager::EValidationError::IncorrectScope);
    }
}
