#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/actions/evolution.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/tags/tags_search.h>


Y_UNIT_TEST_SUITE(B2b) {
    class B2bAccountEmulator {
    public:
        B2bAccountEmulator() = default;

        void CreateOrganization(NDrive::TServerConfigGenerator& gServer) {
            if (OrganizationId != Max<ui64>()) {
                return;
            }

            NJson::TJsonValue resultReport = CreateB2bAccountRequest(gServer);
            OrganizationId = resultReport["account_id"].GetInteger();

            NJson::TJsonValue accountUpdateData =
                NJson::TMapBuilder("balance_change_notification", false)
                    ("daily_balance_notification", false)
                    ("soft_limit", 5000000000)
                    ("refresh_interval", 1)
                    ("expenditure", 0)
                    ("low_balance_notification", false)
                    ("enable_toll_roads", false)
                    ("refresh_policy", "month")
                    ("hard_limit", 5000000000)
                    ("account_id", OrganizationId);

            if (!gServer.UpdateB2bAccount(accountUpdateData)) {
                throw yexception() << "error: b2b update organization data " << accountUpdateData.GetStringRobust();
            }

            NJson::TJsonValue activateData =
                NJson::TMapBuilder("account_id", OrganizationId)
                    ("active_flag", true);

            if (!gServer.ActivateB2bAccount(activateData)) {
                throw yexception() << "error: b2b activate account" << activateData.GetStringRobust();
            }
        }

        void CreateLimit(const TString& limitName, NDrive::TServerConfigGenerator& gServer) {
            if (Limits.find(limitName) != Limits.end()) {
                return;
            }

            NJson::TJsonValue limitUpdateData =
                NJson::TMapBuilder("name", limitName)
                ("soft_limit", 5000000000)
                ("hard_limit", 5000000000)
                ("type", "wallet")
                ("meta", NJson::TMapBuilder("limited_policy",false)
                            ("data_type", "simple")
                            ("offers_filter", "fix")
                            ("parent_id", OrganizationId)
                            ("refresh_policy", "month")
                            ("selectable", true)
                );

            if (!gServer.CreateB2bWalletAccount(limitUpdateData)) {
                throw yexception() << "error: b2b create limit" << limitUpdateData.GetStringRobust();
            }

            UpdateData(gServer);

        }

        void CreateLimitInstance(const TString& limitName, NDrive::TServerConfigGenerator& gServer) {
            if (Limits.find(limitName) == Limits.end()) {
                return;
            }

            NJson::TJsonValue limitInstanceData = NJson::TMapBuilder("name", limitName);
            NJson::TJsonValue resultReport = CreateB2bAccountRequest(gServer, limitInstanceData);
            ui64 limitId = resultReport["account_id"].GetInteger();

            if (!gServer.ActivateB2bAccount(NJson::TMapBuilder("account_id", limitId)("active_flag", true))) {
                throw yexception() << "error: b2b activate limit account";
            }

            UpdateData(gServer);
        }

        TVector<TString> GetLimitsNames() {
            TVector<TString> names;
            names.reserve(Limits.size());
            for (auto&& l : Limits) {
                names.emplace_back(l.first);
            }

            return std::move(names);
        }

        ui64 GetLimitId(const TString& name) {
            if (auto it = Limits.find(name); it != Limits.end()) {
                return it->second;
            }
            return 0;
        }

        ui64 GetOrganizationId() {return OrganizationId;}

        void LinkUser(NDrive::TServerConfigGenerator& gServer, const TString& id, const TString& limitName) {
            NJson::TJsonValue linkData =
                NJson::TMapBuilder("action", "link")
                    ("force", false)
                    ("active_flag", false)
                    ("user_id", id)
                    ("name", limitName);

            if (!gServer.LinkB2bAccount(linkData)) {
                throw yexception() << "error: b2b link user to account";
            }
        }

    protected:
        NJson::TJsonValue CreateB2bAccountRequest(NDrive::TServerConfigGenerator& gServer, const NJson::TJsonValue& payload = NJson::TJsonValue()) {
            NJson::TJsonValue resultReport = gServer.CreateB2bAccount(payload);
            if (resultReport == NJson::JSON_NULL) {
                throw yexception() << "error: b2b create organization";
            }

            return std::move(resultReport);
        }

        void UpdateData(NDrive::TServerConfigGenerator& gServer) {
            Limits.clear();
            LimitInstances.clear();

            TStringStream ss;
            ss << "&parent_id=" << OrganizationId;
            ss << "&access_tags=organization_creator";

            NJson::TJsonValue resultReport = gServer.GetB2bAccountsDescription(ss);
            if (resultReport == NJson::JSON_NULL) {
                throw yexception() << "error: b2b update description";
            }

            for (auto&& l : resultReport["accounts"].GetArray())
                Limits[l["name"].GetString()] = l["id"].GetInteger();

            for (const auto& l : Limits) {
                NJson::TJsonValue resultReport = gServer.GetB2bAccounts(l.first);
                if (resultReport == NJson::JSON_NULL) {
                    throw yexception() << "error: b2b get accounts";
                }

                for (auto&& inst : resultReport["accounts"].GetArray()) {
                    LimitInstances[l.first].emplace_back(inst["data"]["id"].GetInteger());
                }
            }

        }

    private:
        ui64 OrganizationId = Max<ui64>();
        TMap<TString, ui64> Limits;
        TMap<TString, TVector<ui64>> LimitInstances;

    };

    void RegisterOrganizationCreatorTag(const NDrive::TServerGuard& server) {
        auto organizationCreatorTag = MakeAtomicShared<TWalletAccessTag::TDescription>();
        organizationCreatorTag->SetName("organization_creator");
        organizationCreatorTag->SetType(TWalletAccessTag::TypeName);
        RegisterTag(*server, organizationCreatorTag);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
    }

    Y_UNIT_TEST(EvolveB2BFToCrmFormTag) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TString payload = R"(
                {
                    "fields":[
                        {"value":"Тест Драйв","key":"name"},
                        {"value":"Сделка Тест Драйв","key":"company"},
                        {"value":"","key":"phone"},
                        {"value":"drayv.t@yandex.com","key":"email"},
                        {"value":"0061j00000Jg0jYAAR","key":"record_id"},
                        {"value":"some@some.ru","key":"federation_id"}
                    ]
                }
            )";

        NJson::TJsonValue payloadJson = NJson::JSON_NULL;
        UNIT_ASSERT(NJson::ReadJsonTree(payload, &payloadJson));

        //pure crm flow
        {
            auto userId = eg.CreateUser("test-org-user1", true, "active");
            {
                TUserRole userRole;
                auto session = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
                userRole.SetRoleId("root").SetUserId(userId);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
            }

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            NJson::TJsonValue resultReport = gServer.CrmCreateLead(userId, payloadJson);
            UNIT_ASSERT_C(resultReport["tag_id"].IsString(), resultReport["tag_id"].GetStringRobust());
            UNIT_ASSERT_C(resultReport["user_id"].IsString(), resultReport["user_id"].GetStringRobust());

            TString lastTagId = resultReport["tag_id"].GetString();

            resultReport = gServer.CrmCreateLead(userId, payloadJson);
            UNIT_ASSERT_C(resultReport["tag_id"].IsString(), resultReport["tag_id"].GetStringRobust());
            UNIT_ASSERT_C(resultReport["user_id"].IsString(), resultReport["user_id"].GetStringRobust());
            UNIT_ASSERT_C(resultReport["tag_id"].GetString() == lastTagId, resultReport["user_id"].GetStringRobust());
        }

        //evolve from b2b to crm
        {
            auto userId = eg.CreateUser("test-org-user2", true, "active");
            {
                TUserRole userRole;
                auto session = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
                userRole.SetRoleId("root").SetUserId(userId);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
            }

            const TString b2bAppFormTagName = "b2b_application_form";

            UNIT_ASSERT(gServer.AddTag(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(b2bAppFormTagName),
                userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto session = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
            TVector<TDBTag> organizationTags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreTagsRobust({userId}, {b2bAppFormTagName}, organizationTags, session));

            TDBTag* b2bLeadTag = nullptr;
            for (auto&& tag : organizationTags) {
                if (tag->GetName() == b2bAppFormTagName){
                    b2bLeadTag = &tag;
                }
            }

            UNIT_ASSERT(b2bLeadTag != nullptr);

            NJson::TJsonValue resultReport = gServer.CrmCreateLead(userId, payloadJson);
            UNIT_ASSERT_C(resultReport["tag_id"].IsString(), resultReport["tag_id"].GetStringRobust());
            UNIT_ASSERT_C(resultReport["user_id"].IsString(), resultReport["user_id"].GetStringRobust());
            UNIT_ASSERT_C(resultReport["tag_id"].GetString() == b2bLeadTag->GetTagId(), resultReport["tag_id"].GetStringRobust());
        }
    }

    Y_UNIT_TEST(CreateOrganization) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        try {
            B2bAccountEmulator emulator;
            emulator.CreateOrganization(gServer);
            emulator.CreateLimit("PrimaryLimit", gServer);
            UNIT_ASSERT_C(!emulator.GetLimitsNames().empty(), "no limits created");
            emulator.CreateLimitInstance(emulator.GetLimitsNames().back(), gServer);
        } catch(const yexception& e) {
            UNIT_ASSERT_C(false, e.what());
        }

    }

    Y_UNIT_TEST(AccountSessionList) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TString limit("PrimaryLimit");
        const TString model1("bmw_520i_b");
        const TString model2("skoda_rapid");

        B2bAccountEmulator emulator;

        const TInstant startRides = Now() - TDuration::Hours(12);
        NDrive::NTest::TScript script(configGenerator, startRides);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCommonChecker>(
            [&emulator, &limit](NDrive::NTest::TRTContext& context) {
                NDrive::TServerConfigGenerator& configGenerator = context.GetConfigGenerator();
                try {
                    emulator.CreateOrganization(configGenerator);
                    emulator.CreateLimit(limit, configGenerator);
                    UNIT_ASSERT_C(!emulator.GetLimitsNames().empty(), "no limits created");
                    emulator.CreateLimitInstance(emulator.GetLimitsNames().back(), configGenerator);
                } catch(const yexception& e) {
                    UNIT_ASSERT_C(false, e.what());
                }
            });

        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(limit).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>(
            [&limit](NDrive::NTest::TRTContext& context) {
                auto session = context.GetDriveAPI().GetBillingManager().BuildSession(true);
                auto accounts = context.GetDriveAPI().GetBillingManager().GetAccountsManager().GetUserAccounts(USER_ID_DEFAULT, session);
                UNIT_ASSERT(accounts);
                TSet<TString> accNames;
                for (const auto& acc : *accounts) {
                    accNames.emplace(acc->GetUniqueName());
                }
                UNIT_ASSERT_C(accNames.contains(limit), TStringBuilder() << "error: no " << limit << " linked to user.");
        });

        auto createRide = [&](const TString& carModel, const TString& limitName, TGeoCoord rideTo, TGeoCoord rideFrom) {
            script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<NDrive::NTest::TCreateCar>().SetModel(carModel).SetPosition(rideFrom);
            script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetAccountName(limitName).SetUserDestination(rideTo).SetUserPosition(rideFrom);
            script.Add<NDrive::NTest::TInitSessionId>();
            script.Add<NDrive::NTest::TAccept>();
            script.Add<NDrive::NTest::TRide>(TDuration::Hours(1)).SetCarPosition(rideTo);
            script.Add<NDrive::NTest::TCommonChecker>(
                [&](NDrive::NTest::TRTContext& context) {
                    auto bSessions = context.GetDriveAPI().GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
                    auto sessions = bSessions->GetSessionsActual();
                    for (auto&& session : sessions) {
                        if (session->GetSessionId() != context.GetCurrentSessionIdUnsafe()) {
                            continue;
                        }

                        auto billingSession = dynamic_cast<const TBillingSession*>(session.Get());
                        if (!billingSession) {
                            continue;
                        }

                        auto offer = billingSession->GetCurrentOffer();
                        if (!offer) {
                            continue;
                        }

                        INFO_LOG << "Actual session info: " << "\n"
                            << "    Session id: " << billingSession->GetSessionId()  << "\n"
                            << "    ObjectModel: " << offer->GetObjectModel()  << "\n"
                            << "    Selected charge: "<< offer->GetSelectedCharge()  << "\n";

                        UNIT_ASSERT_VALUES_EQUAL(offer->GetSelectedCharge(), limitName);
                    }
                });

            script.Add<NDrive::NTest::TCommonChecker>(
                [&](NDrive::NTest::TRTContext& context) {
                    NDrive::TServerConfigGenerator& gServer = context.GetConfigGenerator();
                    NJson::TJsonValue resultReport =
                        gServer.GetB2bSessionsList(TStringBuilder() << "parent_id=" << emulator.GetOrganizationId());
                    UNIT_ASSERT_C(resultReport["actual_orders"].GetArray().size() == 1, TStringBuilder() << "actual_orders array count mismatch " << resultReport.GetStringRobust());
                });

            script.Add<NDrive::NTest::TPark>().SetCarPosition(rideTo);
            script.Add<NDrive::NTest::TDrop>();
            script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
            script.Add<NDrive::NTest::TDropCache>();

            const auto sessionHistoryChecker =
                [&](const NDrive::NTest::TRTContext& context, const NJson::TJsonValue& json)->bool {
                    INFO_LOG << json.GetStringRobust() << Endl;
                    UNIT_ASSERT_VALUES_EQUAL(json["sessions"].GetArraySafe().size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(json["sessions"].GetArraySafe()[0]["offer_proto"]["OfferId"].GetStringRobust(), context.GetCurrentSessionIdUnsafe());
                    return true;
                };

            script.Add<NDrive::NTest::TCheckSessionHistoryWaiting>().SetChecker(sessionHistoryChecker);
            script.Add<NDrive::NTest::TDropCache>();
            script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto& billingManager = context.GetDriveAPI().GetBillingManager();
                TCachedPayments snapshot;
                auto session = billingManager.BuildSession(true);
                UNIT_ASSERT(billingManager.GetPaymentsManager().GetPayments(snapshot, context.GetCurrentSessionIdUnsafe(), session));
                UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
                for (auto&& paymentTask : snapshot.GetPayments()) {
                    INFO_LOG << "PaymentTask: " << "\n"
                        << "    AccountId: " << paymentTask.GetAccountId() << "\n"
                        << "    PayMethod: " << paymentTask.GetPayMethod() << "\n"
                        << "    PaymentId: " << paymentTask.GetPaymentId() << "\n"
                        << "    SessionId: " << paymentTask.GetSessionId() << "\n"
                        << "    Sum: " << paymentTask.GetSum() << "\n"
                        << "    PaymentErrorDescription: " << paymentTask.GetPaymentErrorDescription() << "\n";
                }
                UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetPaymentType(), NDrive::NBilling::EAccount::Wallet);
            });

            script.Add<NDrive::NTest::TRunBillingCycle>();

            script.Add<NDrive::NTest::TCommonChecker>(
                [&](NDrive::NTest::TRTContext& context) {
                    NDrive::TServerConfigGenerator& gServer = context.GetConfigGenerator();

                    auto options = TBillingHistoryManager::TQueryOptions(1, true);
                    options.AddGenericCondition("session_id", context.GetCurrentSessionIdUnsafe());
                    options.SetOrderBy({ "history_event_id" });

                    auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
                    auto lastEvent = context.GetDriveAPI().GetBillingManager().GetHistoryManager().GetEvents({}, {}, session, options);
                    INFO_LOG << "Last Event info: " << "\n"
                        << "    Size: " << lastEvent->size() << "\n"
                        << "    GetHistoryAction: " << ::ToString(lastEvent->front().GetHistoryAction()) << "\n";

                    UNIT_ASSERT(gServer.CreateCompiledBill(NJson::TMapBuilder("session_id", context.GetCurrentSessionIdUnsafe())));
                });

            script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        };

        TGeoCoord geoFrom(37.5848674, 55.7352435);
        TGeoCoord geoTo(37.5675511, 55.7323499);
        createRide(model1, limit, geoTo, geoFrom);
        createRide(model2, limit, geoFrom, geoTo);

        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>(
            [&](NDrive::NTest::TRTContext& context) {
                NDrive::TServerConfigGenerator& gServer = context.GetConfigGenerator();

                auto cgiBuilder =
                    [](ui64 since, ui64 until, ui32 limit, ui64 parentId, const TString& searchStr = "") {
                        return std::move(TStringBuilder()
                            << "&since=" << since
                            << "&until=" << until
                            << "&limit=" << limit
                            << "&parent_id=" << parentId
                            << "&has_all_of=" << searchStr);
                    };
                {
                    NJson::TJsonValue resultReport = gServer.GetB2bSessionsList(cgiBuilder(startRides.Seconds(), Now().Seconds(), 3, emulator.GetOrganizationId()));
                    UNIT_ASSERT_C(resultReport["orders"].GetArray().size() == 2, TStringBuilder() << "orders array count mismatch " << resultReport.GetStringRobust());
                }

                {
                    NJson::TJsonValue resultReport = gServer.GetB2bSessionsList(cgiBuilder(startRides.Seconds(), Now().Seconds(), 1, emulator.GetOrganizationId()));
                    UNIT_ASSERT_C(resultReport["orders"].GetArray().size() == 1, TStringBuilder() << "orders array count mismatch " << resultReport.GetStringRobust());
                }

                {
                    NJson::TJsonValue resultReport = gServer.GetB2bSessionsList(std::move(cgiBuilder(startRides.Seconds(), Now().Seconds(), 3, emulator.GetOrganizationId(), model1)));
                    UNIT_ASSERT_C(resultReport["orders"].GetArray().size() == 1, TStringBuilder() << "array count mismatch " << resultReport.GetStringRobust());
                }

                {
                    NJson::TJsonValue resultReport = gServer.GetB2bSessionsList(std::move(cgiBuilder(startRides.Seconds(), Now().Seconds(), 3, emulator.GetOrganizationId(), model2)));
                    UNIT_ASSERT_C(resultReport["orders"].GetArray().size() == 1, TStringBuilder() << "array count mismatch " << resultReport.GetStringRobust());
                }
            });

        script.Execute();
    }

}
