#include <drive/backend/data/account_tags.h>
#include <drive/backend/ut/library/helper2.h>

Y_UNIT_TEST_SUITE(BalanceClient) {

    inline const TString SimpleExpiredDebtTag = "simple_expired_debt_tag";

    const TString partnerBalanceXml = R"(
        <value><struct>
        <member><name>Currency</name><value><string>RUB</string></value></member>
        <member><name>Amount</name><value><int>0</int></value></member>
        <member><name>ActSum</name><value><string>24945.47</string></value></member>
        <member><name>LastActDT</name><value><string>2022-05-31T00:00:00</string></value></member>
        <member><name>ReceiptSum</name><value><string>25200</string></value></member>
        <member><name>ConsumeSum</name><value><string>24945.47</string></value></member>
        <member><name>PersonalAccountExternalID</name><value><string>ЛСДР-4054476748-1</string></value></member>
        <member><name>DT</name><value><string>2022-06-23T14:31:28.733083</string></value></member>
        <member><name>ContractID</name><value><int>5230212</int></value></member>
        </struct></value>
    )";

    const TString partnerBalanceExpiredXml = R"(
        <value><struct>
        <member><name>Currency</name><value><string>RUB</string></value></member>
        <member><name>Amount</name><value><int>0</int></value></member>
        <member><name>ActSum</name><value><string>24945.47</string></value></member>
        <member><name>LastActDT</name><value><string>2022-05-31T00:00:00</string></value></member>
        <member><name>ReceiptSum</name><value><string>25200</string></value></member>
        <member><name>ConsumeSum</name><value><string>24945.47</string></value></member>
        <member><name>PersonalAccountExternalID</name><value><string>ЛСДР-4054476748-1</string></value></member>
        <member><name>DT</name><value><string>2022-06-23T14:31:28.733083</string></value></member>
        <member><name>ContractID</name><value><int>5230212</int></value></member>
        <member><name>ExpiredDebtAmount</name><value><string>23036.09</string></value></member>
        <member><name>ExpiredDT</name><value><string>2022-03-31T00:00:00</string></value></member>
        <member><name>FirstDebtAmount</name><value><string>4025.45</string></value></member>
        <member><name>FirstDebtFromDT</name><value><string>2022-03-31T00:00:00</string></value></member>
        <member><name>FirstDebtPaymentTermDT</name><value><string>2022-03-31T00:00:00</string></value></member>
        </struct></value>
    )";

    TBalanceClient::TBalance GetDefaultBalance() {
        TBalanceClient::TBalance balance;
        UNIT_ASSERT(balance.FromXml(NXmlRPC::ParseValue(partnerBalanceXml).Struct()));
        return balance;
    }

    TBalanceClient::TBalance GetExpiredBalance() {
        TBalanceClient::TBalance balance;
        UNIT_ASSERT(balance.FromXml(NXmlRPC::ParseValue(partnerBalanceExpiredXml).Struct()));
        return balance;
    }

    void RegisterTags(const NDrive::IServer &server) {
        auto tag = MakeAtomicShared<TExpiredDebtAccountTag::TDescription>();
        tag->SetName(SimpleExpiredDebtTag);
        tag->SetType(TExpiredDebtAccountTag::TypeName);
        tag->SetDeactivateAccount(true);
        tag->SetDeactivateSinceMonthDay(20);
        tag->SetWarningSinceMonthDay(15);
        RegisterTag(server, tag);
    }

    ui64 PrepareAccount(TTestEnvironment& env) {
        auto driveApi = env.GetServer()->GetDriveAPI();
        auto& billingManager = driveApi->GetBillingManager();
        auto& accountsManager = billingManager.GetAccountsManager();

        {
            auto tx = billingManager.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = NDrive::NBilling::AccountLimit;
            descJson["soft_limit"] = NDrive::NBilling::AccountLimit;
            descJson["name"] = "corp.wallet";
            descJson["meta"]["hr_name"] = "corp.wallet";
            descJson["meta"]["refresh_policy"] = ToString(NDrive::NBilling::TRefreshSchedule::EPolicy::Month);
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());
        }

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto tx = driveApi->GetBillingManager().BuildSession();
        auto baDescription = accountsManager.GetDescriptionByName("corp.wallet");
        UNIT_ASSERT(baDescription.Defined());
        auto orgAccount = accountsManager.GetOrCreateAccount(USER_ROOT_DEFAULT, baDescription.GetRef(), USER_ROOT_DEFAULT, tx);
        UNIT_ASSERT(tx.Commit());
        UNIT_ASSERT(orgAccount->GetId());

        return orgAccount->GetId();
    }

    void UpdateAccountWithBalance(const ui32& accountId, const TBalanceClient::TBalance& balanceStruct, const NDrive::IServer &server) {
        const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
        const auto& accountsTags = tagsManager.GetAccountTags();
        const auto& tagsMeta = tagsManager.GetTagsMeta();
        const auto& billingManager = server.GetDriveAPI()->GetBillingManager();
        const auto& accountsManager = billingManager.GetAccountsManager();

        auto session = server.GetDriveAPI()->BuildTx<NSQL::Writable>();
        auto taggedAccount = accountsTags.RestoreObject(::ToString(accountId), session);
        UNIT_ASSERT_C(taggedAccount, "cannot find account with id: " << accountId);

        TExpiredDebtAccountTag::TPtr expiredDebtTagImpl;
        auto expiredDebtTag = taggedAccount->GetTag(SimpleExpiredDebtTag);
        if (expiredDebtTag) {
            expiredDebtTagImpl = std::dynamic_pointer_cast<TExpiredDebtAccountTag>(expiredDebtTag->GetData());
        }

        if (expiredDebtTagImpl && !balanceStruct.HasExpiredDT()) {
            UNIT_ASSERT_C(accountsTags.RemoveTags({*expiredDebtTag}, USER_ROOT_DEFAULT, &server, session), "cannot remove expired debt tag from account with id: " << accountId << " " << session.GetStringReport());
            expiredDebtTagImpl.Reset();
        } else if (!expiredDebtTagImpl && balanceStruct.HasExpiredDT()) {
            expiredDebtTagImpl = tagsMeta.CreateTagAs<TExpiredDebtAccountTag>(SimpleExpiredDebtTag);
            expiredDebtTagImpl->OptionalFirstExpiredDT() = balanceStruct.OptionalFirstDebtFromDT();
            expiredDebtTagImpl->OptionalDebtAmount() = balanceStruct.OptionalExpiredDebtAmount();
            UNIT_ASSERT_C(expiredDebtTagImpl->CalculateValues(&server, session), "error while calculating tag values account id: " << accountId);
            UNIT_ASSERT_C(accountsTags.AddTag(expiredDebtTagImpl, USER_ROOT_DEFAULT, ::ToString(accountId), &server, session), "cannot add tag account id: " << accountId << " " << session.GetStringReport());
        } else if (expiredDebtTagImpl) {
            expiredDebtTagImpl->OptionalFirstExpiredDT() = balanceStruct.OptionalFirstDebtFromDT();
            expiredDebtTagImpl->OptionalDebtAmount() = balanceStruct.OptionalExpiredDebtAmount();
            UNIT_ASSERT_C(expiredDebtTagImpl->CalculateValues(&server, session), "error while calculating tag values account id: " << accountId);
            UNIT_ASSERT_C(accountsTags.UpdateTagData(*expiredDebtTag, USER_ROOT_DEFAULT, session), "cannot update expired debt tag, account id: " << accountId << " " << session.GetStringReport());
        }

        if (expiredDebtTagImpl && expiredDebtTagImpl->MustBeDeactivated(&server, session)) {
            UNIT_ASSERT_C(accountsManager.ActivateAccounts({accountId}, false, USER_ROOT_DEFAULT, session), "cannot deactivate account: " << accountId);
            UNIT_ASSERT_C(expiredDebtTagImpl->GetWarningDateRef() == TInstant::Seconds(1649970000), "invalid warning date: " << expiredDebtTagImpl->GetWarningDateRef() << "!=" << TInstant::Seconds(1649970000));
            UNIT_ASSERT_C(expiredDebtTagImpl->GetDeactivateDateRef() == TInstant::Seconds(1650402000), "invalid deactivate date: " << expiredDebtTagImpl->GetDeactivateDateRef() << "!=" << TInstant::Seconds(1650402000));
        } else if (!expiredDebtTagImpl) {
            UNIT_ASSERT_C(accountsManager.ActivateAccounts({accountId}, true, USER_ROOT_DEFAULT, session), "cannot activate account: " << accountId);
        }


        UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
    }

    Y_UNIT_TEST(ExpiredBalance) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        auto& server = *env.GetServer();
        RegisterTags(server);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
        const auto& accountsTags = tagsManager.GetAccountTags();
        const auto& billingManager = server.GetDriveAPI()->GetBillingManager();
        const auto& accountsManager = billingManager.GetAccountsManager();

        ui32 accountId = PrepareAccount(env);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto checkDefaultBalance = [&]() {
            UpdateAccountWithBalance(accountId, GetDefaultBalance(), server);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            auto session = server.GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
            auto taggedAccount = accountsTags.RestoreObject(::ToString(accountId), session);
            UNIT_ASSERT_C(taggedAccount, "cannot find account with id: " << accountId);
            auto expiredDebtTag = taggedAccount->GetTag(SimpleExpiredDebtTag);
            UNIT_ASSERT_C(!expiredDebtTag, "error find expired debt tag, account id: " << accountId);

            auto account = accountsManager.GetAccountById(accountId);
            UNIT_ASSERT_C(account, "cannot find account, account id: " << accountId);
            UNIT_ASSERT_C(account->IsActive(), "account cannot be deactivated");
        };

        checkDefaultBalance();

        {
            UpdateAccountWithBalance(accountId, GetExpiredBalance(), server);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            auto session = server.GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
            auto taggedAccount = accountsTags.RestoreObject(::ToString(accountId), session);
            UNIT_ASSERT_C(taggedAccount, "cannot find account with id: " << accountId);
            auto expiredDebtTag = taggedAccount->GetTag(SimpleExpiredDebtTag);
            UNIT_ASSERT_C(expiredDebtTag, "cannot find expired debt tag, account id: " << accountId);
            auto impl = expiredDebtTag->MutableTagAs<TExpiredDebtAccountTag>();
            UNIT_ASSERT_C(impl, "cannot cast toTExpiredDebtAccountTag");
            auto account = accountsManager.GetAccountById(accountId);
            INFO_LOG << "ExpiredBalance Account_id: " << accountId << " " <<  impl->SerializeToLog();
            UNIT_ASSERT_C(account, "cannot find account, account id: " << accountId);
            UNIT_ASSERT_C(!account->IsActive(), "account cannot be active");
            UNIT_ASSERT_C(impl->IsNeedWarning(), "warning is needed");
            UNIT_ASSERT_C(impl->IsNeedDeactivate(), "deactivating in needed");
        }

        checkDefaultBalance();
    }
}
