#include <drive/backend/ut/library/helper.h>

#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/documents_verification/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/yang.h>
#include <drive/backend/yang/client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(Bans) {

    Y_UNIT_TEST(ManagerSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");
        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT(user.GetStatus() != "blocked");
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetUserRegistrationManager()->BanUser(userId, userId, NBans::EReason::Other, session));
            UNIT_ASSERT(session.Commit());
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetUserRegistrationManager()->UnbanUser(userId, userId, session));
            UNIT_ASSERT(session.Commit());
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "onboarding");
        }
    }

    Y_UNIT_TEST(BlockingWindowsCompatibility) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        auto session = configGenerator.GetCurrentSession(userId);
        UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "registration");
        UNIT_ASSERT(!session["user"]["show_chat"]["is_closable"].GetBoolean());
    }

    Y_UNIT_TEST(BlockingWindowsFromTag) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto tag = new TUserChatShowTag("user_chat_show_tag");
            tag->SetIsClosable(true);
            tag->SetTopicLink("support.321");
            UNIT_ASSERT(configGenerator.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        auto session = configGenerator.GetCurrentSession(userId);
        UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "support.321");
        UNIT_ASSERT(session["user"]["show_chat"]["is_closable"].GetBoolean());
    }

    Y_UNIT_TEST(BlockingWindowsFromUserDB) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->SetChatToShow(userId, "support.222", true, "robot-frontend", server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }

        auto session = configGenerator.GetCurrentSession(userId);
        UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "support.222");
        UNIT_ASSERT(session["user"]["show_chat"]["is_closable"].GetBoolean());
    }

    Y_UNIT_TEST(DrivingBanSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");

        {
            auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_VALUES_EQUAL(TUserProblemTag::ShouldBeBlocked(permissions->GetUserId(), *server, session), TUserProblemTag::EBlockedStatus::Ok);
        }

        {
            // Initiate chat for this user. In such way history will contain intro messages and chat robot will 'exist' for user.
            auto messagesJson = configGenerator.GetChatMessages(userId, "high_speed_ban");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
        }

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT(user.GetStatus() != "blocked");
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_speed_asshole", "zz");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, server.Get(), session) && session.Commit());
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            // User should be forwarded to the ban chat after being blocked. It will result into additional messages.
            // Messages depend on entry node specified in the tag and in a robot script.
            auto messagesJson = configGenerator.GetChatMessages(userId, "high_speed_ban");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "Когда запрет будет снят, вы получите смс и пуш-уведомление.");
        }

        {
            auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_VALUES_EQUAL(TUserProblemTag::ShouldBeBlocked(permissions->GetUserId(), *server, session), TUserProblemTag::EBlockedStatus::Blocked);
        }
    }

    Y_UNIT_TEST(DrivingBanSpeedAssholeSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");
        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT(user.GetStatus() != "blocked");
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetUserRegistrationManager()->BanUser(userId, userId, NBans::EReason::SpeedAsshole, session));
            UNIT_ASSERT(session.Commit());
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }
    }

    Y_UNIT_TEST(AutoTagsBan) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");
        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_minor", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_major", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "onboarding");
        }
    }

    Y_UNIT_TEST(AutoTagsBanReturnToActive) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto userId = eGenerator.CreateUser("skulik-was-there");
        {
            auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
            auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
            auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
            auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
            auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().GetResult();

            TString newAssignmentId;
            for (auto&& it : allAssignmentsNew) {
                if (
                    it.second.GetLicenseBackId() == licenseBackId &&
                    it.second.GetLicenseFrontId() == licenseFrontId &&
                    it.second.GetPassportBiographicalId() == passportBiographicalId &&
                    it.second.GetPassportRegistrationId() == passportRegistrationId &&
                    it.second.GetPassportSelfieId() == passportSelfieId
                ) {
                    newAssignmentId = it.second.GetId();
                }
            }
            UNIT_ASSERT(!!newAssignmentId);
            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            Sleep(TDuration::Seconds(10));
        }

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_minor", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_major", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            driveApi.GetUsersData()->FetchInfo(userId);
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }
    }

    Y_UNIT_TEST(AutoTagsBanOldLicense) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);
        {
            auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
            auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto allAssignmentsNew =  docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().GetResult();

            TString newAssignmentId;
            for (auto&& it : allAssignmentsNew) {
                if (
                    it.second.GetLicenseBackId() == licenseBackId &&
                    it.second.GetLicenseFrontId() == licenseFrontId &&
                    it.second.GetPassportBiographicalId() == passportBiographicalId &&
                    it.second.GetPassportRegistrationId() == passportRegistrationId &&
                    it.second.GetPassportSelfieId() == passportSelfieId
                ) {
                    newAssignmentId = it.second.GetId();
                }
            }
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);

            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).GetResult().begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }


        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("blocked_old_license"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString(), configGenerator.GetValidImageString()}));
        }

        TString yaId;
        {
            auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
            auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto allAssignmentsNew =  docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().GetResult();

            TString newAssignmentId;
            for (auto&& it : allAssignmentsNew) {
                if (
                    it.second.GetLicenseBackId() == licenseBackId &&
                    it.second.GetLicenseFrontId() == licenseFrontId &&
                    it.second.GetPassportBiographicalId() == passportBiographicalId &&
                    it.second.GetPassportSelfieId() == passportSelfieId &&
                    it.second.GetPassportRegistrationId() == passportRegistrationId
                ) {
                    newAssignmentId = it.second.GetId();
                }
            }
            UNIT_ASSERT(!!newAssignmentId);
            yaId = newAssignmentId;

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud, true);
        }

        {
            auto ya = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(yaId).GetResult().begin()->second;
            server->GetUserRegistrationManager()->HandleVerifiedAssignment(ya, "unittest", Now());
        }

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }
    }

    Y_UNIT_TEST(AutoTagsBanReturnToOnboarding) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
        {
            auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
            auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
            auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
            auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
            auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto allAssignmentsNew =  docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().GetResult();

            TString newAssignmentId;
            for (auto&& it : allAssignmentsNew) {
                if (
                    it.second.GetLicenseBackId() == licenseBackId &&
                    it.second.GetLicenseFrontId() == licenseFrontId &&
                    it.second.GetPassportBiographicalId() == passportBiographicalId &&
                    it.second.GetPassportRegistrationId() == passportRegistrationId &&
                    it.second.GetPassportSelfieId() == passportSelfieId
                ) {
                    newAssignmentId = it.second.GetId();
                }
            }
            UNIT_ASSERT(!!newAssignmentId);
            eGenerator.CompleteYangAssignment(newAssignmentId, "ooioi", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
        }

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_minor", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
        }

        {
            TString tagId;
            UNIT_ASSERT(configGenerator.GetTagId(userId, "user_problem_tag_major", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(configGenerator.RemoveTag({tagId}, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "onboarding");
        }
    }
}
