#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/billing/ut/library/offer.h>

#include <library/cpp/testing/unittest/registar.h>


Y_UNIT_TEST_SUITE(BillingInfo) {
    using namespace NDrive::NTest;
    using TChecker = std::function<void(const NJson::TJsonValue& response, TRTContext& context)>;

    class TGetBillingPaymentInfo: public TAPIAction {
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TGetBillingPaymentInfo> Registrator;

    private:
        R_FIELD(TString, RequestSessionId);
        R_FIELD(TString, RequestUserId);
        R_FIELD(TChecker, Checker);

    public:
        using TBase::TBase;

        TGetBillingPaymentInfo(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/staff/billing/payment/info>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: user_payments" << Endl;
            ss << "</api/staff/billing/payment/info>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            TString request = "/api/staff/billing/payment/info";
            TString cgi = RequestUserId ? "user_id=" + RequestUserId : "session_id=" + RequestSessionId;
            cgi += "&until=" + ToString(ModelingNow().Seconds() + TDuration::Days(1).Seconds());
            NJson::TJsonValue reply = SendRequest(context, "/api/staff/billing/payment/info", NJson::JSON_NULL, cgi);
            UNIT_ASSERT(GetExpectOK() == !reply.IsNull());
            Checker(reply, context);
        }
    };

    TAPIAction::TFactory::TRegistrator<TGetBillingPaymentInfo> TGetBillingPaymentInfo::Registrator("TGetBillingPaymentInfo");

    class TDBExec: public ITestAction {
        R_FIELD(TString, Query);
        using TBase = ITestAction;

    public:
        using TBase::TBase;

        TDBExec(const TInstant startInstant, const TString& query)
            : TBase(startInstant)
            , Query(query)
        {
        }

    protected:
        void DoExecute(TRTContext& context) override {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession(false);
            session->Exec(Query);
            UNIT_ASSERT(session.Commit());
        }
    };

    const TString DEFAULT_ACCOUNT = "y.drive";

    class CreateCloseSession: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, OfferId);
        R_FIELD(TString, UserId, USER_ID_DEFAULT);
        R_FIELD(TString, Account, DEFAULT_ACCOUNT);
        R_FIELD(double, Money, 10000);

    protected:
        void DoExecute(TRTContext& context) override {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();

            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(OfferId).SetPaymentDiscretization(500);
            TVector<TString> accounts;
            accounts.push_back(Account);
            offer->SetChargableAccounts(accounts);
            {
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.CreateBillingTask(UserId, offer, session, EBillingType::CarUsage));
                UNIT_ASSERT(session.Commit());
            }
            {
                TMap<TString, double> money;
                money[OfferId] = Money;
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.SetBillingInfo(money, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.FinishingBillingTask(offer->GetOfferId(), session));
                UNIT_ASSERT(session.Commit());
                UNIT_ASSERT(billingManager.AddClosedBillingInfo(offer->GetOfferId(), USER_ROOT_DEFAULT));
            }
            billingManager.WaitBillingCycle();
        }
    public:
        using TBase::TBase;

        CreateCloseSession(const TInstant startInstant, const TString& offerId = "")
            : TBase(startInstant)
            , OfferId(offerId ? offerId : ICommonOffer::CreateOfferId())
        {
        }
    };

    Y_UNIT_TEST(PaymentInfo) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        const TString& offerId = script.Add<CreateCloseSession>().GetOfferId();
        script.Add<NDrive::NTest::TDropCache>();
        TString sessionId = ICommonOffer::CreateOfferId();
        script.Add<NDrive::NTest::TCommonChecker>([&offerId, &sessionId](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            {
                auto session = billingManager.BuildSession(false);
                TBillingTask task;
                task.SetId(sessionId).SetUserId(USER_ID_DEFAULT).SetBill(0).SetDeposit(1000).SetBillingType(EBillingType::TollRoad).SetRealSessionId(offerId);
                TSet<TString> accounts;
                accounts.emplace("card");
                task.SetChargableAccounts(accounts);
                task.SetLastUpdate(ModelingNow());

                UNIT_ASSERT(billingManager.CreateBillingTask(task, USER_ID_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            billingManager.WaitBillingCycle();
        });
        auto checker = [&offerId, &sessionId](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            INFO_LOG << response << Endl;
            UNIT_ASSERT_VALUES_EQUAL(response["compiled_history"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(response["compiled_history"].GetArray()[0]["session_id"].GetString(), offerId);
            UNIT_ASSERT_VALUES_EQUAL(response["compiled_history"].GetArray()[0]["meta"].GetMap().at("real_session_id"), offerId);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["session_id"].GetString(), sessionId);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["meta"].GetMap().at("real_session_id"), offerId);
        };
        auto current_only_checker = [&offerId, &sessionId](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            INFO_LOG << response << Endl;
            UNIT_ASSERT_VALUES_EQUAL(response["compiled_history"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["session_id"].GetString(), sessionId);
            UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["meta"].GetMap().at("real_session_id"), offerId);
        };
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId).SetChecker(checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(sessionId).SetChecker(current_only_checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TDBExec>("DELETE FROM compiled_bills WHERE session_id='" + offerId + "'");
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId).SetChecker(checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TDBExec>("DELETE FROM billing_tasks_history WHERE history_action='add' AND session_id='" + offerId + "'");
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId).SetChecker(checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TDBExec>("DELETE FROM billing_tasks_history WHERE history_action='update_data' AND session_id='" + offerId + "'");
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId).SetChecker(checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TDBExec>("DELETE FROM billing_tasks_history WHERE session_id='" + offerId + "'");
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(sessionId).SetChecker(current_only_checker).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(current_only_checker).SetUserId(USER_ROOT_DEFAULT);
        script.Execute();
    }


    class TPatchHashes: public ITestAction {
        R_FIELD(TString, UserId);
        R_FIELD(TString, TestPassNumberHash);
        using TBase = ITestAction;

    public:
        using TBase::TBase;

        TPatchHashes(const TInstant startInstant, const TString& userId, const TString& hash)
            : TBase(startInstant)
            , UserId(userId)
            , TestPassNumberHash(hash)
        {
        }

    protected:
        void DoExecute(TRTContext& context) override {
            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto users = context.GetServer()->GetDriveAPI()->GetUsersData()->FetchInfo(UserId, session);
            UNIT_ASSERT_C(users, session.GetStringReport());
            auto userData = users.GetResult().begin()->second;
            userData.SetPassportNumberHash(TestPassNumberHash);
            userData.SetDrivingLicenseNumberHash(TestPassNumberHash);
            userData.SetPassportNamesHash(TestPassNumberHash);
            UNIT_ASSERT(context.GetServer()->GetDriveAPI()->GetUsersData()->UpdateUser(userData, "robot-frontend", session));
            UNIT_ASSERT(session.Commit());
        }
    };

    Y_UNIT_TEST(PaymentInfoFromDuplicates) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TSetSetting>().Set("handlers.api/staff/billing/payment/info.same_persons", "true");

        auto getChecker = [] (const TString& offer) {
            return [&offer](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                INFO_LOG << response << Endl;
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["session_id"].GetString(), offer);
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["meta"].GetMap().at("real_session_id"), offer);
            };
        };
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);
        const TString& offerId = script.Add<CreateCloseSession>().SetMoney(NotAuthorizedSum).GetOfferId();
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId).SetChecker(getChecker(offerId)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(getChecker(offerId)).SetUserId(USER_ROOT_DEFAULT);

        const TString account1 = DEFAULT_ACCOUNT + "1";
        script.Add<NDrive::NTest::TCreateAccount>(account1);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(account1).SetUserId(USER_ID_DEFAULT1);
        const TString& offerId1 = script.Add<CreateCloseSession>().SetUserId(USER_ID_DEFAULT1).SetAccount(account1).SetMoney(NotAuthorizedSum).GetOfferId();
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId1).SetChecker(getChecker(offerId1)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT1).SetChecker(getChecker(offerId1)).SetUserId(USER_ROOT_DEFAULT);

        const TString testPassNumberHash = "f3c266214dfa332bd6386c215c34c3043bb03291";
        script.Add<TPatchHashes>(USER_ID_DEFAULT1, testPassNumberHash);
        script.Add<TPatchHashes>(USER_ID_DEFAULT, testPassNumberHash);

        auto getCheckerWithDuplicates = [] (const TString& offer, const TString& offerSecond, const TString& userId) {
            return [&offer, &offerSecond, &userId](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                INFO_LOG << response << Endl;
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["session_id"].GetString(), offer);
                UNIT_ASSERT_VALUES_EQUAL(response["current"].GetArray()[0]["meta"].GetMap().at("real_session_id"), offer);

                UNIT_ASSERT(response.Has("same_persons"));
                UNIT_ASSERT_VALUES_EQUAL(response["same_persons"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(response["same_persons"].GetArray()[0]["session_id"].GetString(), offerSecond);
                UNIT_ASSERT_VALUES_EQUAL(response["same_persons"].GetArray()[0]["user_id"].GetString(), userId);
            };
        };

        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(getCheckerWithDuplicates(offerId, offerId1, USER_ID_DEFAULT1)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT1).SetChecker(getCheckerWithDuplicates(offerId1, offerId, USER_ID_DEFAULT)).SetUserId(USER_ROOT_DEFAULT);


        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(context.GetServer()->GetUserRegistrationManager()->BanUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, NBans::EReason::Auto, session));
            UNIT_ASSERT(session.Commit());
        });

        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(getCheckerWithDuplicates(offerId, offerId1, USER_ID_DEFAULT1)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT1).SetChecker(getCheckerWithDuplicates(offerId1, offerId, USER_ID_DEFAULT)).SetUserId(USER_ROOT_DEFAULT);

        const TString account2 = DEFAULT_ACCOUNT + "2";
        script.Add<NDrive::NTest::TCreateAccount>(account2);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(account2).SetUserId(USER_ID_DEFAULT2);
        const TString& offerId2 = script.Add<CreateCloseSession>().SetUserId(USER_ID_DEFAULT2).SetAccount(account2).SetMoney(NotAuthorizedSum).GetOfferId();
        script.Add<TGetBillingPaymentInfo>().SetRequestSessionId(offerId2).SetChecker(getChecker(offerId2)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT2).SetChecker(getChecker(offerId2)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session =  context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(context.GetServer()->GetUserRegistrationManager()->BanUser(USER_ID_DEFAULT2, USER_ROOT_DEFAULT, NBans::EReason::Auto, session));
            UNIT_ASSERT(session.Commit());
        });
        script.Add<TPatchHashes>(USER_ID_DEFAULT2, testPassNumberHash);

        auto checkSameUsers = [](const TSet<TString>& samePersons) {
            return [samePersons](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                INFO_LOG << response << Endl;
                UNIT_ASSERT(response.Has("same_persons"));
                UNIT_ASSERT(response["same_persons"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(response["same_persons"].GetArray().size(), samePersons.size());
                for (ui32 i = 0; i < samePersons.size(); ++i) {
                    UNIT_ASSERT(samePersons.contains(response["same_persons"].GetArray()[i]["user_id"].GetString()));
                }
            };
        };
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(checkSameUsers({USER_ID_DEFAULT1, USER_ID_DEFAULT2})).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT1).SetChecker(checkSameUsers({USER_ID_DEFAULT, USER_ID_DEFAULT2})).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT2).SetChecker(checkSameUsers({USER_ID_DEFAULT1, USER_ID_DEFAULT})).SetUserId(USER_ROOT_DEFAULT);

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto users = context.GetServer()->GetDriveAPI()->GetUsersData()->FetchInfo(USER_ID_DEFAULT, session);
            UNIT_ASSERT_C(users, session.GetStringReport());
            auto userData = users.GetResult().begin()->second;
            userData.SetStatus(NDrive::UserStatusDeleted);
            UNIT_ASSERT_C(context.GetServer()->GetDriveAPI()->GetUsersData()->UpdateUser(userData, "robot-frontend", session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        });

        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT).SetChecker(checkSameUsers({USER_ID_DEFAULT1, USER_ID_DEFAULT2})).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT1).SetChecker(checkSameUsers({USER_ID_DEFAULT2})).SetUserId(USER_ROOT_DEFAULT);
        script.Add<TGetBillingPaymentInfo>().SetRequestUserId(USER_ID_DEFAULT2).SetChecker(checkSameUsers({USER_ID_DEFAULT1})).SetUserId(USER_ROOT_DEFAULT);

        script.Execute();
    }

    class TGetDebtWithDuplicates: public TAPIAction {
        R_FIELD(TChecker, Checker);

        using TBase = TAPIAction;
        static TFactory::TRegistrator<TGetDebtWithDuplicates> Registrator;

    public:
        using TBase::TBase;

        TGetDebtWithDuplicates(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/yandex/user/debt>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: personal_debt_info" << Endl;
            ss << "    OverrideCgiPart: use_same_persons=true" << Endl;
            ss << "</api/yandex/user/debt>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue reply = SendRequest(context, "/api/yandex/user/debt");
            UNIT_ASSERT(GetExpectOK() == !reply.IsNull());
            Checker(reply, context);
        }
    };

    TAPIAction::TFactory::TRegistrator<TGetDebtWithDuplicates> TGetDebtWithDuplicates::Registrator("TGetDebtWithDuplicates");

    class TBoostUserBillingTask: public TAPIAction {
        R_FIELD(TString, PayMethod);
        R_FIELD(TString, SamePerson);

        using TBase = TAPIAction;
        static TFactory::TRegistrator<TBoostUserBillingTask> Registrator;

    public:
        using TBase::TBase;

        TBoostUserBillingTask(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/yandex/card>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: set_card" << Endl;
            ss << "</api/yandex/card>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue postData;
            postData["paymethod_id"] = PayMethod;
            postData["same_person"] = SamePerson;
            SendRequest(context, "/api/yandex/card", postData);
        }
    };

    TAPIAction::TFactory::TRegistrator<TBoostUserBillingTask> TBoostUserBillingTask::Registrator("TBoostUserBillingTask");

    Y_UNIT_TEST(BoostDuplicateSession) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<TGetDebtWithDuplicates>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            UNIT_ASSERT_C(response.Has("debt") || response["debt"].Has("amount"), response.GetStringRobust());
            UNIT_ASSERT_VALUES_EQUAL_C(response["debt"]["amount"].GetIntegerRobust(), 0, response.GetStringRobust());
        }).SetUserId(USER_ID_DEFAULT1);

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.GetEGenerator().GetBillingMock().SetReply(R"(
                {
                    "status": "success",
                    "bound_payment_methods": []
                }
            )");
        });

        TString sessionId = ICommonOffer::CreateOfferId();
        script.Add<NDrive::NTest::TCommonChecker>([&sessionId](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            {
                auto session = billingManager.BuildSession(false);
                TBillingTask task;
                task.SetId(sessionId).SetUserId(USER_ID_DEFAULT1).SetBill(20000).SetDiscretization(1000).SetBillingType(EBillingType::CarUsage);
                task.SetQueue(EBillingQueue::Tests).SetNextQueue(EBillingQueue::Tests);
                TSet<TString> accounts;
                accounts.emplace("card");
                task.SetChargableAccounts(accounts);
                task.SetLastUpdate(ModelingNow());

                UNIT_ASSERT(billingManager.CreateBillingTask(task, USER_ID_DEFAULT1, session));
                UNIT_ASSERT(billingManager.FinishingBillingTask(sessionId, session));
                UNIT_ASSERT(session.Commit());
                UNIT_ASSERT(billingManager.AddClosedBillingInfo(sessionId, USER_ROOT_DEFAULT));
            }
            billingManager.WaitBillingCycle();
            {
                auto session = billingManager.BuildSession(true);
                auto tasks = billingManager.GetActiveTasksManager().GetSessionsTasks({sessionId}, session);
                UNIT_ASSERT(tasks);
                UNIT_ASSERT(!tasks->empty());
            }
            {
                auto session = billingManager.BuildSession(true);
                auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks({USER_ID_DEFAULT1}, session);
                UNIT_ASSERT(tasks);
                UNIT_ASSERT(!tasks->empty());
            }
        });

        script.Add<TGetDebtWithDuplicates>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            UNIT_ASSERT_C(response.Has("debt") || response["debt"].Has("amount"), response.GetStringRobust());
            UNIT_ASSERT_VALUES_UNEQUAL_C(response["debt"]["amount"].GetIntegerRobust(), 0, response.GetStringRobust());
        }).SetUserId(USER_ID_DEFAULT1);
        script.Add<TGetDebtWithDuplicates>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            UNIT_ASSERT_C(response.Has("debt") || response["debt"].Has("amount"), response.GetStringRobust());
            UNIT_ASSERT_VALUES_EQUAL_C(response["debt"]["amount"].GetIntegerRobust(), 0, response.GetStringRobust());
        }).SetUserId(USER_ID_DEFAULT2);

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.GetEGenerator().GetBillingMock().SetReply(R"(
                {
                    "status": "success",
                    "bound_payment_methods": [{
                        "region_id": 225,
                        "payment_method": "card",
                        "expiration_month": "07",
                        "binding_ts": "1536324485.656",
                        "id": "card-x12345",
                        "expired": false,
                        "card_bank": "TINKOFF BANK",
                        "system": "VISA",
                        "account": "510000****0257",
                        "expiration_year": "2021"
                    },
                    {
                        "region_id": 225,
                        "payment_method": "card",
                        "expiration_month": "07",
                        "binding_ts": "1536324485.656",
                        "id": "card-x12",
                        "expired": false,
                        "card_bank": "TINKOFF BANK",
                        "system": "VISA",
                        "account": "511100****0257",
                        "expiration_year": "2021"
                    }
                ]
                }
            )");
        });

        script.Add<TBoostUserBillingTask>().SetPayMethod("card-x12345").SetSamePerson(USER_ID_DEFAULT1).SetUserId(USER_ID_DEFAULT2);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession(true);
            auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks({USER_ID_DEFAULT1}, session);
            UNIT_ASSERT(tasks);
            UNIT_ASSERT(!tasks->empty());
            const auto task = tasks->front();
            UNIT_ASSERT_VALUES_UNEQUAL(task.GetExecContext().GetUserId(), USER_ID_DEFAULT2);
            UNIT_ASSERT_VALUES_UNEQUAL(task.GetExecContext().GetPaymethod(), "card-x12345");
        });

        const TString testPassNumberHash = "f3c266214dfa332bd6386c215c34c3043bb03291";
        script.Add<TPatchHashes>(USER_ID_DEFAULT1, testPassNumberHash);
        script.Add<TPatchHashes>(USER_ID_DEFAULT2, testPassNumberHash);
        script.Add<TGetDebtWithDuplicates>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            UNIT_ASSERT_C(response.Has("debt") || response["debt"].Has("amount"), response.GetStringRobust());
            UNIT_ASSERT_VALUES_UNEQUAL_C(response["debt"]["amount"].GetIntegerRobust(), 0, response.GetStringRobust());
        }).SetUserId(USER_ID_DEFAULT1);
        script.Add<TGetDebtWithDuplicates>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
            UNIT_ASSERT_C(response.Has("debt") || response["debt"].Has("amount"), response.GetStringRobust());
            UNIT_ASSERT_VALUES_UNEQUAL_C(response["debt"]["amount"].GetIntegerRobust(), 0, response.GetStringRobust());
        }).SetUserId(USER_ID_DEFAULT2);

        script.Add<TBoostUserBillingTask>().SetPayMethod("card-x12345").SetSamePerson(USER_ID_DEFAULT1).SetUserId(USER_ID_DEFAULT2);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession(true);
            auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks({USER_ID_DEFAULT1}, session);
            UNIT_ASSERT(tasks);
            UNIT_ASSERT(!tasks->empty());
            const auto task = tasks->front();
            UNIT_ASSERT_VALUES_EQUAL(task.GetExecContext().GetUserId(), USER_ID_DEFAULT2);
            UNIT_ASSERT_VALUES_EQUAL(task.GetExecContext().GetPaymethod(), "card-x12345");
        });

        script.Execute();
    }
}
