#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/complaint_tags/car_complaint.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/digest/md5/md5.h>

Y_UNIT_TEST_SUITE(CarComplaint) {
    using namespace NDrive::NTest;

    void RegisterComplaintTag(NDrive::TServerGuard& server) {
        auto description = MakeHolder<TCarComplaintTag::TDescription>();
        description->SetType(TCarComplaintTag::TypeName);
        description->SetName("car_complaint");
        description->SetMaxSessionAge(TDuration::Days(2));
        description->SetMinComplaintWeight(4);
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(std::move(description), USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
    }

    Y_UNIT_TEST(MergeOnAdd) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        RegisterComplaintTag(server);

        auto car = eGenerator.CreateCar();
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetUserId("111").SetTopicLink("").SetSourceName("test_source").SetAddedWeight(1);
            tag->SetComplaintSources({source});
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetStatus(), TCarComplaintTag::EComplaintStatus::Created);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetProblemUserId(), "");
        }
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetUserId("222").SetTopicLink("").SetSourceName("test_source_2").SetAddedWeight(3);
            tag->SetComplaintSources({source});
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 4);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test_source_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test_source");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetStatus(), TCarComplaintTag::EComplaintStatus::ConfirmedNoUser);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetProblemUserId(), "");
        }
    }

    Y_UNIT_TEST(MergeOnUpdate) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        RegisterComplaintTag(server);

        auto car = eGenerator.CreateCar();
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetUserId("111").SetAddedWeight(1).SetSourceName("test_source");
            tag->SetComplaintSources({source});
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            TCarComplaintTag::TComplaintSource source;
            source.SetUserId("222").SetAddedWeight(5).SetSourceName("test_source_2");
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            tag->SetComplaintSources({source});
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(dbTags.front(), std::move(tag), USER_ROOT_DEFAULT, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 6);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test_source_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test_source");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetStatus(), TCarComplaintTag::EComplaintStatus::ConfirmedNoUser);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetProblemUserId(), "");
        }
    }

    Y_UNIT_TEST(FindProblemSession) {
        NDrive::TServerConfigGenerator configGenerator;
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());

        TString user;
        script.Add<TCommonChecker>([&user](TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("complaint_user", true, "active");
            if (!!userId) {
                context.SetUserId(userId);
            }
            user = userId;
            TUserRole userRole;
            userRole.SetRoleId("standart_access").SetUserId(userId);
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        });
        TGeoCoord from(37.5848674, 55.7352435);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Minutes(1));
        script.Add<TCommonChecker>([](TRTContext& context) {
            RegisterComplaintTag(context.GetServer());
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, context.GetCar().Id, context.GetServer().Get(), session) && session.Commit(), session.GetStringReport());
        });
        script.Add<TCommonChecker>([&user](TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(context.GetDriveAPI().GetTagsManager().GetDeviceTags().RestoreTags({ context.GetCar().Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetStatus(), TCarComplaintTag::EComplaintStatus::Created);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetProblemUserId(), user);
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ActionObject) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        auto complainedUser1 = eGenerator.CreateUser("complained_user_1", true, "active");
        auto complainedUser2 = eGenerator.CreateUser("complained_user_2", true, "active");
        auto problemUser = eGenerator.CreateUser("car_problem_user", true, "active");
        auto customUser = eGenerator.CreateUser("custom_user_939393", true, "active");

        auto description = MakeHolder<TCarComplaintTag::TDescription>();
        description->SetType(TCarComplaintTag::TypeName);
        description->SetName("car_complaint");
        description->SetMaxSessionAge(TDuration::Days(2));
        description->SetMinComplaintWeight(4);

        TCarComplaintTag::TAction addTagActionOnComplained;
        addTagActionOnComplained.SetActionType(TCarComplaintTag::EActionType::AddTag);
        addTagActionOnComplained.SetFromStatus(TCarComplaintTag::EComplaintStatus::Created);
        addTagActionOnComplained.SetToStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        addTagActionOnComplained.SetObject("complained_users");
        addTagActionOnComplained.SetTag("{\"tag\":\"user_problem_tag_minor\"}");
        TCarComplaintTag::TAction addTagActionOnProblem;
        addTagActionOnProblem.SetActionType(TCarComplaintTag::EActionType::AddTag);
        addTagActionOnProblem.SetFromStatus(TCarComplaintTag::EComplaintStatus::Created);
        addTagActionOnProblem.SetToStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        addTagActionOnProblem.SetObject("problem_user");
        addTagActionOnProblem.SetTag("{\"tag\":\"support_chat_dtp\"}");
        TCarComplaintTag::TAction addTagActionOnCustom;
        addTagActionOnCustom.SetActionType(TCarComplaintTag::EActionType::AddTag);
        addTagActionOnCustom.SetFromStatus(TCarComplaintTag::EComplaintStatus::Created);
        addTagActionOnCustom.SetToStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        addTagActionOnCustom.SetObject(customUser);
        addTagActionOnCustom.SetTag("{\"tag\":\"support_chat_2_line\"}");
        description->SetStatusChangedActions({ addTagActionOnComplained, addTagActionOnProblem, addTagActionOnCustom });

        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(std::move(description), USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto car = eGenerator.CreateCar();
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source1;
            source1.SetAddedWeight(3).SetSourceName("test_source").SetUserId(complainedUser1);
            TCarComplaintTag::TComplaintSource source2;
            source2.SetAddedWeight(2).SetSourceName("test_source").SetUserId(complainedUser2);

            TVector<TCarComplaintTag::TComplaintSource> sources({source1, source2});
            tag->SetComplaintSources(sources);
            tag->SetProblemUserId(problemUser);
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ complainedUser1, complainedUser2 }, { "user_problem_tag_minor" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 2);
            UNIT_ASSERT((dbTags.front().GetObjectId() == complainedUser1 && dbTags.back().GetObjectId() == complainedUser2) || (dbTags.front().GetObjectId() == complainedUser2 && dbTags.back().GetObjectId() == complainedUser1));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ complainedUser1, complainedUser2 }, { "support_chat_dtp" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ complainedUser1, complainedUser2 }, { "support_chat_2_line" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ problemUser }, { "user_problem_tag_minor" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ problemUser }, { "support_chat_dtp" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags.front().GetObjectId(), problemUser);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ problemUser }, { "support_chat_2_line" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ customUser }, { "user_problem_tag_minor" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ customUser }, { "support_chat_dtp" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ customUser }, { "support_chat_2_line" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags.front().GetObjectId(), customUser);
        }
    }

    Y_UNIT_TEST(MoveToNodeAction) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        auto complainedUser = eGenerator.CreateUser("complained_user_move", true, "active");
        auto problemUser = eGenerator.CreateUser("problem_user_move", true, "active");

        auto description = MakeHolder<TCarComplaintTag::TDescription>();
        description->SetType(TCarComplaintTag::TypeName);
        description->SetName("car_complaint");
        description->SetMaxSessionAge(TDuration::Days(2));
        description->SetMinComplaintWeight(4);

        TCarComplaintTag::TAction moveToNodeComplained;
        moveToNodeComplained.SetActionType(TCarComplaintTag::EActionType::MoveChatToNode);
        moveToNodeComplained.SetFromStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        moveToNodeComplained.SetToStatus(TCarComplaintTag::EComplaintStatus::Resolved);
        moveToNodeComplained.SetObject("complained_users");
        moveToNodeComplained.SetTopicLink("support_modern.383838383992");
        moveToNodeComplained.SetNodeId("ive_got_debt");
        moveToNodeComplained.SetSources({"test_source", "other_source"});

        TCarComplaintTag::TAction moveToNodeProblem;
        moveToNodeProblem.SetActionType(TCarComplaintTag::EActionType::MoveChatToNode);
        moveToNodeProblem.SetFromStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        moveToNodeProblem.SetToStatus(TCarComplaintTag::EComplaintStatus::Resolved);
        moveToNodeProblem.SetObject("problem_user");
        moveToNodeProblem.SetTopicLink("support_modern.383838383992");
        moveToNodeProblem.SetNodeId("ive_got_debt");
        moveToNodeProblem.SetSources({ "other_source" });
        description->SetStatusChangedActions({ moveToNodeComplained, moveToNodeProblem });


        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(std::move(description), USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto car = eGenerator.CreateCar();
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetAddedWeight(5).SetSourceName("test_source").SetUserId(complainedUser);
            tag->SetComplaintSources({ source });
            tag->SetProblemUserId(problemUser);
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }

        {
            auto chatsJson = gServer.GetChatsList(complainedUser, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
            chatsJson = gServer.GetChatsList(problemUser, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            tag->SetStatus(TCarComplaintTag::EComplaintStatus::Resolved);
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(dbTags.front(), std::move(tag), USER_ROOT_DEFAULT, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto chatsJson = gServer.GetChatsList(complainedUser, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            auto messagesJson = gServer.GetChatMessages(complainedUser, "support_modern.383838383992");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 1);
            UNIT_ASSERT(messagesJson["messages"].GetArray()[0]["text"].GetString().StartsWith("Вы можете самостоятельно погасить задолженность"));

            chatsJson = gServer.GetChatsList(problemUser, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(UpdateComplaintSource) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        RegisterComplaintTag(server);

        auto car = eGenerator.CreateCar();
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetAddedWeight(1).SetSourceName("test_source").SetUserId("111").SetConfirmed(false);
            tag->SetComplaintSources({source});
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            TCarComplaintTag::TComplaintSource source;
            source.SetUserId("222").SetAddedWeight(5).SetSourceName("test_source_2");
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            tag->SetComplaintSources({source});
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(dbTags.front(), std::move(tag), USER_ROOT_DEFAULT, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test_source_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test_source");
        }
        {
            auto tag = MakeHolder<TCarComplaintTag>("car_complaint");
            TCarComplaintTag::TComplaintSource source;
            source.SetAddedWeight(2).SetSourceName("test_source").SetUserId("111").SetConfirmed(true);
            tag->SetComplaintSources({source});
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(std::move(tag), USER_ROOT_DEFAULT, car.Id, server.Get(), session) && session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ car.Id }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test_source_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test_source");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->IsConfirmed(), true);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetAddedWeight(), 2);
        }
    }

    Y_UNIT_TEST(UploadComplaintPhoto) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        RegisterComplaintTag(server);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        {
            TBlob image = TBlob::FromString("aaaaaaaaaaaaaaaaaaaaaaaaaaaaa");
            UNIT_ASSERT(configGenerator.RegisterComplaintPhoto(USER_ID_DEFAULT, OBJECT_ID_DEFAULT, "support.123", "photo_id", MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())), "test_marker", "test", 1));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ OBJECT_ID_DEFAULT }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->IsConfirmed(), false);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetImagePaths().size(), 1);
        }

        {
            TBlob image = TBlob::FromString("bbbbbbbbbbbbbbbbbbbbbbbbbbbbb");
            UNIT_ASSERT(configGenerator.RegisterComplaintPhoto(USER_ID_DEFAULT, OBJECT_ID_DEFAULT, "support.123", "photo_id", MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())), "test_marker", "test_2", 1));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ OBJECT_ID_DEFAULT }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 2);
        }

        {
            TBlob image = TBlob::FromString("ccccccccccccccccccccccccccccc");
            UNIT_ASSERT(configGenerator.RegisterComplaintPhoto(USER_ID_DEFAULT, OBJECT_ID_DEFAULT, "support.123", "photo_id", MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())), "test_marker", "test", 2));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ OBJECT_ID_DEFAULT }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), "test");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetSourceName(), "test_2");
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 3);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetImagePaths().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[1]->GetImagePaths().size(), 1);
        }
    }

    Y_UNIT_TEST(UploadPhotoActions) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        auto complainedUserId = eGenerator.CreateUser("complained_user_move", true, "active");

        auto description = MakeHolder<TCarComplaintTag::TDescription>();
        description->SetType(TCarComplaintTag::TypeName);
        description->SetName("car_complaint");
        description->SetMaxSessionAge(TDuration::Days(2));
        description->SetMinComplaintWeight(4);

        TCarComplaintTag::TAction moveToNodeComplained;
        TString topicLink = "support_modern.383838383992";
        TString sourceName = "test_source";
        moveToNodeComplained.SetActionType(TCarComplaintTag::EActionType::MoveChatToNode);
        moveToNodeComplained.SetFromStatus(TCarComplaintTag::EComplaintStatus::None);
        moveToNodeComplained.SetToStatus(TCarComplaintTag::EComplaintStatus::Created);
        moveToNodeComplained.SetObject("complained_users");
        moveToNodeComplained.SetTopicLink(topicLink);
        moveToNodeComplained.SetNodeId("ive_got_debt");
        moveToNodeComplained.SetSources({sourceName});
        description->SetStatusChangedActions({moveToNodeComplained});

        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(std::move(description), USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            auto chatsJson = gServer.GetChatsList(complainedUserId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        {
            TBlob image = TBlob::FromString("aaaaaaaaaaaaaaaaaaaaaaaaaaaaa");
            UNIT_ASSERT(gServer.RegisterComplaintPhoto(complainedUserId, OBJECT_ID_DEFAULT, topicLink, "photo_id", MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())), "test_marker", sourceName, 1));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ OBJECT_ID_DEFAULT }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetUserId(), complainedUserId);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetSourceName(), sourceName);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->IsConfirmed(), false);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetWeight(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources()[0]->GetImagePaths().size(), 1);
        }

        {
            auto chatsJson = gServer.GetChatsList(complainedUserId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            auto messagesJson = gServer.GetChatMessages(complainedUserId, topicLink);
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 1);
            UNIT_ASSERT(messagesJson["messages"].GetArray()[0]["text"].GetString().StartsWith("Вы можете самостоятельно погасить задолженность"));
        }
    }

    Y_UNIT_TEST(LostItem) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        auto complainedUserId = eGenerator.CreateUser("complained_user_move", true, "active");

        auto description = MakeHolder<TCarComplaintTag::TDescription>();
        description->SetType(TCarComplaintTag::TypeName);
        description->SetName("car_complaint");
        description->SetMaxSessionAge(TDuration::Days(2));
        description->SetMinComplaintWeight(4);

        TString topicLink = "support_modern.383838383992";
        TString sourceName = "test_source";

        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(std::move(description), USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            TBlob image = TBlob::FromString("bbbbbbbbbbbbbbbbbbbb");
            UNIT_ASSERT(gServer.RegisterComplaintPhoto(
                complainedUserId,
                OBJECT_ID_DEFAULT,
                topicLink,
                "photo_id",
                MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())),
                "test_marker",
                sourceName,
                1,
                true));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ OBJECT_ID_DEFAULT }, { "car_complaint" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front().GetTagAs<TCarComplaintTag>();
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetComplaintSources().size(), 1);
            auto lostItemSource = dynamic_cast<TCarComplaintTag::TLostItemSource*>(tag->GetComplaintSources()[0].Get());
            UNIT_ASSERT(lostItemSource);
            UNIT_ASSERT_VALUES_EQUAL(lostItemSource->GetLostItemType(), TCarComplaintTag::ELostItemType::ForgottenItem);
            UNIT_ASSERT(lostItemSource->GetUserAvailable());
            UNIT_ASSERT_VALUES_EQUAL(lostItemSource->GetLocation(), "somewhere");
            UNIT_ASSERT_VALUES_EQUAL(lostItemSource->GetItemDescription(), "passport");
            UNIT_ASSERT_VALUES_EQUAL(lostItemSource->GetDocument(), true);
        }
    }
}
