#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/rt_background/attachments_expiration_watcher/watcher.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

#define NEW_CAR_ID_DEFAULT "c64943b5-b3a1-422d-8878-9c9ccf82c34d"

Y_UNIT_TEST_SUITE(CarPreparation) {

    Y_UNIT_TEST(CarCreationUnsuccessful) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);
        {
            auto upsertCarResponse = configGenerator.UpsertCar("{}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 400);
        }
    }

    Y_UNIT_TEST(CarCreationFullCycle) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);

        // set up default tags with redactor in advance
        {
            NJson::TJsonValue tagDescription;
            tagDescription["tag_name"] = "new_car";
            tagDescription["comment"] = "new car uploaded through admin portal";
            tagDescription["priority"] = 1000;

            NJson::TJsonValue tagsDescription = NJson::JSON_ARRAY;
            tagsDescription.AppendValue(tagDescription);

            NJson::TJsonValue payload;
            payload["default_tags"] = tagsDescription;

            UNIT_ASSERT(configGenerator.ModifyCarModel("renault_kaptur", "Renault Kaptur", "Renault", USER_ROOT_DEFAULT, payload));
            auto modelData = driveApi.GetModelsData()->FetchInfo("renault_kaptur").MutableResult().begin()->second;
            modelData.SetRegistryManufacturer("Renault");
            modelData.SetRegistryModel("Kaptur");
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetModelsData()->Upsert(modelData, session) && session.Commit());
        }

        // basic creation
        {
            // create car
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"AAAA\",\"model_code\": \"renault_kaptur\",\"number\": \"12345\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            // find this car
            auto gCarsData = driveApi.GetCarsData()->FetchInfo(NEW_CAR_ID_DEFAULT);
            auto result = gCarsData.GetResult();
            UNIT_ASSERT(!result.empty());
            UNIT_ASSERT_VALUES_EQUAL(result.begin()->second.GetVin(), "AAAA");
        }

        {
            TCarGenericAttachment attachment;
            TCarRegistryDocument currentRegistryDocument;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(NEW_CAR_ID_DEFAULT, EDocumentAttachmentType::CarRegistryDocument, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarRegistryDocument*>(attachment.Get());
                if (!!implPtr) {
                    currentRegistryDocument = *implPtr;
                }
            }

            UNIT_ASSERT_VALUES_EQUAL(currentRegistryDocument.GetManufacturer(), "Renault");
            UNIT_ASSERT_VALUES_EQUAL(currentRegistryDocument.GetModel(), "Kaptur");
        }

        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto attachments = driveApi.GetCarAttachmentAssignments().GetAttachmentOfType(NEW_CAR_ID_DEFAULT, EDocumentAttachmentType::CarRegistryDocument, tx);
            UNIT_ASSERT(attachments);
            UNIT_ASSERT_VALUES_EQUAL(attachments->size(), 1);
            auto impl = dynamic_cast<const TCarRegistryDocument*>(attachments->front().Get());
            UNIT_ASSERT(impl);
            TCarRegistryDocument currentRegistryDocument(*impl);
            UNIT_ASSERT_VALUES_EQUAL(currentRegistryDocument.GetManufacturer(), "Renault");
            UNIT_ASSERT_VALUES_EQUAL(currentRegistryDocument.GetModel(), "Kaptur");
        }

        {
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"AAAA\",\"model_code\": \"renault_kaptur\",\"number\": null}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            auto gCarsData = driveApi.GetCarsData()->FetchInfo(NEW_CAR_ID_DEFAULT);
            auto result = gCarsData.GetResult();
            UNIT_ASSERT(!result.empty());
            UNIT_ASSERT_VALUES_EQUAL(result.begin()->second.GetNumber(), "");
        }

        // check that the history has this car
        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto optionalEvents = driveApi.GetCarManager().GetHistoryManager().GetEvents(NEW_CAR_ID_DEFAULT, {}, tx);
            UNIT_ASSERT(optionalEvents);
            UNIT_ASSERT_VALUES_EQUAL(optionalEvents->size(), 2);
        }

        // change VIN-number
        {
            // ensure that the request is successful
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"BBBB\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            // find car and check vin
            auto gCarsData = driveApi.GetCarsData()->FetchInfo(NEW_CAR_ID_DEFAULT);
            auto result = gCarsData.GetResult();
            UNIT_ASSERT(!result.empty());
            UNIT_ASSERT_VALUES_EQUAL(result.begin()->second.GetVin(), "BBBB");
        }

        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto optionalEvents = driveApi.GetCarManager().GetHistoryManager().GetEvents(NEW_CAR_ID_DEFAULT, {}, tx);
            UNIT_ASSERT(optionalEvents);
            UNIT_ASSERT_VALUES_EQUAL(optionalEvents->size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(optionalEvents->at(0).GetHistoryAction(), EObjectHistoryAction::Add);
            UNIT_ASSERT_VALUES_EQUAL(optionalEvents->at(1).GetHistoryAction(), EObjectHistoryAction::UpdateData);
        }
    }

    Y_UNIT_TEST(CarModificationHistoryEmpty) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);

        TString carId = eGenerator.CreateCar().Id;
        {
            auto getHistoryResponseJSON = configGenerator.GetCarModificationHistory(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(getHistoryResponseJSON != NJson::JSON_NULL, TStringBuilder() << getHistoryResponseJSON);
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(CarModificationHistoryPresent) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);

        // make some historical events
        {
            // create car
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"BBBB\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"MODIFIED_VIN\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"registration_id\": \"987654321\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
        }

        // get history
        {
            auto getHistoryResponseJSON = configGenerator.GetCarModificationHistory(NEW_CAR_ID_DEFAULT, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(getHistoryResponseJSON != NJson::JSON_NULL, TStringBuilder() << getHistoryResponseJSON);
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray().size(), 3);

            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[0]["object"]["vin"], "BBBB");
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[1]["object"]["vin"], "MODIFIED_VIN");
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[2]["object"]["vin"], "MODIFIED_VIN");

            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[0]["object"]["registration_id"], "");
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[1]["object"]["registration_id"], "");
            UNIT_ASSERT_VALUES_EQUAL(getHistoryResponseJSON["history"].GetArray()[2]["object"]["registration_id"], "987654321");
        }
    }

    Y_UNIT_TEST(CarVINValidation) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        // incorrect length
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"q\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["recognized_model"], NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["validation_status"].GetString(), "invalid");
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().size(), 0);
        }

        // incorrect chars
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"Q94C241ABJR039580\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["recognized_model"], NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["validation_status"].GetString(), "invalid");
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray()[0], 0);
        }
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"Z_4C241ABJR039580\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["recognized_model"], NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["validation_status"].GetString(), "invalid");
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray()[0], 1);
        }

        // suspicious cases
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"Z94C241ABJR039580\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["validation_status"].GetString(), "suspicious");
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().front(), 3);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().back(), 7);
        }
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"Z94C251AB0R008830\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["validation_status"].GetString(), "suspicious");
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray()[0]["details"]["suspicious_indices"].GetArray()[0], 9);
        }

        // ok
        {
            auto response = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"z94c251ABJR008606\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            NJson::TJsonValue result;
            UNIT_ASSERT(NJson::ReadJsonFastTree(response.Content(), &result));

            UNIT_ASSERT_VALUES_EQUAL(result["errors"].GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(OwnerBySlug) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"BBBB\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

        INFO_LOG << configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "7582cd33c548018a38ba2aa1e175402e", true, USER_ROOT_DEFAULT);
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug("7582cd33c548018a38ba2aa1e175402e", Now()), "c64943b5-b3a1-422d-8878-9c9ccf82c34d");
        }
    }

    Y_UNIT_TEST(CarInsuranceAttachment) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"BBBB\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        INFO_LOG << configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "a09xx14ac550", true, USER_ROOT_DEFAULT);
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
        }

        {
            TCarInsurancePolicy* impl = new TCarInsurancePolicy(
                "a", "b", 1, 2, TInstant::Seconds(1514764800), TInstant::Seconds(1546300799)
            );
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, "c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        {
            TCarInsurancePolicy* impl = new TCarInsurancePolicy(
                "c", "d", 1, 2, TInstant::Seconds(1546300800), TInstant::Seconds(1577836800)
            );
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, "c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        driveApi.GetCarGenericAttachments().ForceUpdate(Now());
        driveApi.GetCarAttachmentAssignments().Size(Now());

        NJson::TJsonValue report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 3);

        TCarInsurancePolicy policy;
        UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy("c64943b5-b3a1-422d-8878-9c9ccf82c34d", TInstant::Seconds(1546300805), policy));
        UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "c");

        UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy("c64943b5-b3a1-422d-8878-9c9ccf82c34d", TInstant::Seconds(1514764805), policy));
        UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "a");

        UNIT_ASSERT(!driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy("c64943b5-b3a1-422d-8878-9c9ccf82c34d", TInstant::Seconds(1577836900), policy));
    }

    Y_UNIT_TEST(CarHardwareAttachmentsReport) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"c64943b5-b3a1-422d-8878-9c9ccf82c34d\",\"vin\": \"BBBB\", \"model_code\": \"renault_kaptur\"}", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        INFO_LOG << configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "a09xx14ac550", true, USER_ROOT_DEFAULT);

        NJson::TJsonValue report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);

        UNIT_ASSERT(report != NJson::JSON_NULL);

        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareBeacon)), NJson::JSON_NULL);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareModem)), NJson::JSON_NULL);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareVega)), NJson::JSON_NULL);

        UNIT_ASSERT_C(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareHead)) != NJson::JSON_NULL, TStringBuilder() << report);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareHead))["data"]["head_id"], "a09xx14ac550");
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareHead))["data"]["device_id"], NJson::JSON_NULL);

        auto result = configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "ZXZXZXZX", false, USER_ROOT_DEFAULT);
        INFO_LOG << result << Endl;
        UNIT_ASSERT(result != NJson::JSON_NULL);

        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareBeacon)), NJson::JSON_NULL);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareModem)), NJson::JSON_NULL);
        UNIT_ASSERT_VALUES_EQUAL(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareVega)), NJson::JSON_NULL);
        UNIT_ASSERT_C(TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareHead)) != NJson::JSON_NULL, TStringBuilder() << TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", "head"));

        report = configGenerator.ServiceAppAssignmentHistory("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        INFO_LOG << report << Endl;
        UNIT_ASSERT_VALUES_EQUAL(report["history"].GetArray().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(report["attachments"].GetArray().size(), 2);

        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        INFO_LOG << "Attached hardware list size: " << report.GetArray().size() << Endl;
        TString id = report.GetArray()[0]["id"].GetStringRobust();
        INFO_LOG << "We will detach the attachment with id=" << id << Endl;

        UNIT_ASSERT(configGenerator.DetachAttachment(id, USER_ROOT_DEFAULT));

        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 0);

        result = configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "6362875000007078256", false, USER_ROOT_DEFAULT);
        INFO_LOG << result << Endl;
        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        INFO_LOG << report << Endl;
        UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 1);

        {
            TCarInsurancePolicy* impl = new TCarInsurancePolicy("a", "b", 1, 2);
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, "c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            impl->SetOsagoNumber("123");

            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, "c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "ABCDEF", false, USER_ROOT_DEFAULT);

        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT);
        INFO_LOG << report << Endl;
        UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 3);

        report = configGenerator.ListAttachedHardware("c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, true);
        INFO_LOG << report << Endl;
        UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 4);

        configGenerator.ServiceAppAssignDevice("c64943b5-b3a1-422d-8878-9c9ccf82c34d", "866710035999999", false, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo("c64943b5-b3a1-422d-8878-9c9ccf82c34d").begin()->second.GetIMEI(), "866710035999999");

        // Check city field availability to be upserted and built in report
        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            NJson::TJsonValue registryData(NJson::JSON_MAP);
            registryData.InsertValue("city", "Moscow");
            UNIT_ASSERT(impl->PatchWithRawRegistryData(registryData));
            TCarGenericAttachment attachment(impl);

            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, "c64943b5-b3a1-422d-8878-9c9ccf82c34d", USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());

            auto session3 = driveApi.BuildTx<NSQL::ReadOnly>();
            auto attachments = driveApi.GetCarAttachmentAssignments().GetActiveAttachments("c64943b5-b3a1-422d-8878-9c9ccf82c34d", EDocumentAttachmentType::CarRegistryDocument, session3);
            UNIT_ASSERT(attachments);
            UNIT_ASSERT_VALUES_EQUAL(attachments->size(), 1);

            const auto implPtr = dynamic_cast<const TCarRegistryDocument*>(attachments->at(0).Get());
            UNIT_ASSERT(implPtr);
            UNIT_ASSERT_VALUES_EQUAL(implPtr->GetCity(), "Moscow");

            auto context = TCarAttachmentReportContext(&driveApi);
            context.SetRegistryDocumentTraits(NRegistryDocumentReport::ReportCity);
            auto report = implPtr->BuildReport(context);
            UNIT_ASSERT_VALUES_EQUAL(report["city"].GetStringSafe(), "Moscow");

            context.SetRegistryDocumentTraits(NRegistryDocumentReport::ReportBasicInfo);
            report = implPtr->BuildReport(context);
            UNIT_ASSERT(!report.Has("city"));
        }
    }

    Y_UNIT_TEST(CarUpsertFromServiceApp) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        // check that existing hardware is found all the time
        TCarGenericAttachment attachment;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(!driveApi.GetCarGenericAttachments().UpsertFromServiceApp("ZXZXZXZX", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertFromServiceApp("866710035764478", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_VALUES_EQUAL(attachment.GetId(), "1686124a-772b-4aba-b332-4bd06e92a329");
        }

        // check various hardware parsing cases
        TVector<std::pair<TString, EDocumentAttachmentType>> additionItems = {
            std::make_pair("2120244E00055", EDocumentAttachmentType::CarSignalDevice),
            std::make_pair("514042B401E89,355001098043986,", EDocumentAttachmentType::CarSignalDevice),
            std::make_pair("QR_34f68r1a", EDocumentAttachmentType::CarQrCode),
            std::make_pair("866710035999999", EDocumentAttachmentType::CarHardwareVega),
            std::make_pair("MT-32K LTE;866766666699990", EDocumentAttachmentType::CarHardwareVega),
            std::make_pair("a09xxxxxx551", EDocumentAttachmentType::CarHardwareHead),
            std::make_pair("SN008521,869696000052864", EDocumentAttachmentType::CarHardwareBeacon),
            std::make_pair("89701026947689999", EDocumentAttachmentType::CarHardwareModem),
            std::make_pair("101007786184200192", EDocumentAttachmentType::CarHardwareHeadSerialNumber),
            std::make_pair("DVM201802080929", EDocumentAttachmentType::CarHardwareHeadSerialNumber),
            std::make_pair("7582cd33c548018a38ba2aa1e175402e", EDocumentAttachmentType::CarHardwareHeadNew),
            std::make_pair("6362875000001111111", EDocumentAttachmentType::CarTransponderSpb),
            std::make_pair("7005830009874744", EDocumentAttachmentType::CarFuelCard),
            std::make_pair("7826010104866351", EDocumentAttachmentType::CarFuelCard),
            std::make_pair("7013420002084945", EDocumentAttachmentType::CarFuelCard),
            std::make_pair("7824861090007231801", EDocumentAttachmentType::CarFuelCard)
        };

        for (auto item : additionItems) {
            size_t oldSize = driveApi.GetCarGenericAttachments().Size(Now());
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT_C(driveApi.GetCarGenericAttachments().UpsertFromServiceApp(item.first, attachment, USER_ROOT_DEFAULT, session, item.second), session.GetStringReport());
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
                UNIT_ASSERT_C(!!attachment.GetId(), TStringBuilder() << attachment.GetId());
                UNIT_ASSERT_VALUES_EQUAL(attachment.GetType(), item.second);
            }

            {
                size_t newSize = driveApi.GetCarGenericAttachments().Size(Now());
                UNIT_ASSERT_VALUES_EQUAL(newSize, oldSize + 1);
            }
        }
    }

    Y_UNIT_TEST(CarSimAttachmentConflict) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TCarGenericAttachment attachment;

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto carId = eGenerator.CreateCar().Id;

        // Register some Vega
        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(carId, "MT-32K LTE;866710035999990;89701026947689999;89701026167689999", true, USER_ROOT_DEFAULT).Has("error_details"));
        }

        // Fail to add modem with same SIM card: it's in Vega already
        {
            UNIT_ASSERT(configGenerator.ServiceAppAssignDevice(carId, "89701026947689999", true, USER_ROOT_DEFAULT).Has("error_details"));
        }

        // Now detach that SIM card from conflicting Vega
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertFromServiceApp("MT-32K LTE;866710035999990", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());

            auto vegaImpl = std::dynamic_pointer_cast<TCarHardwareVega>(attachment.GetImpl());
            UNIT_ASSERT(vegaImpl);
            vegaImpl->SetPrimarySim(nullptr);

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session2));
            UNIT_ASSERT(session2.Commit());
        }

        // And now the modem is created
        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(carId, "89701026947689999", true, USER_ROOT_DEFAULT).Has("error_details"));
        }

        //
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertFromServiceApp("101007786184200192", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        //
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertFromServiceApp("DVM201802080929", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        //
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().UpsertFromServiceApp("7582cd33c548018a38ba2aa1e175402e", attachment, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(CarHeadAttachmentConflict) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TString head1 = "a09dc14azzz1";
        TString head2 = "a09dc14azzz2";

        TVector<TString> cars;
        for (size_t i = 0; i < 3; ++i) {
            TString id = "c64943b5-b3a1-422d-8878-9c9ccf80000" + TString((char)(i + '0'));
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"" + id + "\",\"vin\": \"AAAA\",\"model_code\": \"renault_kaptur\",\"number\": \"12345\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            cars.push_back(id);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NJson::TJsonValue response = configGenerator.ServiceAppAssignDevice(cars[0], head1, false, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(head1, Now()), cars[0]);

        response = configGenerator.ServiceAppAssignDevice(cars[0], head2, false, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(head1, Now()), "");
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(head2, Now()), cars[0]);

        response = configGenerator.ServiceAppAssignDevice(cars[1], head1, false, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(head1, Now()), cars[1]);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(head2, Now()), cars[0]);

        response = configGenerator.ServiceAppAssignDevice(cars[2], head1, false, USER_ROOT_DEFAULT);
        UNIT_ASSERT(response.Has("error_details"));
    }

    Y_UNIT_TEST(CarMassiveHardwareAssignment) {
        srand(1995);
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        /*
            First of all, create a pool of devices, which will then be assigned to different cars.
        */

        TVector<TString> deviceTypes = {"vega", "beacon", "modem", "head", "malformed"};
        TMap<TString, TVector<TString>> devicePool;
        for (size_t i = 0; i < 10; ++i) {
            devicePool["vega"].push_back("86671003599999" + TString((char)(i + '0')));
            devicePool["beacon"].push_back("SN008521,86969604879998" + TString((char)(i + '0')));
            devicePool["modem"].push_back("8970102694768997" + TString((char)(i + '0')));
            devicePool["head"].push_back("a09dc14azzz" + TString((char)(i + '0')));
            devicePool["malformed"].push_back("ZXZXZXZXZ" + TString((char)(i + 'a')));
        }

        /*
            Then, create a pool of blank cars, to which these devices will be assigned.
        */

        TVector<TString> carsPool;
        for (size_t i = 0; i < 10; ++i) {
            TString id = "c64943b5-ffff-ffff-ffff-9c9ccf80000" + TString((char)(i + '0'));
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"" + id + "\",\"vin\": \"AAAA\",\"model_code\": \"renault_kaptur\",\"number\": \"12345\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
            carsPool.push_back(id);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        /*
            Now, pick a random device and assign it to a random car 1000 times.
        */

        TMap<TString, TString> deviceOwner;
        TMap<std::pair<TString, TString>, TString> lastDeviceOfType;
        for (size_t it = 0; it < 100; ++it) {
            TString deviceType = deviceTypes[rand() % deviceTypes.size()];
            TString deviceCode = devicePool[deviceType][rand() % devicePool[deviceType].size()];
            TString carId = carsPool[rand() % carsPool.size()];
            bool isForce = rand() % 2;

            ERROR_LOG << "Iteration:" << it << " " << deviceType << " " << deviceCode << " " << carId << " " << isForce << Endl;

            NJson::TJsonValue response = configGenerator.ServiceAppAssignDevice(carId, deviceCode, isForce, USER_ROOT_DEFAULT);
            if (deviceType == "malformed") {
                UNIT_ASSERT_VALUES_EQUAL(response["error_details"]["ui_message"], NDrive::TLocalization::UnableToUpsertDevice());
            } else if (isForce) {
                if (deviceOwner[deviceCode] == carId) {
                    UNIT_ASSERT_C(response.Has("error_details"), TStringBuilder() << response);
                } else {
                    UNIT_ASSERT_C(!response.Has("error_details"), TStringBuilder() << response);

                    if (deviceOwner.contains(deviceCode)) {
                        auto key = std::make_pair(deviceOwner[deviceCode], deviceType);
                        if (lastDeviceOfType.contains(key)) {
                            INFO_LOG << lastDeviceOfType[key] << " " << deviceOwner[lastDeviceOfType[key]] << Endl;
                            deviceOwner.erase(deviceOwner.find(lastDeviceOfType[key]));
                            lastDeviceOfType.erase(lastDeviceOfType.find(key));
                        }
                    }

                }
                auto key = std::make_pair(carId, deviceType);
                if (lastDeviceOfType.contains(key)) {
                    INFO_LOG << lastDeviceOfType[key] << " " << deviceOwner[lastDeviceOfType[key]] << Endl;
                    deviceOwner.erase(deviceOwner.find(lastDeviceOfType[key]));
                    lastDeviceOfType.erase(lastDeviceOfType.find(key));
                }
                deviceOwner[deviceCode] = carId;
                lastDeviceOfType[key] = deviceCode;
            } else {
                if (!deviceOwner.contains(deviceCode)) {
                    UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
                    auto key = std::make_pair(carId, deviceType);
                    if (lastDeviceOfType.contains(key)) {
                        INFO_LOG << lastDeviceOfType[key] << " " << deviceOwner[lastDeviceOfType[key]] << Endl;
                        deviceOwner.erase(deviceOwner.find(lastDeviceOfType[key]));
                        lastDeviceOfType.erase(lastDeviceOfType.find(key));
                    }
                    deviceOwner[deviceCode] = carId;
                    lastDeviceOfType[key] = deviceCode;
                } else if (deviceOwner[deviceCode] == carId) {
                    UNIT_ASSERT_VALUES_EQUAL(response["error_details"]["ui_message"], NDrive::TLocalization::DeviceAlreadyAttached());
                } else if (deviceOwner[deviceCode] != carId) {
                    UNIT_ASSERT_VALUES_EQUAL(response["error_details"]["ui_message"], NDrive::TLocalization::DeviceAttachedToDifferentCar());
                }
            }
        }

        for (auto car : carsPool) {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(car, USER_ROOT_DEFAULT);
            for (auto& head : devicePool["head"]) {
                if (deviceOwner[head] == car) {
                    UNIT_ASSERT_VALUES_EQUAL(head, TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareHead))["data"]["head_id"].GetStringRobust());
                }
            }
            for (auto& vega : devicePool["vega"]) {
                if (deviceOwner[vega] == car) {
                    UNIT_ASSERT_VALUES_EQUAL(vega, TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareVega))["data"]["imei"].GetStringRobust());
                }
            }
            for (auto& modem : devicePool["modem"]) {
                if (deviceOwner[modem] == car) {
                    UNIT_ASSERT_VALUES_EQUAL(modem, TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareModem))["data"]["sim"]["icc"].GetStringRobust());
                }
            }
            for (auto& beacon : devicePool["beacon"]) {
                if (deviceOwner[beacon] == car) {
                    auto tokens = SplitString(beacon, ",");
                    UNIT_ASSERT_VALUES_EQUAL(tokens[0], TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareBeacon))["data"]["serial_number"].GetStringRobust());
                    UNIT_ASSERT_VALUES_EQUAL(tokens[1], TValidationHelpers::GetFlatJsonItem(report.GetArray(), "type", CarAttachmentType(EDocumentAttachmentType::CarHardwareBeacon))["data"]["imei"].GetStringRobust());
                }
            }
        }
    }

    Y_UNIT_TEST(CarVegaAttachedTwice) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        {
            TCarHardwareSim* impl = new TCarHardwareSim(TCarHardwareSim::NormalizeICC("897010269200757803"), "+79179999999");
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);

            TCarHardwareSim* impl2 = new TCarHardwareSim(TCarHardwareSim::NormalizeICC("8970199171227765084"), "+79178888888");
            TCarGenericAttachment attachment2(impl2);
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment2, USER_ROOT_DEFAULT, session);

            UNIT_ASSERT(session.Commit());
        }

        auto c1 = eGenerator.CreateCar();
        auto c2 = eGenerator.CreateCar();

        NJson::TJsonValue response = configGenerator.ServiceAppAssignDevice(c1.Id, "MT-32K LTE;866766666699990;897010269200757803;8970199171227765084", true, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo(c1.Id).begin()->second.GetIMEI(), "866766666699990");
        UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");

        auto attachedVega = configGenerator.ListAttachedHardware(c1.Id, USER_ROOT_DEFAULT).GetArray()[0];
        UNIT_ASSERT_VALUES_EQUAL(attachedVega["data"]["primary_sim"]["phone_number"].GetString(), "+79179999999");
        UNIT_ASSERT_VALUES_EQUAL(attachedVega["data"]["secondary_sim"]["phone_number"].GetString(), "+79178888888");

        response = configGenerator.ServiceAppAssignDevice(c2.Id, "MT-32K LTE;866766666699990", false, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo(c1.Id).begin()->second.GetIMEI(), "866766666699990");
        UNIT_ASSERT(driveApi.GetCarsData()->FetchInfo(c2.Id).begin()->second.GetIMEI() != "866766666699990");
        UNIT_ASSERT(response["status"] != "success");
        response = configGenerator.ServiceAppAssignDevice(c2.Id, "MT-32K LTE;866766666699990", true, USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo(c2.Id).begin()->second.GetIMEI(), "866766666699990");
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo(c1.Id).begin()->second.GetIMEI(), "");
        UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
    }

    Y_UNIT_TEST(DeviceAssignmentByServiceAppSlug) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const THeadAccountManager& headManager = driveApi.GetHeadAccountManager();

        TEnvironmentGenerator eGenerator(*server.Get());
        auto car = eGenerator.CreateCar();

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(headManager.FinishSession(car.Id, session));
            UNIT_ASSERT(headManager.UpdateCarHeadId(car.Id, HEAD_ID_DEFAULT, USER_ID_DEFAULT, session, server.Get()));
            UNIT_ASSERT(session.Commit());
        }

        UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().GetCarByHeadId(HEAD_ID_DEFAULT, Now()) == car.Id);
    }

    Y_UNIT_TEST(CarsBatchUpload) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        size_t carDocumentsBeforeCreation = driveApi.GetCarGenericAttachments().Size(Now());
        TString id1, id2;

        // set up default tags with redactor in advance
        {
            NJson::TJsonValue tagDescription;
            tagDescription["tag_name"] = "simple1";
            tagDescription["comment"] = "new car uploaded through admin portal";
            tagDescription["priority"] = 1000;

            NJson::TJsonValue tagsDescription = NJson::JSON_ARRAY;
            tagsDescription.AppendValue(tagDescription);

            NJson::TJsonValue payload;
            payload["default_tags"] = tagsDescription;
            payload["registry_manufacturer"] = "Renault";
            payload["registry_model"] = "Kaptur";
            UNIT_ASSERT(configGenerator.ModifyCarModel("renault_kaptur", "Renault Kaptur", "Renault", USER_ROOT_DEFAULT, payload));

            TDriveModelData model = driveApi.GetModelsData()->FetchInfo("renault_kaptur").GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(model.GetRegistryManufacturer(), "Renault");
            UNIT_ASSERT_VALUES_EQUAL(model.GetRegistryModel(), "Kaptur");

            payload["registry_manufacturer"] = "KIA";
            payload["registry_model"] = "Rio";
            UNIT_ASSERT(configGenerator.ModifyCarModel("kia_rio", "KIA Rio", "KIA", USER_ROOT_DEFAULT, payload));

            model = driveApi.GetModelsData()->FetchInfo("kia_rio").GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(model.GetRegistryManufacturer(), "KIA");
            UNIT_ASSERT_VALUES_EQUAL(model.GetRegistryModel(), "Rio");
        }

        {
            NJson::TJsonValue requestPayload;
            NJson::TJsonValue requestCarsData = NJson::JSON_ARRAY;

            NJson::TJsonValue car1;
            car1["VIN"] = "vin1";
            car1["Марка"] = "Renault";
            car1["Модель"] = "Kaptur";

            NJson::TJsonValue car2;
            car2["VIN"] = "vin2";
            car2["Марка"] = "KIA";
            car2["Модель"] = "Rio";
            car2["ГРЗ"] = "zx123qfd";
            car2["СТС"] = "123";

            requestCarsData.AppendValue(car1);
            requestCarsData.AppendValue(car2);

            requestPayload.InsertValue("cars", std::move(requestCarsData));

            UNIT_ASSERT(configGenerator.BatchUploadCars(requestPayload, USER_ROOT_DEFAULT));

            size_t timesMatched = 0;
            auto allCars = driveApi.GetCarsData()->FetchInfo().GetResult();
            for (auto&& entry : allCars) {
                if (entry.second.GetVin() == "VIN1" || entry.second.GetVin() == "VIN2") {
                    ++timesMatched;
                }
                if (entry.second.GetVin() == "VIN1") {
                    id1 = entry.second.GetId();
                }
                if (entry.second.GetVin() == "VIN2") {
                    id2 = entry.second.GetId();
                    UNIT_ASSERT_VALUES_EQUAL(entry.second.GetNumber(), "zx123qfd");
                    UNIT_ASSERT_VALUES_EQUAL(entry.second.GetRegistrationID(), 123);
                }
            }

            UNIT_ASSERT_VALUES_EQUAL(timesMatched, 2);

            size_t carDocumentsAfterCreation = driveApi.GetCarGenericAttachments().Size(Now());
            UNIT_ASSERT_VALUES_EQUAL(carDocumentsAfterCreation, carDocumentsBeforeCreation + 2);

            {
                auto carDocuments = configGenerator.ListAttachedHardware(id2, USER_ROOT_DEFAULT, true);
                for (auto&& doc : carDocuments.GetArray()) {
                    ERROR_LOG << "document attached: " << doc["type"].GetString() << Endl;
                }
                UNIT_ASSERT_VALUES_EQUAL(carDocuments.GetArray().size(), 1);
            }
        }

        {
            NJson::TJsonValue requestPayload;
            NJson::TJsonValue requestCarsData = NJson::JSON_ARRAY;

            NJson::TJsonValue car1;
            car1["vin"] = "VIN1";
            car1["ГРЗ"] = "a123BC";
            car1["СТС"] = "1234567890";

            NJson::TJsonValue car2;
            car2["vin"] = "VIN2";
            car2["ГРЗ"] = "zx123qff";

            NJson::TJsonValue car3;
            car3["VIN"] = "VIN2";
            car3["Марка"] = "KIA";
            car3["Модель"] = "Rio";
            car3["ГРЗ"] = "zx123qff";
            car3["СТС"] = "123";

            requestCarsData.AppendValue(car1);
            requestCarsData.AppendValue(car2);
            requestCarsData.AppendValue(car3);

            requestPayload.InsertValue("cars", std::move(requestCarsData));

            UNIT_ASSERT(configGenerator.BatchUploadCars(requestPayload, USER_ROOT_DEFAULT));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));

            {
                auto carDocuments = configGenerator.ListAttachedHardware(id2, USER_ROOT_DEFAULT, true);
                for (auto&& doc : carDocuments.GetArray()) {
                    ERROR_LOG << "document attached: " << doc["type"].GetString() << Endl;
                }
                UNIT_ASSERT_VALUES_EQUAL(carDocuments.GetArray().size(), 1);
            }

            auto currentDocuments = driveApi.GetCarGenericAttachments().Size(Now());
            UNIT_ASSERT_VALUES_EQUAL(currentDocuments, carDocumentsBeforeCreation + 4);

            TDriveCarInfo car = driveApi.GetCarsData()->FetchInfo(id1).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(car.GetNumber(), "a123bc");
            UNIT_ASSERT_VALUES_EQUAL(car.GetRegistrationID(), 1234567890);

            car = driveApi.GetCarsData()->FetchInfo(id2).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(car.GetNumber(), "zx123qff");
            UNIT_ASSERT_VALUES_EQUAL(car.GetRegistrationID(), 123);
        }

        {
            bool hasInitialTag = false;
            NJson::TJsonValue report = configGenerator.ListTags(id1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
            NJson::TJsonValue::TArray arr;
            report["records"].GetArray(&arr);
            for (auto&& i : arr) {
                if (i["tag"].GetString() == "simple1") {
                    hasInitialTag = true;
                }
            }
            UNIT_ASSERT(hasInitialTag);
        }

        {
            bool hasInitialTag = false;
            NJson::TJsonValue report = configGenerator.ListTags(id2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
            NJson::TJsonValue::TArray arr;
            report["records"].GetArray(&arr);
            for (auto&& i : arr) {
                if (i["tag"].GetString() == "simple1") {
                    hasInitialTag = true;
                }
            }
            UNIT_ASSERT(hasInitialTag);
        }

        {
            auto upsertCarResponse = configGenerator.UpsertCar("{\"id\": \"" + id1 + "\",\"number\": \"x567yz799\"}", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(upsertCarResponse.Code(), 200);
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

            TCarGenericAttachment attachment;
            TCarRegistryDocument currentRegistryDocument;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(id1, EDocumentAttachmentType::CarRegistryDocument, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarRegistryDocument*>(attachment.Get());
                if (!!implPtr) {
                    currentRegistryDocument = *implPtr;
                }
            }

            UNIT_ASSERT_VALUES_EQUAL(currentRegistryDocument.GetNumber(), "x567yz799");
        }
    }

    Y_UNIT_TEST(AttachmentExpirationWatcher) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        auto carId = eGenerator.CreateCar().Id;
        auto carId2 = eGenerator.CreateCar().Id;

        ERROR_LOG << "carId = " << carId << Endl;

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description;
            description.SetName("expiration_tag").SetType("simple_car_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session) && session.Commit());
            Sleep(TDuration::Seconds(10));
        }

        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            impl->SetRegistrationDate(Now() - TDuration::Days(30 * 12));
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            impl->SetRegistrationId(123456789);
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId2, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        TAttachmentWatcherConfig watcherConfig;
        watcherConfig.SetTagName("expiration_tag");
        watcherConfig.SetTagComment("something expires soon");
        watcherConfig.SetAttachmentType(EDocumentAttachmentType::CarRegistryDocument);
        watcherConfig.SetInstantFieldName("registration_date");
        watcherConfig.AddTriggerInterval(-1000000000000000000LL, -30 * 11 * 86400);

        TSet<TString> allCarIds;
        {
            auto carsFetchResult = driveApi.GetCarsData()->FetchInfo();
            for (auto&& it : carsFetchResult) {
                allCarIds.insert(it.first);
            }
        }

        TSimpleAttachmentWatcher watcher(*server.Get(), watcherConfig);
        watcher.ActualizeTags(allCarIds);

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {"expiration_tag"}, dbTags, session));
            UNIT_ASSERT_C(dbTags.size() > 0, TStringBuilder() << dbTags.size());
            bool hasCarId = false;
            bool hasCarId2 = false;
            for (auto&& tag : dbTags) {
                if (tag.GetObjectId() == carId) {
                    hasCarId = true;
                }
                if (tag.GetObjectId() == carId2) {
                    hasCarId2 = true;
                }
            }
            UNIT_ASSERT(hasCarId);
            UNIT_ASSERT(!hasCarId2);
        }

        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            impl->SetRegistrationDate(Now() - TDuration::Days(30 * 6));
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        watcher.ActualizeTags(allCarIds);

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {"expiration_tag"}, dbTags, session));
            bool hasCarId = false;
            bool hasCarId2 = false;
            for (auto&& tag : dbTags) {
                if (tag.GetObjectId() == carId) {
                    hasCarId = true;
                }
                if (tag.GetObjectId() == carId2) {
                    hasCarId2 = true;
                }
            }
            UNIT_ASSERT(!hasCarId);
            UNIT_ASSERT(!hasCarId2);
        }

        // create another watcher and test that backwards variant also works
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description;
            description.SetName("expiration_tag_two").SetType("simple_car_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session) && session.Commit());
            Sleep(TDuration::Seconds(10));
        }

        TAttachmentWatcherConfig watcherConfig2;
        watcherConfig2.SetTagName("expiration_tag_two");
        watcherConfig2.SetTagComment("something expires soon");
        watcherConfig2.SetAttachmentType(EDocumentAttachmentType::CarRegistryDocument);
        watcherConfig2.SetInstantFieldName("registration_date");
        watcherConfig2.AddTriggerInterval(0, 30 * 86400);

        {
            TCarRegistryDocument* impl = new TCarRegistryDocument();
            impl->SetRegistrationDate(Now() + TDuration::Days(15));
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }

        TSimpleAttachmentWatcher watcher2(*server.Get(), watcherConfig2);
        watcher2.ActualizeTags(allCarIds);

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {"expiration_tag_two"}, dbTags, session));
            UNIT_ASSERT_C(dbTags.size() > 0, TStringBuilder() << dbTags.size());
            bool hasCarId = false;
            bool hasCarId2 = false;
            for (auto&& tag : dbTags) {
                if (tag.GetObjectId() == carId) {
                    hasCarId = true;
                }
                if (tag.GetObjectId() == carId2) {
                    hasCarId2 = true;
                }
            }
            UNIT_ASSERT(hasCarId);
            UNIT_ASSERT(!hasCarId2);
        }
    }

    Y_UNIT_TEST(AbsenceTagMaintenance) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        auto carId = eGenerator.CreateCar().Id;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description;
            description.SetName("need_vega").SetType("simple_car_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session) && session.Commit());
            Sleep(TDuration::Seconds(10));
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag("need_vega", "set automatically");
            UNIT_ASSERT(!!tag);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, "robot-frontend", carId, server.Get(), session) && session.Commit());
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"need_vega"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
        {
            NJson::TJsonValue response = configGenerator.ServiceAppAssignDevice(carId, "MT-32K LTE;866766666699990;897010269200757803;8970199171227765084", true, USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(driveApi.GetCarsData()->FetchInfo(carId).begin()->second.GetIMEI(), "866766666699990");
            UNIT_ASSERT_VALUES_EQUAL(response["status"], "success");
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"need_vega"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
        {
            auto report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            INFO_LOG << "Attached hardware list size: " << report.GetArray().size() << Endl;
            TString id = report.GetArray()[0]["id"].GetStringRobust();
            INFO_LOG << "We will detach the attachment with id=" << id << Endl;
            UNIT_ASSERT(configGenerator.DetachAttachment(id, USER_ROOT_DEFAULT));
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"need_vega"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(AbsenceTagMaintenanceMultipleDevices) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        auto carId = eGenerator.CreateCar().Id;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description;
            description.SetName("need_insurance").SetType("simple_car_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session) && session.Commit());
            Sleep(TDuration::Seconds(10));
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag("need_insurance", "set automatically");
            UNIT_ASSERT(!!tag);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, "robot-frontend", carId, server.Get(), session) && session.Commit());
        }
        {
            TCarInsurancePolicy* impl = new TCarInsurancePolicy(
                "a", "b", 1, 2, TInstant::Seconds(1514764800), TInstant::Seconds(1546300799)
            );
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }
        {
            TCarInsurancePolicy* impl = new TCarInsurancePolicy(
                "c", "d", 1, 2, TInstant::Seconds(1546300800), TInstant::Seconds(1577836800)
            );
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }
        TString id1, id2;
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 2);
            id1 = report.GetArray()[0]["id"].GetString();
            id2 = report.GetArray()[1]["id"].GetString();
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"need_insurance"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
        UNIT_ASSERT(configGenerator.DetachAttachment(id1, USER_ROOT_DEFAULT));
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({carId}, {"need_insurance"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
        UNIT_ASSERT(configGenerator.DetachAttachment(id2, USER_ROOT_DEFAULT));
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"need_insurance"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(PresenceTagMaintenance) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        auto carId = eGenerator.CreateCar().Id;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description;
            description.SetName("transponder").SetType("additional_feature_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session) && session.Commit());
            Sleep(TDuration::Seconds(10));
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"transponder"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
        {
            INFO_LOG << configGenerator.ServiceAppAssignDevice(carId, "6362875000001111111", true, USER_ROOT_DEFAULT) << Endl;
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"transponder"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 1);
            auto id = report.GetArray()[0]["id"].GetString();
            UNIT_ASSERT(configGenerator.DetachAttachment(id, USER_ROOT_DEFAULT));
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"transponder"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
    }

    Y_UNIT_TEST(InsuranceBatchUploadSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TEnvironmentGenerator::TCar car;
        TEnvironmentGenerator::TCar car2;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD");
            car2 = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD2");
            UNIT_ASSERT(session.Commit());
        }

        NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;

        // the per-minute prices are chosen on purpose, don't change them
        {
            NJson::TJsonValue insurance;
            insurance["AgreementPartnerNum "] = "qwe";
            insurance["AgreementNum"] = "rty";
            insurance["Дата начала страхования"] = "43318";
            insurance["Дата окончания страхования"] = "43682";
            insurance["Поминутное"] = "0.58";  // int(0.58 * 100) = 57
            insurance["VIN"] = car.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }
        {
            NJson::TJsonValue insurance;
            insurance["AgreementPartnerNum "] = "asd";
            insurance["AgreementNum"] = "fgh";
            insurance["Дата начала страхования"] = "43318";
            insurance["Дата окончания страхования"] = "43682";
            insurance["Поминутное"] = "0.60";  // int(0.60 * 100) = 60
            insurance["VIN"] = car2.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }

        NJson::TJsonValue payload;
        payload["cars"] = std::move(insuranceArr);
        payload["type"] = "renins";
        payload["base_cost"] = 666;

        UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));

        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Renins);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "qwe");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementNumber(), "rty");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetPerMinuteCost(), 58);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetBaseCost(), 666);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetValidFrom().Seconds(), 1533502800);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetValidUntil().Seconds(), 1565038799);

            INFO_LOG << policy.BuildReport(TCarAttachmentReportImplContext()) << Endl;
        }
        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car2.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Renins);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "asd");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementNumber(), "fgh");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetPerMinuteCost(), 60);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetBaseCost(), 666);
        }

        insuranceArr = NJson::JSON_ARRAY;
        {
            NJson::TJsonValue insurance;
            insurance["AgreementPartnerNum "] = "qweeee";
            insurance["AgreementNum"] = "rtyyyy";
            insurance["Дата начала страхования"] = "43318";
            insurance["Дата окончания страхования"] = "43682";
            insurance["Поминутное"] = "0.77";
            insurance["VIN"] = car.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }
        payload = NJson::JSON_MAP;
        payload["cars"] = std::move(insuranceArr);
        payload["type"] = "renins";

        UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Renins);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "qweeee");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementNumber(), "rtyyyy");
            UNIT_ASSERT_VALUES_EQUAL(policy.GetPerMinuteCost(), 77);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetBaseCost(), 35);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetValidFrom().Seconds(), 1533502800);
            UNIT_ASSERT_VALUES_EQUAL(policy.GetValidUntil().Seconds(), 1565038799);
        }
    }

    Y_UNIT_TEST(InsuranceBatchUploadFake) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TEnvironmentGenerator::TCar car;
        TEnvironmentGenerator::TCar car2;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD");
            car2 = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD2");
            UNIT_ASSERT(session.Commit());
        }

        NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;

        {
            NJson::TJsonValue insurance;
            insurance["VIN"] = car.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }
        {
            NJson::TJsonValue insurance;
            insurance["VIN"] = car2.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }

        NJson::TJsonValue payload;
        payload["cars"] = std::move(insuranceArr);
        payload["type"] = "fake";

        UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Fake);
        }
        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car2.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Fake);
        }
    }

    Y_UNIT_TEST(InsuranceBatchUploadIngos) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TEnvironmentGenerator::TCar car;
        TEnvironmentGenerator::TCar car2;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD");
            car2 = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD2");
            UNIT_ASSERT(session.Commit());
        }

        NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;

        {
            NJson::TJsonValue insurance;
            insurance["AgreementPartnerNum"] = "qwe";
            insurance["AgreementNum"] = "rty";
            insurance["Дата начала страхования"] = "43318";
            insurance["Дата окончания страхования"] = "43682";
            insurance["Поминутное"] = "0,58";
            insurance["VIN"] = car.Vin;
            insuranceArr.AppendValue(std::move(insurance));
        }

        NJson::TJsonValue payload;
        payload["cars"] = std::move(insuranceArr);
        payload["type"] = "ingos";

        UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        {
            TCarGenericAttachment attachment;
            TCarInsurancePolicy policy;
            if (driveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(car.Id, EDocumentAttachmentType::CarInsurancePolicy, attachment, Now())) {
                auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachment.Get());
                UNIT_ASSERT(implPtr);
                policy = *implPtr;
            } else {
                UNIT_ASSERT_C(false, TStringBuilder() << "can't acquire uploaded insurance policy");
            }

            UNIT_ASSERT_VALUES_EQUAL(policy.GetProvider(), NDrive::EInsuranceProvider::Ingos);
        }
    }

    Y_UNIT_TEST(InsuranceBatchUploadDrop) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD");
            UNIT_ASSERT(session.Commit());
        }

        {
            NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;
            {
                NJson::TJsonValue insurance;
                insurance["AgreementPartnerNum "] = "qweeee";
                insurance["AgreementNum"] = "rtyyyy";
                insurance["Дата начала страхования"] = "43318";
                insurance["Дата окончания страхования"] = "43682";
                insurance["Поминутное"] = "0.77";
                insurance["VIN"] = car.Vin;
                insuranceArr.AppendValue(std::move(insurance));
            }

            NJson::TJsonValue payload;
            payload["cars"] = std::move(insuranceArr);
            payload["type"] = "renins";

            UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        }

        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto attachments = driveApi.GetCarAttachmentAssignments().GetActiveAttachments(car.Id, EDocumentAttachmentType::CarInsurancePolicy, tx);
            UNIT_ASSERT(attachments);
            UNIT_ASSERT_VALUES_EQUAL(attachments->size(), 1);
            auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachments->at(0).Get());
            UNIT_ASSERT(implPtr);
            UNIT_ASSERT_VALUES_EQUAL(implPtr->GetProvider(), NDrive::EInsuranceProvider::Renins);
        }

        {
            NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;
            {
                NJson::TJsonValue insurance;
                insurance["VIN"] = car.Vin;
                insuranceArr.AppendValue(std::move(insurance));
            }
            NJson::TJsonValue payload;
            payload["cars"] = std::move(insuranceArr);
            payload["type"] = "fake";

            UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        }

        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto attachments = driveApi.GetCarAttachmentAssignments().GetActiveAttachments(car.Id, EDocumentAttachmentType::CarInsurancePolicy, tx);
            UNIT_ASSERT(attachments);
            UNIT_ASSERT_VALUES_EQUAL(attachments->size(), 1);
            auto implPtr = dynamic_cast<const TCarInsurancePolicy*>(attachments->at(0).Get());
            UNIT_ASSERT(implPtr);
            UNIT_ASSERT_VALUES_EQUAL(implPtr->GetProvider(), NDrive::EInsuranceProvider::Fake);
        }
    }

    Y_UNIT_TEST(InsuranceBatchUploadActual) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD");
            UNIT_ASSERT(session.Commit());
        }

        {
            NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;
            {
                NJson::TJsonValue insurance;
                insurance["AgreementPartnerNum "] = "qweeee";
                insurance["AgreementNum"] = "rtyyyy";
                insurance["Дата начала страхования"] = "43318";
                insurance["Дата окончания страхования"] = "43682";
                insurance["Поминутное"] = "0.77";
                insurance["VIN"] = car.Vin;
                insuranceArr.AppendValue(std::move(insurance));
            }

            NJson::TJsonValue payload;
            payload["cars"] = std::move(insuranceArr);
            payload["type"] = "renins";

            UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        }

        {
            NJson::TJsonValue insuranceArr = NJson::JSON_ARRAY;
            {
                NJson::TJsonValue insurance;
                insurance["AgreementPartnerNum "] = "qwe";
                insurance["AgreementNum"] = "rty";
                insurance["Дата начала страхования"] = "43683";
                insurance["Дата окончания страхования"] = "44047";
                insurance["Поминутное"] = "0.77";
                insurance["VIN"] = car.Vin;
                insuranceArr.AppendValue(std::move(insurance));
            }

            NJson::TJsonValue payload;
            payload["cars"] = std::move(insuranceArr);
            payload["type"] = "renins";

            UNIT_ASSERT(configGenerator.BatchUploadInsurance(payload, USER_ROOT_DEFAULT));
        }

        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto attachments = driveApi.GetCarAttachmentAssignments().GetActiveAttachments(car.Id, EDocumentAttachmentType::CarInsurancePolicy, tx);
            UNIT_ASSERT(attachments);
            UNIT_ASSERT_VALUES_EQUAL(attachments->size(), 2);
        }

        {
            driveApi.GetCarGenericAttachments().ForceUpdate(Now());
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Size(Now()));
            TCarInsurancePolicy policy;
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy(car.Id, TInstant::Seconds(1543536000), policy));
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "qweeee");

            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy(car.Id, TInstant::Seconds(1575072000), policy));
            UNIT_ASSERT_VALUES_EQUAL(policy.GetAgreementPartnerNumber(), "qwe");

            UNIT_ASSERT(!driveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy(car.Id, TInstant::Seconds(1606694400), policy));
        }
    }

    Y_UNIT_TEST(CarDeviceRepeatedAttachment) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TString carId = eGenerator.CreateCar().Id;
        TString carId2 = eGenerator.CreateCar().Id;
        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(carId, "MT-32K LTE V2;867962043174286;897010269627041096;8970199151041501246", true, USER_ROOT_DEFAULT).Has("error_details"));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));
        }

        TString assignmentId;
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 1);
            assignmentId = report.GetArray()[0]["id"].GetString();
        }

        {
            UNIT_ASSERT(configGenerator.DetachAttachment(assignmentId, USER_ROOT_DEFAULT));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));
        }

        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 0);
        }

        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(carId2, "MT-32K LTE V2;867962043174287;897010269627041096;8970199151041541246", true, USER_ROOT_DEFAULT).Has("error_details"));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));
        }

        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId2, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(CarDeviceRepeatedForbiddenAttachment) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TString carId = eGenerator.CreateCar().Id;
        TString carId2 = eGenerator.CreateCar().Id;
        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(carId, "MT-32K LTE V2;867962043174286;897010269627041096;8970199151041501246", true, USER_ROOT_DEFAULT).Has("error_details"));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));
        }

        TString assignmentId;
        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 1);
            assignmentId = report.GetArray()[0]["id"].GetString();
        }

        {
            UNIT_ASSERT(configGenerator.ServiceAppAssignDevice(carId2, "MT-32K LTE V2;867962043174287;897010269627041096;8970199151041541246", true, USER_ROOT_DEFAULT).Has("error_details"));
            UNIT_ASSERT(driveApi.GetCarGenericAttachments().RefreshCache(Now()));
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().RefreshCache(Now()));
        }

        {
            NJson::TJsonValue report = configGenerator.ListAttachedHardware(carId2, USER_ROOT_DEFAULT);
            UNIT_ASSERT(report != NJson::JSON_NULL);
            UNIT_ASSERT_VALUES_EQUAL(report.GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(CarSearchAfterVega) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        // Add user with default roles
        TString userId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");

        // Add car
        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "renault_kaptur", "", "XW8ZZZ61ZKG919376");
            UNIT_ASSERT(session.Commit());
        }

        // Refresh cache in order not to wait for 1s auto-refresh period
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
        UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));

        // Search for the car by VIN. Should be visible
        {
            auto searchResult = configGenerator.DoSearch(
                TVector<TString>({car.Vin}),
                TVector<TString>(),
                TVector<TString>({"cars"}),
                20,
                userId
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result.GetStringRobust());
        }

        // Attach vega to the car.
        {
            UNIT_ASSERT(!configGenerator.ServiceAppAssignDevice(car.Id, "MT-32K LTE V2;867962043174287;897010269627041096;8970199151041541246", true, USER_ROOT_DEFAULT).Has("error_details"));
            UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
        }

        // Search this car by VIN again. Should still be visible.
        {
            auto searchResult = configGenerator.DoSearch(
                TVector<TString>({car.Vin}),
                TVector<TString>(),
                TVector<TString>({"cars"}),
                20,
                userId
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result.GetStringRobust());
        }

        // Search by IMEI. Also available.
        {
            auto searchResult = configGenerator.DoSearch(
                TVector<TString>({"867962043174287"}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result.GetStringRobust());
        }
    }

    Y_UNIT_TEST(CarSpecifications) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto newSpecName = "my_fancy_spec";
        auto newSpecValue = "hello there";

        auto carList = env->Request(USER_ROOT_DEFAULT, "/service_app/car/list?model_traits=ReportModelSpecifications&car_id=" + OBJECT_ID_DEFAULT);
        auto views = carList["views"];
        UNIT_ASSERT(views.IsArray());
        UNIT_ASSERT_VALUES_EQUAL(views.GetArray().size(), 1);
        auto modelSpecifications = views[0]["model_specifications"];
        const auto sizeBefore = modelSpecifications.GetArray().size();
        {
            auto car = carList["cars"][0];
            UNIT_ASSERT(car.IsMap());
            UNIT_ASSERT_VALUES_EQUAL(car["id"].GetStringRobust(), OBJECT_ID_DEFAULT);
            car["model_specifications"].AppendValue(NJson::TMapBuilder
                ("name", newSpecName)
                ("value", newSpecValue)
            );

            auto upsertReply = env->UpsertCar(car.GetStringRobust(), USER_ROOT_DEFAULT);
            UNIT_ASSERT(upsertReply.IsSuccessReply());
        }
        carList = env->Request(USER_ROOT_DEFAULT, "/service_app/car/list?model_traits=ReportModelSpecifications&car_id=" + OBJECT_ID_DEFAULT);
        views = carList["views"];
        UNIT_ASSERT(views.IsArray());
        UNIT_ASSERT_VALUES_EQUAL(views.GetArray().size(), 1);
        modelSpecifications = views[0]["model_specifications"];
        const auto sizeAfter = modelSpecifications.GetArray().size();
        UNIT_ASSERT_VALUES_EQUAL(sizeBefore + 1, sizeAfter);

        auto modelSpecificationStat = carList["model_specifications"];
        UNIT_ASSERT(modelSpecificationStat.IsArray());
        bool found = false;
        for (auto&& spec : modelSpecificationStat.GetArray()) {
            auto name = spec["name"].GetStringSafe();
            if (name == newSpecName) {
                auto values = spec["values"].GetArray();
                UNIT_ASSERT_VALUES_EQUAL(values.size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(values[0]["value"].GetStringSafe(), newSpecValue);
                found = true;
                break;
            }
        }
        UNIT_ASSERT(found);
    }

    Y_UNIT_TEST(CheckSignalqLocation) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server.Get());

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "renault_kaptur", "", "XW8ZZZ61ZKG919376");
            UNIT_ASSERT(session.Commit());
        }
        TString carId = car.Id;

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            TCarSignalDevice* impl = new TCarSignalDevice("514042B401E89", "355001098043986");
            TCarGenericAttachment attachment(impl);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());
        }
        Sleep(TDuration::Seconds(2));

        const auto& snapshotsManager = server->GetSnapshotsManager();
        const auto snapshot = snapshotsManager.GetSnapshot(carId);
        auto location = snapshot.GetLocation();
        auto l = NDrive::TLocation(32.8333, 44.3333, 123.22, 12, NDrive::TLocation::GPSCurrent, TInstant::Seconds(1646920800));
        UNIT_ASSERT(snapshot.IsGettingSignalqLocation());
        UNIT_ASSERT_VALUES_EQUAL(location->ToJson(), l.ToJson());
        const auto& speed = snapshot.GetSpeed();
        UNIT_ASSERT(speed);
        UNIT_ASSERT_DOUBLES_EQUAL(*speed, 53.12, 0.01);

        {
            TTelematicServerBuilder tmBuilder;
            tmBuilder.Run();
            auto emulator = tmBuilder.BuildEmulator(driveApi.GetIMEI(carId));
            UNIT_ASSERT(tmBuilder.SetSensorDoubleValue(*emulator, VEGA_SPEED, 200.41));
            emulator->GetContext().TrySetSpeed(11.33);
            UNIT_ASSERT(configGenerator.WaitSensor(carId, "speed", "200.41"));
            Sleep(TDuration::Seconds(2));

            auto snapshot2 = snapshotsManager.GetSnapshot(carId);
            const auto l2 = NDrive::TLocation(11.8333, 22.3333, 44.22, 0, NDrive::TLocation::GPSCurrent, TInstant::Zero());
            UNIT_ASSERT(snapshot2.UpdateLinkedLocation(l2, TInstant::Max()));

            const auto& speed2 = snapshot2.GetSpeed();
            UNIT_ASSERT(speed2);
            UNIT_ASSERT_DOUBLES_EQUAL(*speed2, 200.41, 0.01);
        }
    }

    Y_UNIT_TEST(CheckSignalqTagsAttachments) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server.Get());
        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "renault_kaptur", "", "XW8ZZZ61ZKG919376");
            UNIT_ASSERT(session.Commit());
        }
        TString carId = car.Id;

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(driveApi.GetIMEI(carId));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = driveApi.BuildTx<NSQL::ReadOnly>();
            TDBTags dbTags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreTags({carId}, {"leasing_has_signalq_tag"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        // Check signalq tag is set after attaching
        TCarSignalDevice* impl = new TCarSignalDevice("514042B401E89", "355001098043986");
        TCarGenericAttachment attachment(impl);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            driveApi.GetCarGenericAttachments().UpsertAttachment(attachment, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());

            auto session2 = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Attach(attachment, carId, USER_ROOT_DEFAULT, session2, server.Get()));
            UNIT_ASSERT(session2.Commit());

            auto session3 = driveApi.BuildTx<NSQL::ReadOnly>();
            TDBTags dbTags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreTags({carId}, {"leasing_has_signalq_tag"}, dbTags, session3));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        driveApi.GetCarGenericAttachments().ForceUpdate(Now());
        driveApi.GetCarAttachmentAssignments().Size(Now());

        // Check signalq tag is removed after detaching
        {
            auto session4 = driveApi.BuildTx<NSQL::Writable>();
            auto optionalAssignment = driveApi.GetCarAttachmentAssignments().GetActiveAttachmentAssignment(attachment.GetId(), session4);
            UNIT_ASSERT(optionalAssignment);
            const auto assignment = *optionalAssignment;
            UNIT_ASSERT(assignment);

            UNIT_ASSERT(driveApi.GetCarAttachmentAssignments().Detach(assignment.GetId(), USER_ROOT_DEFAULT, session4, server.Get()));
            UNIT_ASSERT(session4.Commit());

            auto session5 = driveApi.BuildTx<NSQL::ReadOnly>();
            TDBTags dbTags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreTags({carId}, {"leasing_has_signalq_tag"}, dbTags, session5));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
    }
}
