#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/registrar/hasher.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(ChatLandings) {

    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "test_introscreen", "min_cost_zones"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "test_introscreen");
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["id"].GetString(), "test_introscreen");
        }

        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 0);
        }
    }

    Y_UNIT_TEST(Auditory) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        NJson::TJsonValue hasTagConditionJson;
        hasTagConditionJson["type"] = "has_tag";
        hasTagConditionJson["tag_name"] = "user_registered_manually";

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "test_introscreen", "min_cost_zones", hasTagConditionJson));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 0);
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        UNIT_ASSERT(configGenerator.AddTag(new TRegistrationUserTag("user_registered_manually"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(LandingOnSetting) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        NJson::TJsonValue landingConditionJson;
        landingConditionJson["type"] = "dictionary_tag_value";
        landingConditionJson["field"] = "home_work_specified";
        landingConditionJson["policy"] = "equals";
        landingConditionJson["tag_name"] = "user_app_flags";
        landingConditionJson["value"] = "0";

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "test_introscreen", "min_cost_zones", landingConditionJson));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 0);
        }
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        auto userId2 = eGenerator.CreateUser("skulik-was-there", true, "active");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto& userTagsManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();

            TDBTag settingsTag = server->GetDriveAPI()->GetUserAppFlags(userId2, session);
            UNIT_ASSERT_C(!!settingsTag.HasData(), TStringBuilder() << "incorrect settings configuration");

            TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
            UNIT_ASSERT_C(!!settingsData, TStringBuilder() << "incorrect settings configuration");
            settingsData->SetField("home_work_specified", "0");

            UNIT_ASSERT(userTagsManager.AddTag(settingsTag.GetData(), userId2, userId2, server.Get(), session) && session.Commit());
        }

        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId2, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
        }
        {
            auto chatsJson = configGenerator.GetChatsList(userId2, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(StaticAnnouncement) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        {
            auto messages = configGenerator.GetChatMessages(userId, "static_announcements.nda_referral_program");
            UNIT_ASSERT_VALUES_EQUAL(messages["messages"].GetArray().size(), 3);
        }
        {
            auto messages = configGenerator.GetChatMessages(userId, "static_announcements.test_lsmding");
            UNIT_ASSERT_VALUES_EQUAL(messages["messages"].GetArray().size(), 4);
        }

        {
            auto staticChatRobot = server->GetChatRobotsManager()->GetChatRobot("static_announcements");
            auto searchId = staticChatRobot->GetChatSearchId(userId, "");
            NDrive::NChat::TChat chat;
            UNIT_ASSERT(!server->GetChatEngine()->GetChats().GetChatFromTable(searchId, chat));
        }
        {
            auto staticChatRobot = server->GetChatRobotsManager()->GetChatRobot("static_announcements");
            auto searchId = staticChatRobot->GetChatSearchId(userId, "nda_referral_program");
            NDrive::NChat::TChat chat;
            UNIT_ASSERT(!server->GetChatEngine()->GetChats().GetChatFromTable(searchId, chat));
        }
        {
            auto staticChatRobot = server->GetChatRobotsManager()->GetChatRobot("static_announcements");
            auto searchId = staticChatRobot->GetChatSearchId(userId, "test_lsmding");
            NDrive::NChat::TChat chat;
            UNIT_ASSERT(!server->GetChatEngine()->GetChats().GetChatFromTable(searchId, chat));
        }
    }

    Y_UNIT_TEST(StaticAnnouncementOrder) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("stories_users_123", true, "active");

        TInstant deadline = Now() + TDuration::Days(1);
        TInstant created = Now() - TDuration::Days(1);
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "nda_referral_program", NJson::JSON_NULL, true, deadline, created - TDuration::Days(1)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", NJson::JSON_NULL, true, deadline, created - TDuration::Days(2)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone",NJson::JSON_NULL,  true, deadline, created - TDuration::Days(3)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing4", "{'text':'landing4'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "test_lsmding", NJson::JSON_NULL, true, deadline, created - TDuration::Days(4)));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("landing").SetUserId(userId);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(configGenerator.AcceptLandings({ "landing1" }, userId));
        UNIT_ASSERT(configGenerator.AcceptLandings({ "landing2" }, userId));
        UNIT_ASSERT(configGenerator.AcceptLandings({ "landing3" }, userId));
        UNIT_ASSERT(configGenerator.AcceptLandings({ "landing4" }, userId));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("user_landings");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 4);
            auto chats = chatsJson["chats"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(chats[0]["id"], "static_announcements.nda_referral_program");
            UNIT_ASSERT_VALUES_EQUAL(chats[1]["id"], "static_announcements.home_work");
            UNIT_ASSERT_VALUES_EQUAL(chats[2]["id"], "static_announcements.leaf_ttk_allow_drop_zone");
            UNIT_ASSERT_VALUES_EQUAL(chats[3]["id"], "static_announcements.test_lsmding");
        }
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "nda_referral_program", NJson::JSON_NULL, true, deadline, TInstant::Now() - TDuration::Days(4)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", NJson::JSON_NULL, true, deadline, TInstant::Now() - TDuration::Days(3)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone", NJson::JSON_NULL, true, deadline, TInstant::Now() - TDuration::Days(2)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing4", "{'text':'landing4'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "test_lsmding", NJson::JSON_NULL, true, deadline, TInstant::Now() - TDuration::Days(1)));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 4);
            auto chats = chatsJson["chats"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(chats[0]["id"], "static_announcements.test_lsmding");
            UNIT_ASSERT_VALUES_EQUAL(chats[1]["id"], "static_announcements.leaf_ttk_allow_drop_zone");
            UNIT_ASSERT_VALUES_EQUAL(chats[2]["id"], "static_announcements.home_work");
            UNIT_ASSERT_VALUES_EQUAL(chats[3]["id"], "static_announcements.nda_referral_program");
        }
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", NJson::JSON_NULL, true, TInstant::Now() - TDuration::Days(2)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone", NJson::JSON_NULL, true, TInstant::Now() - TDuration::Days(2)));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            auto chats = chatsJson["chats"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(chats[0]["id"], "static_announcements.test_lsmding");
            UNIT_ASSERT_VALUES_EQUAL(chats[1]["id"], "static_announcements.nda_referral_program");
        }
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "nda_referral_program", NJson::JSON_NULL, false, TInstant::Now() - TDuration::Days(2)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", NJson::JSON_NULL, false, TInstant::Now() + TDuration::Days(2)));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone", NJson::JSON_NULL, true, TInstant::Now() + TDuration::Days(2)));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            auto chats = chatsJson["chats"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(chats[0]["id"], "static_announcements.leaf_ttk_allow_drop_zone");
            UNIT_ASSERT_VALUES_EQUAL(chats[1]["id"], "static_announcements.test_lsmding");
        }
    }

    Y_UNIT_TEST(MultipleLandingsPriority) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        TInstant deadline = Now() + TDuration::Days(1);
        TInstant created = Now() - TDuration::Days(1);
        auto userId = eGenerator.CreateUser("multiple_landings_test9872", true, "active");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("landing").SetUserId(userId);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "nda_referral_program", NJson::JSON_NULL, true, deadline, created - TDuration::Days(1), -100));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", NJson::JSON_NULL, true, deadline, created - TDuration::Days(2), 20));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone", NJson::JSON_NULL, true, deadline, created - TDuration::Days(3), -30));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.nda_referral_program");
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.leaf_ttk_allow_drop_zone");
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.home_work");
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc);
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 0);
        }
    }

    Y_UNIT_TEST(LandingHeadersFilter) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        TInstant deadline = Now() + TDuration::Days(1);
        auto userId = eGenerator.CreateUser("multiple_landings_test9872", true, "active");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("landing").SetUserId(userId);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        NJson::TJsonValue androidConditionJson;
        androidConditionJson["type"] = "platform";
        androidConditionJson["name"] = "android_client";
        NJson::TJsonValue businessConditionJson;
        businessConditionJson["type"] = "platform";
        businessConditionJson["name"] = "business";

        UNIT_ASSERT(configGenerator.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "nda_referral_program", androidConditionJson, true, deadline, {}, -80, true));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing2", "{'text':'landing2'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "home_work", businessConditionJson, true, deadline, {}, -100, true));
        UNIT_ASSERT(configGenerator.ModifyLanding("landing3", "{'text':'landing3'}", true, USER_ROOT_DEFAULT, {}, "static_announcements", "leaf_ttk_allow_drop_zone", NJson::JSON_NULL, true, deadline, {}, 0, true));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc, "AC_AppBuild");
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.nda_referral_program");
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc, "AC_AppBuild");
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.leaf_ttk_allow_drop_zone");
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc, "AC_AppBuild");
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 0);
        }
        {
            NJson::TJsonValue landings = configGenerator.GetChatLandings(userId, &gc, "BC_AppBuild");
            UNIT_ASSERT(landings.Has("chat_landings"));
            UNIT_ASSERT(landings["chat_landings"].IsArray());
            auto landingsArray = landings["chat_landings"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(landingsArray.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(landingsArray[0]["id"], "static_announcements.home_work");
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("user_landings");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get(), {}, "BC_AppBuild");
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            auto chats = chatsJson["chats"].GetArray();
            TSet<TString> chatIds { chats[0]["id"].GetString(), chats[1]["id"].GetString() };
            UNIT_ASSERT(chatIds.contains("static_announcements.leaf_ttk_allow_drop_zone"));
            UNIT_ASSERT(chatIds.contains("static_announcements.home_work"));
        }
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get(), {}, "AC_AppBuild");
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            auto chats = chatsJson["chats"].GetArray();
            TSet<TString> chatIds { chats[0]["id"].GetString(), chats[1]["id"].GetString() };
            UNIT_ASSERT(chatIds.contains("static_announcements.leaf_ttk_allow_drop_zone"));
            UNIT_ASSERT(chatIds.contains("static_announcements.nda_referral_program"));
        }
    }
}
