#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/users/user_documents_check.h>
#include <drive/backend/experiments/context.h>
#include <drive/backend/data/dictionary_tags.h>

#include <rtline/util/types/uuid.h>

Y_UNIT_TEST_SUITE(ChatMath) {
    static int checkCounter = 0;
    void MessageChecker(const TString& option, const TString& userId, NDrive::TServerConfigGenerator& configGenerator) {
        checkCounter++;
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_math_checks", option, {}));
        auto messagesJson = configGenerator.GetChatMessages(userId, "support_math_checks");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2 * checkCounter);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2 * checkCounter - 1]["text"].GetString(), "Success");
    };

    Y_UNIT_TEST(Basics) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("Pythagoras", false, "active");

        checkCounter = 0;
        MessageChecker("add_two_numbers", userId, configGenerator);
        MessageChecker("order", userId, configGenerator);
        MessageChecker("counter", userId, configGenerator);
    }

    Y_UNIT_TEST(Advance) {
       NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("Newton", false, "active");

        checkCounter = 0;
        MessageChecker("integral", userId, configGenerator);
        MessageChecker("series", userId, configGenerator);
    }

    Y_UNIT_TEST(SkipActionHistory) {
       NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("Cauchy", false, "active");

        const auto& chatRobotManger = *Yensured(server->GetChatRobotsManager());
        const auto& robotStateStorage = *Yensured(chatRobotManger.GetChatRobotStateStorage());
        const auto& stateHistoryManager = robotStateStorage.GetHistoryManager();

        auto GetEvents = [&]() {
            NSQL::TQueryOptions options;
            options.AddGenericCondition("talk_id", userId + "-support_math_checks");
            options.SetOrderBy({"history_event_id"});
            options.SetDescending(true);
            options.SetLimit(10);
            auto session = Yensured(server->GetChatEngine())->BuildSession(true);
            auto optionalEvents = stateHistoryManager.GetEvents({}, {}, session, options);
            UNIT_ASSERT(optionalEvents && !optionalEvents->empty());
            return *optionalEvents;
        };

        checkCounter = 0;
        MessageChecker("add_two_numbers", userId, configGenerator);
        ui64 initialEventId = GetEvents().front().GetHistoryEventId();

        MessageChecker("counter", userId, configGenerator);
        ui64 finalEventId = GetEvents().front().GetHistoryEventId();
        UNIT_ASSERT(finalEventId - initialEventId == 96);

        auto events = GetEvents();
        UNIT_ASSERT_EQUAL(events[0].GetCurrentStep(), "intro");
        UNIT_ASSERT_EQUAL(events[1].GetCurrentStep(), "success");
        UNIT_ASSERT_EQUAL(events[2].GetCurrentStep(), "count");
        UNIT_ASSERT_EQUAL(events[3].GetCurrentStep(), "repeat_count");
        UNIT_ASSERT_EQUAL(events[4].GetCurrentStep(), "count");
        UNIT_ASSERT_EQUAL(events[5].GetCurrentStep(), "repeat_count");
    }
}
