#include <drive/backend/ut/library/helper.h>

#include <drive/library/cpp/datasync_queue/client.h>
#include <drive/backend/rt_background/datasync_manager/manager.h>
#include <drive/library/cpp/yang/client.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <rtline/util/types/uuid.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(DatasyncQueue) {

    bool EmptyQueue(const TDriveAPI& driveApi) {
        const auto queueClient = driveApi.GetDatasyncQueueClient();
        auto session = queueClient->BuildSession();
        while (auto assignment = queueClient->Get(1, session)) {
            if (!queueClient->Delete(assignment.begin()->first, session)) {
                return false;
            }
        }
        if (!session.Commit()) {
            return false;
        }
        return true;
    }

    TDriveUserData GetUser(const TString& userId, const TDriveAPI& DriveApi) {
        auto session = DriveApi.BuildTx<NSQL::ReadOnly>();
        auto fetchUserResult = DriveApi.GetUsersData()->FetchInfo(userId, session);
        UNIT_ASSERT(fetchUserResult && fetchUserResult.size() == 1);

        return fetchUserResult.begin()->second;
    }

    void SetGvars(const NDrive::TServerGuard& server) {
        {
            NJson::TJsonValue settings;
            settings["OK?"][0] = "OK";
            settings["OK?"][1] = "NEED_INFO";
            settings["OK?"][2] = "DISCARDED";

            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        }

        {
            NJson::TJsonValue photoChecksSettings = NJson::JSON_ARRAY;
            {
                NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                verdicts[0] = "OK";
                NJson::TJsonValue pair;
                pair["OK"] = std::move(verdicts);
                photoChecksSettings.AppendValue(std::move(pair));

            }
            {
                NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                verdicts[0] = "NEED_INFO";
                NJson::TJsonValue pair;
                pair["NEED_INFO"] = std::move(verdicts);
                photoChecksSettings.AppendValue(std::move(pair));

            }
            {
                NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                verdicts[0] = "DISCARDED";
                NJson::TJsonValue pair;
                pair["DISCARDED"] = std::move(verdicts);
                photoChecksSettings.AppendValue(std::move(pair));

            }
            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.photo_settings", photoChecksSettings.GetStringRobust(), USER_ROOT_DEFAULT));
        }

        {
            NJson::TJsonValue genericChecks;
            genericChecks["OK?"] = {"passport_biographical"};
            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.generic_checks", genericChecks.GetStringRobust(), USER_ROOT_DEFAULT));
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
    }

    const TString FALLBACK_TAG_NAME = "simple_user_tag";

    void ConfigureFallbackPolicy(NJson::TJsonValue& config) {
        config["fallback_type"] = "add_tag";
        config["fallback_tag_name"] = FALLBACK_TAG_NAME;
        config["remove_from_queue"] = true;
        config["attempt_limit"] = 3;
    }

    TRTBackgroundProcessContainer BuildRobot(bool configureFallback = false) {
        TAtomicSharedPtr<TRTDatasyncManager> actor(new TRTDatasyncManager);

        NJson::TJsonValue robotConfig;
        robotConfig["batch_size"] = 5;
        robotConfig["request_timeout"] = "10s";
        robotConfig["checks_settings"] = "user_documents_checks.settings";
        robotConfig["generic_checks"] = "user_documents_checks.generic_checks";
        robotConfig["checks_priority"] = "user_documents_checks.photo_settings";
        robotConfig["pb_path"] = "solutions.[0].output_values.documents.passport_biographical";
        robotConfig["pr_path"] = "solutions.[0].output_values.documents.passport_registration";
        robotConfig["lb_path"] = "solutions.[0].output_values.documents.license_back";
        robotConfig["lf_path"] = "solutions.[0].output_values.documents.license_front";
        robotConfig["user_id_path"] = "tasks.[0].input_values.user_id";
        robotConfig["secret_id_path"] = "tasks.[0].input_values.secret";
        robotConfig["verdicts_path"] = "solutions.[0].output_values.verdicts";
        robotConfig["comment_path"] = "solutions.[0].output_values.other";
        if (configureFallback) {
            ConfigureFallbackPolicy(robotConfig);
        }
        UNIT_ASSERT(actor->DeserializeFromJson(robotConfig));
        actor->SetPeriod(TDuration::Seconds(0.5));
        actor->SetEnabled(true);

        TRTBackgroundProcessContainer container(actor);
        container.SetName("yang_to_datasync_queue_manager");

        return container;
    }

    TString PrepareDocuments(TString& userId, const NDrive::TServerGuard& server, TString* photoId = nullptr) {
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        userId = eGenerator.CreateUser("steap-was-here", false);
        Sleep(TDuration::Seconds(2));

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin()->second;
            user.SetPassportDatasyncRevision("aa");
            user.SetDrivingLicenseDatasyncRevision("aa");
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(user, "robot-frontend", session) && session.Commit());
            driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin();
        }

        if (photoId) {
            *photoId = passportBiographicalId;
        }

        auto allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();

        TString assignmentId;
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                assignmentId = it.second.GetId();
            }
        }

        return assignmentId;
    }

    Y_UNIT_TEST(Dispatcher) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        SetGvars(server);
        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);

        auto passport = TDocumentsHelper::CreateRandomPassportData();

        NJson::TJsonValue payload;
        payload["passport_biographical"] = passport.SerializeBioToYang(false);
        TString testComment = "'\"тест ТЕСТ test\"'\\";
        payload["verification_statuses"]["comment"] = testComment;
        payload["passport_biographical"]["data"]["number"] = "HB2352296";
        payload["verification_statuses"]["passport_biographical_status"] = "OK";

        UNIT_ASSERT(configGenerator.PostYangAssignmentData(assignmentId, assignmentId, payload, false, true));

        {
            const auto queueClient = driveApi.GetDatasyncQueueClient();
            auto session = queueClient->BuildSession(true);
            auto assignmentEntries = queueClient->Get(5, session);
            UNIT_ASSERT(assignmentEntries.size() == 1);
            UNIT_ASSERT(assignmentEntries.contains(assignmentId));
        }
    };

    Y_UNIT_TEST(ManagerPop) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        SetGvars(server);
        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);

        auto robot = BuildRobot();

        NJson::TJsonValue assignment;
        {
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            assignment["status"] = "ACCEPTED";
            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";

            NJson::TJsonValue passport;
            passport["first_name"] = "боб";
            passport["middle_name"] = "том";
            passport["last_name"] = "вест";
            passport["gender"] = "MALE";
            passport["citizenship"] = "US";
            passport["birth_place"] = "гор. сан-хосе";
            passport["birth_date"] = "1864-09-20";
            passport["number"] = "81919991991";
            passport["issue_date"] = "2020-11-11";
            passport["subdivision_code"] = "000-000";

            assignment["solutions"][0]["output_values"]["documents"]["passport_biographical"] = passport;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto user = GetUser(userId, driveApi);
            UNIT_ASSERT(user);
            const auto& privateDataClient = driveApi.GetPrivateDataClient();
            TUserPassportData passport;
            UNIT_ASSERT(privateDataClient.GetPassportSync(user, assignmentId, passport));

            UNIT_ASSERT(passport.GetLastName() == "вест");
        }
    };

    Y_UNIT_TEST(ManagerReturnFailedTask) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        SetGvars(server);

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        auto robot = BuildRobot();

        TString assignmentId = NUtil::CreateUUID();
        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT_VALUES_EQUAL(entries.size(), 1);
            UNIT_ASSERT(entries.begin()->second.GetCreatedAt() != entries.begin()->second.GetProcessAt());
            UNIT_ASSERT_VALUES_EQUAL(entries.begin()->second.GetAttempt(), 2);
        }
    };

    Y_UNIT_TEST(ManagerReturnEmptyAssignment) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        SetGvars(server);
        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);

        auto robot = BuildRobot();

        NJson::TJsonValue assignment;
        {
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;
            assignment["status"] = "ACCEPTED";
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT_VALUES_EQUAL(entries.size(), 1);
            UNIT_ASSERT(entries.begin()->second.GetCreatedAt() != entries.begin()->second.GetProcessAt());
            UNIT_ASSERT_VALUES_EQUAL(entries.begin()->second.GetAttempt(), 2);
        }
    };

    Y_UNIT_TEST(SequentialUpdate) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        SetGvars(server);
        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);

        auto robot = BuildRobot();

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            assignment["status"] = "ACCEPTED";
            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"] = NJson::TJsonArray({"OK"});

            NJson::TJsonValue license;
            license["middle_name"] = "том";
            license["last_name"] = "вест";

            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";
            assignment["solutions"][0]["output_values"]["documents"]["license_front"] = license;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto user = GetUser(userId, driveApi);
            UNIT_ASSERT(user);
            const auto& privateDataClient = driveApi.GetPrivateDataClient();
            TUserDrivingLicenseData license;
            UNIT_ASSERT(privateDataClient.GetDrivingLicenseSync(user, assignmentId, license));

            UNIT_ASSERT_VALUES_EQUAL(license.GetLastName(), "вест");
            UNIT_ASSERT_VALUES_EQUAL(license.GetMiddleName(), "том");
            UNIT_ASSERT_VALUES_EQUAL(license.GetFirstName(), "");
        }

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            NJson::TJsonValue license;
            license["first_name"] = "боб";

            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";
            assignment["solutions"][0]["output_values"]["documents"]["license_front"] = license;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto user = GetUser(userId, driveApi);
            UNIT_ASSERT(user);
            const auto& privateDataClient = driveApi.GetPrivateDataClient();
            TUserDrivingLicenseData license;
            UNIT_ASSERT(privateDataClient.GetDrivingLicenseSync(user, assignmentId, license));

            UNIT_ASSERT_VALUES_EQUAL(license.GetLastName(), "вест");
            UNIT_ASSERT_VALUES_EQUAL(license.GetMiddleName(), "том");
            UNIT_ASSERT_VALUES_EQUAL(license.GetFirstName(), "боб");
        }
    };

    Y_UNIT_TEST(BatchMerge) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        SetGvars(server);
        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);
        TString additionalAssignmentId = NUtil::CreateUUID();

        auto robot = BuildRobot();

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            assignment["status"] = "ACCEPTED";
            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";

            NJson::TJsonValue license;
            license["middle_name"] = "том";
            license["last_name"] = "вест";

            assignment["solutions"][0]["output_values"]["documents"]["license_front"] = license;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        {
            NJson::TJsonValue assignment;
            assignment["id"] = additionalAssignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            NJson::TJsonValue license;
            license["country"] = "эстония";

            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";
            assignment["solutions"][0]["output_values"]["documents"]["license_back"] = license;
            yangClient.InsertAssignment(additionalAssignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();

        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(queueClient->Push(additionalAssignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto user = GetUser(userId, driveApi);
            UNIT_ASSERT(user);
            const auto& privateDataClient = driveApi.GetPrivateDataClient();
            TUserDrivingLicenseData license;
            UNIT_ASSERT(
                privateDataClient.GetDrivingLicenseSync(user, assignmentId, license) ||
                privateDataClient.GetDrivingLicenseSync(user, additionalAssignmentId, license)
            );

            UNIT_ASSERT_VALUES_EQUAL(license.GetLastName(), "вест");
            UNIT_ASSERT_VALUES_EQUAL(license.GetMiddleName(), "том");
            UNIT_ASSERT_VALUES_EQUAL(license.GetCountry(), "");
            UNIT_ASSERT_VALUES_EQUAL(license.GetBackCountry(), "эстония");
        }
    };

    Y_UNIT_TEST(SetStatus) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");

        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        UNIT_ASSERT(server->GetUserDocumentsChecksManager());
        const auto* checksMananger = server->GetUserDocumentsChecksManager();

        SetGvars(server);
        TString userId;
        TString photoId;
        TString assignmentId = PrepareDocuments(userId, server, &photoId);
        UNIT_ASSERT(!!assignmentId);

        const auto& photoManager = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB();
        auto statusChecker = [
            &userId,
            &photoManager
        ] (
            const TString& photoId,
            const NUserDocument::EVerificationStatus expectedPhotoStatus,
            const NUserDocument::EType documentType,
            NDrive::TEntitySession& session) -> void
        {
            auto photos = photoManager.GetUsersWithPhotoIds({photoId}, session);
            UNIT_ASSERT(photos && photos->contains(userId));

            for (const auto& photo : (*photos)[userId]) {
                if (photo.GetType() == documentType) {
                    UNIT_ASSERT_VALUES_EQUAL(photo.GetVerificationStatus(), expectedPhotoStatus);
                    return;
                }
            }
            UNIT_ASSERT_C(false, "Photo is not found");
        };

        auto robot = BuildRobot();

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            assignment["status"] = "ACCEPTED";
            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "DISCARDED";

            NJson::TJsonValue passport;
            passport["first_name"] = "-";
            passport["middle_name"] = "-";
            passport["last_name"] = "_";
            passport["subdivision_code"] = "000-000";

            assignment["solutions"][0]["output_values"]["documents"]["passport_biographical"] = passport;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = server->GetDriveAPI()->BuildSession(NSQL::ReadOnly);
            auto checks = checksMananger->GetChecks(userId, {"OK?"}, session);
            UNIT_ASSERT(checks && checks->empty());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto session = server->GetDriveAPI()->BuildSession(NSQL::ReadOnly);
            auto checks = checksMananger->GetChecks(userId, {"OK?"}, session);
            UNIT_ASSERT(checks);
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetType(), "OK?");
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetStatus(), "DISCARDED");
            statusChecker(
                photoId,
                NUserDocument::EVerificationStatus::Discarded,
                NUserDocument::EType::PassportBiographical,
                session
            );
        }

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;

            assignment["status"] = "ACCEPTED";
            assignment["solutions"][0]["output_values"]["verdicts"]["OK?"][0] = "OK";
            assignment["solutions"][0]["output_values"]["other"] = "Nice passport";

            NJson::TJsonValue passport;
            passport["first_name"] = "боб";
            passport["middle_name"] = "том";
            passport["last_name"] = "вест";
            passport["gender"] = "MALE";
            passport["citizenship"] = "US";
            passport["birth_place"] = "гор. сан-хосе";
            passport["birth_date"] = "1864-09-20";
            passport["number"] = "81919991991";
            passport["issue_date"] = "2020-11-11";
            passport["subdivision_code"] = "000-000";

            assignment["solutions"][0]["output_values"]["documents"]["passport_biographical"] = passport;
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT(entries.empty());
        }

        {
            auto session = server->GetDriveAPI()->BuildSession(NSQL::ReadOnly);
            auto checks = checksMananger->GetChecks(userId, {"OK?"}, session);
            UNIT_ASSERT(checks);
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetType(), "OK?");
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetComment(), "Nice passport");

            statusChecker(
                photoId,
                NUserDocument::EVerificationStatus::Ok,
                NUserDocument::EType::PassportBiographical,
                session
            );
        }
    };

    Y_UNIT_TEST(Fallback) {
        TStringStream ss;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        SetGvars(server);

        UNIT_ASSERT_C(EmptyQueue(driveApi), "Failed to prepare queue");
        UNIT_ASSERT(server->GetYangClient());
        auto& yangClient = *dynamic_cast<const TFakeYangClient*>(server->GetYangClient());

        auto robot = BuildRobot(true);

        TString userId;
        TString assignmentId = PrepareDocuments(userId, server);
        UNIT_ASSERT(!!assignmentId);

        {
            NJson::TJsonValue assignment;
            assignment["id"] = assignmentId;
            assignment["tasks"][0]["input_values"]["secret"] = assignmentId;
            assignment["tasks"][0]["input_values"]["user_id"] = userId;
            assignment["status"] = "ACCEPTED";
            yangClient.InsertAssignment(assignmentId, assignment);
        }

        const auto queueClient = driveApi.GetDatasyncQueueClient();
        {
            auto session = queueClient->BuildSession();
            UNIT_ASSERT(queueClient->Push(assignmentId, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT_VALUES_EQUAL(entries.size(), 1);
            UNIT_ASSERT(entries.begin()->second.GetCreatedAt() != entries.begin()->second.GetProcessAt());
            UNIT_ASSERT_VALUES_EQUAL(entries.begin()->second.GetAttempt(), 2);
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT_VALUES_EQUAL(entries.size(), 1);
        }

        {
            UNIT_ASSERT(robot->Execute(nullptr, {*server}));
            auto session = queueClient->BuildSession(true);
            auto entries = queueClient->Get(5, session);
            UNIT_ASSERT_VALUES_EQUAL(entries.size(), 0);
        }

        {
            TVector<TDBTag> tags;
            auto session = driveApi.BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreTags({userId}, {FALLBACK_TAG_NAME}, tags, session));
            UNIT_ASSERT_EQUAL(tags.size(), 1);
            const auto fallbackTag = tags.front().GetTagAs<NDrive::ITag>();
            UNIT_ASSERT(fallbackTag);
            UNIT_ASSERT_STRING_CONTAINS(
                fallbackTag->GetComment(),
                "Failed to get any documents data from the assignment " + assignmentId
            );
        }
    };
}
