#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/actions/filter.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/offers/actions/dedicated_fleet.h>
#include <drive/backend/sessions/manager/billing.h>

#include <library/cpp/iterator/enumerate.h>
#include <library/cpp/testing/unittest/registar.h>


inline const TString SimpleDedicatedFleetOfferName = "simple_dedicated_fleet_offer";
inline const TString SimpleDedicatedFleetUnitOfferName = "simple_dedicated_fleet_unit_offer";

inline const TString DedicatedFleetOfferHolderTagName = "dedicated_fleet_offer_holder_tag";
inline const TString DFSupportTagName = "dedicated_fleet_support_tag";

inline const TString SpecialFleetTagName = TSpecialFleetTag::Actual;
inline const TString SimpleCarTagName = "dedicated_fleet_simple_on_hold_tag";

inline const TString EvolveTargetTagName = "evolve_target";
inline const TString EvolveSuffix = "_fleet";


Y_UNIT_TEST_SUITE(DedicatedFleetOffer) {
    TInstant Day(int days) {
        return TInstant::ParseIso8601("2021-01-01T00:00:00") + TDuration::Days(days);
    }

    void EvolveUserTag(const TString& sourceId, const NJson::TJsonValue& data, TTestEnvironment& env) {
        const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
        const auto& tagsMeta = env.GetServer()->GetDriveDatabase().GetTagsManager().GetTagsMeta();

        auto tx = userTagManager.BuildTx<NSQL::Writable>();

        auto source = userTagManager.RestoreTag(sourceId, tx);
        UNIT_ASSERT(source);

        auto dstTag = tagsMeta.CreateTagAs<TFleetSupportOutgoingCommunicationTag>(source->GetData()->GetName());
        UNIT_ASSERT(dstTag);
        TMessagesCollector errors;
        UNIT_ASSERT(dstTag->SpecialDataFromJson(data, &errors));
        auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
        UNIT_ASSERT(env.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(*source, dstTag, *permissions, env.GetServer().Get(), tx));
        UNIT_ASSERT(tx.Commit());
    }

    void RegisterDedicatedFleetOffers(const NDrive::IServer& server) {
        auto fleetUnitBuilderAction = MakeAtomicShared<TDedicatedFleetUnitOfferBuilder>();
        fleetUnitBuilderAction->SetName(SimpleDedicatedFleetUnitOfferName);
        fleetUnitBuilderAction->SetDescription(SimpleDedicatedFleetUnitOfferName);
        fleetUnitBuilderAction->SetDeltaCost(500);

        auto fleetBuilderAction = MakeAtomicShared<TDedicatedFleetOfferBuilder>();
        fleetBuilderAction->SetName(SimpleDedicatedFleetOfferName);
        fleetBuilderAction->SetDescription(SimpleDedicatedFleetOfferName);
        fleetBuilderAction->SetUnitOfferBuilder(SimpleDedicatedFleetUnitOfferName);
        fleetBuilderAction->SetCancellationOnPrepareCost(100000);

        {
            using namespace NDedicatedFleet;

            {
                TCommonFleetValuesAdapter commonFleet;
                TCommonFleetValuesAdapter::TConstants c;
                commonFleet.SetConstants(c);
                fleetBuilderAction->SetCommonFleetData(commonFleet);
            }

            {
                TTimeIntervalValuesAdapter timeInterval;
                TTimeIntervalValuesAdapter::TConstants c;
                timeInterval.SetConstants(c);
                fleetBuilderAction->SetTimeIntervalData(timeInterval);
            }
        }

        RegisterAction(server, fleetUnitBuilderAction, "default_user");
        RegisterAction(server, fleetBuilderAction, "default_user");
    }

    void RegisterTags(const NDrive::IServer& server) {
        TTagDescription::TPtr evolveTargetTag = new TTagDescription();
        evolveTargetTag->SetType(TDeviceTagRecord::TypeName);
        evolveTargetTag->SetName(EvolveTargetTagName);
        RegisterTag(server, evolveTargetTag);
        evolveTargetTag->SetType(TDeviceTagRecord::TypeName);
        evolveTargetTag->SetName(EvolveTargetTagName + EvolveSuffix);
        RegisterTag(server, evolveTargetTag);

        auto dfSupportTag = MakeAtomicShared<TFleetSupportOutgoingCommunicationTag::TDescription>();
        dfSupportTag->SetName(DFSupportTagName);
        dfSupportTag->SetType(TFleetSupportOutgoingCommunicationTag::TypeName);
        dfSupportTag->SetLinkedTagName(TDedicatedFleetOfferHolderTag::Type());
        RegisterTag(server, dfSupportTag);

        auto simpleOnHoldTag = MakeAtomicShared<TTagDescription>(SimpleCarTagName, "simple_car_tag");
        RegisterTag(server, simpleOnHoldTag);

        auto fleetTag = MakeAtomicShared<TTagDescription>(SpecialFleetTagName, TSpecialFleetTag::TypeName);
        RegisterTag(server, fleetTag);

        auto basic = MakeAtomicShared<TDedicatedFleetOfferHolderTag::TDescription>();
        basic->SetName(TDedicatedFleetOfferHolderTag::Type());
        basic->SetType(TDedicatedFleetOfferHolderTag::Type());
        basic->SetCommunicationTag(DFSupportTagName);
        basic->SetFleetTag(SpecialFleetTagName);
        basic->SetCarTags({SimpleCarTagName});
        basic->SetEvolveTargetSuffix(EvolveSuffix);
        basic->SetTagsToEvolveWithSuffix({EvolveTargetTagName});
        RegisterTag(server, basic);
    }

    ui64 PrepareAccount(TTestEnvironment& env) {
        auto driveApi = env.GetServer()->GetDriveAPI();
        auto& billingManager = driveApi->GetBillingManager();
        auto& accountsManager = billingManager.GetAccountsManager();

        {
            auto tx = billingManager.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = NDrive::NBilling::AccountLimit;
            descJson["soft_limit"] = NDrive::NBilling::AccountLimit;
            descJson["name"] = "corp.wallet";
            descJson["meta"]["hr_name"] = "corp.wallet";
            descJson["meta"]["refresh_policy"] = ToString(NDrive::NBilling::TRefreshSchedule::EPolicy::Month);
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());
        }

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto tx = driveApi->GetBillingManager().BuildSession();
        auto baDescription = driveApi->GetBillingManager().GetAccountsManager().GetDescriptionByName("corp.wallet");
        UNIT_ASSERT(baDescription.Defined());
        auto walletAccount = driveApi->GetBillingManager().GetAccountsManager().GetOrCreateAccount(USER_ROOT_DEFAULT, baDescription.GetRef(), USER_ROOT_DEFAULT, tx);
        UNIT_ASSERT(tx.Commit());
        UNIT_ASSERT(walletAccount->GetId());

        return walletAccount->GetId();
    }

    Y_UNIT_TEST(BuildSimpleFleetOffer) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterDedicatedFleetOffers(*env.GetServer());
        RegisterTags(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto userId = TString{USER_ID_DEFAULT};

        const ui64 accountId = PrepareAccount(env);

        TInstantGuard ig(Day(1));

        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleDedicatedFleetOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("account_id", accountId)
                ("count", 4)
            )
        );
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT_C(offerId, TStringBuilder() << "emtpy offer id " << createdOffers.GetStringRobust());
        auto restored = env->Request(userId, "/api/yandex/offers/restore", "offer_id=" + offerId);
        UNIT_ASSERT(restored.IsDefined());

        UNIT_ASSERT_C(restored["offers"][0]["offer_id"].GetString() == offerId, TStringBuilder() << "invalid offer id " << restored.GetStringRobust());
        UNIT_ASSERT_C(restored["offers"][0]["units_offers"].GetArray().size() == 4, TStringBuilder() << "invalid units offers size " << restored.GetStringRobust());

        UNIT_ASSERT(env->BookOffer(offerId, userId));

        TString holderTagId;
        UNIT_ASSERT(env->GetTagId(::ToString(accountId), DedicatedFleetOfferHolderTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Account, holderTagId));
        UNIT_ASSERT(holderTagId);

        {
            const auto& accountTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetAccountTags();
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();

            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
            auto optionalTag = accountTagManager.RestoreTag(holderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto fleetHolderTag = optionalTag->GetTagAs<TDedicatedFleetOfferHolderTag>();
            UNIT_ASSERT(fleetHolderTag);
            auto&& supportTagId = fleetHolderTag->GetCommunicationTagId();
            UNIT_ASSERT(supportTagId);
            auto supportDbTag = userTagManager.RestoreTag(supportTagId, tx);
            UNIT_ASSERT(supportDbTag);
            auto supportTag = supportDbTag->GetTagAs<TFleetSupportOutgoingCommunicationTag>();
            UNIT_ASSERT(supportTag);
            UNIT_ASSERT(tx.Commit());

            auto car1 = env.GetEnvironmentGenerator().CreateCar();
            auto car2 = env.GetEnvironmentGenerator().CreateCar();
            for (auto&& id : {car1.Id, car2.Id}){
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                auto session = env.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto tag = env.GetServer()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(EvolveTargetTagName, "expected car tag");
                UNIT_ASSERT(tag);
                UNIT_ASSERT_C(deviceTags.AddTag(tag, userId, id, env.GetServer().Get(), session) && session.Commit(), session.GetStringReport());
            }

            // check add_car
            {
                NJson::TJsonValue fleetConfiguration =
                    NJson::TMapBuilder(restored["offers"][0]["units_offers"][0]["offer_id"].GetString(), NJson::TMapBuilder("car_id", car1.Id))
                        (restored["offers"][0]["units_offers"][1]["offer_id"].GetString(), NJson::TMapBuilder("car_id", car2.Id));

                EvolveUserTag(supportTagId, fleetConfiguration, env);
            }
            {
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                TVector<TDBTag> dbTags;
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTags.RestoreTags({car1.Id}, {SpecialFleetTagName, SimpleCarTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 2, dbTags.size());
            }
            {
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                TVector<TDBTag> dbTags;
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {SpecialFleetTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 1, dbTags.size());
                dbTags.clear();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {EvolveTargetTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 0, dbTags.size());
                dbTags.clear();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {EvolveTargetTagName + EvolveSuffix}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 1, dbTags.size());
            }

            {
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                auto tagOpt = accountTagManager.RestoreTag(holderTagId, session);
                UNIT_ASSERT_C(tagOpt, session.GetStringReport());
                auto tag = tagOpt->GetTagAs<TDedicatedFleetOfferHolderTag>();
                UNIT_ASSERT(tag);
                UNIT_ASSERT_C(tag->GetFleetTagIds().size() == 2, tag->GetStringData());
            }

            {
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                TVector<TDBTag> dbTags;
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {SpecialFleetTagName, SimpleCarTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 2, dbTags.size());
            }

            {
                NJson::TJsonValue fleetConfiguration =
                    NJson::TMapBuilder(restored["offers"][0]["units_offers"][0]["offer_id"].GetString(), "");
                EvolveUserTag(supportTagId, fleetConfiguration, env);
            }

            {
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                auto tagOpt = accountTagManager.RestoreTag(holderTagId, session);
                UNIT_ASSERT_C(tagOpt, session.GetStringReport());
                auto tag = tagOpt->GetTagAs<TDedicatedFleetOfferHolderTag>();
                UNIT_ASSERT(tag);
                UNIT_ASSERT_C(tag->GetFleetTagIds().size() == 1, tag->GetStringData());
            }

            {
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                TVector<TDBTag> dbTags;
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTags.RestoreTags({car1.Id}, {SpecialFleetTagName, SimpleCarTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 0, dbTags.size());
            }

            {
                ig.Set(Day(5));
                auto sessionReport = env->GetSessionReport("session_id=" + holderTagId, USER_ROOT_DEFAULT);
                UNIT_ASSERT_C(!sessionReport.IsNull(), sessionReport.GetStringRobust());
                INFO_LOG << "Session report: " << sessionReport.GetStringRobust() << '\n';
            }

            {
               ig.Set(Day(10));
                auto sessionReport = env->GetSessionReport("session_id=" + holderTagId, USER_ROOT_DEFAULT);
                UNIT_ASSERT_C(!sessionReport.IsNull(), sessionReport.GetStringRobust());
                INFO_LOG << "Session report: " << sessionReport.GetStringRobust() << '\n';
            }

            {
                auto sessionReport = env->GetSessionReport("session_id=" + holderTagId, USER_ID_DEFAULT);
                UNIT_ASSERT_C(sessionReport.IsNull(), sessionReport.GetStringRobust());
            }

            {
                auto sessionReport = env->GetSessionReport("account_id=" + ::ToString(accountId), USER_ROOT_DEFAULT);
                UNIT_ASSERT_C(!sessionReport.IsNull(), sessionReport.GetStringRobust());
            }

            {
                env->Request(USER_ROOT_DEFAULT, "/b2b/dedicated_fleet/session/drop", "session_id=" + holderTagId);
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                auto optionalTag = accountTagManager.RestoreTag(holderTagId, session);
                UNIT_ASSERT_C(optionalTag, session.GetStringReport());
                auto pTag = dynamic_cast<const TDedicatedFleetOfferHolderTag*>(optionalTag->GetData().Get());
                UNIT_ASSERT(pTag == nullptr);
            }

            {
                const auto& deviceTags = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                TVector<TDBTag> dbTags;
                auto session = env.GetServer()->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {SpecialFleetTagName, SimpleCarTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 0, dbTags.size());
                dbTags.clear();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {EvolveTargetTagName}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 1, dbTags.size());
                dbTags.clear();
                UNIT_ASSERT(deviceTags.RestoreTags({car2.Id}, {EvolveTargetTagName + EvolveSuffix}, dbTags, session));
                UNIT_ASSERT_C(dbTags.size() == 0, dbTags.size());
            }
        }
    }

}
