#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>
#include <drive/backend/ut/library/scripts/delegation.h>

#include <drive/backend/actions/evolution_policy.h>
#include <drive/backend/background/car_markers/config.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(Delegation) {
    using namespace NDrive::NTest;
    Y_UNIT_TEST(FreeControl) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TFreeCarDelegation>().SetExpectOK(false);
        script.Add<TPark>();
        script.Add<TFreeCarDelegation>();
        script.Add<TCheckCarDelegation>("free");
        script.Add<TDropCarDelegation>();
        script.Add<TCheckCarDelegation>();
        script.Add<TDrop>();
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Free) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TFreeCarDelegation>().SetExpectOK(false);
        script.Add<TCheckCarDelegation>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TFreeCarDelegation>().SetExpectOK(true);
        script.Add<TCheckCarDelegation>("free");
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetWaitingDuration(TDuration::Max()).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarDelegation>("free").SetFinished(true).SetActionUserId(USER_ID_DEFAULT);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FreeAtomicEndpoint) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord outside(35.917933, 56.855074);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        {
            NJson::TJsonValue jsonLanding;
            jsonLanding["text"] = "some alert text";
            jsonLanding["payload_patch"]["delegation"] = TCarDelegationPolicy(USER_ID_DEFAULT).SerializeToJson();
            script.Add<TAddLanding>("suspended_rent_only", jsonLanding.GetStringRobust());
        }
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TFreeCarDelegation>().SetExpectOK(false);
        script.Add<TCheckCarDelegation>();
        script.Add<TDrop>().SetCarPosition(outside).SetExpectOK(false);
        script.Add<TDrop>().SetCarPosition(outside).SetAlertLandingId("!dont_patch suspended_rent_only").SetExpectOK(false);
        script.Add<TDrop>().SetCarPosition(outside).SetAlertLandingId("suspended_rent_only").SetExpectOK(true);
        script.Add<TCheckCarDelegation>("free");
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckHasTag>().SetTagName("delegation_finished").SetExpectOK(false);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetWaitingDuration(TDuration::Max()).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckHasTag>().SetTagName("delegation_finished");
        script.Add<TCheckCarDelegation>("free").SetFinished(true).SetActionUserId(USER_ID_DEFAULT);
        script.Add<TCheckCarDelegation>("free").SetFinished(true).SetActionUserId(USER_ID_DEFAULT);
        script.Add<TViewFinishedDelegation>();
        script.Add<TCheckCarDelegation>();
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(outside).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FromParking) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord outside(35.917933, 56.855074);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        {
            NJson::TJsonValue jsonLanding;
            jsonLanding["text"] = "some alert text";
            jsonLanding["payload_patch"]["delegation"] = TCarDelegationPolicy(USER_ID_DEFAULT).SerializeToJson();
            script.Add<TAddLanding>("suspended_rent_only", jsonLanding.GetStringRobust());
        }
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TFreeCarDelegation>().SetExpectOK(false);
        script.Add<TDrop>().SetCarPosition(outside).SetExpectOK(false);
        script.Add<TDrop>().SetCarPosition(outside).SetAlertLandingId("suspended_rent_only").SetExpectOK(true);
        script.Add<TCheckCarDelegation>("free");
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(DelegationSamePerson) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord outside(35.917933, 56.855074);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        {
            NJson::TJsonValue jsonLanding;
            jsonLanding["text"] = "some alert text";
            jsonLanding["payload_patch"]["delegation"] = TCarDelegationPolicy(USER_ID_DEFAULT).SerializeToJson();
            script.Add<TAddLanding>("suspended_rent_only", jsonLanding.GetStringRobust());
        }
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TFreeCarDelegation>().SetExpectOK(false);
        script.Add<TCheckCarDelegation>();
        script.Add<TDrop>().SetCarPosition(outside).SetExpectOK(false);
        script.Add<TDrop>().SetCarPosition(outside).SetAlertLandingId("suspended_rent_only").SetExpectOK(true);
        script.Add<TCheckCarDelegation>("free");
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TRide>();
        script.Add<TCheckCarDelegation>();
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2P) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT1);
        script.Add<TCheckCarDelegation>("p2p");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetUserId(USER_ID_DEFAULT1).SetExpectOK(false);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT1).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    // Y_UNIT_TEST(P2PWithServiceTag) {
    //     NDrive::TServerConfigGenerator configGenerator;
    //     configGenerator.SetLogLevel(6);
    //     TGeoCoord from(37.5848674, 55.7352435);
    //     TGeoCoord to(37.5675511, 55.7323499);
    //     auto serviceTag = MakeAtomicShared<TDeviceTagRecord>("simple1");;
    //     serviceTag->SetTagPriority(1);
    //     NDrive::NTest::TScript script(configGenerator);
    //     const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
    //     script.Add<TBuildEnv>(TDuration::Zero());
    //     script.Add<TCreateCar>().SetPosition(from);
    //     script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
    //     script.Add<TAccept>();
    //     script.Add<TRide>();
    //     script.Add<TAddTag>(serviceTag);
    //     script.Add<TPark>();
    //     script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
    //     script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
    //     script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
    //     script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
    //     script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
    //     script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT1);
    //     script.Add<TCheckCarDelegation>("p2p");
    //     script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("minutki_test").SetUserPosition(from).SetUserId(USER_ID_DEFAULT1).SetExpectOK(false);
    //     script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("minutki_test").SetUserPosition(from).SetUserId(USER_ID_DEFAULT2);
    //     script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
    //     script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT1).SetExpectOK(false);
    //     script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
    //     UNIT_ASSERT(script.Execute());
    // }

    Y_UNIT_TEST(P2PRejectByDelegator) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckCarDelegation>("p2p");
        script.Add<TRide>();
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(false);
        script.Add<TPark>();
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TRejectOutgoingDelegation>().SetUserId(USER_ID_DEFAULT);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PBookAfterReject) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckCarDelegation>("p2p");
        script.Add<TRide>();
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(false);
        script.Add<TPark>();
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);

        script.Add<TRejectAndBookDelegatedOffer>().SetDelegatorId(USER_ID_DEFAULT).SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PRejectByDelegatee) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TRejectIncomingDelegation>().SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckDelegationRejected>().SetRejected(true).SetExpired(false);
        //
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TRejectIncomingDelegation>().SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        //
        script.Add<TCheckHasTag>().SetTagName(TDelegationUserTag::TypeName).SetTargetCount(1);
        script.Add<TCheckDelegationRejected>().SetRejected(true).SetExpired(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PRejectByFinishRide) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckCarDelegation>("p2p");
        script.Add<TDrop>();
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PExpired) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        {
            THolder<TRideCompletionEvolutionPolicyAction> action(new TRideCompletionEvolutionPolicyAction("ride_completion_action"));
            action->SetFreeDelegationEnabled(true);
            action->SetFreeDelegationConfirmationLanding("suspended_rent_only");
            action->SetP2PDelegationDuration(TDuration::Zero());
            script.Add<TUpsertAction>().SetAction(action.Release());
        }
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetExpired(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(true);
        script.Add<TCheckCarDelegation>("p2p");
        script.Add<TDrop>();
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPossibility) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);

        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();

        // use phone of USER_ID_DEFAULT
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_ID_DEFAULT_PHONE).SetExpectedError(TDriveAPI::EDelegationAbility::SameUser).SetExpectOK(false);

        // use definitely non-existent phone
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone("+xxxx").SetExpectedError(TDriveAPI::EDelegationAbility::UserNotRegistered).SetExpectOK(false);

        // use phone of blocked user
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_PHONE_BLOCKED).SetExpectedError(TDriveAPI::EDelegationAbility::TargetUserBlocked).SetExpectOK(false);

        // use phone of onboarding user
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_PHONE_ONBOARDING).SetExpectedError(TDriveAPI::EDelegationAbility::TargetUserOnboarding).SetExpectOK(false);

        // use phone of regular user
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(1).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);

        // now the same, but with different phone number format
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_ID_DEFAULT_PHONE_ALT).SetExpectedError(TDriveAPI::EDelegationAbility::SameUser).SetExpectOK(false);
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_PHONE_BLOCKED_ALT).SetExpectedError(TDriveAPI::EDelegationAbility::TargetUserBlocked).SetExpectOK(false);
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_PHONE_ONBOARDING_ALT).SetExpectedError(TDriveAPI::EDelegationAbility::TargetUserOnboarding).SetExpectOK(false);
        script.Add<TP2PCarDelegationCheck>().SetTargetUserPhone(USER_ID_DEFAULT2_PHONE_ALT).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPassOfferNotPossibleWithDebt) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TBillingEmptyCardsReply>();
        script.Add<TRunBillingCycle>();
        script.Add<TPark>();

        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(2).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(true).SetExpectOK(false);
        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(false).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPossibilityWithServiceTag) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        auto serviceTag = MakeAtomicShared<TDeviceTagRecord>("simple1");;
        serviceTag->SetTagPriority(1);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TAddTag>(serviceTag);
        script.Add<TRide>();
        // use phone of regular user
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(1).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPackOffer) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(2).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(true).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetExpired(false).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckCarDelegation>("p2p_pass_offer");

        script.Add<TBookDelegatedOffer>().SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT1).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TViewFinishedDelegation>().SetP2PUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPackOfferQR) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();

        script.Add<TP2PCarDelegation>().SetP2PUserId("").SetPreserveOffer(true).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckCarDelegation>("p2p_pass_offer");

        script.Add<TBookDelegatedOffer>().SetDelegatorId(USER_ID_DEFAULT).SetUseQR(true).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckIncomingCarDelegation>().SetExists(false).SetUserId(USER_ID_DEFAULT2);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT1).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetUseSessionId(false).SetCarPosition(to).SetActionUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TViewFinishedDelegation>().SetP2PUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PB2BOffer) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor_b2b").SetAccountName("y.drive").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(2).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(true).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetExpired(false).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckCarDelegation>("p2p_pass_offer");
        script.Add<TBookDelegatedOffer>().SetUserId(USER_ID_DEFAULT2).SetExpectOK(false);

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT2);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<TBookDelegatedOffer>().SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCommonChecker>([](TRTContext& context) {
            TVector<TAtomicSharedPtr<const ISession>> sessions;
            UNIT_ASSERT(context.GetDriveAPI().GetCurrentUserSessions(USER_ID_DEFAULT2, sessions, TInstant::Now()));
            UNIT_ASSERT_EQUAL(sessions.size(), 1);
            auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(sessions.front());
            UNIT_ASSERT(billingSession);
            auto currentOffer = billingSession->GetCurrentOffer();
            UNIT_ASSERT(currentOffer);
            UNIT_ASSERT_EQUAL(currentOffer->GetSelectedCharge(), "y.drive");
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPackOfferNotWallet) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetAccountName("y.drive").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(2).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(true).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetExpired(false).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckCarDelegation>("p2p_pass_offer");
        script.Add<TBookDelegatedOffer>().SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(P2PPackOfferWalletAccess) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT2);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetAccountName("y.drive").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TP2PCarDelegationCheck>().SetTotalOptions(2).SetTargetUserPhone(USER_ID_DEFAULT2_PHONE).SetExpectOK(true);

        script.Add<TP2PCarDelegation>().SetP2PUserId(USER_ID_DEFAULT2).SetPreserveOffer(true).SetExpectOK(true);
        script.Add<TCheckIncomingCarDelegation>().SetExists(true).SetExpired(false).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<TCheckDelegationRejected>().SetRejected(false).SetExpired(false);
        script.Add<TCheckCarDelegation>("p2p_pass_offer");
        script.Add<TBookDelegatedOffer>().SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<TCommonChecker>([](TRTContext& context) {
            TVector<TAtomicSharedPtr<const ISession>> sessions;
            UNIT_ASSERT(context.GetDriveAPI().GetCurrentUserSessions(USER_ID_DEFAULT2, sessions, TInstant::Now()));
            UNIT_ASSERT_EQUAL(sessions.size(), 1);
            auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(sessions.front());
            UNIT_ASSERT(billingSession);
            auto currentOffer = billingSession->GetCurrentOffer();
            UNIT_ASSERT(currentOffer);
            UNIT_ASSERT_EQUAL(currentOffer->GetSelectedCharge(), "y.drive");
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FreeB2BOffer) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TInstant now = Now();
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor_b2b").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TPark>();
        script.Add<TFreeCarDelegation>().SetExpectOK(true);
        script.Add<TCheckCarDelegation>("free");

        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetWaitingDuration(TDuration::Max()).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarDelegation>("free").SetFinished(true).SetActionUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(script.Execute());
    }
}
