#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/users/user_documents_check.h>
#include <drive/backend/experiments/context.h>
#include <drive/backend/data/dictionary_tags.h>

#include <rtline/util/types/uuid.h>

Y_UNIT_TEST_SUITE(DocumentsChecks) {
    bool CheckStatus(const TUserDocumentsChecksManager* manager, const TString& userId, const TString& type, const TString& expectedStatus, NDrive::TEntitySession& session) {
        auto status = manager->GetStatus(userId, type, session);
        return status && *status == expectedStatus;
    }

    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check1", false, "active");
        auto userId2 = eGenerator.CreateUser("user-with-documents-check2", false, "active");

        TVector<TString> allowedTypes = {"new", "created", "ok"};
        NJson::TJsonValue settings;
        NJson::InsertField(settings, "license_front", allowedTypes);
        NJson::InsertField(settings, "license_back", allowedTypes);
        UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license_front", "new", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license_back", "new", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId2, "license_back", "new", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId2, "license_front", "created", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(!documentsChecksManager->SetStatus(userId, "license_front", "bad", USER_ROOT_DEFAULT, *server.Get(), session));
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "license_front", "new", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "license_back", "new", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "license_front_2", "", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId2, "license_front", "created", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId2, "license_back", "new", session));

            auto checks = documentsChecksManager->GetChecks(userId2, {"license_front", "license_back", "other", "passport"}, session);
            UNIT_ASSERT(checks);

            UNIT_ASSERT_VALUES_EQUAL(checks->size(), 2);
            for (auto&& check : *checks) {
                UNIT_ASSERT_VALUES_EQUAL(check.GetUserId(), userId2);
                if (check.GetType() == "license_front") {
                    UNIT_ASSERT_VALUES_EQUAL(check.GetStatus(), "created");
                } else if (check.GetType() == "license_back") {
                    UNIT_ASSERT_VALUES_EQUAL(check.GetStatus(), "new");
                } else {
                    UNIT_ASSERT(false);
                }
            }
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId2, "license_back", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId2, "license_back", "ok", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "license_back", "new", session));
        }
    }

    Y_UNIT_TEST(Conditions) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check2", false, "active");

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "talk_about_documents_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["text"].GetString(), "option_2");
        }

        TVector<TString> allowedTypes = {"new", "ok", "bad"};
        NJson::TJsonValue settings;
        NJson::InsertField(settings, "license", allowedTypes);
        NJson::InsertField(settings, "passport", allowedTypes);
        NJson::InsertField(settings, "selfie", allowedTypes);
        UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license", "bad", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "selfie", "new", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "option_3");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "new", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "selfie", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"].GetString(), "option_1");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "bad", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["text"].GetString(), "option_4");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 11);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[10]["text"].GetString(), "option_2");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "selfie", "bad", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_checks", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 13);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[12]["text"].GetString(), "option_5");
        }
    }

    Y_UNIT_TEST(Resubmit) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check3", false, "active");

        TVector<TString> allowedTypes = {"new", "ok", "bad"};
        NJson::TJsonValue settings;
        NJson::InsertField(settings, "license", allowedTypes);
        NJson::InsertField(settings, "passport", allowedTypes);
        NJson::InsertField(settings, "selfie", allowedTypes);
        UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "bad", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "selfie", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "test_resubmits", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_biographical");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "", {configGenerator.GetValidImageString()}));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "Спасибо. Я отправил фотографию на проверку");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license", "bad", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "test_resubmits", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "", {configGenerator.GetValidImageString()}));
        }
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 11);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "", {configGenerator.GetValidImageString()}));
        }
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 14);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[12]["text"].GetString(), "Спасибо. Я отправил фотографии на проверку");
        }
    }

    Y_UNIT_TEST(PrivateDataMatches) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check3.5", false, "active");

        TDriveUserData user;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto fetchUserResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
            UNIT_ASSERT(fetchUserResult && fetchUserResult.size() == 1);
            user = fetchUserResult.begin()->second;
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            user.SetPassportDatasyncRevision(NUtil::CreateUUID());
            user.SetDrivingLicenseDatasyncRevision(NUtil::CreateUUID());
            server->GetDriveAPI()->GetUsersData()->UpdateUser(user, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
        }

        {
            auto passport = TDocumentsHelper::CreateRandomPassportData();
            passport.SetFirstName("Федор");
            passport.SetLastName("Муртазин");
            passport.SetMiddleName("Валерьевич");
            server->GetDriveAPI()->GetPrivateDataClient().UpdatePassport(user, user.GetPassportDatasyncRevision(), passport).Wait();
        }

        {
            auto license = TDocumentsHelper::CreateRandomDrivingLicenseData();
            license.SetFirstName("Федор");
            license.SetLastName("Muртазин");
            server->GetDriveAPI()->GetPrivateDataClient().UpdateDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision(), license).Wait();

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "test_documents_match", {}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["text"].GetString(), "latin");

            license.SetFirstName("Федор");
            license.SetLastName("Муртазин");
            server->GetDriveAPI()->GetPrivateDataClient().UpdateDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision(), license).Wait();
        }

        {
            auto passport = TDocumentsHelper::CreateRandomPassportData();
            passport.SetFirstName("Федор");
            passport.SetLastName("Муртазин");
            passport.SetMiddleName("+");
            server->GetDriveAPI()->GetPrivateDataClient().UpdatePassport(user, user.GetPassportDatasyncRevision(), passport).Wait();

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_match", {}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "only");
        }

        {
            auto passport = TDocumentsHelper::CreateRandomPassportData();
            passport.SetFirstName("Атонеон");
            passport.SetLastName("Муртазин");
            passport.SetMiddleName("Петрович-Иванович");
            server->GetDriveAPI()->GetPrivateDataClient().UpdatePassport(user, user.GetPassportDatasyncRevision(), passport).Wait();

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_match", {}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"].GetString(), "complex_batya");
        }

        {
            auto passport = TDocumentsHelper::CreateRandomPassportData();
            passport.SetFirstName("неофан");
            passport.SetLastName("андерсон");
            passport.SetMiddleName("неогимович");
            server->GetDriveAPI()->GetPrivateDataClient().UpdatePassport(user, user.GetPassportDatasyncRevision(), passport).Wait();

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_match", {}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["text"].GetString(), "mr_anderson_welcome_back");
        }
    }

    Y_UNIT_TEST(SetStatusFromChat) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check4", false, "active");
        TVector<TString> allowedTypes = {"new", "ok", "bad"};
        NJson::TJsonValue settings;
        NJson::InsertField(settings, "license", allowedTypes);
        NJson::InsertField(settings, "passport", allowedTypes);
        NJson::InsertField(settings, "selfie", allowedTypes);
        UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "passport", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "license", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(userId, "selfie", "ok", USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "test_check_status", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["text"].GetString(), "Обновляем статус");
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "license", "new", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "passport", "new", session));
            UNIT_ASSERT(CheckStatus(documentsChecksManager, userId, "selfie", "ok", session));
        }
    }

    Y_UNIT_TEST(CheckContextRoles) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        TDriveRoleHeader role("RonFuckingSwanson");
        UNIT_ASSERT(configGenerator.RegisterRole(role, USER_ROOT_DEFAULT));

        NJson::TJsonValue headerJson;
        headerJson[0]["HANDLER"] = NDrive::ExperimentHandler;

        TMap<TString, TString> headers;
        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("ron-swanson", false, "active");

        TDriveUserData user;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto fetchUserResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
            UNIT_ASSERT(fetchUserResult && fetchUserResult.size() == 1);
            user = fetchUserResult.begin()->second;
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            user.SetPassportDatasyncRevision(NUtil::CreateUUID());
            server->GetDriveAPI()->GetUsersData()->UpdateUser(user, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(session.Commit());
        }

        {
            headerJson[0]["CONTEXT"][NDrive::ExperimentHandler]["enabled_roles"][0] = "Harry";
            headers["X-Yandex-ExpFlags"] = Base64Encode(headerJson.GetStringRobust());
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "check_ron_swanson", {}, 0, 0, {}, "", headers));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["text"].GetString(), "impostor");
        }

        {
            Sleep(TDuration::Seconds(0.5));
            headerJson[0]["CONTEXT"][NDrive::ExperimentHandler]["enabled_roles"][0] = "RonFuckingSwanson";
            headers["X-Yandex-ExpFlags"] = Base64Encode(headerJson.GetStringRobust());
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "to_intro", {}, 0, 0, {}, "", headers));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_documents_checks", "check_ron_swanson", {}, 0, 0, {}, "", headers));
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_documents_checks");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"].GetString(), "duke_silver_welcome_back");
        }
    }

    Y_UNIT_TEST(Meta) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check", false, "active");

        TVector<TString> allowedTypes = {"new", "created", "ok"};
        NJson::TJsonValue settings;
        NJson::InsertField(settings, "license_front", allowedTypes);
        NJson::InsertField(settings, "license_back", allowedTypes);
        UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        const TString poolId = NUtil::CreateUUID();
        const TString workerId = NUtil::CreateUUID();
        const TString comment = "Nice license";

        TUserDocumentsCheck check;
        check.SetUserId(userId);
        check.SetType("license_front");
        check.SetStatus("new");
        check.SetPoolId(NUtil::CreateUUID());
        check.SetWorkerId(NUtil::CreateUUID());
        check.SetComment(comment);
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(check, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto checks = documentsChecksManager->GetChecks(userId, {"license_front"}, session);
            UNIT_ASSERT_C(checks && checks->size() == 1, session.GetStringReport());
            UNIT_ASSERT_VALUES_UNEQUAL(checks->front().GetPoolId(), poolId);
            UNIT_ASSERT_VALUES_UNEQUAL(checks->front().GetWorkerId(), workerId);
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetComment(), comment);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            check.SetPoolId(poolId);
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(check, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto checks = documentsChecksManager->GetChecks(userId, {"license_front"}, session);
            UNIT_ASSERT_C(checks && checks->size() == 1, session.GetStringReport());
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetPoolId(), poolId);
            UNIT_ASSERT_VALUES_UNEQUAL(checks->front().GetWorkerId(), workerId);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            check.SetWorkerId(workerId);
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(check, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto checks = documentsChecksManager->GetChecks(userId, {"license_front"}, session);
            UNIT_ASSERT_C(checks && checks->size() == 1, session.GetStringReport());
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetPoolId(), poolId);
            UNIT_ASSERT_VALUES_EQUAL(checks->front().GetWorkerId(), workerId);
        }
    }

    Y_UNIT_TEST(RegistrationTagMerge) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId = eGenerator.CreateUser("user-with-documents-check", false, "active");


        {
            TVector<TString> allowedTypes = {"new", "created", "ok"};
            NJson::TJsonValue settings;
            NJson::InsertField(settings, "license_back_fill", allowedTypes);
            NJson::InsertField(settings, "passport_biographical", allowedTypes);

            NJson::TJsonValue registrationTagMapping;
            registrationTagMapping["license_back_fill"] = "lb_check";
            registrationTagMapping["passport_biographical"] = "pb_check";

            NJson::TJsonValue registrationTagSettings;
            registrationTagSettings["name"] = "registration_info";
            registrationTagSettings["mapping"] = registrationTagMapping;

            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSettings().SetValue("handlers.api/support_staff/documents_checks/get.merge_registration_tag", "true", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.registration_tag_settings", registrationTagSettings.GetStringRobust(), USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        TString poolId = NUtil::CreateUUID();
        TString workerId = NUtil::CreateUUID();
        TString assignmentId = NUtil::CreateUUID();

        TUserDocumentsCheck check;
        check.SetUserId(userId);
        check.SetType("license_back_fill");
        check.SetStatus("ok");
        check.SetPoolId(poolId);

        {
            TAtomicSharedPtr<TUserDictionaryTag> tag = new TUserDictionaryTag("registration_info");
            TString assignment = "[{\
                \"assignmentId\":\"" + assignmentId + "\",\
                \"create_ts\":1633234230713432,\
                \"document_types\":[\"license_back\"],\
                \"result\":{\"other\":{},\"verdicts\":{\"license_back_fill\":[\"new\"]}},\
                \"worker_id\":\"" + workerId +"\"\
            }]";
            tag->SetField("lb_check", assignment);
            UNIT_ASSERT(configGenerator.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(check, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto checks = configGenerator.GetDocumentCheck(userId, "license_back_fill");
            UNIT_ASSERT_C(checks.IsDefined() && checks[1].IsDefined(), session.GetStringReport());
            UNIT_ASSERT_VALUES_EQUAL(checks[0]["pool_id"].GetString(), poolId);
            UNIT_ASSERT_VALUES_EQUAL(checks[0]["status"].GetString(), "ok");
            UNIT_ASSERT_VALUES_EQUAL(checks[1]["worker_id"].GetString(), workerId);
            UNIT_ASSERT_VALUES_EQUAL(checks[1]["assignment_id"].GetString(), assignmentId);
        }
    }

    Y_UNIT_TEST(ChecksHistory) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);
        auto userId1 = eGenerator.CreateUser("user-with-documents-check-1", false, "active");
        auto userId2 = eGenerator.CreateUser("user-with-documents-check-2", false, "active");

        {
            TVector<TString> allowedTypes = {"new", "created", "ok", "tag_ok"};
            NJson::TJsonValue settings;
            NJson::InsertField(settings, "license_back_fill", allowedTypes);
            NJson::InsertField(settings, "passport_biographical", allowedTypes);

            NJson::TJsonValue registrationTagMapping;
            registrationTagMapping["license_back_fill"] = "lb_check";
            registrationTagMapping["passport_biographical"] = "pb_check";

            NJson::TJsonValue registrationTagSettings;
            registrationTagSettings["name"] = "registration_info";
            registrationTagSettings["mapping"] = registrationTagMapping;

            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSettings().SetValue("handlers.api/support_staff/documents_checks/get_history.merge_registration_tag", "true", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.registration_tag_settings", registrationTagSettings.GetStringRobust(), USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        TString poolId = NUtil::CreateUUID();
        TString workerId = NUtil::CreateUUID();
        TString assignmentId = NUtil::CreateUUID();

        TUserDocumentsCheck checkUser1;
        checkUser1.SetUserId(userId1);
        checkUser1.SetType("license_back_fill");
        checkUser1.SetStatus("new");
        checkUser1.SetWorkerId(workerId);

        TUserDocumentsCheck checkUser2;
        checkUser2.SetUserId(userId2);
        checkUser2.SetType("passport_biographical");
        checkUser2.SetStatus("new");
        checkUser2.SetWorkerId(workerId);
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(checkUser1, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(checkUser2, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

        {
            auto history = configGenerator.GetDocumentChecksHistory(0, userId1, "");
            UNIT_ASSERT(history[0].IsMap());
            UNIT_ASSERT(!history[1].IsDefined());
            UNIT_ASSERT_VALUES_EQUAL(history[0]["status"].GetString(), "new");
        }

        {
            Sleep(TDuration::Seconds(1));
            TAtomicSharedPtr<TUserDictionaryTag> tag = new TUserDictionaryTag("registration_info");
            TString assignment = "[{\
                \"assignmentId\":\"" + assignmentId + "\",\
                \"create_ts\":1633234230713432,\
                \"document_types\":[\"license_back\"],\
                \"result\":{\"other\":{},\"verdicts\":{\"license_back_fill\":[\"tag_ok\"]}},\
                \"worker_id\":\"" + workerId +"\"\
            }]";
            tag->SetField("lb_check", assignment);
            UNIT_ASSERT(configGenerator.AddTag(tag, userId1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            auto history = configGenerator.GetDocumentChecksHistory(0, userId1, "license_back_fill");
            UNIT_ASSERT(history[1].IsMap());
            UNIT_ASSERT(!history[2].IsDefined());

            UNIT_ASSERT_EQUAL(history[0]["status"].GetString(), "new");
            UNIT_ASSERT_EQUAL(history[0]["type"].GetString(), "license_back_fill");
            UNIT_ASSERT_EQUAL(history[1]["status"].GetString(), "tag_ok");
            UNIT_ASSERT_EQUAL(history[1]["type"].GetString(), "lb_check");
        }

        {
            Sleep(TDuration::Seconds(1));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            checkUser1.SetStatus("ok");
            UNIT_ASSERT_C(documentsChecksManager->SetStatus(checkUser1, USER_ROOT_DEFAULT, *server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

        {
            auto history = configGenerator.GetDocumentChecksHistory(0, userId1, "license_back_fill");
            UNIT_ASSERT(history[2].IsMap());
            UNIT_ASSERT(!history[3].IsDefined());

            UNIT_ASSERT_EQUAL(history[1]["status"].GetString(), "tag_ok");
            UNIT_ASSERT_EQUAL(history[2]["status"].GetString(), "ok");
        }

        {
            auto history = configGenerator.GetDocumentChecksHistory(0, userId2, "passport_biographical");
            UNIT_ASSERT(history[0].IsMap());
            UNIT_ASSERT(!history[1].IsDefined());

            UNIT_ASSERT_EQUAL(history[0]["status"].GetString(), "new");
        }
    }

    Y_UNIT_TEST(PhotoStatus) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto documentsChecksManager = server->GetUserDocumentsChecksManager();
        UNIT_ASSERT(documentsChecksManager);

        UNIT_ASSERT(server->GetDriveAPI()->HasDocumentPhotosManager());
        const auto& photoManager = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB();

        auto userId = eGenerator.CreateUser("user-with-documents-check", false, "active");

        {
            {
                TVector<TString> allowedTypes = {"OK", "BAD", "AWFUL"};
                NJson::TJsonValue settings;
                NJson::InsertField(settings, "license_back_selfie", allowedTypes);
                NJson::InsertField(settings, "license_back_fill", allowedTypes);
                NJson::InsertField(settings, "license_front_selfie", allowedTypes);
                NJson::InsertField(settings, "license_front_fill", allowedTypes);
                UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.settings", settings.GetStringRobust(), USER_ROOT_DEFAULT));
            }

            {
                NJson::TJsonValue genericChecks;
                genericChecks["license_back_fill"] = {"license_back"};
                genericChecks["license_front_fill"] = {"license_front"};
                genericChecks["license_back_selfie"] = {"license_back"};
                genericChecks["license_front_selfie"] = {"license_front"};
                UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.generic_checks", genericChecks.GetStringRobust(), USER_ROOT_DEFAULT));
            }

            {
                NJson::TJsonValue photoChecksSettings = NJson::JSON_ARRAY;
                {
                    NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                    verdicts[0] = "OK";
                    NJson::TJsonValue pair;
                    pair["OK"] = std::move(verdicts);
                    photoChecksSettings.AppendValue(std::move(pair));

                }
                {
                    NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                    verdicts[0] = "BAD";
                    NJson::TJsonValue pair;
                    pair["NEED_INFO"] = std::move(verdicts);
                    photoChecksSettings.AppendValue(std::move(pair));

                }
                {
                    NJson::TJsonValue verdicts = NJson::JSON_ARRAY;
                    verdicts[0] = "AWFUL";
                    NJson::TJsonValue pair;
                    pair["DISCARDED"] = std::move(verdicts);
                    photoChecksSettings.AppendValue(std::move(pair));

                }
                UNIT_ASSERT(server->GetSettings().SetValue("user_documents_checks.photo_settings", photoChecksSettings.GetStringRobust(), USER_ROOT_DEFAULT));
            }

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        auto photoCreator = [
            &userId,
            &eGenerator,
            &photoManager
        ] (const NUserDocument::EType documentType) -> TString
        {
            auto photoId = eGenerator.CreateUserDocumentPhoto(userId, documentType);
            {
                auto photos = photoManager.GetPhotosForUsers({userId}, documentType);
                UNIT_ASSERT_VALUES_EQUAL(photos.size(), 1);
                for (const auto& [_, photo] : photos) {
                    UNIT_ASSERT_VALUES_EQUAL(photo.GetType(), documentType);
                    UNIT_ASSERT_VALUES_EQUAL(photo.GetVerificationStatus(), NUserDocument::EVerificationStatus::NotYetProcessed);
                }
            }
            return photoId;
        };

        TUserDocumentsChecksManager::TChecksConfiguration checksConfig;
        TMessagesCollector errors;
        UNIT_ASSERT(checksConfig.Init(
            "user_documents_checks.settings",
            "user_documents_checks.generic_checks",
            "user_documents_checks.photo_settings",
            "user_documents_checks.chats_to_move",
            *server,
            errors
        ));

        auto statusChecker = [
            &userId,
            &server,
            &photoManager,
            &documentsChecksManager,
            &checksConfig
        ] (
            const TString& checkType,
            const TString& checkStatus,
            const TString& photoId,
            const NUserDocument::EVerificationStatus expectedPhotoStatus,
            const NUserDocument::EType documentType,
            NDrive::TEntitySession& chatSession,
            NDrive::TEntitySession& tagSession) -> void
        {
            TUserDocumentsCheck check(userId, checkType, checkStatus);
            UNIT_ASSERT(documentsChecksManager->SetStatus(check, userId, {photoId}, checksConfig, *server, chatSession, chatSession));

            auto photos = photoManager.GetUsersWithPhotoIds({photoId}, tagSession);
            UNIT_ASSERT(photos && photos->contains(userId));

            for (const auto& photo : (*photos)[userId]) {
                if (photo.GetType() == documentType) {
                    UNIT_ASSERT_VALUES_EQUAL(photo.GetVerificationStatus(), expectedPhotoStatus);
                    return;
                }
            }
            UNIT_ASSERT_C(false, "Photo is not found");
        };

        {
            auto chatSession = server->GetChatEngine()->BuildTx<NSQL::Writable>();
            // auto tagSession = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
            auto photoType = NUserDocument::EType::LicenseBack;
            auto licenseBackId = photoCreator(photoType);
            statusChecker("license_back_selfie", "OK", licenseBackId, NUserDocument::EVerificationStatus::Ok, photoType, chatSession, chatSession);
            statusChecker("license_back_fill", "BAD", licenseBackId, NUserDocument::EVerificationStatus::NeedInfo, photoType, chatSession, chatSession);
            statusChecker("license_back_fill", "OK", licenseBackId, NUserDocument::EVerificationStatus::Ok, photoType, chatSession, chatSession);
        }

        {
            auto chatSession = server->GetChatEngine()->BuildTx<NSQL::Writable>();
            // auto tagSession = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
            auto photoType = NUserDocument::EType::LicenseFront;
            auto licenseFrontId = photoCreator(photoType);
            statusChecker("license_front_selfie", "AWFUL", licenseFrontId, NUserDocument::EVerificationStatus::Discarded, photoType, chatSession, chatSession);
            statusChecker("license_front_fill", "BAD", licenseFrontId, NUserDocument::EVerificationStatus::Discarded, photoType, chatSession, chatSession);
            statusChecker("license_front_selfie", "OK", licenseFrontId, NUserDocument::EVerificationStatus::NeedInfo, photoType, chatSession, chatSession);
        }
    }
}
