#include <drive/backend/ut/library/helper.h>

#include <drive/backend/data/device_tags.h>
#include <drive/backend/documents_verification/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/yang.h>
#include <drive/backend/yang/client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

#define ASSIGNMENT_TO_BE_RECREATED "f53ce6ca-1871-4ac1-979e-f283089670f4"
#define ASSIGNMENT_TO_BE_PROCESSED "5311acd9-53d8-4c92-a90b-2f202bdc1399"

Y_UNIT_TEST_SUITE(DocumentsVerification) {

    Y_UNIT_TEST(LaunchCreationDaemon) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto initialNumberOfAssignments = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().size();

        auto userId = eGenerator.CreateUser("zxqfd-was-there", false);
        Sleep(TDuration::Seconds(2));
        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto allAssignmentsNew =  docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();
        auto newNumberOfAssignments = allAssignmentsNew.size();

        TString newAssignmentId;
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                newAssignmentId = it.second.GetId();
            }
        }
        UNIT_ASSERT(!!newAssignmentId);
        UNIT_ASSERT_VALUES_EQUAL(newNumberOfAssignments, initialNumberOfAssignments + 1);

        /*
            Wait for 3 more launches, nothing should be added.
        */

        eGenerator.CreateNewYangAssignments(server.Get());
        eGenerator.CreateNewYangAssignments(server.Get());
        eGenerator.CreateNewYangAssignments(server.Get());
        UNIT_ASSERT_VALUES_EQUAL(docPhotoManager.GetDocumentVerificationAssignments().FetchInfo().size(), initialNumberOfAssignments + 1);
    }

    Y_UNIT_TEST(FaceMatchingAssignment) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto userId = eGenerator.CreateUser("zxqfd-was-there", false);
        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        // Create assignment
        TVector<TString> userNewAssignments;
        {
            auto presentAssignments = docPhotoManager.GetDocumentVerificationAssignments().GetNotVerifiedSelfieAssignments();
            TSet<TString> usedAssignmentIds;
            for (auto&& it : presentAssignments) {
                usedAssignmentIds.insert(it.first);
            }
            UNIT_ASSERT(configGenerator.SubmitUserDocumentPhoto(userId, NUserDocument::EType::Selfie, configGenerator.GetValidImageString()));
            UNIT_ASSERT(configGenerator.SubmitUserDocumentPhoto(userId, NUserDocument::EType::Selfie, configGenerator.GetValidImageString()));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            UNIT_ASSERT(eGenerator.CreateNewSelfieYangAssignments(server.Get()));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto newAssignments = docPhotoManager.GetDocumentVerificationAssignments().GetNotVerifiedSelfieAssignments();
            for (auto&& it : newAssignments) {
                if (usedAssignmentIds.contains(it.first)) {
                    continue;
                }
                if (it.second.GetUserId() != userId) {
                    continue;
                }
                userNewAssignments.push_back(it.first);
            }
            UNIT_ASSERT_VALUES_EQUAL(userNewAssignments.size(), 2);
        }

        // Submit assignment results
        TString testComment = "'\"тест ТЕСТ test\"'\\";
        {
            NJson::TJsonValue payload;
            payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

            payload["verification_statuses"]["comment"] = testComment;

            payload["verification_statuses"]["passport_selfie_status"] = "OK";
            payload["verification_statuses"]["selfie_status"] = "OK";
            UNIT_ASSERT(configGenerator.SubmitSelfieVerificationResults(userNewAssignments.front(), "assignment1", payload));
        }

        // Check that verification results are now reflected
        {
            auto fr = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(userNewAssignments.front());
            UNIT_ASSERT_VALUES_EQUAL(fr.size(), 1);
            auto assignment = std::move(fr.begin()->second);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetIsFraud(), TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetComments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetComments().front(), testComment);

            auto photoFR = docPhotoManager.GetUserPhotosDB().FetchInfo(assignment.GetSelfieId());
            UNIT_ASSERT_VALUES_EQUAL(photoFR.size(), 1);
            auto photo = std::move(photoFR.begin()->second);
            UNIT_ASSERT_VALUES_EQUAL(photo.GetVerificationStatus(), NUserDocument::EVerificationStatus::Ok);
        }
    }

    Y_UNIT_TEST(ReYang) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto userId = eGenerator.CreateUser("zxqfd-was-there", false);
        Sleep(TDuration::Seconds(2));
        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();

        TString newAssignmentId;
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                newAssignmentId = it.second.GetId();
            }
        }
        UNIT_ASSERT(!!newAssignmentId);

        {
            auto initialAssignmentData = configGenerator.GetYangAssignmentData(newAssignmentId, "0");
            UNIT_ASSERT(!initialAssignmentData["license_back"]["data"].Has("has_at_mark"));
            INFO_LOG << "proxy data: " << initialAssignmentData.GetStringRobust() << Endl;
            UNIT_ASSERT(initialAssignmentData != NJson::JSON_NULL);
        }

        auto passport = TDocumentsHelper::CreateRandomPassportData();
        auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();

        NJson::TJsonValue payload;
        payload["passport_biographical"] = passport.SerializeBioToYang(false);
        payload["passport_registration"] = passport.SerializeRegToYang(false);
        payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
        payload["license_back"] = drivingLicense.SerializeBackToYang(false);
        payload["license_back"]["data"]["has_at_mark"] = true;
        payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

        TString testComment = "'\"тест ТЕСТ test\"'\\";
        payload["verification_statuses"]["comment"] = testComment;

        payload["verification_statuses"]["passport_biographical_status"] = "OK";
        payload["verification_statuses"]["passport_registration_status"] = "OK";
        payload["verification_statuses"]["license_front_status"] = "UNRECOGNIZABLE";
        payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
        payload["verification_statuses"]["passport_selfie_status"] = "OK";

        UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload));
        Sleep(TDuration::Seconds(2));

        auto licenseFrontId2 = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto selfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::Selfie);

        TString userId2 = eGenerator.CreateUser("skulik-was-there", false);
        auto u2licenseBackId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::LicenseBack);
        auto u2licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::LicenseFront);
        auto u2passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::PassportBiographical);
        auto u2passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::PassportRegistration);
        auto u2passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::PassportSelfie);
        auto u2selfieId = eGenerator.CreateUserDocumentPhoto(userId2, NUserDocument::EType::Selfie);

        TString userId3 = eGenerator.CreateUser("skulik1-was-there", false);
        auto u3licenseBackId = eGenerator.CreateUserDocumentPhoto(userId3, NUserDocument::EType::LicenseBack);
        auto u3licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId3, NUserDocument::EType::LicenseFront);
        auto u3passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId3, NUserDocument::EType::PassportBiographical);
        auto u3passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId3, NUserDocument::EType::PassportRegistration);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TTagDescription description1;
            description1.SetName("nothing_to_do_with_documents").SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description1), USER_ROOT_DEFAULT, session));
            TTagDescription description2;
            description2.SetName("resend_documents_to_yang").SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description2), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            UNIT_ASSERT(configGenerator.AddTag(new TSimpleUserTag("nothing_to_do_with_documents"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(configGenerator.AddTag(new TSimpleUserTag("resend_documents_to_yang"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(configGenerator.AddTag(new TSimpleUserTag("resend_documents_to_yang"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(configGenerator.AddTag(new TSimpleUserTag("resend_documents_to_yang"), userId3, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        auto dvc = eGenerator.BuildDefaultDocumentsVerificationConfig();
        dvc.SetDocumentTypes({
            NUserDocument::EType::LicenseBack,
            NUserDocument::EType::LicenseFront,
            NUserDocument::EType::PassportBiographical,
            NUserDocument::EType::PassportRegistration,
            NUserDocument::EType::PassportSelfie,
            NUserDocument::EType::Selfie
        });
        TDocumentsVerificationManager manager(dvc, docPhotoManager);
        UNIT_ASSERT(manager.ResendToYang(TSet<TString>({ userId, userId2, userId3 }), "resend_documents_to_yang", "unittest", server.Get()));
        Sleep(TDuration::Seconds(2));

        bool isU1Ok = false;
        bool isU2Ok = false;
        bool isU3Ok = true;

        allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId2 &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId &&
                it.second.GetSelfieId() == selfieId
            ) {
                isU1Ok = true;
            }
            if (
                it.second.GetLicenseBackId() == u2licenseBackId &&
                it.second.GetLicenseFrontId() == u2licenseFrontId &&
                it.second.GetPassportBiographicalId() == u2passportBiographicalId &&
                it.second.GetPassportRegistrationId() == u2passportRegistrationId &&
                it.second.GetPassportSelfieId() == u2passportSelfieId &&
                it.second.GetSelfieId() == u2selfieId
            ) {
                isU2Ok = true;
            }
            if (it.second.GetLicenseBackId() == u3licenseBackId) {
                isU3Ok = false;
            }
        }
        INFO_LOG << "per user results:" << isU1Ok << " " << isU2Ok << " " << isU3Ok << Endl;
        UNIT_ASSERT(isU1Ok);
        UNIT_ASSERT(isU2Ok);
        UNIT_ASSERT(isU3Ok);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreTags({ userId, userId2, userId3 }, {"nothing_to_do_with_documents"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag.GetObjectId(), userId);

            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreTags({ userId, userId2, userId3 }, {"resend_documents_to_yang"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag.GetObjectId(), userId3);

        }
    }

    Y_UNIT_TEST(YangProxyArraySerialization) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto userId = eGenerator.CreateUser("zxqfd-was-there", false);
        Sleep(TDuration::Seconds(2));
        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();

        TString newAssignmentId;
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                newAssignmentId = it.second.GetId();
            }
        }
        UNIT_ASSERT(!!newAssignmentId);

        {
            auto initialAssignmentData = configGenerator.GetYangAssignmentData(newAssignmentId, "0");
            UNIT_ASSERT(!initialAssignmentData["license_back"]["data"].Has("has_at_mark"));
            INFO_LOG << "proxy data: " << initialAssignmentData.GetStringRobust() << Endl;
            UNIT_ASSERT(initialAssignmentData != NJson::JSON_NULL);
        }

        auto passport = TDocumentsHelper::CreateRandomPassportData();
        auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();

        NJson::TJsonValue payload;
        payload["passport_biographical"] = passport.SerializeBioToYang(false);
        payload["passport_registration"] = passport.SerializeRegToYang(false);
        payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
        payload["license_back"] = drivingLicense.SerializeBackToYang(false);
        payload["license_back"]["data"]["has_at_mark"] = true;
        payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

        TString testComment = "111";
        payload["verification_statuses"]["comment"] = testComment;

        payload["verification_statuses"]["passport_biographical_status"] = "OK";
        payload["verification_statuses"]["passport_registration_status"] = "OK";
        payload["verification_statuses"]["license_front_status"] = "NEED_INFO";
        payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
        payload["verification_statuses"]["passport_selfie_status"] = "OK";

        UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload, true));

        {
            auto ya = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).MutableResult().begin()->second;
            ya.SetComments({"ошибка видео паспорт лицевая, прописка и селфи"});
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(docPhotoManager.GetDocumentVerificationAssignments().Upsert(ya, session) && session.Commit());
        }

        {
            auto ya = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).MutableResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(ya.GetComments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(ya.GetComments().front(), TString("ошибка видео паспорт лицевая, прописка и селфи"));
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(docPhotoManager.GetDocumentVerificationAssignments().Upsert(ya, session) && session.Commit());
        }
    }

    Y_UNIT_TEST(YangProxy) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        auto userId = eGenerator.CreateUser("zxqfd-was-there", false);
        Sleep(TDuration::Seconds(2));
        auto licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        auto licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        auto passportBiographicalId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportBiographical);
        auto passportRegistrationId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportRegistration);
        auto passportSelfieId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::PassportSelfie);

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            auto user = driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin()->second;
            user.SetPassportDatasyncRevision("aa");
            user.SetDrivingLicenseDatasyncRevision("aa");
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(user, "robot-frontend", session) && session.Commit());
            driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin();
        }

        auto allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();

        TString newAssignmentId;
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                newAssignmentId = it.second.GetId();
            }
        }
        UNIT_ASSERT(!!newAssignmentId);

        {
            auto initialAssignmentData = configGenerator.GetYangAssignmentData(newAssignmentId, "0");
            UNIT_ASSERT(!initialAssignmentData["license_back"]["data"].Has("has_at_mark"));
            INFO_LOG << "proxy data: " << initialAssignmentData.GetStringRobust() << Endl;
            UNIT_ASSERT(initialAssignmentData != NJson::JSON_NULL);
        }

        auto passport = TDocumentsHelper::CreateRandomPassportData();
        auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();

        NJson::TJsonValue payload;
        payload["passport_biographical"] = passport.SerializeBioToYang(false);
        payload["passport_registration"] = passport.SerializeRegToYang(false);
        payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
        payload["license_back"] = drivingLicense.SerializeBackToYang(false);
        payload["license_back"]["data"]["has_at_mark"] = true;
        payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

        TString testComment = "'\"тест ТЕСТ test\"'\\";
        payload["verification_statuses"]["comment"] = testComment;

        NJson::TJsonValue fraudReasons = NJson::JSON_ARRAY;
        fraudReasons.AppendValue("fr1");
        fraudReasons.AppendValue("fr2");
        payload["verification_statuses"]["fraud_reasons"] = std::move(fraudReasons);
        payload["passport_biographical"]["data"]["number"] = "HB2352296";

        payload["verification_statuses"]["passport_biographical_status"] = "OK";
        payload["verification_statuses"]["passport_registration_status"] = "OK";
        payload["verification_statuses"]["license_front_status"] = "NEED_INFO";
        payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
        payload["verification_statuses"]["passport_selfie_status"] = "OK";

        UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload));

        {
            auto assignmentFR = driveApi.GetDocumentPhotosManager().GetDocumentVerificationAssignments().FetchInfo(newAssignmentId);
            auto assignmentPtr = assignmentFR.GetResultPtr(newAssignmentId);
            UNIT_ASSERT_VALUES_EQUAL(assignmentPtr->GetComments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignmentPtr->GetComments().front(), testComment);

            UNIT_ASSERT_VALUES_EQUAL(assignmentPtr->GetFraudReasons().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(assignmentPtr->GetFraudReasons()[0], "fr1");
            UNIT_ASSERT_VALUES_EQUAL(assignmentPtr->GetFraudReasons()[1], "fr2");
        }

        {
            auto result = configGenerator.GetHistoricalFillingData(newAssignmentId, "1", "0");
            INFO_LOG << result << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result["passport_biographical"]["data"]["number"], "HB2352296");
        }

        {
            auto userFR = driveApi.GetUsersData()->FetchInfo(userId);
            auto userPtr = userFR.GetResultPtr(userId);
            UNIT_ASSERT(userPtr->GetHasATMark());
        }

        licenseBackId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseBack);
        licenseFrontId = eGenerator.CreateUserDocumentPhoto(userId, NUserDocument::EType::LicenseFront);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        newAssignmentId = "";
        allAssignmentsNew = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();
        for (auto&& it : allAssignmentsNew) {
            if (
                it.second.GetLicenseBackId() == licenseBackId &&
                it.second.GetLicenseFrontId() == licenseFrontId &&
                it.second.GetPassportBiographicalId() == passportBiographicalId &&
                it.second.GetPassportRegistrationId() == passportRegistrationId &&
                it.second.GetPassportSelfieId() == passportSelfieId
            ) {
                newAssignmentId = it.second.GetId();
            }
        }

        UNIT_ASSERT(!!newAssignmentId);

        {
            auto initialAssignmentData = configGenerator.GetYangAssignmentData(newAssignmentId, "1");
            UNIT_ASSERT(initialAssignmentData["passport_selfie"]["is_verified"].GetBoolean());
            UNIT_ASSERT(initialAssignmentData["passport_biographical"]["is_verified"].GetBoolean());
            UNIT_ASSERT(initialAssignmentData["passport_registration"]["is_verified"].GetBoolean());
            UNIT_ASSERT(!initialAssignmentData["license_front"]["is_verified"].GetBoolean());
            UNIT_ASSERT(!initialAssignmentData["license_back"]["is_verified"].GetBoolean());
            INFO_LOG << "proxy data 2: " << initialAssignmentData.GetStringRobust() << Endl;
            UNIT_ASSERT_C(initialAssignmentData["license_back"]["data"]["has_at_mark"].GetBoolean(), TStringBuilder() << initialAssignmentData["license_back"]["data"]["has_at_mark"].GetStringRobust());
            UNIT_ASSERT(initialAssignmentData != NJson::JSON_NULL);
        }
    }

    Y_UNIT_TEST(YangProxyDatasyncFallback) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetPrivateDataClientType("fake");
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        auto client = dynamic_cast<const TFakePrivateDataStorage*>(&driveApi.GetPrivateDataClient());
        UNIT_ASSERT(!!client);
        client->SetIsFaulty(true);

        TString newAssignmentId = "";
        {

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            auto passport = TDocumentsHelper::CreateRandomPassportData();
            auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();
            NJson::TJsonValue payload;
            payload["passport_biographical"] = passport.SerializeBioToYang(false);
            payload["passport_registration"] = passport.SerializeRegToYang(false);
            payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
            payload["license_back"] = drivingLicense.SerializeBackToYang(false);
            payload["license_back"]["data"]["has_at_mark"] = true;
            payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

            TString testComment = "'\"тест ТЕСТ test\"'\\";
            payload["verification_statuses"]["comment"] = testComment;

            payload["verification_statuses"]["passport_biographical_status"] = "OK";
            payload["verification_statuses"]["passport_registration_status"] = "OK";
            payload["verification_statuses"]["license_front_status"] = "NEED_INFO";
            payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
            payload["verification_statuses"]["passport_selfie_status"] = "OK";

            UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 31);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["messages"].GetArray().size());
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 2);
            TString newAssignmentId2 = newAssignmentIds[0] != newAssignmentId ? newAssignmentIds[0] : newAssignmentIds[1];
            UNIT_ASSERT(!!newAssignmentId2);
            UNIT_ASSERT(newAssignmentId2 != newAssignmentId);

            // Although reask worked, the data was not saved because storage is faulty
            auto assignmentData = configGenerator.GetYangAssignmentData(newAssignmentId2, "1");
            UNIT_ASSERT_C(!assignmentData["license_front"]["data"]["first_name"].IsDefined(), TStringBuilder() << assignmentData.GetStringRobust());

            auto payload = TDocumentsHelper::ConstructConsistentYangPayload();

            UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId2, "1", payload));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "enter_map");

            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT(user.GetHasATMark());
            UNIT_ASSERT(!user.IsMTAllowed());
            UNIT_ASSERT_VALUES_EQUAL(user.GetPassportNumberHash(), "f3c266214dfa332bd6386c215c34c3043bb03291");
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
            UNIT_ASSERT(user.GetFirstName() != "");
            UNIT_ASSERT(user.GetLastName() != "");
            UNIT_ASSERT(user.GetPName() != "");
        }
    }

    Y_UNIT_TEST(FraudReasons) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetPrivateDataClientType("fake");
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        auto client = dynamic_cast<const TFakePrivateDataStorage*>(&driveApi.GetPrivateDataClient());
        UNIT_ASSERT(!!client);
        client->SetIsFaulty(true);

        TString newAssignmentId = "";
        {

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            auto passport = TDocumentsHelper::CreateRandomPassportData();
            auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();
            NJson::TJsonValue payload;
            payload["passport_biographical"] = passport.SerializeBioToYang(false);
            payload["passport_registration"] = passport.SerializeRegToYang(false);
            payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
            payload["license_back"] = drivingLicense.SerializeBackToYang(false);
            payload["license_back"]["data"]["has_at_mark"] = true;
            payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

            TString testComment = "'\"тест ТЕСТ test\"'\\";
            payload["verification_statuses"]["comment"] = testComment;

            NJson::TJsonValue fraudReasons = NJson::JSON_ARRAY;
            fraudReasons.AppendValue("PHOTO_DIFF_PERSON");
            fraudReasons.AppendValue("REG_SIGNATURE_MISSING");
            payload["verification_statuses"]["fraud_reasons"] = std::move(fraudReasons);

            payload["verification_statuses"]["passport_biographical_status"] = "OK";
            payload["verification_statuses"]["passport_registration_status"] = "OK";
            payload["verification_statuses"]["license_front_status"] = "NEED_INFO";
            payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
            payload["verification_statuses"]["passport_selfie_status"] = "OK";

            UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[messagesJson["messages"].GetArray().size() - 3]["text"].GetString(), "Не хватает подписи в штапме с адресом прописки. Поставьте её, и сделайте новую фотографию.");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 32);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["messages"].GetArray().size());
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 2);
            TString newAssignmentId2;
            UNIT_ASSERT(!!newAssignmentId);

            for (auto&& id : newAssignmentIds) {
                if (id != newAssignmentId) {
                    newAssignmentId2 = id;
                    break;
                }
            }

            UNIT_ASSERT(!!newAssignmentId2);

            // Although reask worked, the data was not saved because storage is faulty
            auto assignmentData = configGenerator.GetYangAssignmentData(newAssignmentId2, "1");
            UNIT_ASSERT_C(!assignmentData["license_front"]["data"]["first_name"].IsDefined(), TStringBuilder() << assignmentData.GetStringRobust());

            auto passport = TDocumentsHelper::CreateRandomPassportData();
            auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();

            NJson::TJsonValue payload;
            payload["passport_biographical"] = passport.SerializeBioToYang(false);
            payload["passport_registration"] = passport.SerializeRegToYang(false);
            payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
            payload["license_back"] = drivingLicense.SerializeBackToYang(false);

            payload["license_front"]["data"]["first_name"] = passport.GetFirstName();
            payload["license_front"]["data"]["last_name"] = passport.GetLastName();
            payload["license_front"]["data"]["birth_date"] = passport.GetBirthDate();

            payload["license_back"]["data"]["categories_b_valid_to_date"] = TInstant::Seconds(Max<i32>()).ToString();
            payload["passport_biographical"]["data"]["number"] = "HB2352296";

            payload["license_back"]["data"]["has_at_mark"] = true;
            payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

            TString testComment = "'\"тест ТЕСТ test\"'\\";
            payload["verification_statuses"]["comment"] = testComment;

            payload["verification_statuses"]["passport_biographical_status"] = "OK";
            payload["verification_statuses"]["passport_registration_status"] = "OK";
            payload["verification_statuses"]["license_front_status"] = "OK";
            payload["verification_statuses"]["license_back_status"] = "OK";
            payload["verification_statuses"]["passport_selfie_status"] = "OK";

            UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId2, "1", payload));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "enter_map");

            auto user = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second;
            UNIT_ASSERT(user.GetHasATMark());
            UNIT_ASSERT(!user.IsMTAllowed());
            UNIT_ASSERT_VALUES_EQUAL(user.GetPassportNumberHash(), "f3c266214dfa332bd6386c215c34c3043bb03291");
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
            UNIT_ASSERT(user.GetFirstName() != "");
            UNIT_ASSERT(user.GetLastName() != "");
            UNIT_ASSERT(user.GetPName() != "");
        }
    }
}
