#include <drive/backend/ut/library/script.h>

#include <library/cpp/testing/unittest/registar.h>


namespace NDrive::NTest {
    class TCheckMultipleSessions: public TCheckCurrentSessionWaitingImpl {
        R_FIELD(ui32, Count, 1);
        R_FIELD(TVector<TString>, ChagredCars);
    private:
        using TBase = TCheckCurrentSessionWaitingImpl;
    protected:
        TMaybe<TString> ExptractSessoinId(ui32 index, const NJson::TJsonValue& sessionsReport) const {
            if (!sessionsReport.IsArray() || sessionsReport.GetArray().size() <= index) {
                return Nothing();
            }
            const auto& session = sessionsReport.GetArray()[index];
            return session["segment"]["meta"]["session_id"].GetString();
        }

        TMaybe<TString> ExtractObjectId(ui32 index, const NJson::TJsonValue& sessionsReport) const {
            if (!sessionsReport.IsArray() || sessionsReport.GetArray().size() <= index) {
                return Nothing();
            }
            const auto& session = sessionsReport.GetArray()[index];
            return session["segment"]["meta"]["object_id"].GetString();
        }

        TMaybe<ui32> ExtractTotalSum(ui32 index, const NJson::TJsonValue& sessionsReport) const {
            if (!sessionsReport.IsArray() || sessionsReport.GetArray().size() <= index) {
                return Nothing();
            }
            const auto& session = sessionsReport.GetArray()[index];
            ui32 totalSum = session["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe();
            return totalSum;
        }

        bool Check(const TRTContext& context, const NJson::TJsonValue& json) const override {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (!context.GetMultiSession() || !json.Has("sessions")) {
                return false;
            }
            ui32 sessionsCount = json["sessions"].GetArray().size();
            UNIT_ASSERT_VALUES_EQUAL(sessionsCount, Count);
            for (auto&& carName : ChagredCars) {
                auto car = context.GetNamedCar(carName);
                UNIT_ASSERT(car);
                for (ui32 i = 0; i < Count; ++i) {
                    auto carId = ExtractObjectId(i, json["sessions"]);
                    UNIT_ASSERT(carId.Defined());
                    if (*carId == car->Id) {
                        auto totalPrice = ExtractTotalSum(i, json["sessions"]);
                        UNIT_ASSERT(totalPrice.Defined());
                        UNIT_ASSERT(*totalPrice > 0);
                    }
                }
            }
            return true;
        }

    public:
        TCheckMultipleSessions(const TInstant startInstant)
            : TBase(startInstant) {
        }
    };
}

Y_UNIT_TEST_SUITE(DoubleRent) {
    Y_UNIT_TEST(DoubleRentPricing) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TSetMultiSession>(true);
        script.Add<NDrive::NTest::TSetSetting>().Set("handlers.api/yandex/offers/book.multi_rent", "true");
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from).SetNamedCar("first");
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from).SetNamedCar("second");
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetNamedCar("first");
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero()).SetNamedCar("first");
        script.Add<NDrive::NTest::TCheckMultipleSessions>().SetCount(1);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetNamedCar("second");
        script.Add<NDrive::NTest::TRide>(TDuration::Minutes(1)).SetNamedCar("first");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCheckMultipleSessions>().SetCount(2).MutableChagredCars().push_back("first");
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(30)).SetCarPosition(to).SetNamedCar("first");
        script.Add<NDrive::NTest::TCheckMultipleSessions>().SetCount(1).MutableChagredCars().push_back("second");
        UNIT_ASSERT(script.Execute());
    }
}
